
const AWS = require("aws-sdk");
const utils = require('./utility');
const SNS = new AWS.SNS();
let returnError = function(callback, errorMessage) {
    let response = {
        statusCode: 400,
        headers: {
            "Content-Type": "application/json",
            "Access-Control-Allow-Origin" : "*",
            "Access-Control-Allow-Credentials" : true
        },
        body: JSON.stringify({"error": errorMessage})
    };
    console.log("response: " + JSON.stringify(response));
    callback(null, response);
}


// function to create deployment group entry in asset service
async function addEntryToDeploymentGroup(buildNumber, deploymentType) {
  console.log("Enter saving group");
    try {
      const path = "restaurant_assets/deployment_groups";
      const bodyJSON = {};
      const dataJSON = {};
      dataJSON.type = "deployment_groups";
      const attributeJSON = {};
      attributeJSON.deployedVersion = buildNumber;
      attributeJSON.type = deploymentType;
      dataJSON.attributes = attributeJSON;
      bodyJSON.data = dataJSON;
      console.log(bodyJSON);
      const resultJSON = await utils.assetServiceRequest('POST',path,bodyJSON);
      console.log(resultJSON);
      const id = resultJSON.data.id;
      if (id === null || id === undefined) {
        throw new Error(`Unable to create deployment group`);
      }
      return id;
    }
    catch (err) {
      console.log(err.message);
      throw err;
    }
}

  // function to create deployment history entry in asset service
async function addEntryToDeploymentHistory(groupId, restaurantId, componentId) {
    try {
      const path = "restaurant_assets/deployment_history";
      const bodyJSON = {};
      const dataJSON = {};
      const relationShipJSON = {};
      dataJSON.type = "deployment_history";
      const attributeJSON = {};
      attributeJSON.status = "In-Progress";
      dataJSON.attributes = attributeJSON;

      //add component relationship
      const componentJSON = {};
      const componentDataJSON = {};
      componentDataJSON.id = componentId;
      componentDataJSON.type = "components";
      componentJSON.data = componentDataJSON;

      //add component relationship
      const groupJSON = {};
      const groupDataJSON = {};
      groupDataJSON.id = groupId;
      groupDataJSON.type = "deployment_groups";
      groupJSON.data = groupDataJSON;

      //add component relationship
      const restaurantJSON = {};
      const restaurantDataJSON = {};
      restaurantDataJSON.id = restaurantId;
      restaurantDataJSON.type = "restaurants";
      restaurantJSON.data = restaurantDataJSON;

      // add all relations in relationshipJSON
      relationShipJSON.component = componentJSON;
      relationShipJSON.deployment_group = groupJSON;
      relationShipJSON.restaurant = restaurantJSON;
      dataJSON.relationships = relationShipJSON;
      bodyJSON.data = dataJSON;
      const resultJSON = await utils.assetServiceRequest('POST',path,bodyJSON);
      console.log(resultJSON);
      const id = resultJSON.data.id;
      if (id === null || id === undefined) {
        throw new Error(`Unable to add history for component ${componentId}`);
      }
      return id;
    }
    catch (err) {
      console.log(err.message);
      throw err;
    }
}

// function to get component id from component name and restaurant id
async function getComponentId(restaurantId, podName) {
    try {
      let path = "restaurant_assets/components";
      //append query params
      path = path + "?filter%5Brestaurant.id%5D=" + restaurantId;
      path = path + "&filter%5Bname%5D=" + podName;
      path = path + "&fields%5Bcomponents%5D=id";
      const resultJSON = await utils.assetServiceRequest('GET',path,null);
      console.log(resultJSON);
      const id = resultJSON.data[0].id;
      if (id === null || id === undefined) {
        throw new Error(`Unable to fetch component id for restaurant ${restaurantId}`);
      }
      return id;
    }
    catch (err) {
      console.log(err.message);
      throw err;
    }
}

// fucntion to delete deployment group
async function deleteDeploymentGroup(groupId){
      try {
        let path = `/restaurant_assets/deployment_groups/${groupId}`;
        const resultJSON = await utils.assetServiceRequest('DELETE',path,null);
        console.log(resultJSON);
      }
      catch (err) {
        console.log(err.message);
        throw err;
      }
}


// function to update history for each component
async function updateComponentHistory(restid, deploymentGroupId, podName) {
    let updateFlag = true;
    try {
      const componentId = await getComponentId(restid, podName);
      await addEntryToDeploymentHistory(deploymentGroupId, restid, componentId);
    }
    catch (err) {

      updateFlag=false;
    }
    return updateFlag;
}

  // function to store deployment history in asset service
async function storeDeploymentHistory(podName, buildNumber, restaurantIdArray, deploymentType) {
  console.log("Inside deployment function");
  console.log('restid',JSON.stringify(restaurantIdArray));
    let deploymentGroupId;
    try{
        //create deployment group 
        deploymentGroupId = await addEntryToDeploymentGroup(buildNumber, deploymentType);
        console.log('deploymentGroupId generated: ', deploymentGroupId);
          //fetch component id for each restaurant and store the deployment history
          for (let i in restaurantIdArray) {
            const restid = restaurantIdArray[i];
            const isHistoryUpdated = await updateComponentHistory(restid,deploymentGroupId,podName);
            if(!isHistoryUpdated){
              //delete deployment group and exit
              await deleteDeploymentGroup(deploymentGroupId);
              deploymentGroupId=undefined;
              break;
            }
          }
      }
    catch(err){
      console.error(err.message);
      deploymentGroupId=undefined;
    }
    return deploymentGroupId;
}

exports.handler =  async (event, context, callback) => {
    console.log("ENVIRONMENT VARIABLES: " + JSON.stringify(process.env, null, 2));
    console.info("EVENT: " + JSON.stringify(event, null, 2));

    let body = event.body ? JSON.parse(event.body) : event;
    let topicArn = process.env.TOPIC_ARN;

    let podName = body.podName;
    let deploymentType = body.deploymentType;
    let imageURL = body.imageURL;
    let replaceVersion = body.replaceVersion;
    let buildNumber = body.buildNumber;
    let restaurantId = body.restaurantId;
    let hardwareVersion = body.hardwareVersion;
    let deviceType = body.deviceType;
    let vendor = body.vendor;
    let msg='';
    let response = null;
    if ( podName ) {
        console.log(`Processing podName = ${podName}`)
    } else {
        msg = `podName does not exist in the event payload`
        console.error(msg)
        returnError(callback, msg)
    }

   // let sns = new AWS2.SNS();
    
    let messageAttributes;

    if(event.deploymentType == 'firmware') {
        if ( podName && deploymentType && imageURL && replaceVersion && hardwareVersion && deviceType && restaurantId && buildNumber && vendor ) {
            console.log(`All required parameters contain necessary values. Proceeding to deploying message to SNS`)
            messageAttributes = {
                "componentName": {
                    "DataType": "String",
                    "StringValue": podName
                },
                "podName": {
                    "DataType": "String",
                    "StringValue": podName
                },
                "deploymentType": {
                    "DataType": "String",
                    "StringValue": deploymentType
                },
                "imageURL": {
                    "DataType": "String",
                    "StringValue": imageURL
                },
                "replaceVersion": {
                    "DataType": "String",
                    "StringValue": replaceVersion
                },
                "buildNumber": {
                    "DataType": "String",
                    "StringValue": buildNumber
                },
                "hardwareVersion": {
                    "DataType": "String",
                    "StringValue": hardwareVersion
                },
                "deviceType": {
                    "DataType": "String",
                    "StringValue": deviceType
                },
                "vendor": {
                    "DataType": "String",
                    "StringValue": vendor
                },
                "restaurantId": {
                    "DataType": "String.Array",
                    "StringValue": JSON.stringify(restaurantId)
                }
            };
        }
        else {
            msg = `One or more inputs for deploymentType of Firmware are invalid or are not defined`
            console.error(msg)
            returnError(callback, msg)
        }
    } else {
        if ( podName && deploymentType && imageURL && replaceVersion && buildNumber && restaurantId ) {
            console.log(`All required parameters contain necessary values. Proceeding to deploying message to SNS`)
            messageAttributes = {
                "componentName": {
                    "DataType": "String",
                    "StringValue": podName
                },
                "podName": {
                    "DataType": "String",
                    "StringValue": podName
                },
                "deploymentType": {
                    "DataType": "String",
                    "StringValue": deploymentType
                },
                "imageURL": {
                    "DataType": "String",
                    "StringValue": imageURL
                },
                "replaceVersion": {
                    "DataType": "String",
                    "StringValue": replaceVersion
                },
                "buildNumber": {
                    "DataType": "String",
                    "StringValue": buildNumber
                },
                "restaurantId": {
                    "DataType": "String.Array",
                    "StringValue": JSON.stringify(restaurantId)
                }
            };
        }
        else {
            msg = `One or more inputs for deploymentType of Docker are invalid or are not defined`
            console.error(msg)
            returnError(callback, msg)
        }
    }

    //store deployment history details
    const resultGroupID = await storeDeploymentHistory(podName, buildNumber, restaurantId, deploymentType);
    if(resultGroupID===undefined || resultGroupID===null){
        msg = `Error while saving deployment details`;
        console.error("Error while saving deployment details");
        console.error(msg)
        returnError(callback, msg)
    }
    else{
     console.info("Sending message to SNS");
      let subject = `deployment-${deploymentType}-${podName}-${buildNumber}`;
      //add deployment group in message Attributes
      const deployJSON={};
      deployJSON.DataType="String";
      deployJSON.StringValue=resultGroupID;
      messageAttributes.deploymentGroupId=deployJSON;

      let params = {
          Message: "Deployment was triggered through the deployment message service",
          Subject: subject,
          TopicArn: topicArn,
          MessageAttributes: messageAttributes
      };
      console.info(`SNS Message : ${messageAttributes}`);
      try{
      let  publishTextPromise = await SNS.publish(params).promise();
      response = {
        "statusCode": 200,
        "headers": {
            "Content-Type": "application/json",
            "Access-Control-Allow-Origin" : "*",
            "Access-Control-Allow-Credentials": true
        },
        body: JSON.stringify({"msg": `published: ${subject}`})
    };
        console.log(publishTextPromise);
        return response;
      }
      catch (e){
        console.log(e);
        msg = `Failed to send message ${subject}`;
        returnError(callback, msg);
        
      }
    }

};