resource "aws_security_group" "lambda_sg" {
  name_prefix = "${var.tags["Name"]}-LAMBDA-SG"
  vpc_id      = var.vpc_id
  description = "Security group used for the ${var.tags["Name"]}(Managed by Terraform)"

  egress {
    from_port   = 0
    to_port     = 0
    protocol    = "-1"
    cidr_blocks = ["0.0.0.0/0"]
  }

  tags = var.tags

  timeouts {
    delete = "40m"
  }
}

resource "aws_sqs_queue" "gitops_deadletter_queue" {
  name                        = var.gitops_deadletter_queue
  fifo_queue                  = true
  delay_seconds               = 90
  max_message_size            = 262144
  message_retention_seconds   = 86400
  receive_wait_time_seconds   = 10
  visibility_timeout_seconds  = 300
  content_based_deduplication = true
  tags                        = var.tags
}

resource "aws_sqs_queue" "gitops_queue" {
  name = var.gitops_queue
  depends_on = [
    aws_sqs_queue.gitops_deadletter_queue,
  ]
  fifo_queue                  = true
  delay_seconds               = 90
  max_message_size            = 262144
  message_retention_seconds   = 86400
  receive_wait_time_seconds   = 10
  visibility_timeout_seconds  = 300
  content_based_deduplication = true
  redrive_policy = jsonencode({
    deadLetterTargetArn = aws_sqs_queue.gitops_deadletter_queue.arn
    maxReceiveCount     = 2
  })
  tags = var.tags
}

locals {
  isStage = var.tags["Name"] == "US-EAST-PROD-BRE-ARCH-GITOPS-PATCHDEPLOY-STG"
}
resource "aws_lambda_function" "gitops_lambda" {

  function_name    = "${var.tags["Name"]}-LAMBDA"
  role             = local.isStage ? "arn:aws:iam::524430043955:role/GITOPS-LAMBDA-IAM-ROLE" : aws_iam_role.lambda_role[0].arn
  handler          = "handler.handler"
  runtime          = "nodejs18.x"
  timeout          = 300
  memory_size      = 256
  filename         = data.archive_file.lambda_zip.output_path
  source_code_hash = data.archive_file.lambda_zip.output_base64sha256
  layers           = [var.lambdagitlayer]
  tags             = var.tags
  vpc_config {
    subnet_ids         = "${data.aws_subnets.private.ids}"
    security_group_ids = ["${aws_security_group.lambda_sg.id}"]
  }
  environment {
    variables = {
      ASSET_URL              = var.asset_url
      REGION                 = var.region
      ENVIRONMENT            = var.environment
      bitbucketSecret        = var.gitopsRepoAccessSecret
      branch                 = var.branch
      artifactoryURL         = var.artifactoryURL
      EVENT_TARGET_ARN       = "arn:aws:lambda:${var.region}:${data.aws_caller_identity.current.account_id}:function:SCHEDULE-DEVICE-MESSAGE-LAMBDA"
      eventBridgeTime        = var.eventBridgeTime
      gitopsSQSQueueURL      = var.gitopsSQSQueueURL
      gitopsMessageGroup     = var.gitopsMessageGroup
    }
  }
}

resource "aws_iam_role" "lambda_role" {
  count              = local.isStage ? 0 : 1
  name               = "GITOPS-LAMBDA-IAM-ROLE"
  assume_role_policy = data.aws_iam_policy_document.gitops_repo_access.json
  inline_policy {
    name   = "GITOPS-secretmanageraccess"
    policy = data.aws_iam_policy_document.inline_policy.json
  }
}

resource "aws_iam_role_policy_attachment" "AWSLambdaVPCAccessExecutionRole" {
  count      = local.isStage ? 0 : 1
  role       = aws_iam_role.lambda_role[0].name
  policy_arn = "arn:aws:iam::aws:policy/service-role/AWSLambdaVPCAccessExecutionRole"
}

resource "aws_iam_role_policy_attachment" "AWSLambdaBasicExecutionRole" {
  count      = local.isStage ? 0 : 1
  role       = aws_iam_role.lambda_role[0].name
  policy_arn = "arn:aws:iam::aws:policy/service-role/AWSLambdaBasicExecutionRole"
}

resource "aws_iam_role_policy_attachment" "sqs_policy_arn" {
  count      = local.isStage ? 0 : 1
  role       = aws_iam_role.lambda_role[0].name
  policy_arn = var.sqs_policy_arn
}

resource "aws_lambda_event_source_mapping" "event_source_mapping" {
  event_source_arn = aws_sqs_queue.gitops_queue.arn
  enabled          = true
  function_name    = aws_lambda_function.gitops_lambda.arn
  batch_size       = 1
}

resource "aws_iam_role_policy_attachment" "invoke_services_apis_policy_arn" {
  count      = local.isStage ? 0 : 1
  role       = aws_iam_role.lambda_role[0].name
  policy_arn = var.invoke_services_apis_policy_arn
}

resource "aws_iam_role_policy_attachment" "AmazonAPIGatewayInvokeFullAccess" {
  role       = aws_iam_role.lambda_role[0].name
  policy_arn = "arn:aws:iam::aws:policy/AmazonAPIGatewayInvokeFullAccess"
}

resource "aws_iam_role_policy_attachment" "event_bridge_policy_arn" {
  count      = local.isStage ? 0 : 1
  role       = aws_iam_role.lambda_role[0].name
  policy_arn = "arn:aws:iam::${data.aws_caller_identity.current.account_id}:policy/MULTI-DEPLOY-DEVICE-MESSAGE-LAMBDA_SNS_PUBLISH_POLICY"
}
