const fs = require("fs");
const yaml = require("js-yaml");
const log = require("lambda-log");
const AWS = require("aws-sdk");
const eventBridge = new AWS.EventBridge();

/**
 * Replaces all occourances of string
 * @returns {String} URL friendly string
 * @param {String} str input string
 * @param {String} find to be replaced
 * @param {String} replace replacement string
 */
function replaceAll(str, find, replace) {
  while (str.indexOf(find) !== -1) {
    str = str.replace(find, replace);
  }
  return str;
}

/**
 * Replaces special characters with URL Encoded characters
 *
 * @returns {String} URL friendly string
 * @param {String} rawString containing special characters
 */
const cleanPassword = (rawString) => {
  let str = rawString;
  const specialCharMap = {
    "!": "%21",
    "#": "%23",
    $: "%24",
    "&": "%26",
    "'": "%27",
    "(": "%28",
    ")": "%29",
    "*": "%2A",
    "+": "%2B",
    ",": "%2C",
    "/": "%2F",
    ":": "%3A",
    ";": "%3B",
    "=": "%3D",
    "?": "%3F",
    "@": "%40",
    "[": "%5B",
    "]": "%5D",
  };
  Object.keys(specialCharMap).forEach((char) => {
    str = replaceAll(str, char, specialCharMap[char]);
  });
  log.info("Successfully URL encoded string in cleanPassword()");
  return str;
};

/**
 * Set a deeply nested property in an object
 *
 * @returns {Any} returns a value found at path passed. If not found returns {}
 * @param {Array} propertyList value path , ex: a.b.c to be passed as ['a','b','c']
 * @param {Object} object Input object from which property is to be fetched
 */

const get = (propertyList, object) =>
  propertyList.reduce(
    (innerObj, property) =>
      innerObj && innerObj[property] ? innerObj[property] : {},
    object
  );

/**
 * Creates kustomize patch files and adds them to kustomize.yaml for the set of applications and configurations passed to it
 *
 * @returns {Array} returns a list of patches created
 * @param {Array} applications list of applications
 * @param {Array} products list of products
 * @param {String} repoName Name of the repository on which kustomizations exist
 * @param {Array} restaurantsToUpdate list of resturants
 * @param {String} deploymentId deployment id
 */

const patchClusters = async function (
  applications,
  products,
  repoName,
  restaurantsToUpdate,
  deploymentId
) {
  try {
    log.info("Entered :: patchClusters()");
    const appliedPatches = [];
    const repoAbsPath = `/tmp/${repoName}`;
    const clustersPath = `${repoAbsPath}/clusters`;
    const baseUrl = `mcd.jfrog.io/brep-docker/`;
    restaurantsToUpdate.forEach((restaurantId) => {
      if (applications) {
        const releaseTemplatePath = `${clustersPath}/${restaurantId}/releasespec/releasespec.yaml`;
        const kustomizationTemplatePath = `${clustersPath}/${restaurantId}/releasespec/kustomization.yaml`;
        
        const kustomizationTemplateJSON = yaml.load(
          fs.readFileSync(kustomizationTemplatePath, "utf8")
        );
        
        let kustomizationJSON = Object.assign({}, kustomizationTemplateJSON);
        kustomizationJSON["resources"].unshift('releasespec.yaml');
        
        let releaseTemplateJSON;
        if (!fs.existsSync(releaseTemplatePath)) {
          releaseTemplateJSON = yaml.load(
            fs.readFileSync('release-spec-template.yaml', "utf8")
          );
          const kustomizationJSONFileYaml = yaml.dump(kustomizationJSON);
          fs.writeFileSync(kustomizationTemplatePath, kustomizationJSONFileYaml);
          log.info(`Added release spec to kustomization file`);
        } else {
          releaseTemplateJSON = yaml.load(
            fs.readFileSync(releaseTemplatePath, "utf8")
          );
        }
        let releaseSpecApplicationsList = [];          
        let releaseJSON = Object.assign({}, releaseTemplateJSON);

        if (Object.keys(applications).length === 1 && Object.keys(applications)[0].toLowerCase() === "imagecache") {
            const appName = Object.keys(applications)[0].trim();
            const baseLineTemplatePath = `${clustersPath}/${restaurantId}/kustomizations/${appName}/gitrepository.yaml`;
            if (!fs.existsSync(baseLineTemplatePath)) {
              throw new Error(`git repository kustomization yaml not found`);
            }
            const baselineTemplateJSON = yaml.load(
              fs.readFileSync(baseLineTemplatePath, "utf8")
            );
            let baselineJSON = Object.assign({}, baselineTemplateJSON);
            let tag = '';
            Object.entries(applications).map( async([appName, applicationList]) => {
              applicationList.forEach(application => {
                tag = application["applicationVersion"];
              });
            });
            baselineJSON.spec.ref.tag = `${tag}`;
            const kustomizationFileYaml = yaml.dump(baselineJSON);
            fs.writeFileSync(baseLineTemplatePath, kustomizationFileYaml);  
            log.info(`Updated url to gitrepository yaml`);
            appliedPatches.push(`${baseLineTemplatePath}/${tag}`);
        } else {
          Object.entries(applications).map( async([appName, applicationList]) => {
            if (appName.toLowerCase() !== "imagecache") {
              releaseJSON["spec"]["deploymentId"] = deploymentId;
              appName = appName.trim();
              console.log(`${clustersPath}/${restaurantId}/kustomizations/${appName}/`);
              const patchTemplatePath = `${clustersPath}/${restaurantId}/kustomizations/${appName}/flux-kustomization.yaml`;
              if (!fs.existsSync(patchTemplatePath)) {
                throw new Error(`flux kustomization yaml not found`);
              }
              const patchTemplateJSON = yaml.load(
                fs.readFileSync(patchTemplatePath, "utf8")
              );
              let patchedJSON = Object.assign({}, patchTemplateJSON);          
              let url = '';
              let tag = '';

              applicationList.forEach(application => {
                const applicationUrl = application["applicationUrl"]
                releaseSpecApplicationsList.push(applicationUrl);
                url = applicationUrl.split(":")[0];
                tag = applicationUrl.split(":")[1];
                
                if (!patchedJSON["spec"]["images"]) {
                  patchedJSON["spec"]["images"] = [];
                }
                const index = patchedJSON["spec"]["images"].findIndex(
                  (patchImage) => patchImage.name === url
                );
                if (index >= 0) {
                  patchedJSON["spec"]["images"][index].newTag = tag;
                } else {
                  const patchName = {
                    name: url,
                    newTag: tag,
                  };
                  patchedJSON["spec"]["images"].push(patchName);
                } 
              });

              releaseJSON["spec"]["applications"] = releaseSpecApplicationsList;

              const kustomizationFileYaml = yaml.dump(patchedJSON);
              fs.writeFileSync(patchTemplatePath, kustomizationFileYaml);
              log.info(`Added patch to flux-kustomization file`);
              appliedPatches.push(`${patchTemplatePath}/${url}-${tag}`);
            } else {
              log.info(`No action required for imagecache`);
            }
          });
        }
        const releaseSpecFileYaml = yaml.dump(releaseJSON);
        fs.writeFileSync(releaseTemplatePath, releaseSpecFileYaml);
        log.info(`Added applications to release spec file`);
      }
      if(products) {
        Object.entries(products).map( async([appName, productTag]) => {
          appName = appName.trim();
          const baseLineTemplatePath = `${clustersPath}/${restaurantId}/kustomizations/${appName}/gitrepository.yaml`;
          if (!fs.existsSync(baseLineTemplatePath)) {
            throw new Error(`git repository kustomization yaml not found`);
          }
          const baselineTemplateJSON = yaml.load(
            fs.readFileSync(baseLineTemplatePath, "utf8")
          );
          let baselineJSON = Object.assign({}, baselineTemplateJSON);
          const tag = productTag;
          baselineJSON.spec.ref.tag = `${tag}`;

          const kustomizationFileYaml = yaml.dump(baselineJSON);
          fs.writeFileSync(baseLineTemplatePath, kustomizationFileYaml);

          const patchTemplatePath = `${clustersPath}/${restaurantId}/kustomizations/${appName}/flux-kustomization.yaml`;
          const patchTemplateJSON = yaml.load(
            fs.readFileSync(patchTemplatePath, "utf8")
          );
          let patchedJSON = Object.assign({}, patchTemplateJSON);
          patchedJSON["spec"]["images"] = [];

          const fluxKustomizationFileYaml = yaml.dump(patchedJSON);
          fs.writeFileSync(patchTemplatePath, fluxKustomizationFileYaml);

          log.info(`Updated url to gitrepository yaml`);
          appliedPatches.push(`${patchTemplatePath}/${tag}`);
        });
      }
    });
    return appliedPatches;
  } catch (error) {
    log.error("Error in :: patchYaml()");
    console.log(error);
    return error;
  }
};

function cronExpression(date) {
  return `cron(${date.getMinutes()} ${date.getHours()} ${date.getDate()} ${date.getMonth() + 1} ? ${date.getFullYear()})`;
}

async function putRule(deploymentId, scheduledAt) {
  const ruleName = `bre-schedule-deployment-${deploymentId}`;
  const ruleParams = {
    Name: ruleName,
    ScheduleExpression: cronExpression(scheduledAt),
  };
  console.log("Put Rule: " + JSON.stringify(ruleParams));
  return eventBridge.putRule(ruleParams).promise();
}

function putTarget(deploymentId) {
  console.log("Put Target: " + deploymentId);
  const ruleName = `bre-schedule-deployment-${deploymentId}`;
  console.log("Put Target: " + ruleName);
  const eventBody = {
    deploymentId: deploymentId,
    isBundleRelease: true,
    isPreCheck: false
  };
  const targetParams = {
    Rule: ruleName,
    Targets: [
      {
        Id: `${ruleName}-target`,
        Arn: process.env.EVENT_TARGET_ARN,
        Input: JSON.stringify(eventBody),
      },
    ],
  };
  return eventBridge.putTargets(targetParams).promise();
};

async function putPreCheckRule(deploymentId, scheduledAt) {
  console.log('inside putPreCheckRule');
  const ruleName = `pre-check-deployment-${deploymentId}`;
  const ruleParams = {
    Name: ruleName,
    ScheduleExpression: cronExpression(scheduledAt),
  };
  console.log("Put precheck Rule: " + JSON.stringify(ruleParams));
  return eventBridge.putRule(ruleParams).promise();
}

function putPreCheckTarget(deploymentId) {
  console.log("Put Pre check Target: " + deploymentId);
  const ruleName = `pre-check-deployment-${deploymentId}`;
  console.log("Put Target: " + ruleName);
  const eventBody = {
    deploymentId: deploymentId,
    isBundleRelease: true,
    isPreCheck: true
  };
  const targetParams = {
    Rule: ruleName,
    Targets: [
      {
        Id: `${ruleName}-target`,
        Arn: process.env.EVENT_TARGET_ARN,
        Input: JSON.stringify(eventBody),
      },
    ],
  };
  console.log('targetParams',targetParams);
  return eventBridge.putTargets(targetParams).promise();
};

/**
 * Schedule multiple application.
 * @param {[object]]} applications 
 * @param {string} scheduleAt date
 */
const scheduleEvents = async (deploymentId, eventBridgeTime, preDeploymentCheck) => {
  const currDate = Date.now();
  const schedulePreCheckDate = new Date(currDate);
  const scheduleDeploymentDate = new Date(currDate);
  scheduleDeploymentDate.setHours(scheduleDeploymentDate.getHours() + parseInt(eventBridgeTime));
  console.log('schedulePreCheckDate',(schedulePreCheckDate.getHours() + parseInt(eventBridgeTime)) - 2);
  console.log('preDeploymentCheck',preDeploymentCheck);
  schedulePreCheckDate.setHours((schedulePreCheckDate.getHours() + parseInt(eventBridgeTime)) - 2);
  await putRule(deploymentId, scheduleDeploymentDate).then(() => putTarget(deploymentId));
  if(preDeploymentCheck === 'true') {
    await putPreCheckRule(deploymentId, schedulePreCheckDate).then(() => putPreCheckTarget(deploymentId));
  }
};

module.exports = {
  get,
  patchClusters,
  cleanPassword,
  scheduleEvents
};
