const AWS = require("aws-sdk");
const log = require("lambda-log");
const simpleGit = require("simple-git");
const git = simpleGit("/tmp");
const fs = require("fs");
/**
 * Adds necessary updates to user profile object
 *
 * @returns {Promise} returns an promise with aws getSecretValue response
 * @param {String} secretName name of the secret to be fetched
 */
const getAwsSecret = function (secretName) {
  try {
    const client = new AWS.SecretsManager({});
    return client.getSecretValue({ SecretId: secretName }).promise();
  } catch (error) {
    log.error("Call to getAwsSecret  failed:: getAwsSecret()", {
      error: error && error.data ? JSON.stringify(error.data) : error,
    });
    return undefined;
  }
};

/**
 * Converts secret retreival to async
 *
 * @returns {Array} with error and response objects
 * @param {String} secretName name of the secret to be fetched
 */
const getAwsSecretAsync = async function (secretName) {
  try {
    let error;
    log.info(
      `Entered, attempting to fetch secret ${secretName} getAwsSecretAsync()`
    );
    const response = await getAwsSecret(secretName).catch((err) => err);
    return [error, response];
  } catch (error) {
    log.info(
      "Failed to retrieve secretValue in services :: getAwsSecretAsync()",
      {
        error: error && error.data ? JSON.stringify(error.data) : error,
      }
    );
    return undefined;
  }
};

/**
 * Commit and push changes to repository
 *
 * @returns {Boolean} true - success , false - failure
 * @param {String} commitMessage commit message to pass while creating commit
 * @param {String} username of repository
 * @param {String} updatedFolder folder to add during git add
 * @param {String} repoName name of the kustomize repository
 * @param {String} branch repo branch to push to
 */
const pushToRepo = async function (
  commitMessage,
  repoName,
  username,
  updatedFolder,
  branch = "main"
) {
  try {
    log.info("Creating commit in pushToRepo()");
    const gitResult = await simpleGit(`/tmp/${repoName}`, {
      config: [`user.name=${username}`],
    })
      .add([updatedFolder])
      .commit(commitMessage)
      .push("origin", branch);
    log.info(
      "Successfully created commit and pushed to repo in pushToRepo()",
      gitResult
    );
    return true;
  } catch (error) {
    log.info(error);
    return false;
  }
};
/**
 * Clones a repository
 *
 * @returns {Boolean} true - success , false - failure
 * @param {String} username with access to repo
 * @param {String} password of username
 * @param {String} repoURL URL of the repository without https://
 * @param {String} repoName name of the kustomize repository
 * @param {String} branch repo branch to clone
 */
const cloneRepo = async function (
  username,
  password,
  repoURL,
  branch,
  repoName
) {
  log.info(`Attempting to clone repo: ${repoURL}, branch: ${branch}, reponame: ${repoName}`);
  const repoAbsPath = `/tmp/${repoName}`;
  try {
    fs.rmdirSync(repoAbsPath, { recursive: true }, { force: true });
  } catch (error) {
    log.error(error);
  }
  try {
    if (fs.existsSync(repoAbsPath)) {
      let result = await simpleGit(repoAbsPath).pull(branch, [
        "--recurse-submodules",
      ]);
      log.info(
        `Repository already exists in directory, performing pull ${repoAbsPath} in ${result}`
      );
      return true;
    } else {
      let cloneOptions = { "--recurse-submodules": null };
      if (branch !== "main") {
        cloneOptions["--branch"] = branch;
      }
      const remote = `https://${username}:${password}@${repoURL}`;
      let result = await git.clone(remote, `/tmp/${repoName}`, cloneOptions);
      log.info("Repository doesnt Exist, cloning in cloneRepo()", result);
      return true;
    }
  } catch (error) {
    log.error(error);
    return false;
  }
};

const domain = process.env.ASSET_URL;
const region = process.env.REGION;

const assetServiceRequest = (method, path) => new Promise((resolve, reject) => {
  AWS.config.getCredentials(function (err) {
    if (err) {
      console.log('Failed while getting AWS config : ', err.stack);
      reject(err);
    }
    else {
      const endpoint = new AWS.Endpoint(domain);
      let request = new AWS.HttpRequest(endpoint, region);
      // make sure to not add / in the path as it is appended automatically.
      request.path += path;
      request.headers['host'] = domain;
      request.method = method;

      let signer = new AWS.Signers.V4(request, 'execute-api');
      signer.addAuthorization(AWS.config.credentials, new Date());
      let client = new AWS.HttpClient();
      client.handleRequest(request, null, function (response) {
        console.log("passed request");
        console.log(response.statusCode + ' ' + response.statusMessage);
        let buffer = "";
        response.on('data', chunk => buffer += chunk)
        response.on('end', () => resolve(JSON.parse(buffer)))
      }, function (error) {
        console.log('Error: ' + error);
        reject(error);
      });

    }
  })
});

/**
 * Fetch deployment information
 * @param deploymentId integer
 */
const fetchDeploymentInformation = async (deploymentId) => {
  console.info(
    "Request to asset service to fetch records " + JSON.stringify(deploymentId)
  );
  let response;
  let path = `deployment/details/${deploymentId}`;
  try {
    response = await assetServiceRequest(
      "GET",
      path
    );
    if (response.data) {
      return response.data;
    }
    else {
      throw new MCDError(`${deploymentId} not found`);
    }
  } catch (err) {
    console.error(err);
    throw new MCDError("Error in getting data from Asset Service");
  }
};

/**
 * Fetch deployment information
 * @param deploymentId integer
 */
const fetchDeploymentGroup = async (deploymentId) => {
  console.info(
    "Request to asset service to fetch records " + JSON.stringify(deploymentId)
  );
  let response;
  let path = `restaurant_assets/deployment_groups/${deploymentId}`;
  try {
    response = await assetServiceRequest(
      "GET",
      path
    );
    if (response.data) {
      return response.data;
    }
    else {
      throw new MCDError(`${deploymentId} not found`);
    }
  } catch (err) {
    console.error(err);
    throw new MCDError("Error in getting data from Asset Service");
  }
};

const DEFAULT_ERROR_MESSAGE =
  "There was a problem completing your request. Please try again later.";

class MCDError extends Error {
  constructor(message = DEFAULT_ERROR_MESSAGE) {
    super(message);
    this.statusCode = 500;
  }
}

module.exports = { getAwsSecretAsync, cloneRepo, pushToRepo, fetchDeploymentInformation, fetchDeploymentGroup };
