const log = require("lambda-log");
const AWS = require("aws-sdk");

const {
  getAwsSecretAsync,
  cloneRepo,
  pushToRepo,
  fetchDeploymentInformation,
  fetchDeploymentGroup
} = require("./services");
const { cleanPassword, get, patchClusters, scheduleEvents } = require("./utils");

const commitToRepo = async function (
  commitMessages,
  branch,
  repoName,
  uname,
  folder,
  clusters
) {
  try {
    if (
      commitMessages &&
      typeof commitMessages !== "string" &&
      commitMessages.length
    ) {
      const commitMessage = `Updates made on the following application: ${commitMessages.join(
        " | "
      )} on clusters: ${clusters} at ${new Date().toISOString()}`;
      log.info(
        "Applied SQS changes,attempting to push to remote repository in :: commitToRepo() ",
        commitMessage
      );
      folder = folder === "." ? folder : `./${folder}/*`;
      const pushToRepoResult = await pushToRepo(
        commitMessage,
        repoName,
        uname,
        folder,
        branch
      );
      if (pushToRepoResult) {
        log.info(
          "Successfully pushed changes to remote repository :: gitOpsDeployHander()"
        );
        return true;
      } else {
        log.error(
          "Failed to push to remote repository in :: gitOpsDeployHander()"
        );
        return false;
      }
    } else {
      log.info(
        "No updated found  for any of the apps, completing execution :: gitOpsDeployHander()"
      );
      return false;
    }
  } catch (error) {
    log.error("Error in :: commitToRepo()", error);
  }
};

// fetch secrets to access repository
const fetchLambdaSecrets = async function (secretName) {
  try {
    const [error, secret] = await getAwsSecretAsync(secretName);
    if (error) {
      log.error("Failed to get secret in :: fetchLambdaSecrets()");
      throw new Error(error);
    } else {
      const { SecretString } = secret;
      log.info(
        "Successfully fetched secret values in :: fetchLambdaSecrets()",
        {
          secretName,
        }
      );
      return JSON.parse(SecretString);
    }
  } catch (error) {
    log.error("Error in :: fetchLambdaSecrets()", error);
    console.log(error);
  }
};

// main invocation point for lambda code
const gitOpsDeployHander = async function (event, context, callback) {
  try {
    context.callbackWaitsForEmptyEventLoop = false;
    // TODO fetch secrets from aws - username and password
    log.info(
      "Initializing gitops lambda, incoming deployment details in :: gitOpsDeployHander(): ",
      event
    );
    const { Records } = event;
    let sqsinput;
    if (!Records || !Records.length) {
      log.info("No records found in event object in :: gitOpsDeployHander()");
      callback("Unable to parse SQS message");
    } else {
      //assuming FIFO queue
      const messageBody = get(["Records", "0", "body"], event);
      if (typeof messageBody === "string") {
        sqsinput = JSON.parse(messageBody);
      } else {
        sqsinput = messageBody;
      }
    }

    let { deploymentId } = messageDetails(sqsinput);
    const deploymentGroup = await fetchDeploymentGroup(deploymentId);
    const dAttributes = deploymentGroup.attributes;
    if (dAttributes.hasOwnProperty("processing") && dAttributes["processing"] === true) {
      await retryDeploymentLater(sqsinput);
    } else {
      console.log("Ready for getting deployment details")
      await processSQSMesssage(sqsinput, callback);
    }
  } catch (error) {
    log.error("Error in :: gitOpsDeployHander()", error);
    console.log(error);
  }
};

async function retryDeploymentLater(sqsinput) {
  const sqs = new AWS.SQS({ apiVersion: "2012-11-05" });
  const { gitopsSQSQueueURL, gitopsMessageGroup } = process.env;
  addRetryAttemptCount(sqsinput);
  const sqsparams = {
    MessageBody: typeof sqsinput === "string"
      ? sqsinput
      : JSON.stringify(sqsinput),
    MessageGroupId: gitopsMessageGroup,
    QueueUrl: gitopsSQSQueueURL
  };
  console.log(sqsparams);
  const sqsPostResult = await sqs.sendMessage(sqsparams).promise();
  console.log(
    "Successfully posted messages to SQS in ::sendMessageToSQS()-gitops ",
    sqsPostResult
  );
}

async function processSQSMesssage(sqsinput, callback) {
  let uname, password;
  const {
    bitbucketSecret, ENVIRONMENT, branch
  } = process.env;

  const secretObj = await fetchLambdaSecrets(bitbucketSecret);
  uname = secretObj.username;
  password = secretObj.password;
  password = cleanPassword(password);

  let { deploymentId, resturantDetails, applicationDetails, productDetails, eventBridgeTime, preDeploymentCheck } = messageDetails(sqsinput);

  const deploymentResponse = await fetchDeploymentInformation(deploymentId);
  console.log(deploymentResponse);
  const market = deploymentResponse.marketName.toLowerCase();
  resturantDetails = deploymentResponse.storeList;
  applicationDetails = deploymentResponse.applications;
  productDetails = deploymentResponse.products;

  await Promise.all(Object.entries(resturantDetails).map(async ([regionName, storeNo]) => {
    try {
      let region = regionName.split(' ').join('_').toLowerCase();
      region = region.split('_region')[0];
      const repoURL = `github.com/bre-org/brep-gitops-${ENVIRONMENT}-${market}-${region}`;
      const repoName = `brep-gitops-${ENVIRONMENT}-${market}-${region}`;
      console.log(repoName);
      log.info(
        "Attempting to clone remote repository in :: gitOpsDeployHander()",
        { repoURL }
      );
      const repoCloneResult = await cloneRepo(
        uname,
        password,
        repoURL,
        branch,
        repoName
      );
      if (repoCloneResult) {
        log.info(
          "Successfully cloned remote repository, attempting to apply SQS changes in :: gitOpsDeployHander()"
        );
        let commitMessages = [];
        if (applicationDetails || productDetails) {
          const restaurantsToUpdate = storeNo;
          log.info(
            "Following restaurants are part of this update in :: gitOpsDeployHander()",
            restaurantsToUpdate
          );
          commitMessages = await patchClusters(
            applicationDetails,
            productDetails,
            repoName,
            restaurantsToUpdate,
            deploymentId
          );
          const pushToRepoResult = await commitToRepo(
            commitMessages,
            branch,
            repoName,
            uname,
            "clusters",
            restaurantsToUpdate
          );
          if (pushToRepoResult) {
            console.log("Schedule an event");
          } else {
            callback("Failed");
          }
        }
      } else {
        log.error("Failed to clone remote repository in :: gitOpsDeployHander()");
        callback("Failed to clone remote repository");
      }
    } catch (error) {
      console.log('error' + error);
    }
  })).then(async () => {
    if (Object.keys(applicationDetails).length === 1 && Object.keys(applicationDetails)[0].toLowerCase() === "imagecache") {
      await scheduleEvents(deploymentId, eventBridgeTime, preDeploymentCheck);
      callback(null, "Success");
    }
  });
  return deploymentId;
}

module.exports = { gitOpsDeployHander };

function messageDetails(sqsinput) {
  let deploymentId;
  let eventBridgeTime = process.env.eventBridgeTime;
  let resturantDetails = {};
  let applicationDetails = {};
  let productDetails = {};
  let preDeploymentCheck;
  if (typeof sqsinput === "string") {
    let parsedSQS = JSON.parse(sqsinput);
    console.log("parsedSQS" + parsedSQS);
    deploymentId = parsedSQS.deploymentGroupId;
    eventBridgeTime = parsedSQS.eventBridgeTime;
    preDeploymentCheck = parsedSQS.isPreDeploymentCheck;
  } else {
    console.log(sqsinput);
    deploymentId = sqsinput.deploymentGroupId;
    eventBridgeTime = sqsinput.eventBridgeTime;
    preDeploymentCheck = sqsinput.isPreDeploymentCheck;
  }
  return { deploymentId, resturantDetails, applicationDetails, productDetails, eventBridgeTime, preDeploymentCheck };
}

function addRetryAttemptCount(message) {
  if (message.hasOwnProperty("retryAttempt")) {
    // If it exists, increment the retryAttempt value by 1
    message.retryAttempt += 1;
  } else {
    // If it doesn't exist, set retryAttempt to 1
    message.retryAttempt = 1;
  }
}