def call() {
  // Remove last string after last - from the name
  // ex: bre-sample-docker will become bre-sample
  // Values for awsAccountID are populated from this Confluence page. If future tooling account environments are added, please update this pipeline accordingly
  // https://us-confluence.mcd.com/pages/viewpage.action?spaceKey=BREP&title=BRE+AWS+Account+List
  // TODO: We should add a check if we are using multibranch pipeline or not... and adjust the appName accordingly
  // TODO: this is only required if we will not recreate old jenkins jobs

  def jobNameParts = env.JOB_NAME.tokenize('/') as String[] // [TEST Projects, test-pawel-test-1-docker, master]
  jobNameParts.length < 2 ? env.JOB_NAME : jobNameParts[jobNameParts.length - 2]
  def appName = jobNameParts[1].replaceAll('-[^-]*$', '')
  def jenkinsSlaveRoleArn = false
  def awsAccountID

  properties([
    parameters([
      choice(name: 'Environment', choices: ['dev', 'int','stg', 'prod'], description: 'Environment'),
      booleanParam(name: 'Use docker cache', defaultValue: true, description: 'Do you want to use docker cache?')
    ])
  ])

  if (params['Environment'] != null) {
    switch (params['Environment']) {
      case 'dev':
        awsAcccountID = "283388140277"
        jenkinsSlaveRoleArn = "arn:aws:iam::" + awsAcccountID + ":role/bre-jenkins-slave"
        releaseSNSArn = "${releaseDevSNSArn}"
        testSNSArn = "${testDevSNSArn}"
        break
      case 'int':
        awsAcccountID = "593265675765"
        jenkinsSlaveRoleArn = "arn:aws:iam::" + awsAcccountID + ":role/bre-jenkins-slave"
        releaseSNSArn = "${releaseIntSNSArn}"
        testSNSArn = "${testIntSNSArn}"
        break
      case 'stg':
        awsAcccountID = "688810906228"
        jenkinsSlaveRoleArn = "arn:aws:iam::" + awsAcccountID + ":role/bre-jenkins-slave"
        releaseSNSArn = "${releaseStgSNSArn}"
        testSNSArn = "${testStgSNSArn}"
        break
      case 'prod':
        awsAcccountID = "524430043955"
        jenkinsSlaveRoleArn = "arn:aws:iam::" + awsAcccountID + ":role/bre-jenkins-slave"
        break
      }
    } else {
        // Printing message that first time run and params are not taken into effect
        // https://issues.jenkins-ci.org/browse/JENKINS-40574 - it is marked resolved but the last comment says it doesn't work for declaritive pipelines
        jenkinsSlaveRoleArn = "First time build, please rerun to use AWS IAM role"
    }

  pipeline {
    environment {
        REGISTRY = "${SaaSartifactoryUrl}/brep-docker/${appName}"
        SAASARTIFACTORY = credentials('33b0c462-4630-4682-aca5-a512254be0e8') //---> SaaS credentials added 
    }

    agent {
      kubernetes {
        yaml """
          apiVersion: v1
          kind: Pod
          metadata:
            labels:
              name: container-pipeline
            annotations:
              iam.amazonaws.com/role: ${jenkinsSlaveRoleArn}
          spec:
            imagePullSecrets:
              - name: image-pull-secret
            containers:
              - name: build-tools
                image: ${SaaSartifactoryUrl}/brep-docker/bred-build-tools:${buildToolsImageVersion}
                command:
                  - cat
                tty: true
                volumeMounts:
                  - mountPath: /var/run/docker.sock
                    name: docker-sock
                  - mountPath: /root/.m2/repository
                    name: maven-cache
              - name: lambda-app-builder
                image: ${SaaSartifactoryUrl}/brep-docker/bre-lambda-app-builder:${lambdaAppBuilderVersion}
                command:
                  - cat
                tty: true
                volumeMounts:
                  - mountPath: /var/run/docker.sock
                    name: docker-sock
                  - mountPath: /root/.m2/repository
                    name: maven-cache
            volumes:
              - name: docker-sock
                hostPath:
                  path: /var/run/docker.sock
              - name: maven-cache
                hostPath:
                  path: /maven-cache
                  type: DirectoryOrCreate
          """
        }
      }
      tools {
        jfrog 'brep-jfrog-cli-remote'
      }

    stages {
      stage('GIT Checkout') {
        steps {
          script {
            def myRepo = checkout scm
            env.gitCommit = myRepo.GIT_COMMIT
            env.gitBranch = myRepo.GIT_BRANCH
            env.shortGitCommit = "${gitCommit[0..10]}"
            env.previousGitCommit = sh(script: "git rev-parse ${gitCommit}~", returnStdout: true)

            def infoymlcheck = fileExists 'info.yml'
            // Note: When a new release tag is detected, Jenkins will trigger a tag job.
            // During this run, the gitBranch will be the name of the release tag, rather than a branch.
            echo "Current Branch: ${gitBranch}"
            echo "App name: ${appName}"
            if (infoymlcheck) {
              ansiColor('xterm') {
                sh """
                set +x
                RED='\033[0;31m'
                BOLD='\033[1m'
                NC='\033[0m'
                echo -e "\${RED}+==================================[NOTICE]===================================+\${NC}"
                echo -e "\${BOLD}info.yml\${NC} was detected in your repo.\nThis file has been deprecated and can be safely removed from your repository.\n"
                echo -e "For more information about versioning, please reference: https://us-confluence.mcd.com/display/BREP/Development+Guidelines#DevelopmentGuidelines-DevelopmentStandards"
                echo -e "\${RED}+=============================================================================+\${NC}"
                """
              }
            }
          }
        }
      }

      stage('Install Dependencies') {
        when { expression { fileExists "install_deps.sh" }}
        steps {
          container('build-tools'){
            ansiColor('xterm') {
              script {
                if (fileExists("install_deps.sh")){
                  sh "./install_deps.sh"
                }
              }
            }
          }
        }
      }

      stage('Build Image') {
        steps {
          withCredentials([usernamePassword(credentialsId: '33b0c462-4630-4682-aca5-a512254be0e8', passwordVariable: 'SAASARTIFACTORY_PSW', usernameVariable: 'SAASARTIFACTORY_USR')]) {
            container ('build-tools') {
              script {
                ansiColor('xterm') {
                  jfrog "rt config --user=${SAASARTIFACTORY_USR} --password=${SAASARTIFACTORY_PSW} --url=${SaaSartifactoryUrl}"
                  jf "rt dl brep-BREScripts/npmrc.zip"
                  sh 'fastjar -xvf npmrc.zip'
          
                  sh 'docker login -u ${SAASARTIFACTORY_USR} -p ${SAASARTIFACTORY_PSW} ${SaaSartifactoryUrl}'
                  sh 'cp -rp /root/.m2 .m2'
                  envVariable = params['Environment']
                  if (params['Use docker cache'] == true ){
                    //specified docker to use same network where container is running
                    sh "docker build --network=host -t ${REGISTRY} . --build-arg env=${envVariable}"
                  }
                  else {
                    sh "docker build --network=host --no-cache -t ${REGISTRY} . --build-arg env=${envVariable}"
                  }
                  def testContainer = fileExists 'Dockerfile.test'
                  if (testContainer) {
                    // Export the build output to a directory so that any report files created during the build can be accessed
                    sh "DOCKER_BUILDKIT=1 docker build --network=host -f=Dockerfile.test -o build_output ."
                  }
                }
              }
            }
          }
        }
      }
      
      
      
      
      stage('Sonarqube EnterPrise Scan') {
        steps {
          withSonarQubeEnv('Sonar Ent Tooling') {
            container('lambda-app-builder') {
              script {
                sh 'docker login -u ${SAASARTIFACTORY_USR} -p ${SAASARTIFACTORY_PSW} ${SaaSartifactoryUrl}' //docker login

                if (fileExists('testcontainers.properties')) {
                  sh "mv testcontainers.properties /root/.testcontainers.properties"
                }
                
                if (fileExists("build.gradle")){
                  def projectName = env.JOB_NAME.split("/")[1]
                  def orgName = projectName.split("-")[0,1].join()
                  sh """cat << EOF > gradle.properties
sonar.projectKey=com.mcd.${orgName}:${projectName}
sonar.projectName=${projectName}
EOF"""
                  sh './gradlew sonarqube'
                } else if (fileExists("pom.xml")) {
                  sh 'mvn -Dhttp.keepAlive=false -Dmaven.wagon.http.pool=false -Dmaven.wagon.http.retryHandler.count=10 clean install sonar:sonar -X'
                } else {
                  def scannerHome = tool 'Sonar Scanner';
                  def projectName = env.JOB_NAME.split("/")[1]
                  def orgName = projectName.split("-")[0,1].join()
                  sh """cat << EOF > sonar-project.properties
sonar.projectKey=com.mcd.${orgName}:${projectName}
sonar.projectName=${projectName}
EOF"""
                  // If a coverage report is found, add the report path to the properties file
                  def coveragePath = 'build_output/coverage.xml' //Python Coverage report Path
                  sh """
                    if [[ -f "$coveragePath" ]]; then
                      echo sonar.python.coverage.reportPaths=${coveragePath} >> sonar-project.properties
                      echo sonar.coverage.exclusions="**/tests/**,**/test*" >> sonar-project.properties
                      # If a src directory exists, set that as the source for application files, otherwise use the default
                      if [[ -d "src" ]]; then
                        echo sonar.sources="src" >> sonar-project.properties
                      fi
                    fi
                  """
                  def tsCoveragePath = 'build_output/lcov.info'  //TS coverage path
                  sh """
                    if [[ -f "$tsCoveragePath" ]]; then
                      echo sonar.javascript.lcov.reportPaths=${tsCoveragePath} >> sonar-project.properties
                      echo sonar.exclusions=**/node_modules/**,**/*.spec.ts >> sonar-project.properties
                       # If a src directory exists, set that as the source for application files, otherwise use the default
                      if [[ -d "src" ]]; then
                        echo sonar.sources="src" >> sonar-project.properties
                      fi
                    fi 
                  """
                  sh "${scannerHome}/bin/sonar-scanner"
                }
              }
            }
          }
        }
      }

      stage('Sonar Quality Gate') {
        steps {
          script {
            timeout(time: 15, unit: 'MINUTES') { // Just in case something goes wrong, pipeline will be killed after a timeout
              def qg = waitForQualityGate() // Reuse taskId previously collected by withSonarQubeEnv
              if (qg.status != 'OK') {
                // error "Pipeline aborted due to quality gate failure: ${qg.status}" // uncomment this when we want to fail pipeline if sonar fails
                echo "Sonar quality gate failed"
              }
            }
          }
        }
      }


      stage('Tag Docker Image') {
        steps {
          script {
            env.releaseTag = sh(returnStdout: true, script: "git tag -l --points-at ${env.gitCommit} | sed -E 's/release\\/(([0-9]*)(\\.[0-9]+){2})/\\1/' | sort | tail -n 1").trim()
            // In the replaceVersion command, 'env.gitBranch' takes advantage of the fact that during a jenkins bitbucket multi-branch tag job, the branch points to the tag name, rather then an actual branch.
            env.replaceVersion = sh(returnStdout: true, script: "git tag --points-at ${env.gitBranch} --format='%(contents)' | sed -rn 's/^Replaces: ([<>]?=?[0-9]+(\\.[0-9]+){2}.*)/\\1/ip'").trim()

            echo "Commit release tag: ${releaseTag}"
            echo "Release replaces: ${replaceVersion}"

            if (releaseTag.length() > 0 && replaceVersion.length() < 5) {
              // When releasing a new version, require that the applicable versions to replace are specified.
              bumpReleaseTagArray = releaseTag.tokenize(".")
              bumpReleaseTagArray[2]++
              bumpReleaseTag = bumpReleaseTagArray.join(".")
              ansiColor('xterm') {
                sh """
                #!/usr/bin/env bash
                set +x
                RED='\033[0;31m'
                BOLD='\033[1m'
                NC='\033[0m'
                echo -e "\${RED}+===================================[ERROR]===================================+\${NC}"
                echo -e "\${RED}Applicable version replacements were not specified or invalid.\${NC}\\nPossible solutions:\\n    1. Use the 'tag_release.sh' script to appropriately tag commits.\n       (http://ac51f54e7be0e11e9b3460203616ba96-778168875.us-east-1.elb.amazonaws.com/projects/BRE/repos/bre-useful-scripts/browse/tag_release.sh)\\n    2. Ensure your release tag message contains a line with the following contents: 'Replaces: (optional inequality symbols)X.X.X'\\n        Example tag message:\\n            Releasing version ${bumpReleaseTag}\\n            \${BOLD}Replaces: ${releaseTag}\${NC}"
                echo -e "\${RED}+=============================================================================+\${NC}"
                """
                error("Replace version not found or invalid.")
              }
            }
          }
          script {
            container ('build-tools') {
              if (gitBranch.contains("master")) {
                sh'''
                  docker login -u ${SAASARTIFACTORY_USR} -p ${SAASARTIFACTORY_PSW} ${SaaSartifactoryUrl}
                  docker tag ${REGISTRY}:latest ${REGISTRY}:${BUILD_NUMBER}
                '''
              } else if (gitBranch.contains("feature/") || gitBranch.contains("hotfix/") || gitBranch.contains("bugfix/")) {
                // Extracts branch and ticket number from gitBranch if the branch contains feature or hotfix 
                // UPDATE : Added bugfix branch support to the pipeline
                env.extractedBranch = gitBranch.split("/")[0].toLowerCase()
                env.extractedTicketNumber = gitBranch.split("/")[1].toLowerCase()
                sh'''
                  docker login -u ${SAASARTIFACTORY_USR} -p ${SAASARTIFACTORY_PSW} ${SaaSartifactoryUrl}
                  docker tag ${REGISTRY}:latest ${REGISTRY}:${extractedBranch}-${extractedTicketNumber}-${BUILD_NUMBER}
                '''
              } else if (gitBranch.contains("dev/") || gitBranch.contains("stg/")) {
                //New block to identify intermediate branch in pipeline for dev and stg env
                env.extractedBranch = gitBranch.split("/")[0].toLowerCase()
                env.extractedTicketNumber = gitBranch.split("/")[1].toLowerCase()
                sh'''
                  docker login -u ${SAASARTIFACTORY_USR} -p ${SAASARTIFACTORY_PSW} ${SaaSartifactoryUrl}
                  docker tag ${REGISTRY}:latest ${REGISTRY}:${extractedBranch}-${BUILD_NUMBER}
                '''
              } else if (releaseTag.length() > 0) {
                sh'''
                  docker login -u ${SAASARTIFACTORY_USR} -p ${SAASARTIFACTORY_PSW} ${SaaSartifactoryUrl}
                  docker tag ${REGISTRY}:latest ${REGISTRY}:${releaseTag}
                '''
              } else {
                echo "Non-trunk branch build. Tagging skipped."
              }
            }
          }
        }
      }

      stage('Push Image, publish build info & xray scanning') {
        steps {
          container ('build-tools') {
            script {
              def REPO = "brep-docker"
              def PROJECT = "brep"
              if (gitBranch.contains("master") || gitBranch.contains("main")) {
              echo '\033[34m' + 'Preparing to push docker image' + '\033[0m' 
              jf "rt dp ${REGISTRY}:${BUILD_NUMBER} ${REPO} --build-name=${appName} --build-number=${BUILD_NUMBER} --project=${PROJECT}"
              echo '\033[34m' + "Successfully pushed docker image to ${REGISTRY}" + '\033[0m'
              echo '\033[34m' + 'Preparing to publish build info to Artifactory' + '\033[0m'
              jf "rt bp ${appName} ${BUILD_NUMBER} --project=${PROJECT}"
              echo '\033[34m' + "Successfully published build info status to Artifactory for current build ${BUILD_NUMBER}" + '\033[0m'
              //initiate xray scanning  the docker image
              echo '\033[34m' + 'Preparing to initiate Xray Scanning for the module' + '\033[0m'
              jf "bs ${appName} ${BUILD_NUMBER} --project=${PROJECT}"
              echo '\033[34m' + 'Xray scanning hae been successfully completed and results are pushed to artifactory' + '\033[0m'
            }
            else if (gitBranch.contains("feature/") || gitBranch.contains("hotfix/") || gitBranch.contains("bugfix/")) {
              env.extractedBranch = gitBranch.split("/")[0].toLowerCase()
              env.extractedTicketNumber = gitBranch.split("/")[1].toLowerCase()
              echo '\033[34m' + 'Preparing to push docker image' + '\033[0m'
              jf "rt dp ${REGISTRY}:${extractedBranch}-${extractedTicketNumber}-${BUILD_NUMBER} ${REPO} --build-name=${appName} --build-number=${BUILD_NUMBER} --project=${PROJECT}"
              echo '\033[34m' + 'Successfully pushed docker image to ${REGISTRY}' + '\033[0m'
              echo '\033[34m' + 'Preparing to publish build info to Artifactory' + '\033[0m'
              jf "rt bp ${appName} ${BUILD_NUMBER} --project=${PROJECT}"
              echo '\033[34m' + 'Successfully published build info status to Artifactory for current build ${BUILD_NUMBER}' + '\033[0m'
              //initiate xray scanning  the docker image
              echo '\033[34m' + 'Preparing to initiate Xray Scanning for the module' + '\033[0m'
              jf "bs ${appName} ${BUILD_NUMBER} --project=${PROJECT}"
              echo '\033[34m' + 'Xray scanning hae been successfully completed and results are pushed to artifactory' + '\033[0m'
            } 
            else if (gitBranch.contains("dev/") || gitBranch.contains("hotfix/") || gitBranch.contains("stg/")) {
              env.extractedBranch = gitBranch.split("/")[0].toLowerCase()
              env.extractedTicketNumber = gitBranch.split("/")[1].toLowerCase()
              echo '\033[34m' + 'Preparing to push docker image' + '\033[0m'
              jf "rt dp ${REGISTRY}:${extractedBranch}-${BUILD_NUMBER} ${REPO} --build-name=${appName} --build-number=${BUILD_NUMBER} --project=${PROJECT}"
              echo '\033[34m' + 'Successfully pushed docker image to ${REGISTRY}' + '\033[0m'
              echo '\033[34m' + 'Preparing to publish build info to Artifactory' + '\033[0m'
              jf "rt bp ${appName} ${BUILD_NUMBER} --project=${PROJECT}"
              echo '\033[34m' + 'Successfully published build info status to Artifactory for current build ${BUILD_NUMBER}' + '\033[0m'
              //initiate xray scanning  the docker image
              echo '\033[34m' + 'Preparing to initiate Xray Scanning for the module' + '\033[0m'
              jf "bs ${appName} ${BUILD_NUMBER} --project=${PROJECT}"
              echo '\033[34m' + 'Xray scanning hae been successfully completed and results are pushed to artifactory' + '\033[0m'
            }
            else if (releaseTag.length() > 0) {
              echo '\033[34m' + 'Preparing to push docker image' + '\033[0m'
              jf "rt dp ${REGISTRY}:${releaseTag} ${REPO} --build-name=${appName} --build-number=${BUILD_NUMBER} --project=${PROJECT}"
              echo '\033[34m' + 'Successfully pushed docker image to ${REGISTRY}' + '\033[0m'
              echo '\033[34m' + 'Preparing to publish build info to Artifactory' + '\033[0m'
              jf "rt bp ${appName} ${BUILD_NUMBER} --project=${PROJECT}"
              echo '\033[34m' + 'Successfully published build info status to Artifactory for current build ${BUILD_NUMBER}' + '\033[0m'
              //initiate xray scanning  the docker image
              echo '\033[34m' + 'Preparing to initiate Xray Scanning for the module' + '\033[0m'
              jf "bs ${appName} ${BUILD_NUMBER} --project=${PROJECT}"
              echo '\033[34m' + 'Xray scanning hae been successfully completed and results are pushed to artifactory' + '\033[0m'
            }
          }
        }
      }
    }
    

      stage('Modify Kubernetes Opentest Pod') {
        when { expression { fileExists "opentest" }}
        steps {
          container ('build-tools') {
            script {
              writeFile file:"containerbuilder-opentest-pod.yaml", text:libraryResource("containerbuilder-opentest-pod.yaml")
              mydata = readYaml file: "containerbuilder-opentest-pod.yaml"
              if (releaseTag.length() > 0) {
                env.appImageUrl = "${REGISTRY}:${releaseTag}"
                env.appImageTag = "${releaseTag}"
              } else if (gitBranch.contains("master")) {
                env.appImageUrl = "${REGISTRY}:${BUILD_NUMBER}"
                env.appImageTag = "${BUILD_NUMBER}"
              } else if (gitBranch.contains("feature/") || gitBranch.contains("hotfix/") || gitBranch.contains("bugfix/")) {
                env.appImageUrl = "${REGISTRY}:${extractedBranch}-${extractedTicketNumber}-${BUILD_NUMBER}"
                env.appImageTag = "${extractedBranch}-${extractedTicketNumber}-${BUILD_NUMBER}"
              } else if (gitBranch.contains("dev/") || gitBranch.contains("stg/")) { 
                env.appImageUrl = "${REGISTRY}:${extractedBranch}-${BUILD_NUMBER}"
                env.appImageTag = "${extractedBranch}-${BUILD_NUMBER}"
              }
              mydata.spec.containers[2].name = "${appName}"
              mydata.spec.containers[2].image = "${appImageUrl}"
              if (fileExists('opentest/dependencies.yml')) {
                ansiColor('xterm') {
                  writeFile file:"dependencies-schema.yaml", text:libraryResource("dependencies-schema.yaml")

                  print("INFO: Validating opentest/dependencies.yml")

                  def yamaleStatusCode = sh(returnStdout: true, returnStatus: true, script: "yamale -s dependencies-schema.yaml opentest/dependencies.yml --strict")

                  if (yamaleStatusCode == 1){
                    env.imageSha = sh(returnStdout: true, script: """ curl -u $SAASARTIFACTORY_USR:$SAASARTIFACTORY_PSW -sSL -I -H "Accept: application/vnd.docker.distribution.manifest.v2+json" "https://$SaaSartifactoryUrl/artifactory/api/docker/docker-local/v2/$appName/manifests/$appImageTag" | grep -i 'docker-content-digest:' | awk '{print \$2}' """).trim()

                    sh """ curl -u $SAASARTIFACTORY_USR:$SAASARTIFACTORY_PSW -sSL -X DELETE "https://$SaaSartifactoryUrl/artifactory/api/brep-docker-local/v2/$appName/manifests/$imageSha" """

                    ansiColor('xterm') {
                      sh """
                        set +x
                        RED='\033[0;31m'
                        BOLD='\033[1m'
                        NC='\033[0m'
                        echo -e "\${RED}+==================================[ERROR]===================================+\${NC}"
                        echo -e "\t\${BOLD}opentest/dependencies.yml\${NC} validation failed!\n"
                        echo -e "For more information about dependencies.yml, please reference: https://us-confluence.mcd.com/display/BREP/%5BBuild+Type%5D+Docker+-+Opentest"
                        echo -e "\${RED}+=============================================================================+\${NC}"
                      """
                      error('opentest/dependencies.yml validation failed!')
                    }
                  }
                }

                dependencies = readYaml file: "opentest/dependencies.yml"
                def databaseSpec = dependencies.deps.database
                def databaseType = databaseSpec.type.isEmpty() ? "mysql" : "${databaseSpec.type}"
                def databaseVersion = databaseSpec.version.isEmpty() ? "8.0" : "${databaseSpec.version}"
                def databasePort = databaseSpec.port == null ? "3306" : "${databaseSpec.port}"
                def databaseName = databaseSpec.db_name.isEmpty() ? "opentest" : "${databaseSpec.db_name}"

                mydata.spec.containers[2].ports[0].containerPort = dependencies.deps.application.port
                mydata.spec.containers[2].env[2].value = "jdbc:" + databaseType + "://localhost:" + databasePort + "/" + databaseName + "?autoReconnect=true&failOverReadOnly=false&maxReconnects=90&useSSL=false&serverTimezone=UTC"
                // database
                mydata.spec.containers[3].image = databaseType + ':' + databaseVersion
                mydata.spec.containers[3].ports[0].containerPort = databasePort
                mydata.spec.containers[3].ports[0].name = databaseType
                mydata.spec.containers[3].env[1].value = databaseName
              }
              writeYaml file: 'containerbuilder-opentest-pod-new.yaml', data: mydata

              if (fileExists('opentest')) {
                env.podTemplate = sh(returnStdout: true, script: "cat containerbuilder-opentest-pod-new.yaml")
              } else {
                env.podTemplate = ""
              }
            }
          }
        }
      }

      stage('Run testing suite & Allure reporting') {
        when {
          expression { fileExists "opentest" }
          // If this is false, Jenkins will run the agent block first before checking
          // the when condition, causing an error for pipelines not using Opentest
          beforeAgent true
        }
        agent {
          kubernetes {
            label "nested-pod-${appName}"
            defaultContainer 'opentest'
            yaml "${podTemplate}"
          }
        }

        steps {
          container('database'){
            sleep time: 30, unit: 'SECONDS'
            script {
              if (fileExists('opentest/dependencies.yml')) {
                dependencies = readYaml file: "opentest/dependencies.yml"
                def databaseSpec = dependencies.deps.database
                def databaseType = databaseSpec.type.isEmpty() ? "" : "${databaseSpec.type}"
                def databaseName = databaseSpec.db_name.isEmpty() ? "" : "${databaseSpec.db_name}"
                def schemaPath = databaseSpec.schema_path.isEmpty() ? "" : "${databaseSpec.schema_path}"

                if (databaseType == "mysql" && schemaPath != "") {
                  sh """
                    mysql -h localhost -u root --password=password < $schemaPath
                  """
                } else if (databaseType == "postgres" && schemaPath != ""){
                  sh """
                    # TODO: This needs to be updated
                    psql -U username $databaseName < $schemaPath
                  """
                }
              }
            }
          }
          container('opentest'){
            dir('opentest'){
              script {
                env.imageSha = sh(returnStdout: true, script: """ curl -u $SAASARTIFACTORY_USR:$SAASARTIFACTORY_PSW -sSL -I -H "Accept: application/vnd.docker.distribution.manifest.v2+json" "https://$SaaSartifactoryUrl/artifactory/api/brep-docker-local/v2/$appName/manifests/$appImageTag" | grep -i 'docker-content-digest:' | awk '{print \$2}' """).trim()

                if (fileExists('dependencies.yml')) {
                  dependencies = readYaml file: "dependencies.yml"
                  env.appPort = dependencies.deps.application.port
                } else {
                  ansiColor('xterm') {
                    sh """
                    set +x
                    RED='\033[0;31m'
                    BOLD='\033[1m'
                    NC='\033[0m'
                    echo -e "\${RED}+==================================[ERROR]===================================+\${NC}"
                    echo -e "\${BOLD}opentest/dependencies.yml\${NC} is missing in your repo.\nThis file is necessary to run opentest tests against application.\n"
                    echo -e "For more information about dependencies.yml, please reference: https://us-confluence.mcd.com/display/BREP/%5BBuild+Type%5D+Docker+-+Opentest"
                    echo -e "\${RED}+=============================================================================+\${NC}"
                    """
                    error('Missing opentest/dependencies.yml file')
                  }
                }

                sh """
                  echo "Starting the OpenTest server"
                  nohup opentest server server > opentest_server_logs 2>&1 &
                """
                //Start actors
                def test_actors = findFiles(glob: 'actors/**/*.yaml')
                // Iterate through all actor YAMLs
                for (def file : test_actors) {
                  echo "Found Actor Config: ${file.path}"
                  script {
                    sh "BASE_URL=http://localhost:$appPort nohup opentest actor \$(dirname ${file.path}) &"
                  }
                }
                echo "Waiting for OpenTest server to start..."
                // curl the server until it responds or the timeout is reached
                sh '''
                  ATTEMPTS=2
                  TIMEOUT=60
                  OPENTEST_SERVER_URL="http://localhost:3000"
                  until $(curl --output /dev/null --silent --head --fail $OPENTEST_SERVER_URL) || [[ $ATTEMPTS -gt $TIMEOUT ]]; do
                    printf '.'
                    sleep 1
                    ATTEMPTS=$((ATTEMPTS+1))
                  done
                  if [[ $ATTEMPTS -gt $TIMEOUT ]]; then
                    echo "OpenTest server failed to start:"
                    cat opentest_server_logs
                    exit 1
                  else
                    echo "OpenTest server started successfully"
                  fi
                  opentest session create --server $OPENTEST_SERVER_URL --template AllTests --wait --out $WORKSPACE/junit.xml
                '''
              }
            }
          }
        }

        post {
          always {
            junit 'junit.xml'
            sh '''
              DIR=./allure-results
              ls -ltr
              if [[ -d $DIR ]]; then
                echo "$DIR already exists in workspace"
                echo "Copying Xml test file to current workspace for allure report"
                cp junit.xml ./allure-results
                echo "Successfully copied file , allure report will be generated"
              else
                echo "$DIR does not exists , preparing to create"
                mkdir -v allure-results
                chmod 777 allure-results
                echo "Copying Xml test file to current workspace for allure report"
                cp junit.xml ./allure-results
                echo "Successfully copied file , allure report will be generated"
              fi
            '''
            allure([
              includeProperties: false,
              jdk: '',
              properties: [],
              reportBuildPolicy: 'ALWAYS', //report to be generated irrespective of build status 
              results: [[path: 'allure-results']]
            ])
          }

          failure {
            ansiColor('xterm'){
              sh """
                set +x
                RED='\033[0;31m'
                BOLD='\033[1m'
                NC='\033[0m'
                echo -e "\${RED}+==================================[NOTICE]===================================+\${NC}"
                echo -e "\t\t\${BOLD}$appImageTag\${NC} failed OpenTest!\n\tRemoving bad image \${BOLD}$appImageTag\${NC} tag from artifactory.\n"
                echo -e "\${RED}+=============================================================================+\${NC}"

                curl -u $SAASARTIFACTORY_USR:$SAASARTIFACTORY_PSW -sSL -X DELETE "https://$SaaSartifactoryUrl/artifactory/api/brep-docker-local/v2/$appName/manifests/$imageSha"
              """
            }
          }

          unsuccessful {
            ansiColor('xterm'){
              sh """
                set +x
                RED='\033[0;31m'
                BOLD='\033[1m'
                NC='\033[0m'
                echo -e "\${RED}+==================================[NOTICE]===================================+\${NC}"
                echo -e "\t\t\${BOLD}$appImageTag\${NC} failed OpenTest!\n\tRemoving bad image \${BOLD}$appImageTag\${NC} tag from artifactory.\n"
                echo -e "\${RED}+=============================================================================+\${NC}"

                curl -u $SAASARTIFACTORY_USR:$SAASARTIFACTORY_PSW -sSL -X DELETE "https://$SaaSartifactoryUrl/artifactory/api/brep-docker-local/v2/$appName/manifests/$imageSha"
              """
            }
          }

          aborted {
            ansiColor('xterm'){
              sh """
                set +x
                RED='\033[0;31m'
                BOLD='\033[1m'
                NC='\033[0m'
                echo -e "\${RED}+==================================[NOTICE]===================================+\${NC}"
                echo -e "\t\tBuild of \${BOLD}$appImageTag\${NC} was aborted!\n\tRemoving bad image \${BOLD}$appImageTag\${NC} tag from artifactory.\n"
                echo -e "\${RED}+=============================================================================+\${NC}"

                curl -u $SAASARTIFACTORY_USR:$SAASARTIFACTORY_PSW -sSL -X DELETE "https://$SaaSartifactoryUrl/artifactory/api/brep-docker-local/v2/$appName/manifests/$imageSha"
              """
            }
          }
        }
      }

      stage('Remove Unused Docker Image') {
        steps{
          script {
            container ('build-tools') {
              // This will remove images that were created more than 2hrs ago... So we can keep cache for builds
              sh 'docker system prune -a -f --filter "until=2h"'
            }
          }
        }
      }

      stage('Push SNS') {
        steps {
          script {
            container ('build-tools'){
              // this is needed for a first time build for master branch
              def lastSuccessfulBuildID = 0
              def build = currentBuild.previousBuild
              while (build != null) {
                if (build.result == "SUCCESS"){
                  lastSuccessfulBuildID = build.id as Integer
                  break
                }
                build = build.previousBuild
              }

              if (releaseTag.length() > 0) {
                // Publish new release message
                sh"""
                  aws sts get-caller-identity
                """
                sh"""
                  aws --region us-east-1 sns publish --topic-arn "${releaseSNSArn}" --subject "${appName}: Image Build Complete" --message "Build and Publish completed for ${BUILD_URL}" --message-attributes '{ "componentName":{ "DataType":"String","StringValue":"${appName}" }, "podName":{ "DataType":"String","StringValue":"${appName}"}, "imageURL":{ "DataType":"String","StringValue":"${REGISTRY}"}, "newVersion":{ "DataType":"String","StringValue":"${releaseTag}"}, "replaceVersion":{ "DataType":"String","StringValue":"${replaceVersion}"}, "buildNumber":{ "DataType":"String","StringValue":"${BUILD_NUMBER}"} }'
                """
              } else if (gitBranch.contains("master")) {
                // Publish regular build message
                sh"""
                  aws sts get-caller-identity
                """
                sh"""
                  aws --region us-east-1 sns publish --topic-arn "${testSNSArn}" --subject "${appName}: Image Build Complete" --message "Build and Publish completed for ${BUILD_URL}" --message-attributes '{ "componentName":{ "DataType":"String","StringValue":"${appName}" }, "podName":{ "DataType":"String","StringValue":"${appName}"}, "imageURL":{ "DataType":"String","StringValue":"${REGISTRY}"}, "replaceVersion":{ "DataType":"String","StringValue":"${lastSuccessfulBuildID}"}, "buildNumber":{ "DataType":"String","StringValue":"${BUILD_NUMBER}"} }'
                """
              } else if (gitBranch.contains("feature/") || gitBranch.contains("hotfix/")) {
                // Publish feature & hotfix message to deploy SNS
                formattedGitBranch = gitBranch.replaceAll("/","-").toLowerCase()
                sh"""
                  aws sts get-caller-identity
                """
                sh"""
                  aws --region us-east-1 sns publish --topic-arn "${testSNSArn}" --subject "${appName}: Image Build Complete" --message "Build and Publish completed for ${BUILD_URL}" --message-attributes '{ "componentName":{ "DataType":"String","StringValue":"${appName}" }, "podName":{ "DataType":"String","StringValue":"${appName}"}, "imageURL":{ "DataType":"String","StringValue":"${REGISTRY}"}, "replaceVersion":{ "DataType":"String","StringValue":"${formattedGitBranch}-${lastSuccessfulBuildID}"}, "buildNumber":{ "DataType":"String","StringValue":"${formattedGitBranch}-${BUILD_NUMBER}"} }'
                """
              }
            }
          }
        }
      }
    }
    post('Send Email Alerts') { //Email recipients are harcoded below for now , addions/deletions can be done as per requirment
      always {
           emailext attachLog: true, body: '''User, <br/><br/>

This is an auto-triggered email from Jenkins. Please do not respond to this email . <br/><br/>

$PROJECT_NAME - Build # $BUILD_NUMBER- $BUILD_STATUS :  <br/><br/>

Check console output at $BUILD_URL to view the results and debug . <br/><br/>

For questions, please reach out to the BRE Tech Platform team. <br/><br/>

Thank you !''', compressLog: true, subject: 'Jenkins - $PROJECT_NAME - Build # $BUILD_NUMBER - $BUILD_STATUS!', to: 'animesh.mathur@us.mcd.com , aradhya.asthana@us.mcd.com , reeya.sengupta@us.mcd.com , kshitij.singh@us.mcd.com , premnath.ramanathan@us.mcd.com'
      }
    }
  }
}
