init python:
    from renpy.store import Character
    from enum import Enum

    class CharState(Enum):
        NORMAL = "normal"
        ANGRY = "angry"
        ANGRY_TALK = "angry_talk"
        SAD = "sad"
        SAD_TALK = "sad_talk"
        TALK = "talk"
        AROUSED = "aroused"
        AROUSED_TALK = "aroused_talk"
        HAPPY = "happy"
        HAPPY_TALK = "happy_talk"

    class OutfitState(Enum):
        NORMAL = "normal"
        TORN_CLOTHES = "torn_clothes"
        WHITE_CASUAL = "white_casual"
        # Future outfit expansions can be added here
        # Each character will have their specific 6 outfits as mentioned in the game description

    DEFAULT_EMOTION_FILES = {
        CharState.NORMAL: "normal",
        CharState.ANGRY: "angry",
        CharState.ANGRY_TALK: "angry_talk",
        CharState.SAD: "sad",
        CharState.SAD_TALK: "sad_talk",
        CharState.TALK: "normal_talk",
        CharState.AROUSED: "aroused",
        CharState.AROUSED_TALK: "aroused_talk",
        CharState.HAPPY: "happy",
        CharState.HAPPY_TALK: "happy_talk",
    }

    def create_game_character(
        name,
        cb_name,
        callback=None,
        font="fonts/Lobster-Regular.ttf",
        color=None,
    ):
        """
        Creates and returns a Ren'Py Character object.
        - `name` is what appears on-screen when the character speaks.
        - `cb_name` is an internal short code (used if you have a name_callback).
        - `callback` is optional if you use dynamic naming or other logic.
        - `font` is the font used for this character’s dialogue.
        """
        return Character(
            name,
            cb_name=cb_name,
            callback=callback,
            font=font,
            color=color,
        )

    def build_conditions_for_character(short_name, outfit_state_var, emotion_files, highlight_transform):
        """
        Returns a list of (condition_string, displayable) for the given character.
        `short_name` is the character folder name (e.g. 'bull', 'craig').
        `outfit_state_var` is the variable name for outfit state (e.g. 'bull_outfit').
        `emotion_files` is a dict {CharState.X: "filename"}.
        `highlight_transform` is the transform to apply.
        """
        pairs = []

        for state_enum, filename in emotion_files.items():
            # Create conditions for all outfit states dynamically
            for outfit_state in OutfitState:
                condition = f"char_states['{short_name}'] == CharState.{state_enum.name} and {outfit_state_var} == OutfitState.{outfit_state.name}"
                disp = At(
                    Image(f"images/characters/{short_name}/{outfit_state.value}/{filename}.png"),
                highlight_transform
            )
                pairs.append((condition, disp))

        # Fallback to normal state
        fallback_disp = At(
            Image(f"images/characters/{short_name}/normal/normal.png"),
            highlight_transform
        )
        pairs.append((True, fallback_disp))

        return pairs

    def define_character_images(short_name, outfit_state_var, highlight_transform, emotion_files=DEFAULT_EMOTION_FILES):
        """
        Builds and returns a ConditionSwitch for the given character.
        `short_name` is the character folder name (e.g. 'bull', 'craig').
        `outfit_state_var` is the variable name for outfit state (e.g. 'bull_outfit').
        `emotion_files` is a dict {CharState.X: "filename"}.
        `highlight_transform` is the transform to apply.
        """
        conditions = build_conditions_for_character(short_name, outfit_state_var, emotion_files, highlight_transform)

        flattened_args = []
        for cond, disp in conditions:
            flattened_args.append(cond)
            flattened_args.append(disp)

        return ConditionSwitch(*flattened_args)

    # Note: name_callback and type_sound are defined in 00auto-highlight.rpy
    # No need to redefine them here


define v = create_game_character(
    name="Victor",
    cb_name="victor",
    callback=name_callback,
    color="#4169E1",
)

define scarlett = create_game_character(
    name="Scarlett",
    cb_name="scarlett",
    color="#FF1493",
    callback=name_callback,
)

define swift = create_game_character(
    name="Swift",
    cb_name="swift",
    color="#DC143C",
    callback=name_callback,
)

define azami = create_game_character(
    name="Azami",
    cb_name="azami",
    color="#FF8C00",
    callback=name_callback,
)

define adriana = create_game_character(
    name="Adriana",
    cb_name="adriana",
    color="#00CED1",
    callback=name_callback,
)

define kira = create_game_character(
    name="Kira",
    cb_name="kira",
    color="#9370DB",
    callback=name_callback,
)

define sophia = create_game_character(
    name="Sophia",
    cb_name="sophia",
    color="#4682B4",
    callback=name_callback,
)

define mayor = create_game_character(
    name="Mayor Richardson",
    cb_name="mayor",
    color="#8B4513",
    callback=name_callback,
)

define maya = create_game_character(
    name="Maya",
    cb_name="maya",
    color="#FF69B4",
    callback=name_callback,
)

define jake = create_game_character(
    name="Jake",
    cb_name="jake",
    color="#1E90FF",
    callback=name_callback,
)

image victor = define_character_images(
    short_name="victor",
    outfit_state_var="outfit_states['victor']",
    highlight_transform=sprite_highlight("victor"),
    emotion_files={
        CharState.NORMAL: "normal",
        CharState.TALK: "normal_talk",
        CharState.ANGRY: "angry",
        CharState.SAD: "sad",
        CharState.HAPPY: "happy",
        CharState.AROUSED: "aroused",
        CharState.ANGRY_TALK: "angry_talk",
        CharState.SAD_TALK: "sad_talk",
        CharState.HAPPY_TALK: "happy_talk",
        CharState.AROUSED_TALK: "aroused_talk"
    }
)

image scarlett = define_character_images(
    short_name="scarlett",
    outfit_state_var="outfit_states['scarlett']",
    highlight_transform=sprite_highlight("scarlett"),
    emotion_files={
        CharState.NORMAL: "normal",
        CharState.TALK: "normal_talk",
        CharState.ANGRY: "angry",
        CharState.SAD: "sad",
        CharState.HAPPY: "happy",
        CharState.AROUSED: "aroused",
        CharState.ANGRY_TALK: "angry_talk",
        CharState.SAD_TALK: "sad_talk",
        CharState.HAPPY_TALK: "happy_talk",
        CharState.AROUSED_TALK: "aroused_talk"
    }
)

image mayor = define_character_images(
    short_name="mayor",
    outfit_state_var="outfit_states['mayor']",
    highlight_transform=sprite_highlight("mayor"),
    emotion_files={
        CharState.NORMAL: "normal",
        CharState.TALK: "normal_talk",
        CharState.ANGRY: "angry",
        CharState.SAD: "sad",
        CharState.HAPPY: "happy",
        CharState.AROUSED: "aroused",
        CharState.ANGRY_TALK: "angry_talk",
        CharState.SAD_TALK: "sad_talk",
        CharState.HAPPY_TALK: "happy_talk",
        CharState.AROUSED_TALK: "aroused_talk"
    }
)

image swift = define_character_images(
    short_name="swift",
    outfit_state_var="outfit_states['swift']",
    highlight_transform=sprite_highlight("swift"),
    emotion_files={
        CharState.NORMAL: "normal",
        CharState.TALK: "normal_talk",
        CharState.ANGRY: "angry",
        CharState.SAD: "sad",
        CharState.HAPPY: "happy",
        CharState.AROUSED: "aroused",
        CharState.ANGRY_TALK: "angry_talk",
        CharState.SAD_TALK: "sad_talk",
        CharState.HAPPY_TALK: "happy_talk",
        CharState.AROUSED_TALK: "aroused_talk"
    }
)

# For now, other characters use simple image definitions until their expression folders are created
# These can be expanded later with define_character_images when outfit systems are implemented