//================================================================================
/**
 * @file   XyNetVct.cpp
 * @brief  {CX`bgCu
 * @author k.ohno
 * @date   2012.10.07
 */
//================================================================================
#include <nex.h>
#include <NetZ/src/Core/DOCore/StationManager.h>


#include "net/vct/XyNetVct.h"
#include "net/vct/sndmic.h"

GFL_NAMESPACE_BEGIN(xynet)
GFL_NAMESPACE_BEGIN(vct)


u8* XyNetVct::mpSendTempBuffer=NULL;


/*---------------------------------------------------------------------------*
  Name        : EventCallback
  Description : event callback function.
  Arguments   : cid      - src connectionID.
              : result   - event result.
              : *session - session.
              : *param   - parameter.
  Returns     : -
 *---------------------------------------------------------------------------*/
void XyNetVct::EventCallback( u16 cid, VCTEvent result, VCTSession *session, void *param )
{
  NN_UNUSED_VAR( param );

  switch ( result )
    {
    case VCT_EVENT_DISCONNECTED:
      NN_LOG( "VCT disconnected event(%d).\n", cid );
      VCT_StopStreaming( session );
      if ( VCT_DeleteSession( session ) == false )
        {
          NN_LOG( "fail delete session.\n" );
        }
      break;
    case VCT_EVENT_CONNECTED:
      NN_LOG( "VCT connected event(%d).\n", cid );
      if ( VCT_StartStreaming( session ) == false )
        {
          NN_LOG( "fail start streming.\n" );
        }
      break;
    case VCT_EVENT_ABORT:
      NN_LOG( "VCT abort event.\n" );
      break;
    default:
      break;
    }
}


/*---------------------------------------------------------------------------*
  Name        : SendDataCallBack
  Description : send data callback function.
  Arguments   : cid        - destination ConnecntionId.
              : *buff      - send data.
              : size       - send data size.
              : isReliable - (true)reliable, (false)unreliable
  Returns     : true  - succsess.
              : false - fail.
 *---------------------------------------------------------------------------*/
bool XyNetVct::SendDataCallBack( u16 cid, const void *buff, u32 size, bool isReliable )
{
  //@todo 2012.10.07   ʐMSendFunction
  bool ret = false;

  int headsize = sizeof(gflnet::p2p::SendHead);
  u16* pBuff16 = reinterpret_cast<u16*>(mpSendTempBuffer);
  gflnet::p2p::SendHead* pSend = reinterpret_cast<gflnet::p2p::SendHead*>(mpSendTempBuffer);

  GFL_ASSERT(size < LIMITSEND_BUFFER);

  memset(mpSendTempBuffer, 0 ,MAXSEND_BUFFER);
  memcpy(&mpSendTempBuffer[SENDHEAD_SIZE],buff,size);
  pSend->size = size;  //TCY
  pSend->netver  = GFL_P2P_NETVERSION;
  pSend->type  = GFLNET_TYPE_VCT;
  pSend->CRC = gfl::math::Crc::Calc(&pBuff16[1], size + headsize - 2 ); //CRC 

  nn::nex::DirectTransport* pD = nn::nex::DirectTransport::GetInstance();
  
  if(isReliable){
    pD->SendReliable(cid,reinterpret_cast<nn::nex::qByte*>(pBuff16),size + headsize);
  }
  else{
    pD->SendUnreliable(cid,reinterpret_cast<nn::nex::qByte*>(mpSendTempBuffer),size + headsize);
  }
  return ret;
}





XyNetVct::XyNetVct(void)
{
  mbStart=false;
}



XyNetVct::~XyNetVct()
{
}


void XyNetVct::Initialize(gfl::heap::HeapBase* pBase)
{

  _net_info *netInfo = &mNetInfo;
  u32                       vpos           = 0;
  u32                       diplay_number  = 0;
  u32                       audio_info_cnt = 0;



  /* Initialize global... */
  //  p_NetInfra     = global->system.netInfra;
  p_SndMicDevice = GFL_NEW( pBase ) SndMicDevice();
  s_MyCID        = netInfo->my_cid;
  s_VADValue     = 255;

  mpAudioBuffer = (u8*)GflHeapAllocMemoryAlign( pBase,VCT_DEFAULT_AUDIO_BUFFER_SIZE, 32 );
  mpSendTempBuffer = (u8*)GflHeapAllocMemoryAlign( pBase,MAXSEND_BUFFER, 32 );

  
  /* Initialize VCT */
  s_vctConfig.session         = s_session;
  s_vctConfig.numSession      = VCT_MAX_SESSION;
  s_vctConfig.mode            = VCT_MODE_CONFERENCE;
  s_vctConfig.ecMode          = VCT_EC_STANDARD;
  s_vctConfig.cid             = (u16)( s_MyCID & 0xFFFF );
  s_vctConfig.codec           = VCT_CODEC_4BIT_ADPCM;
  s_vctConfig.frameLength     = VCT_AUDIO_FRAME_LENGTH_DEFAULT;
  s_vctConfig.flag            = VCT_CONFIG_FLAG_DEFAULT;
  s_vctConfig.audioBuffer     = mpAudioBuffer;
  s_vctConfig.audioBufferSize = VCT_DEFAULT_AUDIO_BUFFER_SIZE;
  s_vctConfig.callback        = EventCallback;
  s_vctConfig.sendCallback    = SendDataCallBack;
  s_vctConfig.userData        = NULL;

  if ( VCT_Init( &s_vctConfig ) == false ) {
    NN_LOG( "Error:VCT_Init()\n" );
    return;
  }

  /* Set VAD Active gain... */
  VCT_SetVADActiveGain( 350 );

  /* Enable Echo Cancel... */
  VCT_EnableEchoCancel( true );
  mbStart= true;

  {
    _sndmic_config          sndmicConfig;

    /* Set snd/mic config... */
    sndmicConfig.frame_msec = VCT_GetAudioFrameLength();
    sndmicConfig.mic_gain   = 60;
    
    /* Initialize snd/mic... */
    if ( p_SndMicDevice->Initialize( pBase, NULL, &sndmicConfig ) != SNDMIC_ERROR_NONE ){
      NN_LOG( "Error:Initialize snd or mic.\n" );
    }
  }
  
  mpVCTRecvDataListener = GFL_NEW(pBase) VCTRecvDataListener;

}

void XyNetVct::Finalize(void)
{
  VCT_Cleanup();

  if(p_SndMicDevice->Finalize()!=SNDMIC_ERROR_NONE){
    NN_LOG( "Error:p_SndMicDevice()\n" );
    return;
  }
  GFL_DELETE p_SndMicDevice;
  p_SndMicDevice=NULL;

  GFL_DELETE mpVCTRecvDataListener;
  GflHeapFreeMemory(mpAudioBuffer);
  mpAudioBuffer=NULL;
  mbStart= false;
}


bool XyNetVct::Start(void)
{
  for(int i=0;i<VCT_MAX_CONNECT;i++){
    u32 no = mNetInfo.CIDList[i];
    if(no==0){
      continue;
    }
    if(VCT_OnConferenceClient( no )){
      continue;
    }
    if ( VCT_AddConferenceClient(no) != VCT_ERROR_NONE ){
      NN_LOG( "Fail Add Client (%ld).\n", no );
      return false;
    }
  }
  return true;
}




bool XyNetVct::StartConference(void)
{
  /* Start snd/mic... */
  if ( p_SndMicDevice->Start() != SNDMIC_ERROR_NONE ) {
    NN_LOG( "Error:Start snd or mic.\n" );
    return false;
  }
  return true;
}



void XyNetVct::Pause(void)
{
}




bool XyNetVct::StopConference(void)
{
  /* Start snd/mic... */
  if ( p_SndMicDevice->Stop() != SNDMIC_ERROR_NONE ) {
    NN_LOG( "Error:Stop snd or mic.\n" );
    return false;
  }
  return true;
}



bool XyNetVct::End(void)
{
  for(int i=0;i<VCT_MAX_CONNECT;i++){
    u32 no = mNetInfo.CIDList[i];
    if(no==0){
      continue;
    }
    if(!VCT_OnConferenceClient( no )){
      continue;
    }
    if ( VCT_RemoveConferenceClient(no) != VCT_ERROR_NONE ){
      NN_LOG( "Fail Remove Client (%ld).\n", no );
      return false;
    }
    
  }
  return true;
}


void XyNetVct::Main(void)
{
  if(mbStart){
    VCT_Main();
  }
}


void XyNetVct::SignalNoiseSkip(void)
{
  VCT_SignalNoiseSkip();
}


void XyNetVct::SetCID(int index,u32 cid)
{
  mNetInfo.CIDList[index] = cid;
}





GFL_NAMESPACE_END(vct)
GFL_NAMESPACE_END(xynet)

