/*-----------------------------------------------------------------------*
Project: Voice Chat Sample.
File   : sndmic.cpp

Copyright (C)2011 Nintendo. All rights reserved.

These coded instructions, statements, and computer programs contain proprietary
information of Nintendo and/or its licensed developers and are protected by
national and international copyright laws. They may not be disclosed to third
parties or copied or duplicated in any form, in whole or in part, without the
prior written consent of Nintendo.

The content herein is highly confidential and should be handled accordingly.
*----------------------------------------------------------------------------*/

/*---------------------------------------------------------------------------*
    Include
 *---------------------------------------------------------------------------*/
#include "sndmic.h"


GFL_NAMESPACE_BEGIN(xynet)
GFL_NAMESPACE_BEGIN(vct)


/*---------------------------------------------------------------------------*
    Define
 *---------------------------------------------------------------------------*/
#define MIC_CAPTURE_MARGIN_MS     (12)
#define MIC_CAPTURE_MARGIN_SAMPLE (MIC_CAPTURE_MARGIN_MS * VCT_AUDIO_DATA_SIZE_PER_MILLI_SEC / 2)


/*---------------------------------------------------------------------------*
    Prototype
 *---------------------------------------------------------------------------*/
void SoundThreadFunc( SndMicDevice* sndmicDev );
void FrameProcessThreadFunc( SndMicDevice* sndmicDev );


/*---------------------------------------------------------------------------*
    Function
 *---------------------------------------------------------------------------*/
/*---------------------------------------------------------------------------*
  Name        : Initialize
  Description : Initialize Snd & Mic Device
  Arguments   : pAppHeap     - Heap buffer pointer.
              : *config      - sndmic config parameter.
  Returns     : SNDMIC_ERROR_STATUS
              :   SNDMIC_ERROR_NONE  - none error.
              :   SNDMIC_ERROR_PARAM - parameter error.
              :   SND_ERROR_DSP      - dsp error.
              :   MIC_ERROR_LIB      - mic library error.
              :   SND_ERROR_LIB      - snd library error.
 *---------------------------------------------------------------------------*/
SNDMIC_ERROR_STATUS SndMicDevice::Initialize( gfl::heap::HeapBase* pBase, gfl::heap::HeapBase *pDev,  _sndmic_config *config )
{
    nn::Result          Result;
    nn::snd::MixParam   Mix;
    nn::fs::FileReader  fileReader;
    WaveFmt             fmt;

    /* Check argument */
    if (( pAppHeap == NULL ) || ( config == NULL ))
    {
        return SNDMIC_ERROR_PARAM;
    }

    /* Heap Pointer̃Rs[ */
  mpBaseHeap = pBase;
  mpDevHeap = pDev;

    /* Config parameter̃Rs[ */
    m_Config.frame_msec     = config->frame_msec;
    m_Config.mic_gain      = config->mic_gain;

    /* frame size... */
    m_AudioFrameSize   = m_Config.frame_msec * 8 * sizeof(s16);

    if ( m_AudioFrameSize < 1056 )
    {
        m_SoundBufferSize  = 1056;
    }
    else
    {
        m_SoundBufferSize  = m_AudioFrameSize;
    }

    /* Initialize variables... */
    m_SoundStatus              = SND_PROCESS_NONE;
    m_CurrBufferIdx            = 0;
    m_SoundBufferReqIdx        = 0;
    m_MicTimingError           = false;

    m_MicStatus                = MIC_PROCESS_NONE;

    m_MicEvent.Initialize(false);
    m_MicEvent.ClearSignal();


    /**********   MIC   **********/
    if ( nn::mic::Initialize().IsFailure() )
    {
        return MIC_ERROR_LIB;
    }

    /* MIC-Buffer == 4096*2byte */
    if ( nn::mic::SetBuffer(m_MicBuffer, sizeof(m_MicBuffer)).IsFailure() )
    {
        return MIC_ERROR_LIB;
    }

    /* Set MIC-Amp. */
    if ( nn::mic::GetAmpGain(&m_MicGain).IsFailure() )
    {
        return MIC_ERROR_LIB;
    }

    if ( nn::mic::SetAmpGain(m_Config.mic_gain).IsFailure() )
    {
        return MIC_ERROR_LIB;
    }

    /* Power On MIC. */
    if ( nn::mic::SetAmp( true ).IsFailure() )
    {
        return MIC_ERROR_LIB;
    }

    /***** Update process status *****/
    m_MicStatus = MIC_PROCESS_INIT;


    /**********   SND   **********/
    /* Initlaize DSP + Load DSP Component */
    Result = nn::dsp::Initialize();
    if ( Result.IsFailure() )
    {
        return SND_ERROR_DSP;
    }

    Result = nn::dsp::LoadDefaultComponent();
    if ( Result.IsFailure() )
    {
        return SND_ERROR_DSP;
    }

    Result = nn::snd::Initialize();
    if ( Result.IsFailure() )
    {
        return SND_ERROR_LIB;
    }


    /* Sound Buffeȑ */
    for ( int Idx = 0; Idx < SOUND_BUFFER_NUM; Idx++ )
    {
        /* SoundBufferm(Programing Manual̋LqASoundBuffer1025Byteȏ) */
        m_SoundBuffer[Idx] = reinterpret_cast<u8*>( m_pAppHeap->Allocate( m_SoundBufferSize, 32 ) );
        std::memset( m_SoundBuffer[Idx], 0x0, m_SoundBufferSize );
        nn::snd::FlushDataCache( reinterpret_cast<uptr>( m_SoundBuffer[Idx] ), m_SoundBufferSize );
    }

    /* WaveBuffer ̏ */
    for ( int Idx = 0; Idx < SOUND_BUFFER_NUM; Idx++ )
    {
        /* WaveBufferւ̓o^ */
        nn::snd::InitializeWaveBuffer( &m_WaveBuffer[Idx] );
        m_WaveBuffer[Idx].bufferAddress = reinterpret_cast<void*>( m_SoundBuffer[Idx] );
        m_WaveBuffer[Idx].sampleLength  = nn::snd::GetSampleLength( m_AudioFrameSize, nn::snd::SAMPLE_FORMAT_PCM16, 1 );
        m_WaveBuffer[Idx].loopFlag      = false;
    }

    /* {[ Mix ̏ */
    Mix.mainBus[nn::snd::CHANNEL_INDEX_FRONT_LEFT]  = 1.0f;
    Mix.mainBus[nn::snd::CHANNEL_INDEX_FRONT_RIGHT] = 1.0f;

    /* Voice ̏ */
    m_pVoice = nn::snd::AllocVoice( 128, NULL, NULL );
    m_pVoice->SetSampleRate( SOUND_FREQ_8KHZ );                /* TvO[g̐ݒ(8kHz)  */
    m_pVoice->SetSampleFormat( nn::snd::SAMPLE_FORMAT_PCM16 ); /* tH[}bgw(PCM16)       */
    m_pVoice->SetChannelCount( 1 );                            /* `lݒ(MONO)          */
    m_pVoice->SetPitch( 1.0f );                                /* sb`̐ݒ                    */
//    m_pVoice->SetVolume( 10.0f );                              /* {[̐ݒ                */
    m_pVoice->SetVolume( 1.0f );                               /* {[̐ݒ                */
    m_pVoice->SetMixParam( Mix );                              /* {[ Mix ̐ݒ           */
    for ( int Idx = 0; Idx < ADD_BUFFER_NUM; Idx++ )
    {
        m_pVoice->AppendWaveBuffer( &m_WaveBuffer[Idx] );      /* obt@̒ǉ                  */
    }

/* Back Music ̏ */
    /* Back Musicf[^Ǎ */
    Result = fileReader.TryInitialize( ROMFILE_WAVE_2 );
    NN_UTIL_PANIC_IF_FAILED( Result );
    m_pWaveMemory[2] = reinterpret_cast<u8*>( m_pAppHeap->Allocate( (GetWaveLength( fileReader ) + m_AudioFrameSize), 32 ) );
    if ( m_pWaveMemory[2] == NULL )
    {
        NN_LOG( "Error:Allocate memory error.\n" );
        return SNDMIC_ERROR_NONE;
    }
    m_WaveData[2].buf = m_pWaveMemory[2];
    LoadWave( fileReader, &fmt, &m_WaveData[2] );
    fileReader.Finalize();

    nn::snd::FlushDataCache( reinterpret_cast<uptr>( m_WaveData[2].buf ), m_WaveData[2].size );

    /* WaveBufferւ̓o^ */
    nn::snd::InitializeWaveBuffer( &m_MusicWaveBuffer );
    m_MusicWaveBuffer.bufferAddress = reinterpret_cast<void*>( m_WaveData[2].buf );
    m_MusicWaveBuffer.sampleLength  = nn::snd::GetSampleLength( m_WaveData[2].size, nn::snd::SAMPLE_FORMAT_PCM16, 1 );
    m_MusicWaveBuffer.loopFlag      = true;

    /* Back Musicp Voice ̏ */
    m_pMusic = nn::snd::AllocVoice( 128, NULL, NULL );
    m_pMusic->SetSampleRate( SOUND_FREQ_8KHZ );                /* TvO[g̐ݒ(8kHz)  */
    m_pMusic->SetSampleFormat( nn::snd::SAMPLE_FORMAT_PCM16 ); /* tH[}bgw(PCM16)       */
    m_pMusic->SetChannelCount( 1 );                            /* `lݒ(MONO)        */
    m_pMusic->SetPitch( 1.0f );                                /* sb`̐ݒ                    */
    m_pMusic->SetVolume( 0.2f );                               /* {[̐ݒ                */
    m_pMusic->SetMixParam( Mix );                              /* {[ Mix ̐ݒ           */
    m_pMusic->AppendWaveBuffer( &m_MusicWaveBuffer );          /* obt@̒ǉ                  */

    /* }X^[{[ݒ */
    nn::snd::SetMasterVolume( 1.0f );

    /* NOISEf[^Ǎ */
    Result = fileReader.TryInitialize( ROMFILE_WAVE_0 );
    NN_UTIL_PANIC_IF_FAILED( Result );
    m_pWaveMemory[0] = reinterpret_cast<u8*>( m_pAppHeap->Allocate( (GetWaveLength( fileReader ) + m_AudioFrameSize), 32 ) );
    if ( m_pWaveMemory[0] == NULL )
    {
        NN_LOG( "Error:Allocate memory error.\n" );
        return SNDMIC_ERROR_NONE;
    }
    m_WaveData[0].buf = m_pWaveMemory[0];
    LoadWave( fileReader, &fmt, &m_WaveData[0] );
    fileReader.Finalize();
    m_FrameCount[0] = (u32)( (m_WaveData[0].size + m_AudioFrameSize - 1) / m_AudioFrameSize );

    /* BUSYf[^Ǎ */
    Result = fileReader.TryInitialize( ROMFILE_WAVE_1 );
    NN_UTIL_PANIC_IF_FAILED( Result );
    m_pWaveMemory[1] = reinterpret_cast<u8*>( m_pAppHeap->Allocate( (GetWaveLength( fileReader ) + m_AudioFrameSize), 32 ) );
    if ( m_pWaveMemory[1] == NULL )
    {
        NN_LOG( "Error:Allocate memory error.\n" );
        return SNDMIC_ERROR_NONE;
    }
    m_WaveData[1].buf = m_pWaveMemory[1];
    LoadWave( fileReader, &fmt, &m_WaveData[1] );
    fileReader.Finalize();
    m_FrameCount[1] = (u32)( (m_WaveData[1].size + m_AudioFrameSize - 1) / m_AudioFrameSize );

    m_FramePlayReq = false;

    /***** Update process status *****/
    m_SoundStatus = SND_PROCESS_INIT;


    /**********   Xbho^   **********/
    /* DSPR[obN֐o^ */
    m_SoundThread.StartUsingAutoStack( SoundThreadFunc, this, (int)4096, (int)2 );

    /* t[f[^Xbho^ */
    m_FrameProcessThread.StartUsingAutoStack( FrameProcessThreadFunc, this, (int)4096, (int)(NN_OS_DEFAULT_THREAD_PRIORITY - 2));



    return SNDMIC_ERROR_NONE;
}


/*---------------------------------------------------------------------------*
  Name        : Start
  Description : Start playing stream.
  Arguments   : -
  Returns     : SNDMIC_ERROR_STATUS
              :   SNDMIC_ERROR_NONE  - none error.
              :   SND_ERROR_SEQ   - snd sequence error.
 *---------------------------------------------------------------------------*/
SNDMIC_ERROR_STATUS SndMicDevice::Start(void)
{
    nn::Result Result;

    /* MICFrameProcessThreadŃX^[g */
    /* STOP=>START̂ƂɃX^[g^C~O킹邽߃Xe[^XύX */
    if( m_MicStatus == MIC_PROCESS_STOP )
    {
        m_MicStatus = MIC_PROCESS_INIT;
    }


    /**********   SNDX^[g   **********/
    /* Check process status */
    if ( ( m_SoundStatus != SND_PROCESS_INIT ) && ( m_SoundStatus != SND_PROCESS_STOP ) )
    {
        return SND_ERROR_SEQ;
    }

    /* ĐJn */
    m_pVoice->SetState( nn::snd::Voice::STATE_PLAY );

    /* Update process status */
    m_SoundStatus = SND_PROCESS_PLAY;

    return SNDMIC_ERROR_NONE;
}


/*---------------------------------------------------------------------------*
  Name        : Stop
  Description : Stop playing stream.
  Arguments   : -
  Returns     : SNDMIC_ERROR_STATUS
              :   SNDMIC_ERROR_NONE  - none error.
              :   SND_ERROR_SEQ   - sequence error.
 *---------------------------------------------------------------------------*/
SNDMIC_ERROR_STATUS SndMicDevice::Stop(void)
{

    /**********   MICXgbv   **********/
    /* Check process status */
    if ( m_MicStatus != MIC_PROCESS_START )
    {
        return MIC_ERROR_SEQ;
    }

    /* Stop sampling. */
    if ( nn::mic::StopSampling().IsFailure() )
    {
        return MIC_ERROR_LIB;
    }

    /* Update process status */
    m_MicStatus = MIC_PROCESS_STOP;


    /**********   SNDXgbv   **********/
    /* Check process status */
    if ( m_SoundStatus != SND_PROCESS_PLAY )
    {
        return SND_ERROR_SEQ;
    }

    /* ĐI */
    if ( m_pVoice->GetState() == nn::snd::Voice::STATE_PLAY )
    {
        m_pVoice->SetState( nn::snd::Voice::STATE_STOP );
    }

    /* Update process status */
    m_SoundStatus = SND_PROCESS_STOP;


    return SNDMIC_ERROR_NONE;
}


/*---------------------------------------------------------------------------*
  Name        : SetVolume
  Description : Set voice volume.
  Arguments   : volume - value of volume.
  Returns     : SNDMIC_ERROR_STATUS
              :   SNDMIC_ERROR_NONE - none error.
              :   SND_ERROR_SEQ     - snd sequence error.
 *---------------------------------------------------------------------------*/
SNDMIC_ERROR_STATUS SndMicDevice::SetVolume( f32 volume )
{
    nn::Result Result;

    /* Check process status */
    if ( m_SoundStatus != SND_PROCESS_PLAY )
    {
        return SND_ERROR_SEQ;
    }

    /* Set Volume... */
    if ( m_pVoice != NULL )
    {
        m_pVoice->SetVolume( volume );
    }

    return SNDMIC_ERROR_NONE;
}


/*---------------------------------------------------------------------------*
  Name        : SetBGMVolume
  Description : Set BGM volume.
  Arguments   : volume - value of BGM volume.
  Returns     : SNDMIC_ERROR_STATUS
              :   SNDMIC_ERROR_NONE - none error.
              :   SND_ERROR_SEQ     - snd sequence error.
 *---------------------------------------------------------------------------*/
SNDMIC_ERROR_STATUS SndMicDevice::SetBGMVolume( f32 volume )
{
    nn::Result Result;

    /* Check process status */
    if ( m_SoundStatus != SND_PROCESS_PLAY )
    {
        return SND_ERROR_SEQ;
    }

    /* Set Volume... */
    if ( m_pMusic != NULL )
    {
        m_pMusic->SetVolume( volume );
    }

    return SNDMIC_ERROR_NONE;
}


/*---------------------------------------------------------------------------*
  Name        : StartBGM
  Description : Start playing BGM.
  Arguments   : -
  Returns     : SNDMIC_ERROR_STATUS
              :   SNDMIC_ERROR_NONE - none error.
              :   SND_ERROR_SEQ     - snd sequence error.
 *---------------------------------------------------------------------------*/
SNDMIC_ERROR_STATUS SndMicDevice::StartBGM( void )
{
    nn::Result Result;

    /* Check process status */
    if ( m_SoundStatus != SND_PROCESS_PLAY )
    {
        return SND_ERROR_SEQ;
    }

    /* Back MusicĐJn */
    m_pMusic->SetState( nn::snd::Voice::STATE_PLAY );

    return SNDMIC_ERROR_NONE;
}


/*---------------------------------------------------------------------------*
  Name        : StopBGM
  Description : Stop playing BGM.
  Arguments   : -
  Returns     : SNDMIC_ERROR_STATUS
              :   SNDMIC_ERROR_NONE - none error.
              :   SND_ERROR_SEQ     - snd sequence error.
 *---------------------------------------------------------------------------*/
SNDMIC_ERROR_STATUS SndMicDevice::StopBGM( void )
{
    nn::Result Result;

    /* Check process status */
    if ( m_SoundStatus != SND_PROCESS_PLAY )
    {
        return SND_ERROR_SEQ;
    }

    /* Back MusicĐI */
    if ( m_pMusic->GetState() == nn::snd::Voice::STATE_PLAY )
    {
        m_pMusic->SetState( nn::snd::Voice::STATE_PAUSE );
    }

    return SNDMIC_ERROR_NONE;
}


/*---------------------------------------------------------------------------*
  Name        : Finalize
  Description : Finalize Snd & Mic Device
  Arguments   : -
  Returns     : SNDMIC_ERROR_STATUS
              :   SNDMIC_ERROR_NONE  - none error.
              :   SND_ERROR_SEQ   - snd sequence error.
              :   SNDMIC_ERROR_PARAM - parameter error.
              :   SND_ERROR_DSP   - dsp error.
              :   SND_ERROR_LIB   - snd library error.
              :   MIC_ERROR_SEQ   - mic sequence error.
              :   MIC_ERROR_LIB   - mic library error.
 *---------------------------------------------------------------------------*/
SNDMIC_ERROR_STATUS SndMicDevice::Finalize( void )
{
    nn::Result  Result;

    /**********   MICI   **********/
    /* Check process status */
    if ( ( m_MicStatus != MIC_PROCESS_STOP ) && ( m_MicStatus != MIC_PROCESS_INIT ) )
    {
        return MIC_ERROR_SEQ;
    }

    /* Power off mic. */
    if ( nn::mic::SetAmp(false).IsFailure() )
    {
        return MIC_ERROR_LIB;
    }

    /* Return mic-amp setting. */
    if ( nn::mic::SetAmpGain(m_MicGain).IsFailure() )
    {
        return MIC_ERROR_LIB;
    }

    /* Free MIC-Buffer */
    if ( nn::mic::ResetBuffer().IsFailure() )
    {
        return MIC_ERROR_LIB;
    }

    /* Finalize MIC */
    if ( nn::mic::Finalize().IsFailure() )
    {
        return MIC_ERROR_LIB;
    }

    /* Update process status */
    m_MicStatus = MIC_PROCESS_NONE;

    /**********   t[f[^XbhI   **********/
    m_FrameProcessThread.Join();
    m_FrameProcessThread.Finalize();


    /**********   SNDI   **********/
    /* Check process status */
    if ( ( m_SoundStatus != SND_PROCESS_STOP ) && ( m_SoundStatus != SND_PROCESS_INIT ) )
    {
        return SND_ERROR_SEQ;
    }

    /* Update process status */
    m_SoundStatus = SND_PROCESS_NONE;

    /* TEhXbh̔j */
    m_SoundThread.Join();
    m_SoundThread.Finalize();

    /* Voice̔j */
    nn::snd::FreeVoice( m_pVoice );
    nn::snd::FreeVoice( m_pMusic );

    /* TEhobt@̔j */
    for ( int Idx = 0; Idx < SOUND_BUFFER_NUM; Idx++ )
    {
        m_pAppHeap->Free( m_SoundBuffer[Idx] );
    }

    /* SND ̏I */
    Result = nn::snd::Finalize();
    if ( Result.IsFailure() )
    {
        return SND_ERROR_LIB;
    }

    /* DSP ̏I */
    Result = nn::dsp::UnloadComponent();
    if ( Result.IsFailure() )
    {
        return SND_ERROR_DSP;
    }
    nn::dsp::Finalize();


    if ( m_pWaveMemory[0] != NULL )
    {
        m_pAppHeap->Free( m_pWaveMemory[0] );
        m_pWaveMemory[0] = NULL;
    }

    if ( m_pWaveMemory[1] != NULL )
    {
        m_pAppHeap->Free( m_pWaveMemory[1] );
        m_pWaveMemory[1] = NULL;
    }

    if ( m_pWaveMemory[2] != NULL )
    {
        m_pAppHeap->Free( m_pWaveMemory[2] );
        m_pWaveMemory[2] = NULL;
    }



    m_MicEvent.Finalize();

    return SNDMIC_ERROR_NONE;
}


/*---------------------------------------------------------------------------*
  Name        : SoundThreadFuncImpl
  Description : Sound Thread function for send Parameter to DSP.(Core)
  Arguments   : -
  Returns     : -
 *---------------------------------------------------------------------------*/
void SndMicDevice::SoundThreadFuncImpl( void )
{
    bool store_req = false;
    while ( true )
    {
        /* DSP ̃f[^M҂*/
        nn::snd::WaitForDspSync();

        if ( m_SoundStatus == SND_PROCESS_NONE )
        {
            break;
        }

        if( m_SoundStatus == SND_PROCESS_PLAY )
        {

            /* SNDMICȂׂ̃^C~OŃX^[g邽߂ɂMICɃgK */
            if ( m_MicStatus == MIC_PROCESS_INIT )
            {
                m_MicEvent.Signal();
            }

            /* WaveBufferXe[^X̊mF(Đ -> Bufferǉ) */
            if ( ( m_pVoice != NULL ) &&
                 ( m_WaveBuffer[m_CurrBufferIdx].status == nn::snd::WaveBuffer::STATUS_DONE ) &&
                 ( m_WaveBuffer[( m_CurrBufferIdx + ADD_BUFFER_NUM ) % SOUND_BUFFER_NUM].status == nn::snd::WaveBuffer::STATUS_FREE ))
            {
                /* Buffer̍Ēǉ */
                m_pVoice->AppendWaveBuffer( &m_WaveBuffer[( m_CurrBufferIdx + ADD_BUFFER_NUM ) % SOUND_BUFFER_NUM] );
                store_req = true;
                m_SoundBufferReqIdx = m_CurrBufferIdx;
                /* Curr Buffer Indx̍XV */
                m_CurrBufferIdx = ( m_CurrBufferIdx + 1 ) % SOUND_BUFFER_NUM;
            }

            if(store_req == true)
            {
                /* Đf[^擾 */
                if(GetReceiveAudioData(m_SoundBuffer[m_SoundBufferReqIdx], m_AudioFrameSize))
                {
                    if ( m_FramePlayReq == true )
                    {
                        if ( m_FramePlayNum < m_FrameCount[m_WaveNumbber] )
                        {
                            std::memcpy( m_SoundBuffer[m_SoundBufferReqIdx], m_pPlayWave, m_AudioFrameSize );
                            m_FramePlayNum++;
                            m_pPlayWave += m_AudioFrameSize;
                        }
                        else
                        {
                            m_FramePlayReq = false;
                        }
                    }

                    /* Flush Cache */
                    nn::snd::FlushDataCache( reinterpret_cast<uptr>( m_SoundBuffer[m_SoundBufferReqIdx] ), m_AudioFrameSize );

                    /* Buffer̍Ēǉ */
                    nn::snd::InitializeWaveBuffer( &m_WaveBuffer[m_SoundBufferReqIdx] );
                    m_WaveBuffer[m_SoundBufferReqIdx].bufferAddress =
                                                      reinterpret_cast<void*>( m_SoundBuffer[m_SoundBufferReqIdx] );
                    m_WaveBuffer[m_SoundBufferReqIdx].sampleLength  =
                                                      nn::snd::GetSampleLength( m_AudioFrameSize, nn::snd::SAMPLE_FORMAT_PCM16, 1 );
                    m_WaveBuffer[m_SoundBufferReqIdx].loopFlag      = false;

                    store_req = false;
                }
            }

                /* Send reference mixed data... */
            if ( nn::snd::GetMixedBusData( m_MixedData, NN_SND_SAMPLES_PER_FRAME ) == true )
            {
                VCT_HandleReference( (u8 *)m_MixedData, NN_SND_SAMPLES_PER_FRAME*2*sizeof(s16) );
            }

        }
        /* p[^ DSP ɑM */
        nn::snd::SendParameterToDsp();
    }
}


/*---------------------------------------------------------------------------*
  Name        : RomfileWavePlay
  Description : Play Romfile wave data.
  Arguments   : number - wave data number.
  Returns     : -
 *---------------------------------------------------------------------------*/
void SndMicDevice::RomfileWavePlay( u32 number )
{
    m_FramePlayReq = true;
    m_FramePlayNum = 0;
    m_WaveNumbber  = number;
    m_pPlayWave    = m_WaveData[number].buf;
}


/*---------------------------------------------------------------------------*
  Name        : SoundThreadFunc
  Description : Sound Thread function for send Parameter to DSP.(Wrapper)
  Arguments   : sndmicDev - SndMicDevice Class pointer.
  Returns     : -
 *---------------------------------------------------------------------------*/
void SoundThreadFunc( SndMicDevice* sndmicDev )
{
    sndmicDev->SoundThreadFuncImpl();
}


/*---------------------------------------------------------------------------*
  Name        : MicGainSet
  Description : Change MIC Gain.
  Arguments   : gain - set mic gain value.
  Returns     : SNDMIC_ERROR_STATUS
              :  SNDMIC_ERROR_NONE  - none error.
              :  MIC_ERROR_SEQ   - sequence error.
              :  MIC_ERROR_LIB   - library error.
 *---------------------------------------------------------------------------*/
SNDMIC_ERROR_STATUS SndMicDevice::MicGainSet( u8 gain )
{
    /* Check process status */
    if ( m_MicStatus == MIC_PROCESS_NONE )
    {
        return MIC_ERROR_SEQ;
    }

    /* Set mic gain. */
    if ( nn::mic::SetAmpGain( gain ).IsFailure() )
    {
        return MIC_ERROR_LIB;
    }

    return SNDMIC_ERROR_NONE;
}


/*---------------------------------------------------------------------------*
  Name        : MicGainGet
  Description : Get MIC Gain.
  Arguments   : *gain - set mic gain value.
  Returns     : SNDMIC_ERROR_STATUS
              :  SNDMIC_ERROR_NONE  - none error.
              :  MIC_ERROR_SEQ   - sequence error.
              :  MIC_ERROR_LIB   - library error.
 *---------------------------------------------------------------------------*/
SNDMIC_ERROR_STATUS SndMicDevice::MicGainGet( u8 *gain )
{
    /* Check process status */
    if ( m_MicStatus == MIC_PROCESS_NONE )
    {
        return MIC_ERROR_SEQ;
    }

    if ( nn::mic::GetAmpGain( gain ).IsFailure() )
    {
        return MIC_ERROR_LIB;
    }

    return SNDMIC_ERROR_NONE;
}


/*---------------------------------------------------------------------------*
  Name        : GetReceiveAudioData
  Description : Get ReceiveAudio Data.
  Arguments   : *gain - set mic gain value.
  Returns     : true    success
              : false   no data
 *---------------------------------------------------------------------------*/
bool SndMicDevice::GetReceiveAudioData( u8 *audioBuf, u16 length )
{
    /* Check Index */
    if ( m_SndBufReadIdx == m_SndBufWriteIdx )
    {
        return false;
    }
    else
    {
        for(int i = 0;i < length;i++)
        {
            audioBuf[i] = m_SndBuf[m_SndBufReadIdx][i];
        }
        m_SndBufReadIdx = (m_SndBufReadIdx + 1) % 4;

        return true;
    }
}


/*---------------------------------------------------------------------------*
  Name        : GetSndMicErrorFlag
  Description : Get SndMic Error Flag.
  Arguments   : 
  Returns     : true    error occured
              : false   no error
 *---------------------------------------------------------------------------*/
bool SndMicDevice::GetSndMicErrorFlag( void)
{
    bool ErrorFlag;

    if(m_MicTimingError == true)
    {
        ErrorFlag = true;
        m_MicTimingError = false;
    }
    else
    {
        ErrorFlag = false;
    }

    return ErrorFlag;
}


/*---------------------------------------------------------------------------*
  Name        : FrameProcessThreadImpl
  Description : Frame Process Thread function.(Core)
  Arguments   : -
  Returns     : -
 *---------------------------------------------------------------------------*/
void SndMicDevice::FrameProcessThreadImpl( void )
{
    uptr Position;
    u32 micScale;
    u32 micOffset;
    nn::Result  Result;
    u16         outCIDList[VCT_MAX_AUDIO_STREAM];


    nn::os::Timer timer(false);


    m_SndBufWriteIdx  = 0;
    m_SndBufReadIdx   = 0;


    while ( true )
    {
        if ( m_MicStatus == MIC_PROCESS_NONE )
        {
            break;
        }

        /**********   MICX^[g   **********/
        /* Check process status */
        if ( ( m_MicStatus != MIC_PROCESS_INIT ) && ( m_MicStatus != MIC_PROCESS_STOP ) )
        {
            NN_LOG( "MIC_ERROR_SEQ !!\n" );
            break;
        }

        /* TEhXbhSignal҂ */
        m_MicEvent.Wait();

        /* Set Half Position */
        m_MicBuffHalfPosition = (uptr)m_MicBuffer + m_AudioFrameSize;
        m_MicPrevPosition     = 1;
    
        /* Start sampling. */
        Result = nn::mic::StartSampling(
                                            nn::mic::SAMPLING_TYPE_SIGNED_16BIT,
                                            nn::mic::SAMPLING_RATE_8180,
                                            0,
                                            m_AudioFrameSize*2,
                                            true
                                           );
    
        /* StartSampling error check. */
        if ( Result.IsFailure() )
        {
            /* Włɂꍇ̂ŁA͌p     */
            NN_LOG( "StartSampling error !!\n" );
            m_MicTimingError = true;
        }
        else
        {
            m_MicTimingError = false;
        }

        /* Update process status */
        m_MicStatus = MIC_PROCESS_START;


        timer.StartPeriodic(nn::fnd::TimeSpan::FromNanoSeconds(FREQ8KHZ_TIME_NS * 8 * m_Config.frame_msec),
            nn::fnd::TimeSpan::FromNanoSeconds(FREQ8KHZ_TIME_NS * 8 * m_Config.frame_msec));

        while (( m_MicStatus == MIC_PROCESS_START ) && ( m_SoundStatus == SND_PROCESS_PLAY ))
        {
            timer.Wait();        /* t[ɋN */

            if( m_MicTimingError == true)
            {
                /* }CNTvOăX^[g */
                /* Set Half Position */
                m_MicBuffHalfPosition = (uptr)m_MicBuffer + m_AudioFrameSize;
                m_MicPrevPosition     = 0;
            
                /* Start sampling. */
                Result = nn::mic::StartSampling(
                                                    nn::mic::SAMPLING_TYPE_SIGNED_16BIT,
                                                    nn::mic::SAMPLING_RATE_8180,
                                                    0,
                                                    m_AudioFrameSize*2,
                                                    true
                                                   );
            
                /* StartSampling error check. */
                if ( Result.IsFailure() )
                {
                    /* Włɂꍇ̂ŁA͌p     */
                    NN_LOG( "StartSampling error !!\n" );
                    m_MicTimingError = true;
                }
                else
                {
                    m_MicTimingError = false;
                }
            }

            /* }CNTvOƃ^C}[ĂȂ߁A}CN擾Ƀ}[W */
            nn::os::Thread::Sleep( nn::fnd::TimeSpan::FromMilliSeconds( MIC_CAPTURE_MARGIN_MS ) );



            /* Get MIC Sampling Position. */
            if ( nn::mic::GetLastSamplingAddress( &Position ).IsFailure() )
            {
                NN_LOG( "Cannot get Mic address!!\n" );
                break;
            }


            /* Get Mic Data */
            if (m_MicPrevPosition == 1)
            {
                if ((((Position + 2) >= m_MicBuffHalfPosition) && ((Position + 2) < (m_MicBuffHalfPosition + m_AudioFrameSize)))
                       && ( m_MicTimingError == false))
                {
                    /* Handle MIC data. */
                    micOffset = (u32)(Position + 2 - m_MicBuffHalfPosition)/2;
                    micScale = VCT_GetAudioScale( (void *)m_MicBuffer, m_AudioFrameSize );
                    VCT_SendAudioEx( (void *)m_MicBuffer, micScale, micOffset );
                    m_MicPrevPosition = 0;
                }
                else
               {
                    /* }CNTvOƃ^C}[ĂȂ߁Aɂ邱Ƃ */
                    /* WƂɂ     */
#if 1
                    NN_LOG( "Wrong Mic Position 1 !!\n" );
#endif
                    /* f[^n */
                    micOffset = MIC_CAPTURE_MARGIN_SAMPLE;
                    std::memset( m_zero_buf, 0x0, m_AudioFrameSize );
                    micScale = 0;
                    VCT_SendAudioEx( m_zero_buf, micScale, micOffset );

                    /* |WV̂MICTvOX^[g */
                    m_MicTimingError = true;
                }
            }
            else 
            {
                /* m_MicPrevPosition == 0  */
                if ((((Position + 2) < m_MicBuffHalfPosition) || ((Position + 2) == (m_MicBuffHalfPosition + m_AudioFrameSize)))
                       && ( m_MicTimingError == false))
                {
                    /* Handle MIC data. */
                    micOffset = (u32)(Position + 2 - (u32)m_MicBuffer)/2;
                    if((Position + 2) == (m_MicBuffHalfPosition + m_AudioFrameSize))
                    {
                        micOffset = 0;
                    }
                    micScale = VCT_GetAudioScale( (void *)m_MicBuffHalfPosition, m_AudioFrameSize );
                    VCT_SendAudioEx( (void *)m_MicBuffHalfPosition, micScale, micOffset );
                    m_MicPrevPosition = 1;
                }
                else
                {
                    /* }CNTvOƃ^C}[ĂȂ߁Aɂ邱Ƃ */
                    /* WƂɂ     */
#if 1
                    NN_LOG( "Wrong Mic Position 0 !!\n" );
#endif
                    /* f[^n */
                    micOffset = MIC_CAPTURE_MARGIN_SAMPLE;
                    micScale = 0;
                    std::memset( m_zero_buf, 0x0, m_AudioFrameSize );
                    VCT_SendAudioEx( m_zero_buf, micScale, micOffset );

                    /* |WV̂MICTvOX^[g */
                    m_MicTimingError = true;
                }
            }

            /* Set snd data. */
            VCT_ReceiveAudio( (void *)m_SndBuf[m_SndBufWriteIdx], m_AudioFrameSize, outCIDList );
            m_SndBufWriteIdx = (m_SndBufWriteIdx + 1) % 4;
        }

        timer.Stop();
    }

    timer.Finalize();

}


/*---------------------------------------------------------------------------*
  Name        : FrameProcessThreadFunc
  Description : Frame Process Thread function.(Wrapper)
  Arguments   : sndmicDev - SndMicDevice Class pointer.
  Returns     : -
 *---------------------------------------------------------------------------*/
void FrameProcessThreadFunc( SndMicDevice* sndmicDev )
{
    sndmicDev->FrameProcessThreadImpl();
}
/* <<<< EOF >>>> */



GFL_NAMESPACE_END(vct)
GFL_NAMESPACE_END(xynet)
