/*---------------------------------------------------------------------------*
 Project:  NEX
 File:     Validation.cpp

 Copyright (C)2012 Nintendo Co., Ltd.  All rights reserved.

 These coded instructions, statements, and computer programs contain
 proprietary information of Nintendo of America Inc. and/or Nintendo
 Company Ltd., and are protected by Federal copyright law.  They may
 not be disclosed to third parties or copied or duplicated in any form,
 in whole or in part, without the prior written consent of Nintendo.
 *---------------------------------------------------------------------------*/
/*
 * NEX Sample
 *
 * Validation
 *
 * Description:
 *
 *  GTS用ポケモンデータの不正チェックを行うサンプルです。
 *
 *
 * Note:
 *
 *  OMAS申請が完了していないため、実際に認証トークンの取得は行わずにダミーのトークンを設定しています。
 *  Debug用のhttpライブラリを使用しています。正式なサンプルではNEXのhttpライブラリを使用するように変更する予定です。
 *
 */

#include <nex.h>
#include <nn.h>
#include <nn/fs.h>
#include <nn/friends.h>
#include <nn/http.h>
#include "../common/NexUtil.h"

//-----------------------------------------------------------------------------
// 定数定義
//-----------------------------------------------------------------------------
// 不正チェックのURL
static const qChar8 *VALIDATOR_URL = "https://sver1087.tk.mesh.ad.jp/pokemon2/validate";
// チェックするポケモンの数
static const qInt32 POKEMON_NUM = 10;
// インデクスデータのサイズ
static const qInt32 INDEX_SIZE = 160;
// ポケモンデータのサイズ
static const qInt32 POKEMON_SIZE = 232;
// 署名のサイズ
static const qInt32 SIGNATURE_SIZE = 256;
// ルート証明書の格納に使用するバッファサイズ
static const qInt32 CA_BUFFER_SIZE = 1024;
// ROMアーカイブのマウントに使用するバッファサイズ
static const qInt32 ROM_BUFFER_SIZE = 128;
// ルートCAのファイルパス
static const wchar_t *ROOT_CA_PATH = L"rom:/cacert.der";

//-----------------------------------------------------------------------------
// グローバル変数定義
//-----------------------------------------------------------------------------
// POST用バッファ
static qChar8 postBuffer[8192] NN_ATTRIBUTE_ALIGN(4096);
// ルート証明書の格納に使用するバッファ
static u8 caBuffer[CA_BUFFER_SIZE] NN_ATTRIBUTE_ALIGN(4);
// ROMアーカイブのマウントに使用するバッファ
static qChar8 romBuffer[ROM_BUFFER_SIZE] NN_ATTRIBUTE_ALIGN(4);

//-----------------------------------------------------------------------------
// 構造体定義
//-----------------------------------------------------------------------------
// 不正チェックサーバに渡すデータ構造
typedef struct {
    qChar8 authToken[nn::nex::IndependentServer::AUTH_TOKEN_SIZE]; // 認証トークン
    qChar8 terminator; // 認証トークンの終端
    qInt16 gameMode; // ゲームモード
    qByte data[POKEMON_NUM][INDEX_SIZE + POKEMON_SIZE]; // チェック対象のデータ
} POKMEON_CHECKDATA;

//-----------------------------------------------------------------------------
// 関数定義
//-----------------------------------------------------------------------------
void DemoConnectionRead(nn::http::Connection &connection, u8 *pBodyBuf,
        size_t bufLen);
void DemoInitialize();
void DemoFinalize();
bool GetAuthenticationToken(qChar8 *authToken);

//-----------------------------------------------------------------------------
// メイン
//-----------------------------------------------------------------------------
void nnMain() {
    NEX_DEMO_LOG("***************************\n");
    NEX_DEMO_LOG("     Validation sample     \n");
    NEX_DEMO_LOG("***************************\n\n");

    POKMEON_CHECKDATA postData;

    DemoInitialize();

    // 認証トークン取得
    if (GetAuthenticationToken(postData.authToken)) {
        // POST用データ設定
        postData.terminator = '\0';
        postData.gameMode = 0x0100; // ネットワークバイトオーダで1
        for (qInt32 i = 0; i < POKEMON_NUM; i++) {
            memset(postData.data[i], 'a' + i, INDEX_SIZE);
            memset(postData.data[i] + INDEX_SIZE, 'A' + i, POKEMON_SIZE);
        }

        // ルート証明書の読み込み
        NN_MAX_ASSERT(nn::fs::GetRomRequiredMemorySize(1, 1, false), ROM_BUFFER_SIZE);
        NN_PANIC_IF_FAILED(nn::fs::MountRom("rom:", 1, 1, romBuffer, sizeof(romBuffer), false));
        nn::fs::FileInputStream caFile(ROOT_CA_PATH);
        s64 size = caFile.GetSize();
        caFile.Read(caBuffer, caFile.GetSize());
        caFile.Finalize();
        nn::fs::Unmount("rom:");

        nn::http::Connection *connection = qNew nn::http::Connection(
                VALIDATOR_URL, nn::http::REQUEST_METHOD_POST, true);

        // ルート証明書の設定
        NN_PANIC_IF_FAILED(connection->SetRootCa(caBuffer, size));

        // POST用データのエンコード方法の設定
        NN_PANIC_IF_FAILED(
                connection->SetPostDataEncoding(nn::http::ENCODING_TYPE_URL));

        // POST用データの設定
        NN_PANIC_IF_FAILED(
                connection->AddPostDataRaw(&postData, sizeof(postData)));

        // 接続
        NN_PANIC_IF_FAILED(connection->Connect());

        // HTTPステータスコードチェック
        s32 httpStatusCode;
        NN_PANIC_IF_FAILED(connection->GetStatusCode(&httpStatusCode));
        if (httpStatusCode == 200) {
            // 不正チェックのステータスコードチェック
            u8 validationStatusCode;
            DemoConnectionRead(*connection, &validationStatusCode,
                    sizeof(validationStatusCode));
            NEX_DEMO_LOG("Validation status code = %d\n", validationStatusCode);
            // 個々のポケモンデータの結果コード表示
            for (qInt32 i = 0; i < POKEMON_NUM; i++) {
                qInt32 validationResultCode;
                DemoConnectionRead(*connection,
                        reinterpret_cast<u8*>(&validationResultCode),
                        sizeof(validationResultCode));
                NEX_DEMO_LOG(
                        "Validation result code [%d] = %d\n", i, validationResultCode);
            }

            if (validationStatusCode == 0) {
                // 署名データ表示
                qByte signature[SIGNATURE_SIZE];
                DemoConnectionRead(*connection, signature, sizeof(signature));
                NEX_DEMO_LOG("Validation signature = ");
                for (int i = 0; i < SIGNATURE_SIZE; i++) {
                    NEX_DEMO_LOG("%02x", signature[i]);
                }
                NEX_DEMO_LOG("\n");
            }
        } else {
            NEX_DEMO_LOG("HTTP error %d\n", httpStatusCode);
        }

        connection->Finalize();
        NEX_DEMO_LOG("Finalize\n");
        qDelete connection;
    }
    DemoFinalize();
}

//-----------------------------------------------------------------------------
// Connection.Read()実行&結果のチェック
//-----------------------------------------------------------------------------
void DemoConnectionRead(nn::http::Connection &connection, u8 *pBodyBuf,
        size_t bufLen) {
    nn::Result result = connection.Read(pBodyBuf, bufLen);
    if (result != nn::http::ResultBodyBufShortage()) {
        NN_PANIC_IF_FAILED(result);
    }
}

//-----------------------------------------------------------------------------
// 初期化処理
//-----------------------------------------------------------------------------
void DemoInitialize() {
    // fsライブラリの初期化
    nn::fs::Initialize();

    // インターネットに接続する
    if (!nex_demo_util::NetInit()) {
        NEX_DEMO_LOG("nex_demo_util::NetInit() failed!!!\n");
        NEX_DEMO_LOG("Leaving application...\n");
        nn::dbg::Break(nn::dbg::BREAK_REASON_USER);
    }

    // ユーティリティ使用準備
    nex_demo_util::Initialize();

    // NEXのグローバル変数の初期化処理
    nn::nex::GlobalVariables::AcquireInstance();

    // HTTPライブラリの初期化
    NN_PANIC_IF_FAILED(
            nn::http::Initialize(reinterpret_cast<uptr>(postBuffer), sizeof(postBuffer)));

    // ログレベルの設定
    nn::nex::EventLog::GetInstance()->SetLogLevel(nn::nex::EventLog::Debug);
    nn::nex::TraceLog::GetInstance()->SetFlag(TRACE_ALWAYS);
}

//-----------------------------------------------------------------------------
// 終了処理
//-----------------------------------------------------------------------------
void DemoFinalize() {
    // HTTPライブラリの開放
    nn::http::Finalize();

    // NEXのグローバル変数の開放処理
    nn::nex::GlobalVariables::ReleaseInstance();

    // ユーティリティ終了
    nex_demo_util::Finalize();

    // インターネットから切断する
    nex_demo_util::NetTerm();

    NEX_DEMO_LOG("Leaving application...\n");
    nn::dbg::Break(nn::dbg::BREAK_REASON_USER);
}

//-----------------------------------------------------------------------------
// 認証トークンの取得
//-----------------------------------------------------------------------------
bool GetAuthenticationToken(qChar8 *authToken) {
    bool ret = true;
    // フレンドライブラリ初期化
    // フレンドサーバへのログインが不要であっても、ライブラリ内部で nn::friends の関数を
    // 呼び出すため、必ず初期化が必要となります
    NN_PANIC_IF_FAILED(nn::friends::Initialize());

    nn::nex::IndependentServer independentServer;
    nn::nex::ProtocolCallContext context;
    nn::nex::qResult result;

    // OMAS申請が完了していないため、認証トークンはダミーを設定しています
#if 0
    result = independentServer.RequestAuthenticationToken(&context,
            NEXDEMO_GAME_ID, NEXDEMO_KEY_HASH, authToken, true);
    if (result) {
        context.Wait();
        if (context.GetOutcome()) {
            NEX_DEMO_LOG("authentication token = %s\n", authToken);
        } else {
            NEX_DEMO_LOG(
                    "IndependentServer::RequestAuthenticationToken() failed. Error code : %d\n", independentServer.GetLastErrorCode());
            ret = false;
        }
    } else {
        NEX_DEMO_LOG(
                "IndependentServer::RequestAuthenticationToken() failed. Return code : %d\n", result.GetReturnCode());
        ret = false;
    }
#else
    memset(authToken, '#', nn::nex::IndependentServer::AUTH_TOKEN_SIZE);
#endif

    nn::friends::Finalize();

    return ret;
}
