<?php

namespace App\Http\Controllers;

use App\NetworkVariable;
use App\Program;
use App\Token;
use App\User;
use Illuminate\Http\Request;

class ProgramController extends Controller
{
    /**
     * Create a new controller instance.
     *
     * @return void
     */
    public function __construct()
    {
        $this->middleware('auth');
    }

    public function dashboard() {
        $programs = Program::where('owner_id', \Auth::user()->id)->get();

        $secondary = \Auth::user()->admin_programs()->get();

        return view('dashboard', ['programs' => $programs, 'secondary' => $secondary]);
    }

    public function index($id) {
        $program = Program::where('id', $id)->where('owner_id',\Auth::user()->id);
        if ($program->count() == 0) {
            $program = \Auth::user()->admin_programs()->where('program_id', $id);
            if ($program->count() == 0) {
                return redirect(route('dashboard'))->withErrors(['application' => 'Invalid Application']);
            }
        }

        return view('program_view', ['program' => $program->first()]);
    }

    public function settings($id) {
        $program = Program::where('owner_id', \Auth::user()->id)->where('id', $id);
        if ($program->count() == 0) {
            $program = \Auth::user()->admin_programs()->where('program_id', $id);
            if ($program->count() == 0) {
                return redirect(route('dashboard'))->withErrors(['application' => 'Invalid Application']);
            }
            if ($program->first()->pivot->access < 3) {
                return redirect(route('dashboard'))->withErrors(['permissions' => 'Invalid Permissions']);
            }
        }

        $shared_with = $program->first()->shared_with();

        return view('program_settings', ['program' => $program->first(), 'shared_with' => $shared_with]);
    }

    public function save_settings(Request $request, $id) {
        $request->validate([
            'name' => 'required|max:25',
            'hash' => 'nullable|max:32',
            'enabled' => 'required',
            'devmode' => 'required',
            'url' => 'nullable',
            'version' => 'required|max:20'
            ]);
        
        
        $program = Program::where('owner_id', \Auth::user()->id)->where('id', $id);
        if ($program->count() == 0) {
            $program = \Auth::user()->admin_programs()->where('program_id', $id);
            if ($program->count() == 0) {
                return redirect(route('dashboard'))->withErrors(['application' => 'Invalid Application']);
            }
            if ($program->first()->pivot->access < 3) {
                return redirect(route('dashboard'))->withErrors(['permissions' => 'Invalid Permissions']);
            }
        }
        $prog = $program->first();
        $prog->name = $request->name;
        $prog->hash = $request->hash;
        $prog->enabled = $request->enabled;
        $prog->devmode = $request->devmode;
        $prog->url = $request->url;
        $prog->version = $request->version;
        $prog->save();

        return redirect()->back()->withStatus('Saved settings!');
    }

    public function _new(Request $request) {
        $request->validate([
            'program_name' => 'required|max:25'
            ]);
        
        if (\Auth::user()->programs->count() >= \Auth::user()->program_allowance()) {
            return redirect(route('dashboard'))->withErrors(['premium' => 'You\'ve hit your limit for programs!  Buy premium to unlock unlimited services!']);
        }

        $program = Program::create([
            'name' => $request->input('program_name'),
            'key' => \Str::random(40),
            'varkey' => \Str::random(30),
            'owner_id' => \Auth::user()->id
        ]);

        return redirect(route('program_view', $program->id));
    }

    public function _delete($id) {
        $program = Program::where('owner_id', \Auth::user()->id)->where('id', $id);
        if ($program->count() == 0) {
            $program = \Auth::user()->admin_programs()->where('program_id', $id);
            if ($program->count() > 0) {
                return redirect(route('program_view_settings', $id))->withErrors(['permissions' => 'You don\'t own this application!']);
            } else {
                return redirect(route('dashboard'))->withErrors(['application' => 'Invalid Application']);
            }
        }

        $program->first()->delete();
        return redirect(route('dashboard'))->withStatus('Deleted application!');
    }

    public function share(Request $request, $id) {
        $request->validate([
           'user' => 'required|exists:users,name',
           'access' => 'required|integer|between:1,3'
        ]);
        
        $program = Program::where('owner_id', \Auth::user()->id)->where('id', $id);
        if ($program->count() == 0) {
            $program = \Auth::user()->admin_programs()->where('program_id', $id);
            if ($program->count() > 0) {
                return redirect()->back()->withErrors(['permissions' => 'You don\'t own this application!']);
            } else {
                return redirect(route('dashboard'))->withErrors(['application' => 'Invalid Application']);
            }
        }

        $user = $request->input('user', '');
        if (strlen($user) > 0) {
            $user = User::where('name', $user)->orWhere('email', $user);
        } else {
            return redirect()->back()->withErrors(['input' => 'Invalid username/email.']);
        }

        if ($user->count() == 0) {
            return redirect()->back()->withErrors(['input' => 'Invalid user!']);
        }

        $access = $request->input('access', '0');
        if (!($access >= 1 && $access <= 3)) {
            return redirect()->back()->withErrors(['input' => 'Invalid access level!']);
        }

        \DB::table('program_admins')->insertOrIgnore([
            ['user_id' => $user->first()->id, 'program_id' => $id, 'access' => $access]
            ]);

        return redirect()->back()->withStatus('Shared with user successfully!');
    }

    public function share_rem($id, $uid) {
        $program = Program::where('owner_id', \Auth::user()->id)->where('id', $id);
        if ($program->count() == 0) {
            $program = \Auth::user()->admin_programs()->where('program_id', $id);
            if ($program->count() > 0) {
                return redirect()->back()->withErrors(['permissions' => 'You don\'t own this application!']);
            } else {
                return redirect()->back()->withErrors(['application' => 'Invalid Application']);
            }
        }

        $program->first()->shared_with()->detach($uid);

        return redirect()->back()->withStatus('Removed collaborative partner.');
    }

    public function share_access(Request $request, $id, $uid) {
        $request->validate([
            'access' => 'integer|required|between:1,3'
        ]);
        
        $program = Program::where('owner_id', \Auth::user()->id)->where('id', $id);
        if ($program->count() == 0) {
            $program = \Auth::user()->admin_programs()->where('program_id', $id);
            if ($program->count() > 0) {
                return redirect()->back()->withErrors(['permissions' => 'You don\'t own this application!']);
            } else {
                return redirect()->back()->withErrors(['application' => 'Invalid Application']);
            }
        }

        $access = $request->input('access', '0');
        if (!($access >= 1 && $access <= 3)) {
            return redirect()->back()->withErrors(['access' => 'Invalid access level!']);
        }

        $user = $program->first()->shared_with()->wherePivot('user_id', $uid);
        if ($user->count() == 0) {
            return redirect()->back()->withErrors(['input' => 'Invalid collaboration user!']);
        }

        $program->first()->shared_with()->updateExistingPivot($uid, ['access' => $access]);

        return redirect()->back()->withStatus('Updated collaborative access.');
    }
}
