<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\NetworkVariable;

class APIController extends Controller
{
    public function program_list() {
    	$owned = \Auth::user()->programs()->get();
    	$shared = \Auth::user()->admin_programs()->get();
    	$all = collect(['owned' => $owned, 'shared' => $shared]);
    	
    	return response()->json(['status' => 'success', 'data' => $all]);
    }
    
    public function program_user_list($id) {
    	$program = \Auth::user()->programs->find($id);
    	$users = $program->users()->get();
    	
    	return response()->json(['status' => 'success', 'data' => $users]);
    }
    
    public function program_user_get($id, $uid) {
    	$program = \Auth::user()->programs->find($id);
    	$user = $program->users()->findOrFail($uid);
    	
    	return response()->json(['status' => 'success', 'data' => $user]);
    }
    
    public function program_user_save(Request $request, $id, $uid) {
        $program = \Auth::user()->programs->find($id);
        $user = $program->users()->findOrFail($uid);
        
        $json = json_decode($request->getContent(), true);
        
        $validator = \Validator::make($json, [
            'email' => 'nullable|max:255',
            'password' => 'nullable|max:255',
            'hwid' => 'nullable|max:32',
            'level' => 'nullable|between:1,10'
        ]);
        
        if ($validator->fails()) {
            return response()->json(['errors' => $validator->messages(), 'status' => 422], 200);
        }
        
        if (isset($json['password'])) {
            $user->password = \Hash::make($json['password'], ['cost' => 14]);
        }
        
        if(isset($json['email'])) {
            if (strlen($json['email']) > 255)
                return response()->json(['status' => 'failure', 'error' => 'Email too long']);
            $user->email = $json['email'];
        }
        
        if (isset($json['hwid'])) {
            if (strlen($json['hwid']) > 255)
                return response()->json(['status' => 'failure', 'error' => 'HWID too long']);
            $user->hwid = $json['hwid'];
        }
        
        if (isset($json['level'])) {
            if (intval($json['level']) < 1 || intval($json['level']) > 10)
                return response()->json(['status' => 'failure', 'error' => 'Level must be between 1 and 10']);
            $user->level = $json['level'];
        }
        
        $user->save();
        
        return response()->json(['status' => 'success', 'data' => $user]);
    }
    
    public function program_user_hwid(Request $request, $id, $uid) {
        $program = \Auth::user()->programs->find($id);
        $user = $program->users()->findOrFail($uid);
        
        $user->hwid = "";
        $user->save();
        
        return response()->json(['status' => 'success']);
    }
    
    public function program_token_list($id) {
        $program = \Auth::user()->programs->find($id);
        $tokens = $program->tokens()->get();
        
        return response()->json(['status' => 'success', 'data' => $tokens]);
    }
    
    public function program_token_generate(Request $request, $id) {
        $program = \Auth::user()->programs->find($id);
        
        $data = json_decode($request->getContent(), true);
        
        if(intval($data['count']) < 1 || intval($data['count']) > 100) {
            return response()->json(['error' => 'Invalid count', 'status' => 'failure'], 200);
        }
        
        if (intval($data['length']) < 1 || intval($data['length']) > 5) {
            return response()->json(['error' => 'Invalid length', 'status' => 'failure'], 200);
        }
        
        if (intval($data['level']) < 1 || intval($data['level']) > 10) {
            return response()->json(['error' => 'Invalid level', 'status' => 'failure'], 200);
        }
        
        if (!isset($data['prefix'])) {
            $program->generate_tokens($data['count'], $data['length'], $data['level']);
        } else {
            $program->generate_tokens($data['count'], $data['length'], $data['level'], $data['prefix']);
        }
        
        return response()->json(['status' => 'success']);
    }
    
    public function program_token_delete($id, $tid) {
        $program = \Auth::user()->programs->find($id);
        $token = $program->tokens()->findOrFail($tid);
        
        $token->delete();
        
        return response()->json(['status' => 'success']);
    }
    
    public function program_variable_list($id) {
        $program = \Auth::user()->programs->find($id);
        $variables = $program->variables()->get();
        
        return response()->json(['status' => 'success', 'data' => $variables]);
    }
    
    public function program_variable_new(Request $request, $id) {
        $program = \Auth::user()->programs->find($id);
        
        $data = json_decode($request->getContent(), true);
        
        if (!isset($data['name']) || !isset($data['value'])) {
            return response()->json(['status' => 'failure', 'error' => 'Must include both the name and value']);
        }
        
        if (strlen($data['name']) > 255)
            return response()->json(['status' => 'failure', 'error' => 'Name too long']);
        if (strlen($data['value']) > 255)
            return response()->json(['status' => 'failure', 'error' => 'Value too long']);
            
        NetworkVariable::create([
            'name' => $data['name'],
            'value' => $data['value'],
            'program_id' => $id
        ]);
        
        return response()->json(['status' => 'success']);
    }
    
    public function program_variable_delete($id, $vid) {
        $program = \Auth::user()->programs->find($id);
        $nvar = $program->variables()->findOrFail($vid);
        
        $nvar->delete();
        
        return response()->json(['status' => 'success']);
    }
    
    public function program_settings_get($id) {
        $program = \Auth::user()->programs()->findOrFail($id);
        
        return response()->json(['status' => 'success', 'data' => $program]);
    }
    /*
    "name": "Test",
    "hash": "",
    "enabled": 1,
    "devmode": 1,
    "url": "",
    "version": "1.0",
    */
    public function program_settings_save(Request $request, $id) {
        $program = \Auth::user()->programs()->findOrFail($id);
        
        $data = json_decode($request->getContent(), true);
        
        if (isset($data['name'])) {
            if(strlen($data['name']) > 255)
                return response()->json(['status' => 'failure', 'error' => 'Invalid name']);
            $program->name = $data['name'];
        }
        if (isset($data['hash'])) {
            if(strlen($data['hash']) > 255)
                return response()->json(['status' => 'failure', 'error' => 'Invalid hash']);
            $program->hash = $data['hash'];
        }
        if (isset($data['url'])) {
            if(strlen($data['url']) > 255)
                return response()->json(['status' => 'failure', 'error' => 'Invalid url']);
            $program->url = $data['url'];
        }
        if (isset($data['version'])) {
            if(strlen($data['version']) > 255)
                return response()->json(['status' => 'failure', 'error' => 'Invalid version']);
            $program->version = $data['version'];
        }
        if (isset($data['enabled'])) {
            if (intval($data['enabled']) < 0 || intval($data['enabled']) > 1)
                return response()->json(['status' => 'failure', 'error' => 'Invalid enabled state, must be 0 or 1']);
            $program->enabled = intval($data['enabled']);
        }
        if (isset($data['devmode'])) {
            if (intval($data['devmode']) < 0 || intval($data['devmode']) > 1)
                return response()->json(['status' => 'failure', 'error' => 'Invalid devmode state, must be 0 or 1']);
            $program->devmode = intval($data['devmode']);
        }
        
        $program->save();
        
        return response()->json(['status' => 'success', 'data' => $program]);
    }
    
    public function program_client_login(Request $request, $id) {
        $program = \Auth::user()->programs()->findOrFail($id);
        
        $data = json_decode($request->getContent(), true);
        
        if (!isset($data['username']) || !isset($data['password'])) {
            return response()->json(['status' => 'failure', 'error' => 'Invalid Username / Password']);
        }
        
        $user = $program->users()->where('username', $data['username']);
        
        if ($user->count() == 0) {
            return response()->json(['status' => 'failure', 'error' => 'Invalid Username / Password']);
        }
        
        $user = $user->first();
        
        if (\Hash::check($data['password'], $user->password)) {
            return response()->json(['status' => 'success', 'data' => $user]);
        }
        return response()->json(['status' => 'failure', 'error' => 'Invalid Username / Password', 'data' => $data]);
    }
}
