package main

import (
	"database/sql"
	"fmt"
	"time"
)

type FromClient struct {
	Username string `json:"username"`
	Password string `json:"password"`
	Email string `json:"email"`
	HWID string `json:"hwid"`
	PacketType string `json:"packettype"`
	Hash string `json:"hash"`
	Data string `json:"data"`
	Key string `json:"key"`
	VKey string `json:"vkey"`
	Version string `json:"version"`
	Session string `json:"session_id"`
	Salt string `json:"salt"`
}

type ShoppyOrder struct {
	Status      bool `json:"status"`
	WebhookType int  `json:"webhook_type"`
	Data        struct {
		Order struct {
			ID                    string      `json:"id"`
			PayID                 interface{} `json:"pay_id"`
			ProductID             string      `json:"product_id"`
			CouponID              interface{} `json:"coupon_id"`
			Price                 int         `json:"price"`
			Currency              string      `json:"currency"`
			ExchangeRate          interface{} `json:"exchange_rate"`
			Email                 string      `json:"email"`
			Delivered             int         `json:"delivered"`
			Confirmations         int         `json:"confirmations"`
			RequiredConfirmations int         `json:"required_confirmations"`
			TransactionID         string      `json:"transaction_id"`
			CryptoAddress         string      `json:"crypto_address"`
			CryptoAmount          string      `json:"crypto_amount"`
			ShippingAddress       interface{} `json:"shipping_address"`
			Quantity              int         `json:"quantity"`
			Gateway               string      `json:"gateway"`
			CustomFields          []struct {
				Name     string `json:"name"`
				Type     string `json:"type"`
				Required bool   `json:"required"`
				Value    string `json:"value"`
			} `json:"custom_fields"`
			RefundID      interface{}   `json:"refund_id"`
			IsReplacement bool          `json:"is_replacement"`
			ReplacementID interface{}   `json:"replacement_id"`
			PaidAt        string        `json:"paid_at"`
			DisputedAt    interface{}   `json:"disputed_at"`
			CreatedAt     string        `json:"created_at"`
			Coupon        []interface{} `json:"coupon"`
			Product       struct {
				ID           string      `json:"id"`
				AttachmentID interface{} `json:"attachment_id"`
				Title        string      `json:"title"`
				Description  string      `json:"description"`
				Image        interface{} `json:"image"`
				Unlisted     bool        `json:"unlisted"`
				Type         string      `json:"type"`
				Price        int         `json:"price"`
				Currency     string      `json:"currency"`
				Email        struct {
					Enabled bool `json:"enabled"`
				} `json:"email"`
				StockWarning int `json:"stock_warning"`
				Quantity     struct {
					Min int `json:"min"`
					Max int `json:"max"`
				} `json:"quantity"`
				Confirmations int `json:"confirmations"`
				CustomFields  []struct {
					Name     string `json:"name"`
					Type     string `json:"type"`
					Required bool   `json:"required"`
				} `json:"custom_fields"`
				Gateways    []string      `json:"gateways"`
				WebhookUrls []string      `json:"webhook_urls"`
				DynamicURL  string        `json:"dynamic_url"`
				Position    interface{}   `json:"position"`
				CreatedAt   string        `json:"created_at"`
				UpdatedAt   string        `json:"updated_at"`
				Stock       int64         `json:"stock"`
				Accounts    []interface{} `json:"accounts"`
			} `json:"product"`
			Agent struct {
				Geo struct {
					IP         string      `json:"ip"`
					IsoCode    string      `json:"iso_code"`
					Country    string      `json:"country"`
					City       string      `json:"city"`
					State      string      `json:"state"`
					StateName  string      `json:"state_name"`
					PostalCode interface{} `json:"postal_code"`
					Lat        float64     `json:"lat"`
					Lon        float64     `json:"lon"`
					Timezone   string      `json:"timezone"`
					Continent  string      `json:"continent"`
					Currency   string      `json:"currency"`
					Default    bool        `json:"default"`
					Cached     bool        `json:"cached"`
				} `json:"geo"`
				Data struct {
					IsMobile  bool   `json:"is_mobile"`
					IsTablet  bool   `json:"is_tablet"`
					IsDesktop bool   `json:"is_desktop"`
					Platform  string `json:"platform"`
					Browser   struct {
						Name    string `json:"name"`
						Version string `json:"version"`
					} `json:"browser"`
					Device struct {
						Name    string `json:"name"`
						Version bool   `json:"version"`
					} `json:"device"`
				} `json:"data"`
			} `json:"agent"`
			Hash string `json:"hash"`
		} `json:"order"`
	} `json:"data"`
}

type ToClient struct {
	Status string `json:"status"`
	Message string `json:"message"`
	Data string `json:"data"`
	IP string `json:"ip"`
	Expiry time.Time `json:"expiry"`
	Level int `json:"level"`
}

type Program struct {
	Id int
	Name string
	Hash sql.NullString
	Key string
	VarKey string
	Enabled int
	Devmode int
	Url sql.NullString
	Version string
	OwnerID int
	Access int
}

func (p *Program) GetUsers() []*UserAccount {
	stmt, err := DatabaseConnection.Prepare("SELECT id,username,email,hwid,expires FROM program_users WHERE program_id = ?")
	if err != nil {
    	fmt.Println(err.Error())
    	return nil
	}
	
	rows, err := stmt.Query(p.Id)
	if err != nil {
		fmt.Println("stmt err")
	}
	
	var arr []*UserAccount
	for rows.Next() {
		ua := &UserAccount{ProgramID:p.Id}
		err = rows.Scan(&ua.Id, &ua.Username, &ua.Email, &ua.HWID, &ua.Expires)
		if err != nil {
			fmt.Println(err.Error())
			continue
		}
		arr = append(arr, ua)
	}
	return arr
}

func (p *Program) GetUser(name string) *UserAccount {
	stmt, err := DatabaseConnection.Prepare("SELECT id, email, hwid, expires FROM program_users WHERE program_id = ? AND username = ?")
	if err != nil {
		return nil
	}

	user := &UserAccount{
		Username:  name,
		ProgramID: p.Id,
	}
	if row := stmt.QueryRow(p.Id, name); row != nil {
		if err = row.Scan(&user.Id, &user.Email, &user.HWID, &user.Expires); err == nil {
			return user
		}
	}
	return nil
}

type UserAccount struct {
	Id int
	Username string
	Password string
	Email string
	HWID string
	Level int
	Expires time.Time
	ProgramID int
}

func (ua *UserAccount) SetHWID(hwid string) {
	stmt, err := DatabaseConnection.Prepare("UPDATE program_users SET hwid = ? WHERE username = ? AND program_id = ?")
	if err != nil {
    	fmt.Println(err.Error())
		return
  	}
	_, err = stmt.Exec(hwid, ua.Username, ua.ProgramID)
	if err != nil {
		fmt.Println(err.Error())
		return
	}
}

func (ua *UserAccount) UpdateLicense(expiry time.Time) {
	stmt, err := DatabaseConnection.Prepare("UPDATE program_users SET expires = ? WHERE username = ? AND program_id = ?")
	if err != nil {
		fmt.Println(err.Error())
		return
	}

	_, err = stmt.Exec(expiry, ua.Username, ua.ProgramID)
	if err != nil {
		fmt.Println(err.Error())
	}
}

type Token struct {
	Token string
	Level int
	Days int
	Used int
	UsedBy string
	ProgramID int
}

func (t *Token) Use(user *UserAccount) *ToClient {
	stmt, err := DatabaseConnection.Prepare("UPDATE tokens SET used = 1, used_by = ? WHERE token = ? AND program_id = ?")
	if err != nil {
		fmt.Println(err.Error())
		return GenError("database_error")
	}
	res, err := stmt.Exec(user.Id, t.Token, t.ProgramID)
	if err != nil {
		fmt.Println(err.Error())
		return GenError("database_error")
	}
	
	_, err = res.RowsAffected()
	if err != nil {
		fmt.Println(err.Error())
		return GenError("database_error")
	}
	
	stmt, err = DatabaseConnection.Prepare("UPDATE program_users SET expires = ? WHERE username = ? AND program_id = ?")
	if err != nil {
		fmt.Println(err.Error())
		return GenError("database error")
	}
	user.Expires = user.Expires.AddDate(0, 0, t.Days)
	fmt.Println(user.Expires)
	_, err = stmt.Exec(user.Expires, user.Username, t.ProgramID)
	if err != nil {
		fmt.Println(err.Error())
		return GenError("database_error")
	}
	
	return &ToClient{Status: "success", Message: "used_token"}
}

type Var struct {
	Name string
	Value string
	ProgramID int
}
