package main

import (
	"fmt"
	"github.com/bwmarrin/discordgo"
	"strings"
	"strconv"
	"net/http"
	"time"
)

type Command struct {
	ident string
	commands []string
	minArgs int
	args []string
	help string
	usage string
	permission int
	execute func(*discordgo.Session, *discordgo.MessageCreate, []string)
}

type CommandList struct {
	commands []*Command
}

func (list* CommandList) NewCommand(ident string, cmd []string, help string, usage string, minArgs int, permission int, args []string, execute func(*discordgo.Session, *discordgo.MessageCreate, []string)) {
	tmp := &Command{ident: ident, commands:cmd, help:help, usage: usage, minArgs:minArgs, permission:permission, args:args, execute: execute}
	list.commands = append(list.commands, tmp)
}

func (list* CommandList) Contains(cmd string) (*Command, bool) {
	for _, k := range list.commands {
		if k.Contains(cmd) {
			return k, true
		}
	}
	return nil, false
}

func (c *Command) Contains(a string) bool {
	for _, k := range c.commands {
		if k == a {
			return true
		}
	}
	return false
}

func MemberHasPermission(s *discordgo.Session, guildID string, userID string, permission int) (bool, error) {
	member, err := s.State.Member(guildID, userID)
	if err != nil {
		if member, err = s.GuildMember(guildID, userID); err != nil {
			return false, err
		}
	}

	// Iterate through the role IDs stored in member.Roles
	// to check permissions
	for _, roleID := range member.Roles {
		role, err := s.State.Role(guildID, roleID)
		if err != nil {
			return false, err
		}
		if role.Permissions&permission != 0 {
			return true, nil
		}
	}

	return false, nil
}

func HelpCommand(s *discordgo.Session, m *discordgo.MessageCreate, args []string) {
	if len(args) > 0 {
		if cmd, ok := CmdList.Contains(args[0]); ok {
			printHelpForCommand(s,m,cmd)
		} else {
			SendError(s,m,"Invalid command")
		}
	} else {
		emb := NewEmbed().
			SetTitle("Command Help").
			SetDescription("A list of commands this bot can handle").
			SetThumbnail(BotUser.AvatarURL("250x250")).
			SetColor(RandomColor()).
			SetFooter(FooterTimestamp())
		for _, k := range CmdList.commands {
			emb.AddField(strings.Join(k.commands, " | "), k.help)
		}

		_, err := s.ChannelMessageSendEmbed(m.ChannelID, emb.MessageEmbed)
		if err != nil {
			fmt.Println("ERR!\n" + err.Error())
		}
	}
}

func UsageCommand(s *discordgo.Session, m *discordgo.MessageCreate, args []string) {
	if len(args) > 0 {
		if cmd, ok := CmdList.Contains(args[0]); ok {
			printUsageForCommand(s,m,cmd)
		} else {
			SendError(s,m,"Invalid command")
		}
	} else {
		emb := NewEmbed().
			SetTitle("Command Usage").
			SetDescription("A list of commands this bot can handle").
			SetThumbnail(BotUser.AvatarURL("250x250")).
			SetColor(RandomColor()).
			SetFooter(FooterTimestamp())
		for _, k := range CmdList.commands {
			emb.AddField(strings.Join(k.commands, " | "), CMD_PREFIX + k.usage)
		}

		_, err := s.ChannelMessageSendEmbed(m.ChannelID, emb.MessageEmbed)
		if err != nil {
			fmt.Println("ERR!\n" + err.Error())
		}
	}
}


func printHelpForCommand(s *discordgo.Session, m *discordgo.MessageCreate, cmd *Command) {
	emb := NewEmbed().
		SetTitle(cmd.ident).
		SetDescription(cmd.help).
		SetThumbnail(BotUser.AvatarURL("250x250")).
		SetColor(RandomColor()).
		SetFooter(FooterTimestamp())
	_, err := s.ChannelMessageSendEmbed(m.ChannelID, emb.MessageEmbed)
	if err != nil {
		fmt.Println("ERR!\n" + err.Error())
	}
}
func printUsageForCommand(s *discordgo.Session, m *discordgo.MessageCreate, cmd *Command) {
	emb := NewEmbed().
		SetTitle(cmd.ident).
		SetDescription(cmd.usage).
		SetThumbnail(BotUser.AvatarURL("250x250")).
		SetColor(RandomColor()).
		SetFooter(FooterTimestamp())
	_, err := s.ChannelMessageSendEmbed(m.ChannelID, emb.MessageEmbed)
	if err != nil {
		fmt.Println("ERR!\n" + err.Error())
	}
}

func PingCommand(s *discordgo.Session, m *discordgo.MessageCreate, args []string) {
	emb := NewEmbed().
		SetTitle("Ping!").
		SetThumbnail(BotUser.AvatarURL("250x250")).
		AddField("Bot Latency", s.HeartbeatLatency().String()).
		SetColor(RandomColor()).
		SetFooter(FooterTimestamp())
	msg, err := s.ChannelMessageSendEmbed(m.ChannelID, emb.MessageEmbed)
	if err != nil {
		go SelfDestructingMessage(s, m, "Error", "Error pinging", 5) // this shouldn't happen
		return
	}
	ours,err := msg.Timestamp.Parse()
	theirs, err := m.Timestamp.Parse()
	s.ChannelMessageDelete(m.ChannelID, m.ID)
	emb.AddField("Your Latency", (ours.Sub(theirs) - s.HeartbeatLatency()).String()).
		SetFooter(FooterTimestamp())
	s.ChannelMessageEditEmbed(msg.ChannelID, msg.ID, emb.MessageEmbed)
}

func ListProgramsCommand(s *discordgo.Session, m *discordgo.MessageCreate, args []string) {
	emb := NewEmbed().
		SetTitle("Your Programs").
		SetThumbnail(BotUser.AvatarURL("250x250")).
		SetColor(RandomColor()).
		SetFooter(FooterTimestamp())
	
	frid := GetFrontendLinkUser(m.Author.ID)
	if frid == 0 {
		go SelfDestructingMessage(s,m, "Error", "You have not linked your account!", 5)
		return
	}
	progs := GetProgramsByUser(frid)
	
	for _,k := range progs {
		emb.AddField(fmt.Sprintf("[%d] %s", k.Id, k.Name), fmt.Sprintf("%d users", len(k.GetUsers())))
	}
	
	_, err := s.ChannelMessageSendEmbed(m.ChannelID, emb.MessageEmbed)
	if err != nil {
		go SelfDestructingMessage(s, m, "Error", "Error listing programs", 5) // this shouldn't happen
		return
	}
}

func GetDownloadCommand(s *discordgo.Session, m *discordgo.MessageCreate, args []string) {
	frid := GetFrontendLinkUser(m.Author.ID)
        if frid == 0 {
                go SelfDestructingMessage(s,m, "Error", "You have not linked your account!", 5)
                return
        }

	progid, err := strconv.Atoi(args[0])
        if err != nil {
                go SelfDestructingMessage(s,m, "Error", "Invalid program ID", 5)
                return
        }
         
        program := GetProgramByUserAndID(progid, frid)
        if program == nil {
                go SelfDestructingMessage(s,m, "Error", "Invalid program", 5)
                return
        }

	emb := NewEmbed().
                SetTitle(fmt.Sprintf("Download link for %s", program.Name)).
                SetThumbnail(BotUser.AvatarURL("250x250")).
                SetColor(RandomColor()).
                SetFooter(FooterTimestamp()).
		AddField("Link", program.Url.String)

	s.ChannelMessageSendEmbed(m.ChannelID, emb.MessageEmbed)
}

func GenTokensCommand(s *discordgo.Session, m *discordgo.MessageCreate, args []string) {
	emb := NewEmbed().
		SetDescription("Tokens were sent in your dm's!").
		SetThumbnail(BotUser.AvatarURL("250x250")).
		SetColor(RandomColor()).
		SetFooter(FooterTimestamp())
		
	frid := GetFrontendLinkUser(m.Author.ID)
	if frid == 0 {
		go SelfDestructingMessage(s,m, "Error", "You have not linked your account!", 5)
		return
	}
	
	progid, err := strconv.Atoi(args[0])
	if err != nil {
		go SelfDestructingMessage(s,m, "Error", "Invalid program ID", 5)
		return
	}
	
	program := GetProgramByUserAndID(progid, frid)
	if program == nil {
		go SelfDestructingMessage(s,m, "Error", "Invalid program", 5)
		return
	}
	emb.SetTitle(fmt.Sprintf("Generated tokens for %s", program.Name))
	amount := 1
	level := 1
	length := 1
	
	if len(args) > 1 {
		amount, err = strconv.Atoi(args[1])
		if err != nil {
			go SelfDestructingMessage(s,m, "Error", "Invalid amount of tokens", 5)
			return
		}
	}
	
	if amount > 50 {
		amount = 50
	}
	
	if amount < 1 {
		amount = 1
	}
	
	if len(args) > 2 {
		level, err = strconv.Atoi(args[2])
		if err != nil {
			go SelfDestructingMessage(s,m, "Error", "Invalid level", 5)
			return
		}
	}
	
	if len(args) > 3 {
		length, err = strconv.Atoi(args[3])
		if err != nil {
			go SelfDestructingMessage(s,m, "Error", "Invalid length", 5)
			return
		}
	}
	
	switch length {
		case 2:
			length = 3
		case 3:
			length = 7
		case 4:
			length = 30
		case 5:
			length = 90
		case 6:
			length = 365
		case 7:
			length = 99999
		default: 
			length = 1
	}
	
	channel, err := s.UserChannelCreate(m.Author.ID)
	if err != nil {
		go SelfDestructingMessage(s,m, "Error", err.Error(), 5)
		return
	}
	
	msg := fmt.Sprintf("Here are your token(s) for %s\r\n```\r\n", program.Name)
	var inserts []string
	vals := []interface{}{}
	tokens := []string{}
	sqlpiece := "(?, ?, ?, ?)"
	sqlinsert := "INSERT INTO tokens (token, days, level, program_id) VALUES "
	for i:=0; i < amount; i++ {
		inserts = append(inserts, sqlpiece)
		t := RandStringBytesMaskImprSrc(20)
		tokens = append(tokens, t)
		vals = append(vals, t, length, level, program.Id)
	}
	sqlinsert += strings.Join(inserts, ",")
	stmt, _ := DatabaseConnection.Prepare(sqlinsert)
	
	stmt.Exec(vals...)
	
	msg += strings.Join(tokens, "\r\n") + "```"
	s.ChannelMessageSend(channel.ID, msg)
	s.ChannelMessageSendEmbed(m.ChannelID, emb.MessageEmbed)
}

func ResetHWIDCommand(s *discordgo.Session, m *discordgo.MessageCreate, args []string) {
	emb := NewEmbed().
		SetTitle("HWID Reset").
		SetThumbnail(BotUser.AvatarURL("250x250")).
		SetColor(RandomColor()).
		SetFooter(FooterTimestamp())

	frid := GetFrontendLinkUser(m.Author.ID)
	if frid == 0 {
		go SelfDestructingMessage(s,m, "Error", "You have not linked your account!", 5)
		return
	}

	progid, err := strconv.Atoi(args[0])
	if err != nil {
		go SelfDestructingMessage(s,m, "Error", "Invalid program ID", 5)
		return
	}

	program := GetProgramByUserAndID(progid, frid)
	
	if program == nil || program.Access < 2 {
		go SelfDestructingMessage(s,m, "Error", "You don't have permission to do that!", 5)
		return
	}
	
	i := 0
	for _,user := range program.GetUsers() {
		if user.Username == args[1] || user.Email == args[1] {
			user.SetHWID("")
			i=1
			emb.SetDescription(fmt.Sprintf("Reset %s's HWID", user.Username))
		}
	}
	if i == 0 {
		emb.SetDescription("User not found")
	}
	s.ChannelMessageSendEmbed(m.ChannelID, emb.MessageEmbed)
}

func LinkCommand(s *discordgo.Session, m *discordgo.MessageCreate, args []string) {
	emb := NewEmbed().
		SetTitle("Click this to invite me").
		SetDescription("Invite me to manage your programs from your own server!").
		SetURL("https://discordapp.com/oauth2/authorize?client_id=649272218622099461&permissions=10240&scope=bot").
		SetFooter(FooterTimestamp()).
		SetColor(RandomColor())
	s.ChannelMessageSendEmbed(m.ChannelID, emb.MessageEmbed)
}

func QuartzServerCommand(s *discordgo.Session, m *discordgo.MessageCreate, args []string) {
	emb := NewEmbed().
		SetTitle("Click here to join!").
		SetDescription("Join QuartzInc. for updates, beta, giveaways, and more!").
		SetURL("https://discord.gg/Gswnh4u").
		SetFooter(FooterTimestamp()).
		SetColor(RandomColor())
	s.ChannelMessageSendEmbed(m.ChannelID, emb.MessageEmbed)
}

func StatusCommand(s *discordgo.Session, m *discordgo.MessageCreate, args []string) {
	emb := NewEmbed().
		SetTitle("QuartzAuth Status").
		SetColor(RandomColor()).
		SetFooter(FooterTimestamp())
		
		// :white_check_mark:
		yesEmj := discordgo.Emoji{Name: ":white_check_mark:"}
		noEmj := discordgo.Emoji{Name: ":no_entry:"}
		
		if FindProcess("QuartzAuthApi") {
			emb.AddField("Licensing System API", yesEmj.MessageFormat())
		} else {
			emb.AddField("Licensing System API", noEmj.MessageFormat())
		}
		
		frnt := false
		resp, err := http.Get("https://auth.zertex.space")
		if err != nil {
			frnt = false
		}else if resp.StatusCode == 200 {
			frnt = true
		}
		
		if frnt {
			emb.AddField("Licensing System Frontend", yesEmj.MessageFormat())
		} else {
			emb.AddField("Licensing System Frontend", noEmj.MessageFormat())
		}
		
		s.ChannelMessageSendEmbed(m.ChannelID, emb.MessageEmbed)
}

func IncreaseLicenses(s *discordgo.Session, m *discordgo.MessageCreate, args []string) {
	i, err := strconv.Atoi(args[0])
	if err != nil {
		return
	}
	id, err := strconv.Atoi(m.Author.ID)
	if err != nil {
		return
	}
	p := GetProgramByUserAndID(i, id)

	d, err := time.ParseDuration(args[1])
	if err != nil {
		go SelfDestructingMessage(s,m, "Error", err.Error(), 5)
		return
	}

	for _, user := range p.GetUsers() {
		user.UpdateLicense(user.Expires.Add(d))
	}

	emb := NewEmbed().
		SetTitle(fmt.Sprintf("Increase Licenses for %s", p.Name)).
		SetDescription(fmt.Sprintf("Licenses increased by %s", FmtDuration(d))).
		SetColor(RandomColor()).
		SetFooter(FooterTimestamp())

	s.ChannelMessageSendEmbed(m.ChannelID, emb.MessageEmbed)
}
// ~!expiry 57 username 2020-04-18 00:30:16

const (
	layoutISO = "2006-01-02"
)

func SetLicenses(s *discordgo.Session, m *discordgo.MessageCreate, args []string) {
	fmt.Println(args)
	i, err := strconv.Atoi(args[0])
	if err != nil {
		return
	}
	id := GetFrontendLinkUser(m.Author.ID)

	prog := GetProgramByUserAndID(i, id)
	if prog == nil {
		go SelfDestructingMessage(s,m, "Error", "program not found", 5)
		return
	}

	user := prog.GetUser(args[1])

	exp, err := time.Parse(layoutISO, args[2])
	if err != nil {
		fmt.Println(err.Error())
		return
	}

	user.UpdateLicense(exp)
	s.ChannelMessageSend(m.ChannelID, fmt.Sprintf("Set expiry for user `%s` to `%s`", user.Username, args[2]))
}
