﻿using Leaf.xNet;
using System;
using System.Collections.Generic;
using System.IO;
using System.Text;

namespace ZertLib
{
    public class ProxyManager
    {
        private List<Proxy> _list = new List<Proxy>();
        private Random rnd = new Random((int)DateTime.Now.ToBinary());
        private ProxyType ProxyType = ProxyType.HTTP;
        private bool _autoMode = false;
        private bool _proxyless = false;
        private string socks5Uri = "https://api.proxyscrape.com/?request=getproxies&proxytype=socks5&timeout=10000&country={0}";
        private string socks4Uri = "https://api.proxyscrape.com/?request=getproxies&proxytype=socks4&timeout=10000&country={0}";
        private string httpsUri = "https://api.proxyscrape.com/?request=getproxies&proxytype=http&timeout=10000&country={0}&ssl=yes&anonymity=elite";

        public ProxyManager()
        {

        }

        /// <summary>
        /// Creates new ProxyManager, loading from file, defaults to HTTP proxy type
        /// </summary>
        /// <param name="filename">Path to proxyfile</param>
        public ProxyManager(string filename) : this(filename, ProxyType.HTTP)
        { }

        /// <summary>
        /// Creates new ProxyManager, loading from file
        /// </summary>
        /// <param name="filename">Path to proxies file</param>
        /// <param name="proxyType">Type of proxies you're loading</param>
        public ProxyManager(string filename, ProxyType proxyType)
        {
            LoadFromFile(filename, proxyType);
        }

        /// <summary>
        /// Loads proxies from a file
        /// </summary>
        /// <param name="filename">Path to proxies file</param>
        /// <returns>Count of proxies loaded</returns>
        public int LoadFromFile(string filename)
        {
            return LoadFromFile(filename, ProxyType.HTTP);
        }

        /// <summary>
        /// Loads proxies from a file
        /// </summary>
        /// <param name="filename">Path to proxies file</param>
        /// <param name="proxyType">Type of proxies loading</param>
        /// <returns>Count of proxies loaded</returns>
        public int LoadFromFile(string filename, ProxyType proxyType)
        {
            var l = new List<string>(File.ReadAllLines(filename));
            foreach(var a in l)
            {
                _list.Add(new Proxy(a, ProxyType));
            }
            this.ProxyType = proxyType;
            return _list.Count;
        }

        /// <summary>
        /// Set the type of loaded proxies
        /// </summary>
        /// <param name="proxyType">Proxy Type to set</param>
        public void SetProxyType(ProxyType proxyType)
        {
            this.ProxyType = proxyType;
        }

        /// <summary>
        /// Are we in auto mode?
        /// </summary>
        /// <returns>Returns if we are in auto mode</returns>
        public bool GetAutoMode()
        {
            return _autoMode;
        }

        /// <summary>
        /// Get the type of proxies loaded
        /// </summary>
        /// <returns>Type of loaded proxies</returns>
        public ProxyType GetProxyType()
        {
            return this.ProxyType;
        }

        /// <summary>
        /// Gets a random proxy
        /// </summary>
        /// <returns>ProxyClient object of proxy</returns>
        public ProxyClient GetProxyClient()
        {
            try
            {
                return _list.FindAll(k => !k.Banned)[rnd.Next(_list.Count - 1)].Client;
            } catch (Exception e)
            {
                return null; // Out of proxies :D
            }
        }

        public Proxy GetProxy()
        {
            if (_proxyless)
                return null;
            var living = _list.FindAll(k => !k.Banned);
            if (living.Count <= 0)
            {
                ResetProxies();
                return GetProxy();
            }
            return living[rnd.Next(living.Count - 1)];
        }

        /// <summary>
        /// Get random proxy string
        /// </summary>
        /// <returns>Returns ip:port</returns>
        public string GetProxyString()
        {
            var a = _list.FindAll(k => !k.Banned)[rnd.Next(_list.Count - 1)];
            return a.Client.Host +":"+ a.Client.Port;
        }

        /// <summary>
        /// Self explanatory
        /// </summary>
        /// <param name="type">Type of proxies to grab in auto mode</param>
        public void EnterAutoMode(ProxyType type = ProxyType.HTTP)
        {
            this._autoMode = true;
            this.ProxyType = type;
        }

        public int GetLiving()
        {
            return _list.FindAll(k => !k.Banned).Count;
        }

        public int GetDead()
        {
            return _list.FindAll(k => k.Banned).Count;
        }

        /// <summary>
        /// Self explanatory
        /// </summary>
        public void ExitAutoMode()
        {
            this._autoMode = false;
            this._list.Clear();
        }

        /// <summary>
        /// Pulls proxies by country
        /// </summary>
        /// <param name="country">Either regional code (US,EU,etc.) or "all"</param>
        public void PullProxies(string country = "all")
        {
            lock(_list)
            {
                _list.Clear();
                switch(this.GetProxyType())
                {
                    case ProxyType.HTTP:
                        PullHTTP(country);
                        break;
                    case ProxyType.Socks4:
                        PullSocks4(country);
                        break;
                    case ProxyType.Socks5:
                        PullSocks5(country);
                        break;
                }
                //cleanList();
            }
        }

        private void cleanList()
        {
            for (int i=0; i < _list.Count; i++)
            {
                try
                {
                    //_list[i] = _list[i].Substring(0, _list[i] - 1);
                }
                catch { }
            }
        }

        /// <summary>
        /// Pulls HTTP proxies in auto mode
        /// </summary>
        /// <param name="country">Either regional code (US,EU,etc.) or "all"</param>
        public void PullHTTP(string country = "all")
        {
            HttpRequest req = new HttpRequest();
            var l = req.Get(string.Format(httpsUri, country)).ToString().Split('\n');
            _list = new List<Proxy>();
            req.Dispose();
            foreach(var a in l)
            {
                _list.Add(new Proxy(a.Replace("\r", ""), ProxyType.HTTP));
            }
        }

        internal void SetProxyless(bool v)
        {
            this._proxyless = v;
        }

        /// <summary>
        /// Pulls SOCKS4 proxies in auto mode
        /// </summary>
        /// <param name="country">Either regional code (US,EU,etc.) or "all"</param>
        public void PullSocks4(string country = "all")
        {
            HttpRequest req = new HttpRequest();
            _list = new List<Proxy>();
            var l = req.Get(string.Format(socks4Uri, country)).ToString().Split('\n');
            req.Dispose();
            foreach (var a in l)
            {
                _list.Add(new Proxy(a.Replace("\r", ""), ProxyType.Socks4));
            }
        }

        /// <summary>
        /// Pulls SOCKS5 proxies in auto mode
        /// </summary>
        /// <param name="country">Either regional code (US,EU,etc.) or "all"</param>
        public void PullSocks5(string country = "all")
        {
            HttpRequest req = new HttpRequest();
            var lines = req.Get(string.Format(socks5Uri,country)).ToString().Split('\n');
            _list = new List<Proxy>();
            req.Dispose();
            foreach (var line in lines)
            {
                _list.Add(new Proxy(line.Replace("\r", ""), ProxyType.Socks5));
            }
        }

        public void SetBanned(Proxy proxy, bool banned = true)
        {
            lock(_list)
            {
                _list.Find(k => k.Client == proxy.Client).Banned = banned;
            }
        }

        public void ResetProxies()
        {
            lock(_list)
                foreach(var proxy in _list)
                {
                    proxy.Banned = false;
                }
        }

        /// <summary>
        /// Get count of loaded proxies
        /// </summary>
        /// <returns>Proxy count</returns>
        public int Count()
        {
            lock(_list)
            {
                return _list.Count;
            }
        }

        /// <summary>
        /// Clears the proxy list
        /// </summary>
        public void Reset()
        {
            lock(_list)
                _list.Clear();
        }
    }

    public class Proxy
    {
        public ProxyClient Client;
        public bool Banned { get; set; }
        public ProxyType Type;

        public Proxy(string combo, ProxyType type)
        {
            Client = ProxyClient.Parse(type, combo);
        }
    }
}
