package commands

import (
	"git.quartzinc.dev/Zertex/QuartzSB/utils"
	"encoding/json"
	"errors"
	"fmt"
	"github.com/bwmarrin/discordgo"
	"io/ioutil"
	"net"
	"net/http"
)

func NewIPInfoCMD() *Command {
	return &Command{
		cmds: []string{"ip", "iplookup"},
		name:        "IP Info",
		description: "Gets information about an IP",
		exec:        execIPInfoCMD,
		_type:       FUN,
	}
}

func execIPInfoCMD(s *discordgo.Session, m *discordgo.MessageCreate, args []string, ret chan CommandRet) {
	var cRet CommandRet
	defer func() {
		ret <- cRet
	}()
	if len(args) != 1 {
		cRet.error = errors.New("invalid arguments")
	}

	emb := utils.NewEmbed().
		SetTitle(fmt.Sprintf("Information for %s", args[0])).
		SetColor(utils.RandomColor())

	hosts, err := net.LookupHost(args[0])
	if err != nil {
		cRet.error = err
		return
	}
	host := "N/A"
	if len(hosts) > 0 {
		host = hosts[0]
	}

	resp, err := http.Get(fmt.Sprintf("http://ip-api.com/json/%s", args[0]))
	if err != nil {
		emb.SetDescription("There was an error contacting the API")
	} else {
		raw, _ := ioutil.ReadAll(resp.Body)
		resp.Body.Close()

		var res *ipAPIResponse
		if err = json.Unmarshal(raw, &res); err != nil {
			emb.SetDescription("Invalid JSON response")
		} else {
			emb.AddFieldInline("Country", fmt.Sprintf("%s (%s)", res.Country, res.CountryCode))
			emb.AddFieldInline("Region", fmt.Sprintf("%s (%s)", res.RegionName, res.Region))
			emb.AddFieldInline("City", res.City)
			emb.AddFieldInline("Hostname", host)
			emb.AddFieldInline("AS", res.As)
			emb.AddFieldInline("Organization", res.Org)
			emb.AddFieldInline("ISP", res.Isp)
			emb.AddFieldInline("TimeZone", res.Timezone)
			emb.AddFieldInline("Lat/Lon", fmt.Sprintf("%f/%f", res.Lat, res.Lon))
		}
		emb.SetWatermark(1)
		emb.SetThemecolor()
	}

	empty := ""
	s.ChannelMessageEditComplex(&discordgo.MessageEdit{
		Content: &empty,
		Embed:   emb.MessageEmbed,
		ID:      m.ID,
		Channel: m.ChannelID,
	})
}

type ipAPIResponse struct {
	Query       string  `json:"query"`
	Status      string  `json:"status"`
	Country     string  `json:"country"`
	CountryCode string  `json:"countryCode"`
	Region      string  `json:"region"`
	RegionName  string  `json:"regionName"`
	City        string  `json:"city"`
	Zip         string  `json:"zip"`
	Lat         float64 `json:"lat"`
	Lon         float64 `json:"lon"`
	Timezone    string  `json:"timezone"`
	Isp         string  `json:"isp"`
	Org         string  `json:"org"`
	As          string  `json:"as"`
}
