package server

import (
	"errors"
	"fmt"
	"git.quartzinc.dev/Zertex/XDGv2/utils"
	protocol "git.quartzinc.dev/xiQQ/Quartz-X/apiproto"
	_ "github.com/go-sql-driver/mysql"
	"io"
	"log"
	"net"
	"sync"
)

type Client struct {
	Id     string
	conn   net.Conn
	Name   string
	Writer *protocol.CommandWriter
}

type TcpServer struct {
	listener net.Listener
	clients  []*Client
	mutex    *sync.Mutex
}

type API interface {
	Listen(address string) error
	Broadcast(command interface{}) error
	Start()
	Close()
	GetClients() []*Client
}

var APIServer API

var (
	UnknownClient = errors.New("unknown client")
)

func NewServer() *TcpServer {
	return &TcpServer{
		mutex: &sync.Mutex{},
	}
}

func (s *TcpServer) GetClientNameArray() []string {
	cl := []string{}
	for _, k := range s.clients {
		cl = append(cl, k.Name)
	}
	return cl
}

func (s *TcpServer) Listen(address string) error {
	l, err := net.Listen("tcp", address)

	if err == nil {
		s.listener = l
	}

	log.Printf("Listening on %v", address)

	return err
}

func (s *TcpServer) Close() {
	s.listener.Close()
}

func (s *TcpServer) Start() {
	for {
		// XXX: need a way to break the loop
		conn, err := s.listener.Accept()

		if err != nil {
			log.Print(err)
		} else {
			// handle connection
			client := s.accept(conn)
			go s.serve(client)
		}
	}
}

func (s *TcpServer) Broadcast(command interface{}) error {
	for _, client := range s.clients {
		// TODO: handle error here?
		client.Writer.Write(command)
	}

	return nil
}

func (s *TcpServer) GetClients() []*Client {
	return s.clients
}

func (s *TcpServer) Send(name string, command interface{}) error {
	for _, client := range s.clients {
		if client.Name == name {
			return client.Writer.Write(command)
		}
	}

	return UnknownClient
}

func (s *TcpServer) accept(conn net.Conn) *Client {
	log.Printf("Accepting connection from %v, total clients: %v", conn.RemoteAddr().String(), len(s.clients)+1)

	s.mutex.Lock()
	defer s.mutex.Unlock()

	client := &Client{
		Id:     utils.StringOfLength(5),
		conn:   conn,
		Writer: protocol.NewCommandWriter(conn),
	}

	s.clients = append(s.clients, client)

	return client
}

func (s *TcpServer) remove(client *Client) {
	s.mutex.Lock()
	defer s.mutex.Unlock()

	// remove the connections from clients array
	for i, check := range s.clients {
		if check == client {
			s.clients = append(s.clients[:i], s.clients[i+1:]...)
		}
	}

	log.Printf("Closing connection from %v", client.conn.RemoteAddr().String())
	client.conn.Close()
}

func (s *TcpServer) serve(client *Client) {
	cmdReader := protocol.NewCommandReader(client.conn)

	defer func() {
		s.remove(client)
		if len(client.Name) > 0 {
			s.Broadcast(protocol.MessageCommand{
				Name:    client.Name,
				Message: "has left the room.",
			})
			s.Broadcast(protocol.UserList{Users: s.GetClientNameArray()})
		}
	}()

	for {
		cmd, err := cmdReader.Read()

		if err != nil && err != io.EOF {
			if err == protocol.UnknownCommand {
				return
			}
			log.Printf("Read error: %v", err)
		}

		if cmd != nil {
			switch v := cmd.(type) {
			case protocol.Heartbeat:
				client.Writer.Write(heartbeat())
			case protocol.SendCommand:
				go s.Broadcast(protocol.MessageCommand{
					Message: v.Message,
					Name:    client.Name,
				})
			case protocol.LoginCommand:
				ret := login(v)
				client.Writer.Write(ret)
				if ret.Status == "success" {
					utils.LogInfo(fmt.Sprintf("[%s] Authenticated user %s", client.conn.RemoteAddr().String(), v.Username))
					client.Name = v.Username
					go s.Broadcast(protocol.UserList{
						Users: s.GetClientNameArray(),
					})
					go s.Broadcast(protocol.MessageCommand{
						Name:    client.Name,
						Message: "has joined the room.",
					})
				} else {
					utils.LogError(fmt.Sprintf("[%s] User failed authentication %s", client.conn.RemoteAddr().String(), v.Username))
				}
			case protocol.RegisterCommand:
				client.Writer.Write(register(v))
			case protocol.RedeemCommand:
				client.Writer.Write(redeem(v))
			case protocol.VarCommand:
				err = client.Writer.Write(nvar(v))
			case protocol.NameCommand:
				client.Name = v.Name
				go s.Broadcast(protocol.UserList{
					Users: s.GetClientNameArray(),
				})
				go s.Broadcast(protocol.MessageCommand{
					Name:    client.Name,
					Message: "has joined the room.",
				})
			}
		}

		if err == io.EOF {
			break
		}
	}
}
