import requests
import shutil
import subprocess
import sys
import time
import re
import xml.etree.ElementTree as ET
from pathlib import Path
from concurrent.futures import ThreadPoolExecutor, wait, FIRST_COMPLETED
from requests.adapters import HTTPAdapter
from urllib3.util.retry import Retry
from tqdm import tqdm

# --- Configuration ---
LEGACY_BASE_URL = "https://bfgc2.miroppb.com/asot/"
RSS_URL = "https://miroppb.com/rss"
OUTPUT_DIR = Path("downloads")
THREADS = 4 
BITRATE = "96k"

def get_session():
    session = requests.Session()
    retries = Retry(total=3, backoff_factor=1, status_forcelist=[500, 502, 503, 504])
    session.mount("https://", HTTPAdapter(max_retries=retries))
    return session

http = get_session()
HAS_FFMPEG = shutil.which("ffmpeg") is not None

def fetch_rss_data():
    """
    Parses the RSS feed to get a map of {episode_number: download_url}
    and determines the absolute latest episode number.
    """
    print("Checking RSS feed for latest episodes...")
    try:
        r = http.get(RSS_URL, timeout=10)
        r.raise_for_status()
        
        root = ET.fromstring(r.content)
        rss_map = {}
        max_ep = 0
        
        for item in root.findall('./channel/item'):
            title = item.find('title')
            enclosure = item.find('enclosure')
            
            if title is not None and enclosure is not None:
                title_text = title.text
                # Extract "1262" from "Episode 1262"
                match = re.search(r"Episode\s+(\d+)", title_text, re.IGNORECASE)
                
                if match:
                    ep_num = int(match.group(1))
                    url = enclosure.get('url')
                    rss_map[ep_num] = url
                    if ep_num > max_ep:
                        max_ep = ep_num
                        
        return rss_map, max_ep
    except Exception as e:
        print(f"Warning: Could not fetch RSS feed ({e}). Using legacy fallback.")
        return {}, 0

def get_last_downloaded():
    """Scans the output directory for the highest episode number."""
    if not OUTPUT_DIR.exists():
        return 0
    episodes = []
    for f in OUTPUT_DIR.glob("ASOT_Ep_*.*"):
        match = re.search(r"ASOT_Ep_(\d+)", f.name)
        if match:
            episodes.append(int(match.group(1)))
    return max(episodes) if episodes else 0

def process_episode(ep_num, convert, slot_id, rss_map):
    filename = f"ASOT_Ep_{ep_num:03d}.mp3"
    
    # Priority: RSS URL -> Legacy Guess
    if ep_num in rss_map:
        url = rss_map[ep_num]
    else:
        url = f"{LEGACY_BASE_URL}{filename}"

    save_path = OUTPUT_DIR / filename
    temp_path = OUTPUT_DIR / f"{filename}.part"
    opus_path = save_path.with_suffix(".opus")

    # Position 0 is now the first thread, since we removed the main bar
    bar_pos = slot_id

    if opus_path.exists() or (not convert and save_path.exists()):
        return "exists", ep_num, slot_id

    try:
        with http.get(url, stream=True, timeout=15) as r:
            if r.status_code == 404:
                return "404", ep_num, slot_id
            r.raise_for_status()
            
            total_size = int(r.headers.get('content-length', 0))
            
            with tqdm(
                desc=f"Ep {ep_num:03d}",
                total=total_size,
                unit='iB',
                unit_scale=True,
                leave=False,        
                position=bar_pos,   
                mininterval=0.1
            ) as bar:
                with open(temp_path, 'wb') as f:
                    for chunk in r.iter_content(chunk_size=8192):
                        f.write(chunk)
                        bar.update(len(chunk))
        
        temp_path.rename(save_path)

        if convert and HAS_FFMPEG:
            cmd = [
                "ffmpeg", "-y", "-i", str(save_path),
                "-c:a", "libopus",
                "-b:a", BITRATE,
                "-vbr", "on",
                "-compression_level", "5", 
                "-loglevel", "error",
                str(opus_path)
            ]
            subprocess.run(cmd, check=True, stdout=subprocess.DEVNULL, stderr=subprocess.DEVNULL)
            save_path.unlink()
            
        return "success", ep_num, slot_id

    except Exception as e:
        if temp_path.exists(): temp_path.unlink()
        return f"error: {e}", ep_num, slot_id

def main():
    print("=== A State of Trance Downloader (ASOTdl) ===")
    OUTPUT_DIR.mkdir(exist_ok=True, parents=True)
    
    # 1. RSS Fetch
    rss_map, max_rss_ep = fetch_rss_data()
    
    # 2. Input Logic
    last_local_ep = get_last_downloaded()
    suggested_start = last_local_ep + 1 if last_local_ep > 0 else 0
    
    print(f"Latest Episode Found Locally: {last_local_ep} | Latest Episode Uploaded: {max_rss_ep if max_rss_ep else 'Unknown'}")

    start_input = input(f"Start Episode [{suggested_start}]: ").strip()
    start = int(start_input) if start_input else suggested_start
    
    stop_prompt = f"Stop Episode [{max_rss_ep}]" if max_rss_ep else "Stop Episode"
    stop_input = input(f"{stop_prompt}: ").strip()
    
    if stop_input:
        stop = int(stop_input)
    elif max_rss_ep > 0:
        stop = max_rss_ep
    else:
        stop = None

    msg = "Convert to Opus? This saves about 50% in file size, while perserving the sound quality. If file size isn't an issue, press N. [y/n]: "
    do_conv = input(msg).lower().startswith('y')

    print("\nStarting downloads.. (Downloading 4 at a time. Please wait if nothing is shown.)")

    current_ep = start
    consecutive_404s = 0
    active_futures = set()
    available_slots = list(range(THREADS))
    
    with ThreadPoolExecutor(max_workers=THREADS) as executor:
        # Initial fill
        while len(active_futures) < THREADS:
            if stop and current_ep > stop: break
            slot = available_slots.pop(0)
            f = executor.submit(process_episode, current_ep, do_conv, slot, rss_map)
            active_futures.add(f)
            current_ep += 1

        # Process loop
        while active_futures:
            done, active_futures = wait(active_futures, return_when=FIRST_COMPLETED)

            for f in done:
                status, ep_num, slot = f.result()
                available_slots.append(slot)
                available_slots.sort() 

                if "success" in status or "exists" in status:
                    consecutive_404s = 0
                elif "404" in status:
                    consecutive_404s += 1
                
            if not stop and consecutive_404s >= 5:
                print(f"\nStopped: Too many consecutive 404s (reached Ep {current_ep}).")
                break

            while len(active_futures) < THREADS and available_slots:
                if stop and current_ep > stop: break
                if not stop and consecutive_404s >= 5: break
                
                slot = available_slots.pop(0)
                f = executor.submit(process_episode, current_ep, do_conv, slot, rss_map)
                active_futures.add(f)
                current_ep += 1

    # Clear space after bars
    print("\n" * THREADS)
    print(f"Task Complete.")

if __name__ == "__main__":
    main()