import bpy
import math
import json
import collections
import traceback
from math import pi
from bpy.props import StringProperty
from mathutils import Euler, Matrix, Quaternion, Vector
from rna_prop_ui import rna_idprop_quote_path

rig_id = "x4i0wdna76224586"


############################
## Math utility functions ##
############################

def perpendicular_vector(v):
    """ Returns a vector that is perpendicular to the one given.
        The returned vector is _not_ guaranteed to be normalized.
    """
    # Create a vector that is not aligned with v.
    # It doesn't matter what vector.  Just any vector
    # that's guaranteed to not be pointing in the same
    # direction.
    if abs(v[0]) < abs(v[1]):
        tv = Vector((1,0,0))
    else:
        tv = Vector((0,1,0))

    # Use cross product to generate a vector perpendicular to
    # both tv and (more importantly) v.
    return v.cross(tv)


def rotation_difference(mat1, mat2):
    """ Returns the shortest-path rotational difference between two
        matrices.
    """
    q1 = mat1.to_quaternion()
    q2 = mat2.to_quaternion()
    angle = math.acos(min(1,max(-1,q1.dot(q2)))) * 2
    if angle > pi:
        angle = -angle + (2*pi)
    return angle

def find_min_range(f,start_angle,delta=pi/8):
    """ finds the range where lies the minimum of function f applied on bone_ik and bone_fk
        at a certain angle.
    """
    angle = start_angle
    while (angle > (start_angle - 2*pi)) and (angle < (start_angle + 2*pi)):
        l_dist = f(angle-delta)
        c_dist = f(angle)
        r_dist = f(angle+delta)
        if min((l_dist,c_dist,r_dist)) == c_dist:
            return (angle-delta,angle+delta)
        else:
            angle=angle+delta

def ternarySearch(f, left, right, absolutePrecision):
    """
    Find minimum of uni-modal function f() within [left, right]
    To find the maximum, revert the if/else statement or revert the comparison.
    """
    while True:
        #left and right are the current bounds; the maximum is between them
        if abs(right - left) < absolutePrecision:
            return (left + right)/2

        leftThird = left + (right - left)/3
        rightThird = right - (right - left)/3

        if f(leftThird) > f(rightThird):
            left = leftThird
        else:
            right = rightThird

def flatten_children(iterable):
    """Enumerate the iterator items as well as their children in the tree order."""
    for item in iterable:
        yield item
        yield from flatten_children(item.children)


######################
## Keyframing tools ##
######################

def get_keying_flags(context):
    "Retrieve the general keyframing flags from user preferences."
    prefs = context.preferences
    ts = context.scene.tool_settings
    flags = set()
    # Not adding INSERTKEY_VISUAL
    if prefs.edit.use_keyframe_insert_needed:
        flags.add('INSERTKEY_NEEDED')
    if ts.use_keyframe_cycle_aware:
        flags.add('INSERTKEY_CYCLE_AWARE')
    return flags

def get_autokey_flags(context, ignore_keyingset=False):
    "Retrieve the Auto Keyframe flags, or None if disabled."
    ts = context.scene.tool_settings
    if ts.use_keyframe_insert_auto and (ignore_keyingset or not ts.use_keyframe_insert_keyingset):
        flags = get_keying_flags(context)
        if context.preferences.edit.use_keyframe_insert_available:
            flags.add('INSERTKEY_AVAILABLE')
        if ts.auto_keying_mode == 'REPLACE_KEYS':
            flags.add('INSERTKEY_REPLACE')
        return flags
    else:
        return None

def add_flags_if_set(base, new_flags):
    "Add more flags if base is not None."
    if base is None:
        return None
    else:
        return base | new_flags

def get_4d_rot_lock(bone):
    "Retrieve the lock status for 4D rotation."
    if bone.lock_rotations_4d:
        return [bone.lock_rotation_w, *bone.lock_rotation]
    else:
        return [all(bone.lock_rotation)] * 4

def keyframe_transform_properties(obj, bone_name, keyflags, *,
                                  ignore_locks=False, no_loc=False, no_rot=False, no_scale=False):
    "Keyframe transformation properties, taking flags and mode into account, and avoiding keying locked channels."
    bone = obj.pose.bones[bone_name]

    def keyframe_channels(prop, locks):
        if ignore_locks or not all(locks):
            if ignore_locks or not any(locks):
                bone.keyframe_insert(prop, group=bone_name, options=keyflags)
            else:
                for i, lock in enumerate(locks):
                    if not lock:
                        bone.keyframe_insert(prop, index=i, group=bone_name, options=keyflags)

    if not (no_loc or bone.bone.use_connect):
        keyframe_channels('location', bone.lock_location)

    if not no_rot:
        if bone.rotation_mode == 'QUATERNION':
            keyframe_channels('rotation_quaternion', get_4d_rot_lock(bone))
        elif bone.rotation_mode == 'AXIS_ANGLE':
            keyframe_channels('rotation_axis_angle', get_4d_rot_lock(bone))
        else:
            keyframe_channels('rotation_euler', bone.lock_rotation)

    if not no_scale:
        keyframe_channels('scale', bone.lock_scale)

######################
## Constraint tools ##
######################

def get_constraint_target_matrix(con):
    target = con.target
    if target:
        if target.type == 'ARMATURE' and con.subtarget:
            if con.subtarget in target.pose.bones:
                bone = target.pose.bones[con.subtarget]
                return target.convert_space(
                    pose_bone=bone, matrix=bone.matrix, from_space='POSE', to_space=con.target_space)
        else:
            return target.convert_space(matrix=target.matrix_world, from_space='WORLD', to_space=con.target_space)
    return Matrix.Identity(4)

def undo_copy_scale_with_offset(obj, bone, con, old_matrix):
    "Undo the effects of Copy Scale with Offset constraint on a bone matrix."
    inf = con.influence

    if con.mute or inf == 0 or not con.is_valid or not con.use_offset or con.use_add:
        return old_matrix

    tgt_matrix = get_constraint_target_matrix(con)
    tgt_scale = tgt_matrix.to_scale()
    use = [con.use_x, con.use_y, con.use_z]

    if con.use_make_uniform:
        if con.use_x and con.use_y and con.use_z:
            total = tgt_matrix.determinant()
        else:
            total = 1
            for i, use in enumerate(use):
                if use:
                    total *= tgt_scale[i]

        tgt_scale = [abs(total)**(1./3.)]*3
    else:
        for i, use in enumerate(use):
            if not use:
                tgt_scale[i] = 1

    scale_delta = [
        1 / (1 + (math.pow(x, con.power) - 1) * inf)
        for x in tgt_scale
    ]

    return old_matrix @ Matrix.Diagonal([*scale_delta, 1])

def undo_copy_scale_constraints(obj, bone, matrix):
    "Undo the effects of all Copy Scale with Offset constraints on a bone matrix."
    for con in reversed(bone.constraints):
        if con.type == 'COPY_SCALE':
            matrix = undo_copy_scale_with_offset(obj, bone, con, matrix)
    return matrix

###############################
## Assign and keyframe tools ##
###############################

def set_custom_property_value(obj, bone_name, prop, value, *, keyflags=None):
    "Assign the value of a custom property, and optionally keyframe it."
    from rna_prop_ui import rna_idprop_ui_prop_update
    bone = obj.pose.bones[bone_name]
    bone[prop] = value
    rna_idprop_ui_prop_update(bone, prop)
    if keyflags is not None:
        bone.keyframe_insert(rna_idprop_quote_path(prop), group=bone.name, options=keyflags)

def get_transform_matrix(obj, bone_name, *, space='POSE', with_constraints=True):
    "Retrieve the matrix of the bone before or after constraints in the given space."
    bone = obj.pose.bones[bone_name]
    if with_constraints:
        return obj.convert_space(pose_bone=bone, matrix=bone.matrix, from_space='POSE', to_space=space)
    else:
        return obj.convert_space(pose_bone=bone, matrix=bone.matrix_basis, from_space='LOCAL', to_space=space)

def get_chain_transform_matrices(obj, bone_names, **options):
    return [get_transform_matrix(obj, name, **options) for name in bone_names]

def set_transform_from_matrix(obj, bone_name, matrix, *, space='POSE', undo_copy_scale=False,
                              ignore_locks=False, no_loc=False, no_rot=False, no_scale=False, keyflags=None):
    """Apply the matrix to the transformation of the bone, taking locked channels, mode and certain
    constraints into account, and optionally keyframe it."""
    bone = obj.pose.bones[bone_name]

    def restore_channels(prop, old_vec, locks, extra_lock):
        if extra_lock or (not ignore_locks and all(locks)):
            setattr(bone, prop, old_vec)
        else:
            if not ignore_locks and any(locks):
                new_vec = Vector(getattr(bone, prop))

                for i, lock in enumerate(locks):
                    if lock:
                        new_vec[i] = old_vec[i]

                setattr(bone, prop, new_vec)

    # Save the old values of the properties
    old_loc = Vector(bone.location)
    old_rot_euler = Vector(bone.rotation_euler)
    old_rot_quat = Vector(bone.rotation_quaternion)
    old_rot_axis = Vector(bone.rotation_axis_angle)
    old_scale = Vector(bone.scale)

    # Compute and assign the local matrix
    if space != 'LOCAL':
        matrix = obj.convert_space(pose_bone=bone, matrix=matrix, from_space=space, to_space='LOCAL')

    if undo_copy_scale:
        matrix = undo_copy_scale_constraints(obj, bone, matrix)

    bone.matrix_basis = matrix

    # Restore locked properties
    restore_channels('location', old_loc, bone.lock_location, no_loc or bone.bone.use_connect)

    if bone.rotation_mode == 'QUATERNION':
        restore_channels('rotation_quaternion', old_rot_quat, get_4d_rot_lock(bone), no_rot)
        bone.rotation_axis_angle = old_rot_axis
        bone.rotation_euler = old_rot_euler
    elif bone.rotation_mode == 'AXIS_ANGLE':
        bone.rotation_quaternion = old_rot_quat
        restore_channels('rotation_axis_angle', old_rot_axis, get_4d_rot_lock(bone), no_rot)
        bone.rotation_euler = old_rot_euler
    else:
        bone.rotation_quaternion = old_rot_quat
        bone.rotation_axis_angle = old_rot_axis
        restore_channels('rotation_euler', old_rot_euler, bone.lock_rotation, no_rot)

    restore_channels('scale', old_scale, bone.lock_scale, no_scale)

    # Keyframe properties
    if keyflags is not None:
        keyframe_transform_properties(
            obj, bone_name, keyflags, ignore_locks=ignore_locks,
            no_loc=no_loc, no_rot=no_rot, no_scale=no_scale
        )

def set_chain_transforms_from_matrices(context, obj, bone_names, matrices, **options):
    for bone, matrix in zip(bone_names, matrices):
        set_transform_from_matrix(obj, bone, matrix, **options)
        context.view_layer.update()


###########################
## Animation curve tools ##
###########################

def flatten_curve_set(curves):
    "Iterate over all FCurves inside a set of nested lists and dictionaries."
    if curves is None:
        pass
    elif isinstance(curves, bpy.types.FCurve):
        yield curves
    elif isinstance(curves, dict):
        for sub in curves.values():
            yield from flatten_curve_set(sub)
    else:
        for sub in curves:
            yield from flatten_curve_set(sub)

def flatten_curve_key_set(curves, key_range=None):
    "Iterate over all keys of the given fcurves in the specified range."
    for curve in flatten_curve_set(curves):
        for key in curve.keyframe_points:
            if key_range is None or key_range[0] <= key.co[0] <= key_range[1]:
                yield key

def get_curve_frame_set(curves, key_range=None):
    "Compute a set of all time values with existing keys in the given curves and range."
    return set(key.co[0] for key in flatten_curve_key_set(curves, key_range))

def set_curve_key_interpolation(curves, ipo, key_range=None):
    "Assign the given interpolation value to all curve keys in range."
    for key in flatten_curve_key_set(curves, key_range):
        key.interpolation = ipo

def delete_curve_keys_in_range(curves, key_range=None):
    "Delete all keys of the given curves within the given range."
    for curve in flatten_curve_set(curves):
        points = curve.keyframe_points
        for i in range(len(points), 0, -1):
            key = points[i - 1]
            if key_range is None or key_range[0] <= key.co[0] <= key_range[1]:
                points.remove(key, fast=True)
        curve.update()

def nla_tweak_to_scene(anim_data, frames, invert=False):
    "Convert a frame value or list between scene and tweaked NLA strip time."
    if frames is None:
        return None
    elif anim_data is None or not anim_data.use_tweak_mode:
        return frames
    elif isinstance(frames, (int, float)):
        return anim_data.nla_tweak_strip_time_to_scene(frames, invert=invert)
    else:
        return type(frames)(
            anim_data.nla_tweak_strip_time_to_scene(v, invert=invert) for v in frames
        )

def find_action(action):
    if isinstance(action, bpy.types.Object):
        action = action.animation_data
    if isinstance(action, bpy.types.AnimData):
        action = action.action
    if isinstance(action, bpy.types.Action):
        return action
    else:
        return None

def clean_action_empty_curves(action):
    "Delete completely empty curves from the given action."
    action = find_action(action)
    for curve in list(action.fcurves):
        if curve.is_empty:
            action.fcurves.remove(curve)
    action.update_tag()

TRANSFORM_PROPS_LOCATION = frozenset(['location'])
TRANSFORM_PROPS_ROTATION = frozenset(['rotation_euler', 'rotation_quaternion', 'rotation_axis_angle'])
TRANSFORM_PROPS_SCALE = frozenset(['scale'])
TRANSFORM_PROPS_ALL = frozenset(TRANSFORM_PROPS_LOCATION | TRANSFORM_PROPS_ROTATION | TRANSFORM_PROPS_SCALE)

def transform_props_with_locks(lock_location, lock_rotation, lock_scale):
    props = set()
    if not lock_location:
        props |= TRANSFORM_PROPS_LOCATION
    if not lock_rotation:
        props |= TRANSFORM_PROPS_ROTATION
    if not lock_scale:
        props |= TRANSFORM_PROPS_SCALE
    return props

class FCurveTable(object):
    "Table for efficient lookup of FCurves by properties."

    def __init__(self):
        self.curve_map = collections.defaultdict(dict)

    def index_curves(self, curves):
        for curve in curves:
            index = curve.array_index
            if index < 0:
                index = 0
            self.curve_map[curve.data_path][index] = curve

    def get_prop_curves(self, ptr, prop_path):
        "Returns a dictionary from array index to curve for the given property, or Null."
        return self.curve_map.get(ptr.path_from_id(prop_path))

    def list_all_prop_curves(self, ptr_set, path_set):
        "Iterates over all FCurves matching the given object(s) and properties."
        if isinstance(ptr_set, bpy.types.bpy_struct):
            ptr_set = [ptr_set]
        for ptr in ptr_set:
            for path in path_set:
                curves = self.get_prop_curves(ptr, path)
                if curves:
                    yield from curves.values()

    def get_custom_prop_curves(self, ptr, prop):
        return self.get_prop_curves(ptr, rna_idprop_quote_path(prop))

class ActionCurveTable(FCurveTable):
    "Table for efficient lookup of Action FCurves by properties."

    def __init__(self, action):
        super().__init__()
        self.action = find_action(action)
        if self.action:
            self.index_curves(self.action.fcurves)

class DriverCurveTable(FCurveTable):
    "Table for efficient lookup of Driver FCurves by properties."

    def __init__(self, object):
        super().__init__()
        self.anim_data = object.animation_data
        if self.anim_data:
            self.index_curves(self.anim_data.drivers)


##################################
# Common bake operator settings ##
##################################

bpy.types.WindowManager.rigify_transfer_use_all_keys = bpy.props.BoolProperty(
    name="Bake All Keyed Frames",
    description="Bake on every frame that has a key for any of the bones, as opposed to just the relevant ones",
    default=False
)
bpy.types.WindowManager.rigify_transfer_use_frame_range = bpy.props.BoolProperty(
    name="Limit Frame Range", description="Only bake keyframes in a certain frame range", default=False
)
bpy.types.WindowManager.rigify_transfer_start_frame = bpy.props.IntProperty(
    name="Start", description="First frame to transfer", default=0, min=0
)
bpy.types.WindowManager.rigify_transfer_end_frame = bpy.props.IntProperty(
    name="End", description="Last frame to transfer", default=0, min=0
)

class RIGIFY_OT_get_frame_range(bpy.types.Operator):
    bl_idname = "rigify.get_frame_range" + ('_'+rig_id if rig_id else '')
    bl_label = "Get Frame Range"
    bl_description = "Set start and end frame from scene"
    bl_options = {'INTERNAL'}

    def execute(self, context):
        scn = context.scene
        id_store = context.window_manager
        id_store.rigify_transfer_start_frame = scn.frame_start
        id_store.rigify_transfer_end_frame = scn.frame_end
        return {'FINISHED'}

    @staticmethod
    def get_range(context):
        id_store = context.window_manager
        if not id_store.rigify_transfer_use_frame_range:
            return None
        else:
            return (id_store.rigify_transfer_start_frame, id_store.rigify_transfer_end_frame)

    @classmethod
    def draw_range_ui(self, context, layout):
        id_store = context.window_manager

        row = layout.row(align=True)
        row.prop(id_store, 'rigify_transfer_use_frame_range', icon='PREVIEW_RANGE', text='')

        row = row.row(align=True)
        row.active = id_store.rigify_transfer_use_frame_range
        row.prop(id_store, 'rigify_transfer_start_frame')
        row.prop(id_store, 'rigify_transfer_end_frame')
        row.operator(self.bl_idname, icon='TIME', text='')

#######################################
# Keyframe baking operator framework ##
#######################################

class RigifyOperatorMixinBase:
    bl_options = {'UNDO', 'INTERNAL'}

    def init_invoke(self, context):
        "Override to initialize the operator before invoke."

    def init_execute(self, context):
        "Override to initialize the operator before execute."

    def before_save_state(self, context, rig):
        "Override to prepare for saving state."

    def after_save_state(self, context, rig):
        "Override to undo before_save_state."


class RigifyBakeKeyframesMixin(RigifyOperatorMixinBase):
    """Basic framework for an operator that updates a set of keyed frames."""

    # Utilities
    def nla_from_raw(self, frames):
        "Convert frame(s) from inner action time to scene time."
        return nla_tweak_to_scene(self.bake_anim, frames)

    def nla_to_raw(self, frames):
        "Convert frame(s) from scene time to inner action time."
        return nla_tweak_to_scene(self.bake_anim, frames, invert=True)

    def bake_get_bone(self, bone_name):
        "Get pose bone by name."
        return self.bake_rig.pose.bones[bone_name]

    def bake_get_bones(self, bone_names):
        "Get multiple pose bones by name."
        if isinstance(bone_names, (list, set)):
            return [self.bake_get_bone(name) for name in bone_names]
        else:
            return self.bake_get_bone(bone_names)

    def bake_get_all_bone_curves(self, bone_names, props):
        "Get a list of all curves for the specified properties of the specified bones."
        return list(self.bake_curve_table.list_all_prop_curves(self.bake_get_bones(bone_names), props))

    def bake_get_all_bone_custom_prop_curves(self, bone_names, props):
        "Get a list of all curves for the specified custom properties of the specified bones."
        return self.bake_get_all_bone_curves(bone_names, [rna_idprop_quote_path(p) for p in props])

    def bake_get_bone_prop_curves(self, bone_name, prop):
        "Get an index to curve dict for the specified property of the specified bone."
        return self.bake_curve_table.get_prop_curves(self.bake_get_bone(bone_name), prop)

    def bake_get_bone_custom_prop_curves(self, bone_name, prop):
        "Get an index to curve dict for the specified custom property of the specified bone."
        return self.bake_curve_table.get_custom_prop_curves(self.bake_get_bone(bone_name), prop)

    def bake_add_curve_frames(self, curves):
        "Register frames keyed in the specified curves for baking."
        self.bake_frames_raw |= get_curve_frame_set(curves, self.bake_frame_range_raw)

    def bake_add_bone_frames(self, bone_names, props):
        "Register frames keyed for the specified properties of the specified bones for baking."
        curves = self.bake_get_all_bone_curves(bone_names, props)
        self.bake_add_curve_frames(curves)
        return curves

    def bake_replace_custom_prop_keys_constant(self, bone, prop, new_value):
        "If the property is keyframed, delete keys in bake range and re-key as Constant."
        prop_curves = self.bake_get_bone_custom_prop_curves(bone, prop)

        if prop_curves and 0 in prop_curves:
            range_raw = self.nla_to_raw(self.get_bake_range())
            delete_curve_keys_in_range(prop_curves, range_raw)
            set_custom_property_value(self.bake_rig, bone, prop, new_value, keyflags={'INSERTKEY_AVAILABLE'})
            set_curve_key_interpolation(prop_curves, 'CONSTANT', range_raw)

    # Default behavior implementation
    def bake_init(self, context):
        self.bake_rig = context.active_object
        self.bake_anim = self.bake_rig.animation_data
        self.bake_frame_range = RIGIFY_OT_get_frame_range.get_range(context)
        self.bake_frame_range_raw = self.nla_to_raw(self.bake_frame_range)
        self.bake_curve_table = ActionCurveTable(self.bake_rig)
        self.bake_current_frame = context.scene.frame_current
        self.bake_frames_raw = set()
        self.bake_state = dict()

        self.keyflags = get_keying_flags(context)
        self.keyflags_switch = None

        if context.window_manager.rigify_transfer_use_all_keys:
            self.bake_add_curve_frames(self.bake_curve_table.curve_map)

    def bake_add_frames_done(self):
        "Computes and sets the final set of frames to bake."
        frames = self.nla_from_raw(self.bake_frames_raw)
        self.bake_frames = sorted(set(map(round, frames)))

    def is_bake_empty(self):
        return len(self.bake_frames_raw) == 0

    def report_bake_empty(self):
        self.bake_add_frames_done()
        if self.is_bake_empty():
            self.report({'WARNING'}, 'No keys to bake.')
            return True
        return False

    def get_bake_range(self):
        "Returns the frame range that is being baked."
        if self.bake_frame_range:
            return self.bake_frame_range
        else:
            frames = self.bake_frames
            return (frames[0], frames[-1])

    def get_bake_range_pair(self):
        "Returns the frame range that is being baked, both in scene and action time."
        range = self.get_bake_range()
        return range, self.nla_to_raw(range)

    def bake_save_state(self, context):
        "Scans frames and collects data for baking before changing anything."
        rig = self.bake_rig
        scene = context.scene
        saved_state = self.bake_state

        try:
            self.before_save_state(context, rig)

            for frame in self.bake_frames:
                scene.frame_set(frame)
                saved_state[frame] = self.save_frame_state(context, rig)

        finally:
            self.after_save_state(context, rig)

    def bake_clean_curves_in_range(self, context, curves):
        "Deletes all keys from the given curves in the bake range."
        range, range_raw = self.get_bake_range_pair()

        context.scene.frame_set(range[0])
        delete_curve_keys_in_range(curves, range_raw)

        return range, range_raw

    def bake_apply_state(self, context):
        "Scans frames and applies the baking operation."
        rig = self.bake_rig
        scene = context.scene
        saved_state = self.bake_state

        for frame in self.bake_frames:
            scene.frame_set(frame)
            self.apply_frame_state(context, rig, saved_state.get(frame))

        clean_action_empty_curves(self.bake_rig)
        scene.frame_set(self.bake_current_frame)

    @staticmethod
    def draw_common_bake_ui(context, layout):
        layout.prop(context.window_manager, 'rigify_transfer_use_all_keys')

        RIGIFY_OT_get_frame_range.draw_range_ui(context, layout)

    @classmethod
    def poll(cls, context):
        return find_action(context.active_object) is not None

    def execute_scan_curves(self, context, obj):
        "Override to register frames to be baked, and return curves that should be cleared."
        raise NotImplementedError()

    def execute_before_apply(self, context, obj, range, range_raw):
        "Override to execute code one time before the bake apply frame scan."
        pass

    def execute(self, context):
        self.init_execute(context)
        self.bake_init(context)

        curves = self.execute_scan_curves(context, self.bake_rig)

        if self.report_bake_empty():
            return {'CANCELLED'}

        try:
            self.bake_save_state(context)

            range, range_raw = self.bake_clean_curves_in_range(context, curves)

            self.execute_before_apply(context, self.bake_rig, range, range_raw)

            self.bake_apply_state(context)

        except Exception as e:
            traceback.print_exc()
            self.report({'ERROR'}, 'Exception: ' + str(e))

        return {'FINISHED'}

    def invoke(self, context, event):
        self.init_invoke(context)

        if hasattr(self, 'draw'):
            return context.window_manager.invoke_props_dialog(self)
        else:
            return context.window_manager.invoke_confirm(self, event)


class RigifySingleUpdateMixin(RigifyOperatorMixinBase):
    """Basic framework for an operator that updates only the current frame."""

    def execute(self, context):
        self.init_execute(context)
        obj = context.active_object
        self.keyflags = get_autokey_flags(context, ignore_keyingset=True)
        self.keyflags_switch = add_flags_if_set(self.keyflags, {'INSERTKEY_AVAILABLE'})

        try:
            try:
                self.before_save_state(context, obj)
                state = self.save_frame_state(context, obj)
            finally:
                self.after_save_state(context, obj)

            self.apply_frame_state(context, obj, state)

        except Exception as e:
            traceback.print_exc()
            self.report({'ERROR'}, 'Exception: ' + str(e))

        return {'FINISHED'}

    def invoke(self, context, event):
        self.init_invoke(context)

        if hasattr(self, 'draw'):
            return context.window_manager.invoke_props_popup(self, event)
        else:
            return self.execute(context)


#############################
## Generic Snap (FK to IK) ##
#############################

class RigifyGenericSnapBase:
    input_bones:   StringProperty(name="Input Chain")
    output_bones:  StringProperty(name="Output Chain")
    ctrl_bones:    StringProperty(name="Input Controls")

    tooltip:         StringProperty(name="Tooltip", default="FK to IK")
    locks:           bpy.props.BoolVectorProperty(name="Locked", size=3, default=[False,False,False])
    undo_copy_scale: bpy.props.BoolProperty(name="Undo Copy Scale", default=False)

    def init_execute(self, context):
        self.input_bone_list = json.loads(self.input_bones)
        self.output_bone_list = json.loads(self.output_bones)
        self.ctrl_bone_list = json.loads(self.ctrl_bones)

    def save_frame_state(self, context, obj):
        return get_chain_transform_matrices(obj, self.input_bone_list)

    def apply_frame_state(self, context, obj, matrices):
        set_chain_transforms_from_matrices(
            context, obj, self.output_bone_list, matrices,
            undo_copy_scale=self.undo_copy_scale, keyflags=self.keyflags,
            no_loc=self.locks[0], no_rot=self.locks[1], no_scale=self.locks[2],
        )

class POSE_OT_rigify_generic_snap(RigifyGenericSnapBase, RigifySingleUpdateMixin, bpy.types.Operator):
    bl_idname = "pose.rigify_generic_snap_" + rig_id
    bl_label = "Snap Bones"
    bl_description = "Snap on the current frame"

    @classmethod
    def description(cls, context, props):
        return "Snap " + props.tooltip + " on the current frame"

class POSE_OT_rigify_generic_snap_bake(RigifyGenericSnapBase, RigifyBakeKeyframesMixin, bpy.types.Operator):
    bl_idname = "pose.rigify_generic_snap_bake_" + rig_id
    bl_label = "Apply Snap To Keyframes"
    bl_description = "Apply snap to keyframes"

    @classmethod
    def description(cls, context, props):
        return "Apply snap " + props.tooltip + " to keyframes"

    def execute_scan_curves(self, context, obj):
        props = transform_props_with_locks(*self.locks)
        self.bake_add_bone_frames(self.ctrl_bone_list, TRANSFORM_PROPS_ALL)
        return self.bake_get_all_bone_curves(self.output_bone_list, props)


#############################
## Generic Clear Keyframes ##
#############################

class POSE_OT_rigify_clear_keyframes(bpy.types.Operator):
    bl_idname = "pose.rigify_clear_keyframes_" + rig_id
    bl_label = "Clear Keyframes And Transformation"
    bl_options = {'UNDO', 'INTERNAL'}
    bl_description = "Remove all keyframes for the relevant bones and reset transformation"

    bones: StringProperty(name="Bone List")

    @classmethod
    def poll(cls, context):
        return find_action(context.active_object) is not None

    def invoke(self, context, event):
        return context.window_manager.invoke_confirm(self, event)

    def execute(self, context):
        obj = context.active_object
        bone_list = [ obj.pose.bones[name] for name in json.loads(self.bones) ]

        curve_table = ActionCurveTable(context.active_object)
        curves = list(curve_table.list_all_prop_curves(bone_list, TRANSFORM_PROPS_ALL))

        key_range = RIGIFY_OT_get_frame_range.get_range(context)
        range_raw = nla_tweak_to_scene(obj.animation_data, key_range, invert=True)
        delete_curve_keys_in_range(curves, range_raw)

        for bone in bone_list:
            bone.location = bone.rotation_euler = (0,0,0)
            bone.rotation_quaternion = (1,0,0,0)
            bone.rotation_axis_angle = (0,0,1,0)
            bone.scale = (1,1,1)

        clean_action_empty_curves(obj)
        obj.update_tag(refresh={'TIME'})
        return {'FINISHED'}


#########################################
## "Visual Transform" helper functions ##
#########################################

def get_pose_matrix_in_other_space(mat, pose_bone):
    """ Returns the transform matrix relative to pose_bone's current
        transform space.  In other words, presuming that mat is in
        armature space, slapping the returned matrix onto pose_bone
        should give it the armature-space transforms of mat.
    """
    return pose_bone.id_data.convert_space(matrix=mat, pose_bone=pose_bone, from_space='POSE', to_space='LOCAL')


def convert_pose_matrix_via_rest_delta(mat, from_bone, to_bone):
    """Convert pose of one bone to another bone, preserving the rest pose difference between them."""
    return mat @ from_bone.bone.matrix_local.inverted() @ to_bone.bone.matrix_local


def convert_pose_matrix_via_pose_delta(mat, from_bone, to_bone):
    """Convert pose of one bone to another bone, preserving the current pose difference between them."""
    return mat @ from_bone.matrix.inverted() @ to_bone.matrix


def get_local_pose_matrix(pose_bone):
    """ Returns the local transform matrix of the given pose bone.
    """
    return get_pose_matrix_in_other_space(pose_bone.matrix, pose_bone)


def set_pose_translation(pose_bone, mat):
    """ Sets the pose bone's translation to the same translation as the given matrix.
        Matrix should be given in bone's local space.
    """
    pose_bone.location = mat.to_translation()


def set_pose_rotation(pose_bone, mat):
    """ Sets the pose bone's rotation to the same rotation as the given matrix.
        Matrix should be given in bone's local space.
    """
    q = mat.to_quaternion()

    if pose_bone.rotation_mode == 'QUATERNION':
        pose_bone.rotation_quaternion = q
    elif pose_bone.rotation_mode == 'AXIS_ANGLE':
        pose_bone.rotation_axis_angle[0] = q.angle
        pose_bone.rotation_axis_angle[1] = q.axis[0]
        pose_bone.rotation_axis_angle[2] = q.axis[1]
        pose_bone.rotation_axis_angle[3] = q.axis[2]
    else:
        pose_bone.rotation_euler = q.to_euler(pose_bone.rotation_mode)


def set_pose_scale(pose_bone, mat):
    """ Sets the pose bone's scale to the same scale as the given matrix.
        Matrix should be given in bone's local space.
    """
    pose_bone.scale = mat.to_scale()


def match_pose_translation(pose_bone, target_bone):
    """ Matches pose_bone's visual translation to target_bone's visual
        translation.
        This function assumes you are in pose mode on the relevant armature.
    """
    mat = get_pose_matrix_in_other_space(target_bone.matrix, pose_bone)
    set_pose_translation(pose_bone, mat)


def match_pose_rotation(pose_bone, target_bone):
    """ Matches pose_bone's visual rotation to target_bone's visual
        rotation.
        This function assumes you are in pose mode on the relevant armature.
    """
    mat = get_pose_matrix_in_other_space(target_bone.matrix, pose_bone)
    set_pose_rotation(pose_bone, mat)


def match_pose_scale(pose_bone, target_bone):
    """ Matches pose_bone's visual scale to target_bone's visual
        scale.
        This function assumes you are in pose mode on the relevant armature.
    """
    mat = get_pose_matrix_in_other_space(target_bone.matrix, pose_bone)
    set_pose_scale(pose_bone, mat)


##############################
## IK/FK snapping functions ##
##############################

def correct_rotation(view_layer, bone_ik, target_matrix, *, ctrl_ik=None):
    """ Corrects the ik rotation in ik2fk snapping functions
    """

    axis = target_matrix.to_3x3().col[1].normalized()
    ctrl_ik = ctrl_ik or bone_ik

    def distance(angle):
        # Rotate the bone and return the actual angle between bones
        ctrl_ik.rotation_euler[1] = angle
        view_layer.update()

        return -(bone_ik.vector.normalized().dot(axis))

    if ctrl_ik.rotation_mode in {'QUATERNION', 'AXIS_ANGLE'}:
        ctrl_ik.rotation_mode = 'ZXY'

    start_angle = ctrl_ik.rotation_euler[1]

    alpha_range = find_min_range(distance, start_angle)
    alpha_min = ternarySearch(distance, alpha_range[0], alpha_range[1], pi / 180)

    ctrl_ik.rotation_euler[1] = alpha_min
    view_layer.update()


def correct_scale(view_layer, bone_ik, target_matrix, *, ctrl_ik=None):
    """ Correct the scale of the base IK bone. """
    input_scale = target_matrix.to_scale()
    ctrl_ik = ctrl_ik or bone_ik

    for i in range(3):
        cur_scale = bone_ik.matrix.to_scale()

        ctrl_ik.scale = [
            v * i / c for v, i, c in zip(bone_ik.scale, input_scale, cur_scale)
        ]

        view_layer.update()

        if all(abs((c - i)/i) < 0.01 for i, c in zip(input_scale, cur_scale)):
            break


def match_pole_target(view_layer, ik_first, ik_last, pole, match_bone_matrix, length):
    """ Places an IK chain's pole target to match ik_first's
        transforms to match_bone.  All bones should be given as pose bones.
        You need to be in pose mode on the relevant armature object.
        ik_first: first bone in the IK chain
        ik_last:  last bone in the IK chain
        pole:  pole target bone for the IK chain
        match_bone:  bone to match ik_first to (probably first bone in a matching FK chain)
        length:  distance pole target should be placed from the chain center
    """
    a = ik_first.matrix.to_translation()
    b = ik_last.matrix.to_translation() + ik_last.vector

    # Vector from the head of ik_first to the
    # tip of ik_last
    ikv = b - a

    # Get a vector perpendicular to ikv
    pv = perpendicular_vector(ikv).normalized() * length

    def set_pole(pvi):
        """ Set pole target's position based on a vector
            from the arm center line.
        """
        # Translate pvi into armature space
        pole_loc = a + (ikv/2) + pvi

        # Set pole target to location
        mat = get_pose_matrix_in_other_space(Matrix.Translation(pole_loc), pole)
        set_pose_translation(pole, mat)

        view_layer.update()

    set_pole(pv)

    # Get the rotation difference between ik_first and match_bone
    angle = rotation_difference(ik_first.matrix, match_bone_matrix)

    # Try compensating for the rotation difference in both directions
    pv1 = Matrix.Rotation(angle, 4, ikv) @ pv
    set_pole(pv1)
    ang1 = rotation_difference(ik_first.matrix, match_bone_matrix)

    pv2 = Matrix.Rotation(-angle, 4, ikv) @ pv
    set_pole(pv2)
    ang2 = rotation_difference(ik_first.matrix, match_bone_matrix)

    # Do the one with the smaller angle
    if ang1 < ang2:
        set_pole(pv1)

##########
## Misc ##
##########

def parse_bone_names(names_string):
    if names_string[0] == '[' and names_string[-1] == ']':
        return eval(names_string)
    else:
        return names_string



########################
## Limb Snap IK to FK ##
########################

class RigifyLimbIk2FkBase:
    prop_bone:    StringProperty(name="Settings Bone")
    pole_prop:    StringProperty(name="Pole target switch", default="pole_vector")
    fk_bones:     StringProperty(name="FK Bone Chain")
    ik_bones:     StringProperty(name="IK Result Bone Chain")
    ctrl_bones:   StringProperty(name="IK Controls")
    tail_bones:   StringProperty(name="Tail IK Controls", default="[]")
    extra_ctrls:  StringProperty(name="Extra IK Controls")

    def init_execute(self, context):
        if self.fk_bones:
            self.fk_bone_list = json.loads(self.fk_bones)
        self.ik_bone_list = json.loads(self.ik_bones)
        self.ctrl_bone_list = json.loads(self.ctrl_bones)
        self.tail_bone_list = json.loads(self.tail_bones)
        self.extra_ctrl_list = json.loads(self.extra_ctrls)

    def get_use_pole(self, obj):
        bone = obj.pose.bones[self.prop_bone]
        return self.pole_prop in bone and bone[self.pole_prop]

    def save_frame_state(self, context, obj):
        return get_chain_transform_matrices(obj, self.fk_bone_list)

    def compute_base_rotation(self, context, ik_bones, ctrl_bones, matrices, use_pole):
        context.view_layer.update()

        if use_pole:
            match_pole_target(
                context.view_layer,
                ik_bones[0], ik_bones[1], ctrl_bones[1], matrices[0],
                (ik_bones[0].length + ik_bones[1].length)
            )

        else:
            correct_rotation(context.view_layer, ik_bones[0], matrices[0], ctrl_ik=ctrl_bones[0])

    def assign_middle_controls(self, context, obj, matrices, ik_bones, ctrl_bones, *, lock=False, keyflags=None):
        for mat, ik, ctrl in reversed(list(zip(matrices[2:-1], ik_bones[2:-1], ctrl_bones[2:-1]))):
            ctrl.bone.use_inherit_rotation = not lock
            ctrl.bone.inherit_scale = 'NONE' if lock else 'FULL'
            context.view_layer.update()
            mat = convert_pose_matrix_via_rest_delta(mat, ik, ctrl)
            set_transform_from_matrix(obj, ctrl.name, mat, keyflags=keyflags)

    def assign_extra_controls(self, context, obj, all_matrices, ik_bones, ctrl_bones):
        for extra in self.extra_ctrl_list:
            set_transform_from_matrix(
                obj, extra, Matrix.Identity(4), space='LOCAL', keyflags=self.keyflags
            )

    def apply_frame_state(self, context, obj, all_matrices):
        ik_bones = [ obj.pose.bones[k] for k in self.ik_bone_list ]
        ctrl_bones = [ obj.pose.bones[k] for k in self.ctrl_bone_list ]
        tail_bones = [ obj.pose.bones[k] for k in self.tail_bone_list ]

        assert len(all_matrices) >= len(ik_bones) + len(tail_bones)

        matrices = all_matrices[0:len(ik_bones)]
        tail_matrices = all_matrices[len(ik_bones):]

        use_pole = self.get_use_pole(obj)

        # Remove foot heel transform, if present
        self.assign_extra_controls(context, obj, all_matrices, ik_bones, ctrl_bones)

        context.view_layer.update()

        # Set the end control position
        end_mat = convert_pose_matrix_via_pose_delta(matrices[-1], ik_bones[-1], ctrl_bones[-1])

        set_transform_from_matrix(
            obj, self.ctrl_bone_list[-1], end_mat, keyflags=self.keyflags,
            undo_copy_scale=True,
        )

        # Set the base bone position
        ctrl_bones[0].matrix_basis = Matrix.Identity(4)

        set_transform_from_matrix(
            obj, self.ctrl_bone_list[0], matrices[0],
            no_scale=True, no_rot=use_pole,
        )

        # Lock middle control transforms (first pass)
        self.assign_middle_controls(context, obj, matrices, ik_bones, ctrl_bones, lock=True)

        # Adjust the base bone state
        self.compute_base_rotation(context, ik_bones, ctrl_bones, matrices, use_pole)

        correct_scale(context.view_layer, ik_bones[0], matrices[0], ctrl_ik=ctrl_bones[0])

        # Assign middle control transforms (final pass)
        self.assign_middle_controls(context, obj, matrices, ik_bones, ctrl_bones, keyflags=self.keyflags)

        # Assign tail control transforms
        for mat, ctrl in zip(tail_matrices, tail_bones):
            context.view_layer.update()
            set_transform_from_matrix(obj, ctrl.name, mat, keyflags=self.keyflags)

        # Keyframe controls
        if self.keyflags is not None:
            if use_pole:
                keyframe_transform_properties(
                    obj, self.ctrl_bone_list[1], self.keyflags,
                    no_rot=True, no_scale=True,
                )

            keyframe_transform_properties(
                obj, self.ctrl_bone_list[0], self.keyflags,
                no_rot=use_pole,
            )

class POSE_OT_rigify_limb_ik2fk(RigifyLimbIk2FkBase, RigifySingleUpdateMixin, bpy.types.Operator):
    bl_idname = "pose.rigify_limb_ik2fk_" + rig_id
    bl_label = "Snap IK->FK"
    bl_description = "Snap the IK chain to FK result"

class POSE_OT_rigify_limb_ik2fk_bake(RigifyLimbIk2FkBase, RigifyBakeKeyframesMixin, bpy.types.Operator):
    bl_idname = "pose.rigify_limb_ik2fk_bake_" + rig_id
    bl_label = "Apply Snap IK->FK To Keyframes"
    bl_description = "Snap the IK chain keyframes to FK result"

    def execute_scan_curves(self, context, obj):
        self.bake_add_bone_frames(self.fk_bone_list, TRANSFORM_PROPS_ALL)
        return self.bake_get_all_bone_curves(self.ctrl_bone_list + self.extra_ctrl_list, TRANSFORM_PROPS_ALL)


####################
## Toggle IK Pole ##
####################

class RigifyLimbTogglePoleBase(RigifyLimbIk2FkBase):
    use_pole: bpy.props.BoolProperty(name="Use Pole Vector")

    def save_frame_state(self, context, obj):
        return get_chain_transform_matrices(obj, self.ik_bone_list)

    def apply_frame_state(self, context, obj, matrices):
        ik_bones = [ obj.pose.bones[k] for k in self.ik_bone_list ]
        ctrl_bones = [ obj.pose.bones[k] for k in self.ctrl_bone_list ]

        # Set the pole property
        set_custom_property_value(
            obj, self.prop_bone, self.pole_prop, bool(self.use_pole),
            keyflags=self.keyflags_switch
        )

        # Lock middle control transforms
        self.assign_middle_controls(context, obj, matrices, ik_bones, ctrl_bones, lock=True)

        # Reset the base bone rotation
        set_pose_rotation(ctrl_bones[0], Matrix.Identity(4))

        self.compute_base_rotation(context, ik_bones, ctrl_bones, matrices, self.use_pole)

        # Assign middle control transforms (final pass)
        self.assign_middle_controls(context, obj, matrices, ik_bones, ctrl_bones, keyflags=self.keyflags)

        # Keyframe controls
        if self.keyflags is not None:
            if self.use_pole:
                keyframe_transform_properties(
                    obj, self.ctrl_bone_list[1], self.keyflags,
                    no_rot=True, no_scale=True,
                )
            else:
                keyframe_transform_properties(
                    obj, self.ctrl_bone_list[0], self.keyflags,
                    no_loc=True, no_scale=True,
                )

    def init_invoke(self, context):
        self.use_pole = not bool(context.active_object.pose.bones[self.prop_bone][self.pole_prop])

class POSE_OT_rigify_limb_toggle_pole(RigifyLimbTogglePoleBase, RigifySingleUpdateMixin, bpy.types.Operator):
    bl_idname = "pose.rigify_limb_toggle_pole_" + rig_id
    bl_label = "Toggle Pole"
    bl_description = "Switch the IK chain between pole and rotation"

class POSE_OT_rigify_limb_toggle_pole_bake(RigifyLimbTogglePoleBase, RigifyBakeKeyframesMixin, bpy.types.Operator):
    bl_idname = "pose.rigify_limb_toggle_pole_bake_" + rig_id
    bl_label = "Apply Toggle Pole To Keyframes"
    bl_description = "Switch the IK chain between pole and rotation over a frame range"

    def execute_scan_curves(self, context, obj):
        self.bake_add_bone_frames(self.ctrl_bone_list, TRANSFORM_PROPS_ALL)

        rot_curves = self.bake_get_all_bone_curves(self.ctrl_bone_list[0], TRANSFORM_PROPS_ROTATION)
        pole_curves = self.bake_get_all_bone_curves(self.ctrl_bone_list[1], TRANSFORM_PROPS_LOCATION)
        return rot_curves + pole_curves

    def execute_before_apply(self, context, obj, range, range_raw):
        self.bake_replace_custom_prop_keys_constant(self.prop_bone, self.pole_prop, bool(self.use_pole))

    def draw(self, context):
        self.layout.prop(self, 'use_pole')


#######################
## Leg Snap IK to FK ##
#######################

class RigifyLegRollIk2FkBase(RigifyLimbIk2FkBase):
    heel_control: StringProperty(name="Heel")
    use_roll:     bpy.props.BoolVectorProperty(
        name="Use Roll", size=3, default=(True, True, False),
        description="Specifies which rotation axes of the heel roll control to use"
    )

    MODES = {
        'ZXY': ((0, 2), (1, 0, 2)),
        'XZY': ((2, 0), (2, 0, 1)),
    }

    def save_frame_state(self, context, obj):
        return get_chain_transform_matrices(obj, self.fk_bone_list + self.ctrl_bone_list[-1:])

    def assign_extra_controls(self, context, obj, all_matrices, ik_bones, ctrl_bones):
        for extra in self.extra_ctrl_list:
            set_transform_from_matrix(
                obj, extra, Matrix.Identity(4), space='LOCAL', keyflags=self.keyflags
            )

        if any(self.use_roll):
            foot_matrix = all_matrices[len(ik_bones) - 1]
            ctrl_matrix = all_matrices[len(self.fk_bone_list)]
            heel_bone = obj.pose.bones[self.heel_control]
            foot_bone = ctrl_bones[-1]

            # Relative rotation of heel from orientation of master IK control
            # to actual foot orientation.
            heel_rest = convert_pose_matrix_via_rest_delta(ctrl_matrix, foot_bone, heel_bone)
            heel_rot = convert_pose_matrix_via_rest_delta(foot_matrix, ik_bones[-1], heel_bone)

            # Decode the euler decomposition mode
            rot_mode = heel_bone.rotation_mode
            indices, use_map = self.MODES[rot_mode]
            use_roll = [self.use_roll[i] for i in use_map]
            roll, turn = indices

            # If the last rotation (yaw) is unused, move it to be first for better result
            if not use_roll[turn]:
                rot_mode = rot_mode[1:] + rot_mode[0:1]

            local_rot = (heel_rest.inverted() @ heel_rot).to_euler(rot_mode)

            heel_bone.rotation_euler = [
                (val if use else 0) for val, use in zip(local_rot, use_roll)
            ]

            if self.keyflags is not None:
                keyframe_transform_properties(
                    obj, bone_name, self.keyflags, no_loc=True, no_rot=no_rot, no_scale=True
                )

            if 'Toe_Roll' in heel_bone and self.tail_bone_list:
                toe_matrix = all_matrices[len(ik_bones)]
                toe_bone = obj.pose.bones[self.tail_bone_list[0]]

                # Compute relative rotation of heel determined by toe
                heel_rot_toe = convert_pose_matrix_via_rest_delta(toe_matrix, toe_bone, heel_bone)
                toe_rot = (heel_rest.inverted() @ heel_rot_toe).to_euler(rot_mode)

                # Determine how much of the already computed heel rotation seems to be applied
                heel_rot = list(heel_bone.rotation_euler)
                heel_rot[roll] = max(0.0, heel_rot[roll])

                # This relies on toe roll interpolation being done in Euler space
                ratios = [
                    toe_rot[i] / heel_rot[i] for i in (roll, turn)
                    if use_roll[i] and heel_rot[i] * toe_rot[i] > 0
                ]

                val = min(1.0, max(0.0, min(ratios) if ratios else 0.0))
                if val < 1e-5:
                    val = 0.0

                set_custom_property_value(
                    obj, heel_bone.name, 'Toe_Roll', val, keyflags=self.keyflags)

    def draw(self, context):
        row = self.layout.row(align=True)
        row.label(text="Use:")
        row.prop(self, 'use_roll', index=0, text="Rock", toggle=True)
        row.prop(self, 'use_roll', index=1, text="Roll", toggle=True)
        row.prop(self, 'use_roll', index=2, text="Yaw", toggle=True)

class POSE_OT_rigify_leg_roll_ik2fk(
        RigifyLegRollIk2FkBase, RigifySingleUpdateMixin, bpy.types.Operator):
    bl_options = {'REGISTER', 'UNDO', 'INTERNAL'}
    bl_idname = "pose.rigify_leg_roll_ik2fk_" + rig_id
    bl_label = "Snap IK->FK With Roll"
    bl_description = "Snap the IK chain to FK result, using foot roll to preserve the current IK "                     "control orientation as much as possible"

    def invoke(self, context, event):
        self.init_invoke(context)
        return self.execute(context)

class POSE_OT_rigify_leg_roll_ik2fk_bake(
        RigifyLegRollIk2FkBase, RigifyBakeKeyframesMixin, bpy.types.Operator):
    bl_idname = "pose.rigify_leg_roll_ik2fk_bake_" + rig_id
    bl_label = "Apply Snap IK->FK To Keyframes"
    bl_description = "Snap the IK chain keyframes to FK result, using foot roll to preserve the "                     "current IK control orientation as much as possible"

    def execute_scan_curves(self, context, obj):
        self.bake_add_bone_frames(self.fk_bone_list, TRANSFORM_PROPS_ALL)
        self.bake_add_bone_frames(self.ctrl_bone_list[-1:], TRANSFORM_PROPS_ROTATION)
        return self.bake_get_all_bone_curves(
            self.ctrl_bone_list + self.extra_ctrl_list, TRANSFORM_PROPS_ALL)


################################
## Switchable Parent operator ##
################################

class RigifySwitchParentBase:
    bone:         StringProperty(name="Control Bone")
    prop_bone:    StringProperty(name="Property Bone")
    prop_id:      StringProperty(name="Property")
    parent_names: StringProperty(name="Parent Names")
    locks:        bpy.props.BoolVectorProperty(name="Locked", size=3, default=[False,False,False])

    parent_items = [('0','None','None')]

    selected: bpy.props.EnumProperty(
        name='Selected Parent',
        items=lambda s,c: RigifySwitchParentBase.parent_items
    )

    def save_frame_state(self, context, obj):
        return get_transform_matrix(obj, self.bone, with_constraints=False)

    def apply_frame_state(self, context, obj, old_matrix):
        # Change the parent
        set_custom_property_value(
            obj, self.prop_bone, self.prop_id, int(self.selected),
            keyflags=self.keyflags_switch
        )

        context.view_layer.update()

        # Set the transforms to restore position
        set_transform_from_matrix(
            obj, self.bone, old_matrix, keyflags=self.keyflags,
            no_loc=self.locks[0], no_rot=self.locks[1], no_scale=self.locks[2]
        )

    def init_invoke(self, context):
        pose = context.active_object.pose

        if (not pose or not self.parent_names
            or self.bone not in pose.bones
            or self.prop_bone not in pose.bones
            or self.prop_id not in pose.bones[self.prop_bone]):
            self.report({'ERROR'}, "Invalid parameters")
            return {'CANCELLED'}

        parents = json.loads(self.parent_names)
        parent_items = [(str(i), name, name) for i, name in enumerate(parents)]

        RigifySwitchParentBase.parent_items = parent_items

        self.selected = str(pose.bones[self.prop_bone][self.prop_id])


class POSE_OT_rigify_switch_parent(RigifySwitchParentBase, RigifySingleUpdateMixin, bpy.types.Operator):
    bl_idname = "pose.rigify_switch_parent_" + rig_id
    bl_label = "Switch Parent (Keep Transform)"
    bl_options = {'REGISTER', 'UNDO', 'INTERNAL'}
    bl_description = "Switch parent, preserving the bone position and orientation"

    def draw(self, _context):
        col = self.layout.column()
        col.prop(self, 'selected', expand=True)


class POSE_OT_rigify_switch_parent_bake(RigifySwitchParentBase, RigifyBakeKeyframesMixin, bpy.types.Operator):
    bl_idname = "pose.rigify_switch_parent_bake_" + rig_id
    bl_label = "Apply Switch Parent To Keyframes"
    bl_description = "Switch parent over a frame range, adjusting keys to preserve the bone position and orientation"

    def execute_scan_curves(self, context, obj):
        return self.bake_add_bone_frames(self.bone, transform_props_with_locks(*self.locks))

    def execute_before_apply(self, context, obj, range, range_raw):
        self.bake_replace_custom_prop_keys_constant(self.prop_bone, self.prop_id, int(self.selected))

    def draw(self, context):
        self.layout.prop(self, 'selected', text='')



# Helper: Snap bone using temporary COPY_LOCATION and COPY_ROTATION constraints
def snap_bone_with_loc_rot(obj, snap_bone, source_bone_name):
    # Create COPY_LOCATION constraint
    loc_constraint = snap_bone.constraints.new(type='COPY_LOCATION')
    loc_constraint.name = "TempSnapLocation"
    loc_constraint.target = obj
    loc_constraint.subtarget = source_bone_name

    # Create COPY_ROTATION constraint
    rot_constraint = snap_bone.constraints.new(type='COPY_ROTATION')
    rot_constraint.name = "TempSnapRotation"
    rot_constraint.target = obj
    rot_constraint.subtarget = source_bone_name

    # Update the scene so constraints take effect
    bpy.context.view_layer.update()

    # Apply the visual transform
    bpy.ops.pose.select_all(action='DESELECT')
    snap_bone.bone.select = True
    obj.data.bones.active = snap_bone.bone
    bpy.ops.pose.visual_transform_apply()

    # Remove temporary constraints
    snap_bone.constraints.remove(loc_constraint)
    snap_bone.constraints.remove(rot_constraint)




# Helper: Snap bone using temporary COPY_LOCATION and COPY_ROTATION constraints
def snap_bone_with_loc_rot(obj, snap_bone, source_bone_name):
    # Create COPY_LOCATION constraint
    loc_constraint = snap_bone.constraints.new(type='COPY_LOCATION')
    loc_constraint.name = "TempSnapLocation"
    loc_constraint.target = obj
    loc_constraint.subtarget = source_bone_name

    # Create COPY_ROTATION constraint
    rot_constraint = snap_bone.constraints.new(type='COPY_ROTATION')
    rot_constraint.name = "TempSnapRotation"
    rot_constraint.target = obj
    rot_constraint.subtarget = source_bone_name

    # Update the scene so constraints take effect
    bpy.context.view_layer.update()

    # Apply the visual transform
    bpy.ops.pose.select_all(action='DESELECT')
    snap_bone.bone.select = True
    obj.data.bones.active = snap_bone.bone
    bpy.ops.pose.visual_transform_apply()

    # Remove temporary constraints
    snap_bone.constraints.remove(loc_constraint)
    snap_bone.constraints.remove(rot_constraint)




# ---------------------------------------------
# Helper: Snap bone using temporary constraint
def snap_bone_with_constraint(obj, snap_bone, source_bone_name):
    constraint = snap_bone.constraints.new(type='COPY_TRANSFORMS')
    constraint.name = "TempSnap"
    constraint.target = obj
    constraint.subtarget = source_bone_name

    bpy.context.view_layer.update()

    bpy.ops.pose.select_all(action='DESELECT')
    snap_bone.bone.select = True
    obj.data.bones.active = snap_bone.bone
    bpy.ops.pose.visual_transform_apply()

    snap_bone.constraints.remove(constraint)

# ---------------------------------------------
# Base Operator
class SnapBoneListOperator(bpy.types.Operator):
    bl_options = {'REGISTER', 'UNDO'}

    bl_idname = "object.snap_bone_list"
    bl_label = "Snap Bone List"
    bl_description = "Snap bones based on two parallel lists"

    prop_bone: bpy.props.StringProperty(default="")
    prop_name: bpy.props.StringProperty(default="")
    prop_value: bpy.props.StringProperty(default="")

    def get_target_bones(self):
        return []

    def get_snap_bones(self):
        return []

    def execute(self, context):
        obj = context.object
        if not obj or obj.type != 'ARMATURE' or context.mode != 'POSE':
            self.report({'ERROR'}, "Select an armature in pose mode")
            return {'CANCELLED'}

        pose_bones = obj.pose.bones

        # Set custom property and key it
        if self.prop_bone and self.prop_name:
            try:
                prop_bone = pose_bones[self.prop_bone]
            except KeyError:
                self.report({'ERROR'}, f"Bone '{self.prop_bone}' not found")
                return {'CANCELLED'}

            existing_value = prop_bone.get(self.prop_name)

            if existing_value is None:
                self.report({'ERROR'}, f"Property '{self.prop_name}' not found on bone '{self.prop_bone}'")
                return {'CANCELLED'}

            # Try to cast prop_value string to match existing type
            try:
                if isinstance(existing_value, bool):
                    new_value = self.prop_value.lower() in {"true", "1", "yes"}
                elif isinstance(existing_value, int):
                    new_value = int(self.prop_value)
                elif isinstance(existing_value, float):
                    new_value = float(self.prop_value)
                else:
                    new_value = self.prop_value
            except Exception as e:
                self.report({'ERROR'}, f"Failed to parse property value: {e}")
                return {'CANCELLED'}

            prop_bone[self.prop_name] = new_value
            prop_bone.keyframe_insert(data_path=f'["{self.prop_name}"]', group=self.prop_bone)

        # Snap bones
        target_bones = self.get_target_bones()
        snap_bones = self.get_snap_bones()

        if len(target_bones) != len(snap_bones):
            self.report({'ERROR'}, "Bone list lengths do not match")
            return {'CANCELLED'}

        action = obj.animation_data.action if obj.animation_data else None

        for source_name, snap_name in zip(target_bones, snap_bones):
            try:
                source = pose_bones[source_name]
                snap = pose_bones[snap_name]
            except KeyError as e:
                self.report({'WARNING'}, f"Bone not found: {e}")
                continue

            snap_bone_with_loc_rot(obj, snap, source_name)

            # Insert keyframes only if fcurves exist
            if action:
                for channel in ("location", "scale"):
                    path = f'pose.bones["{snap_name}"].{channel}'
                    if any(f.data_path == path for f in action.fcurves):
                        snap.keyframe_insert(data_path=channel, group=snap_name)

                if snap.rotation_mode == 'QUATERNION':
                    path = f'pose.bones["{snap_name}"].rotation_quaternion'
                    if any(f.data_path == path for f in action.fcurves):
                        snap.keyframe_insert(data_path="rotation_quaternion", group=snap_name)
                else:
                    path = f'pose.bones["{snap_name}"].rotation_euler'
                    if any(f.data_path == path for f in action.fcurves):
                        snap.keyframe_insert(data_path="rotation_euler", group=snap_name)

        self.report({'INFO'}, "Snapping completed.")
        return {'FINISHED'}


#########  Finger(index.L) IK to  FK snap operator #######

class POSE_OT_finger_index_L_ik_to_fk(SnapBoneListOperator):
    bl_idname = "pose.finger_index_l_ik_to_fk"
    bl_label = "Finger index.L IK to FK"
    bl_description = "Snap f_index.L from IK to FK"

    def get_target_bones(self):
        return ["MCH-REF-f_index.03IK.L", "MCH-REF-f_index.pole.L"]

    def get_snap_bones(self):
        return ["f_index.03IK.L", "f_index.pole.L"]

    prop_bone: bpy.props.StringProperty(default="PROPERTIES")
    prop_name: bpy.props.StringProperty(default="f_index.L_IK_FK")
    prop_value: bpy.props.FloatProperty(default=0)

#########  Finger(index.L) FK to  IK snap operator #######

class POSE_OT_finger_index_L_fk_to_ik(SnapBoneListOperator):
    bl_idname = "pose.finger_index_l_fk_to_ik"
    bl_label = "Finger index.L FK to IK"
    bl_description = "Snap f_index.L from FK to IK"

    def get_target_bones(self):
        return ["MCH-f_index.01IK.L", "MCH-f_index.02IK.L","MCH-f_index.03IK.L"]

    def get_snap_bones(self):
        return ["f_index.01FK.L", "f_index.02FK.L","f_index.03FK.L"]

    prop_bone: bpy.props.StringProperty(default="PROPERTIES")
    prop_name: bpy.props.StringProperty(default="f_index.L_IK_FK")
    prop_value: bpy.props.FloatProperty(default=1)
    
    #########  Finger(index.R) IK to  FK snap operator #######

class POSE_OT_finger_index_R_ik_to_fk(SnapBoneListOperator):
    bl_idname = "pose.finger_index_r_ik_to_fk"
    bl_label = "Finger index.R IK to FK"
    bl_description = "Snap f_index.R from IK to FK"

    def get_target_bones(self):
        return ["MCH-REF-f_index.03IK.R", "MCH-REF-f_index.pole.R"]

    def get_snap_bones(self):
        return ["f_index.03IK.R", "f_index.pole.R"]

    prop_bone: bpy.props.StringProperty(default="PROPERTIES")
    prop_name: bpy.props.StringProperty(default="f_index.R_IK_FK")
    prop_value: bpy.props.FloatProperty(default=0)

#########  Finger(index.R) FK to  IK snap operator #######

class POSE_OT_finger_index_R_fk_to_ik(SnapBoneListOperator):
    bl_idname = "pose.finger_index_r_fk_to_ik"
    bl_label = "Finger index.R FK to IK"
    bl_description = "Snap f_index.R from FK to IK"

    def get_target_bones(self):
        return ["MCH-f_index.01IK.R", "MCH-f_index.02IK.R","MCH-f_index.03IK.R"]

    def get_snap_bones(self):
        return ["f_index.01FK.R", "f_index.02FK.R","f_index.03FK.R"]

    prop_bone: bpy.props.StringProperty(default="PROPERTIES")
    prop_name: bpy.props.StringProperty(default="f_index.R_IK_FK")
    prop_value: bpy.props.FloatProperty(default=1)
   
    
############  Finger(middle.L) IK to  FK snap operator #######

class POSE_OT_finger_middle_L_ik_to_fk(SnapBoneListOperator):
    bl_idname = "pose.finger_middle_l_ik_to_fk"
    bl_label = "Finger middle.L IK to FK"
    bl_description = "Snap f_middle.L from IK to FK"

    def get_target_bones(self):
        return ["MCH-REF-f_middle.03IK.L", "MCH-REF-f_middle.pole.L"]

    def get_snap_bones(self):
        return ["f_middle.03IK.L", "f_middle.pole.L"]

    prop_bone: bpy.props.StringProperty(default="PROPERTIES")
    prop_name: bpy.props.StringProperty(default="f_middle.L_IK_FK")
    prop_value: bpy.props.FloatProperty(default=0)

#########  Finger(middle.L) FK to  IK snap operator #######

class POSE_OT_finger_middle_L_fk_to_ik(SnapBoneListOperator):
    bl_idname = "pose.finger_middle_l_fk_to_ik"
    bl_label = "Finger middle.L FK to IK"
    bl_description = "Snap f_middle.L from FK to IK"

    def get_target_bones(self):
        return ["MCH-f_middle.01IK.L", "MCH-f_middle.02IK.L","MCH-f_middle.03IK.L"]

    def get_snap_bones(self):
        return ["f_middle.01FK.L", "f_middle.02FK.L","f_middle.03FK.L"]

    prop_bone: bpy.props.StringProperty(default="PROPERTIES")
    prop_name: bpy.props.StringProperty(default="f_middle.L_IK_FK")
    prop_value: bpy.props.FloatProperty(default=1)
    
    #########  Finger(middle.R) IK to  FK snap operator #######

class POSE_OT_finger_middle_R_ik_to_fk(SnapBoneListOperator):
    bl_idname = "pose.finger_middle_r_ik_to_fk"
    bl_label = "Finger middle.R IK to FK"
    bl_description = "Snap f_middle.R from IK to FK"

    def get_target_bones(self):
        return ["MCH-REF-f_middle.03IK.R", "MCH-REF-f_middle.pole.R"]

    def get_snap_bones(self):
        return ["f_middle.03IK.R", "f_middle.pole.R"]

    prop_bone: bpy.props.StringProperty(default="PROPERTIES")
    prop_name: bpy.props.StringProperty(default="f_middle.R_IK_FK")
    prop_value: bpy.props.FloatProperty(default=0)

#########  Finger(middle.R) FK to  IK snap operator #######

class POSE_OT_finger_middle_R_fk_to_ik(SnapBoneListOperator):
    bl_idname = "pose.finger_middle_r_fk_to_ik"
    bl_label = "Finger middle.R FK to IK"
    bl_description = "Snap f_middle.R from FK to IK"

    def get_target_bones(self):
        return ["MCH-f_middle.01IK.R", "MCH-f_middle.02IK.R","MCH-f_middle.03IK.R"]

    def get_snap_bones(self):
        return ["f_middle.01FK.R", "f_middle.02FK.R","f_middle.03FK.R"]

    prop_bone: bpy.props.StringProperty(default="PROPERTIES")
    prop_name: bpy.props.StringProperty(default="f_middle.R_IK_FK")
    prop_value: bpy.props.FloatProperty(default=1)

############  Finger(ring.L) IK to  FK snap operator #######

class POSE_OT_finger_ring_L_ik_to_fk(SnapBoneListOperator):
    bl_idname = "pose.finger_ring_l_ik_to_fk"
    bl_label = "Finger ring.L IK to FK"
    bl_description = "Snap f_ring.L from IK to FK"

    def get_target_bones(self):
        return ["MCH-REF-f_ring.03IK.L", "MCH-REF-f_ring.pole.L"]

    def get_snap_bones(self):
        return ["f_ring.03IK.L", "f_ring.pole.L"]

    prop_bone: bpy.props.StringProperty(default="PROPERTIES")
    prop_name: bpy.props.StringProperty(default="f_ring.L_IK_FK")
    prop_value: bpy.props.FloatProperty(default=0)

#########  Finger(ring.L) FK to  IK snap operator #######

class POSE_OT_finger_ring_L_fk_to_ik(SnapBoneListOperator):
    bl_idname = "pose.finger_ring_l_fk_to_ik"
    bl_label = "Finger ring.L FK to IK"
    bl_description = "Snap f_ring.L from FK to IK"

    def get_target_bones(self):
        return ["MCH-f_ring.01IK.L", "MCH-f_ring.02IK.L","MCH-f_ring.03IK.L"]

    def get_snap_bones(self):
        return ["f_ring.01FK.L", "f_ring.02FK.L","f_ring.03FK.L"]

    prop_bone: bpy.props.StringProperty(default="PROPERTIES")
    prop_name: bpy.props.StringProperty(default="f_ring.L_IK_FK")
    prop_value: bpy.props.FloatProperty(default=1)
    
    #########  Finger(ring.R) IK to  FK snap operator #######

class POSE_OT_finger_ring_R_ik_to_fk(SnapBoneListOperator):
    bl_idname = "pose.finger_ring_r_ik_to_fk"
    bl_label = "Finger ring.R IK to FK"
    bl_description = "Snap f_ring.R from IK to FK"

    def get_target_bones(self):
        return ["MCH-REF-f_ring.03IK.R", "MCH-REF-f_ring.pole.R"]

    def get_snap_bones(self):
        return ["f_ring.03IK.R", "f_ring.pole.R"]

    prop_bone: bpy.props.StringProperty(default="PROPERTIES")
    prop_name: bpy.props.StringProperty(default="f_ring.R_IK_FK")
    prop_value: bpy.props.FloatProperty(default=0)

#########  Finger(ring.R) FK to  IK snap operator #######

class POSE_OT_finger_ring_R_fk_to_ik(SnapBoneListOperator):
    bl_idname = "pose.finger_ring_r_fk_to_ik"
    bl_label = "Finger ring.R FK to IK"
    bl_description = "Snap f_ring.R from FK to IK"

    def get_target_bones(self):
        return ["MCH-f_ring.01IK.R", "MCH-f_ring.02IK.R","MCH-f_ring.03IK.R"]

    def get_snap_bones(self):
        return ["f_ring.01FK.R", "f_ring.02FK.R","f_ring.03FK.R"]

    prop_bone: bpy.props.StringProperty(default="PROPERTIES")
    prop_name: bpy.props.StringProperty(default="f_ring.R_IK_FK")
    prop_value: bpy.props.FloatProperty(default=1)


############  Finger(pinky.L) IK to  FK snap operator #######

class POSE_OT_finger_pinky_L_ik_to_fk(SnapBoneListOperator):
    bl_idname = "pose.finger_pinky_l_ik_to_fk"
    bl_label = "Finger pinky.L IK to FK"
    bl_description = "Snap f_pinky.L from IK to FK"

    def get_target_bones(self):
        return ["MCH-REF-f_pinky.03IK.L", "MCH-REF-f_pinky.pole.L"]

    def get_snap_bones(self):
        return ["f_pinky.03IK.L", "f_pinky.pole.L"]

    prop_bone: bpy.props.StringProperty(default="PROPERTIES")
    prop_name: bpy.props.StringProperty(default="f_pinky.L_IK_FK")
    prop_value: bpy.props.FloatProperty(default=0)

#########  Finger(pinky.L) FK to  IK snap operator #######

class POSE_OT_finger_pinky_L_fk_to_ik(SnapBoneListOperator):
    bl_idname = "pose.finger_pinky_l_fk_to_ik"
    bl_label = "Finger pinky.L FK to IK"
    bl_description = "Snap f_pinky.L from FK to IK"

    def get_target_bones(self):
        return ["MCH-f_pinky.01IK.L", "MCH-f_pinky.02IK.L","MCH-f_pinky.03IK.L"]

    def get_snap_bones(self):
        return ["f_pinky.01FK.L", "f_pinky.02FK.L","f_pinky.03FK.L"]

    prop_bone: bpy.props.StringProperty(default="PROPERTIES")
    prop_name: bpy.props.StringProperty(default="f_pinky.L_IK_FK")
    prop_value: bpy.props.FloatProperty(default=1)
    
    #########  Finger(pinky.R) IK to  FK snap operator #######

class POSE_OT_finger_pinky_R_ik_to_fk(SnapBoneListOperator):
    bl_idname = "pose.finger_pinky_r_ik_to_fk"
    bl_label = "Finger pinky.R IK to FK"
    bl_description = "Snap f_pinky.R from IK to FK"

    def get_target_bones(self):
        return ["MCH-REF-f_pinky.03IK.R", "MCH-REF-f_pinky.pole.R"]

    def get_snap_bones(self):
        return ["f_pinky.03IK.R", "f_pinky.pole.R"]

    prop_bone: bpy.props.StringProperty(default="PROPERTIES")
    prop_name: bpy.props.StringProperty(default="f_pinky.R_IK_FK")
    prop_value: bpy.props.FloatProperty(default=0)

#########  Finger(pinky.R) FK to  IK snap operator #######

class POSE_OT_finger_pinky_R_fk_to_ik(SnapBoneListOperator):
    bl_idname = "pose.finger_pinky_r_fk_to_ik"
    bl_label = "Finger pinky.R FK to IK"
    bl_description = "Snap f_pinky.R from FK to IK"

    def get_target_bones(self):
        return ["MCH-f_pinky.01IK.R", "MCH-f_pinky.02IK.R","MCH-f_pinky.03IK.R"]

    def get_snap_bones(self):
        return ["f_pinky.01FK.R", "f_pinky.02FK.R","f_pinky.03FK.R"]

    prop_bone: bpy.props.StringProperty(default="PROPERTIES")
    prop_name: bpy.props.StringProperty(default="f_pinky.R_IK_FK")
    prop_value: bpy.props.FloatProperty(default=1)
    
    
    #########  Finger(thumb.L) IK to  FK snap operator #######

class POSE_OT_finger_thumb_L_ik_to_fk(SnapBoneListOperator):
    bl_idname = "pose.finger_thumb_l_ik_to_fk"
    bl_label = "Finger thumb.L IK to FK"
    bl_description = "Snap f_thumb.L from IK to FK"

    def get_target_bones(self):
        return ["MCH-REF-thumb.03IK.L", "MCH-REF-f_thumb.pole.L"]

    def get_snap_bones(self):
        return ["f_thumb.03IK.L", "f_thumb.pole.L"]

    prop_bone: bpy.props.StringProperty(default="PROPERTIES")
    prop_name: bpy.props.StringProperty(default="f_thumb.L_IK_FK")
    prop_value: bpy.props.FloatProperty(default=0)

#########  Finger(thumb.L) FK to  IK snap operator #######

class POSE_OT_finger_thumb_L_fk_to_ik(SnapBoneListOperator):
    bl_idname = "pose.finger_thumb_l_fk_to_ik"
    bl_label = "Finger thumb.L FK to IK"
    bl_description = "Snap f_thumb.L from FK to IK"

    def get_target_bones(self):
        return ["MCH-thumb.01IK.L", "MCH-thumb.02IK.L","MCH-thumb.03IK.L"]

    def get_snap_bones(self):
        return ["f_thumb.01FK.L", "f_thumb.02FK.L","f_thumb.03FK.L"]

    prop_bone: bpy.props.StringProperty(default="PROPERTIES")
    prop_name: bpy.props.StringProperty(default="f_thumb.L_IK_FK")
    prop_value: bpy.props.FloatProperty(default=1)

    #########  Finger(thumb.R) IK to  FK snap operator #######

class POSE_OT_finger_thumb_R_ik_to_fk(SnapBoneListOperator):
    bl_idname = "pose.finger_thumb_r_ik_to_fk"
    bl_label = "Finger thumb.R IK to FK"
    bl_description = "Snap f_thumb.R from IK to FK"

    def get_target_bones(self):
        return ["MCH-REF-thumb.03IK.R", "MCH-REF-f_thumb.pole.R"]

    def get_snap_bones(self):
        return ["f_thumb.03IK.R", "f_thumb.pole.R"]

    prop_bone: bpy.props.StringProperty(default="PROPERTIES")
    prop_name: bpy.props.StringProperty(default="f_thumb.R_IK_FK")
    prop_value: bpy.props.FloatProperty(default=0)

#########  Finger(thumb.R) FK to  IK snap operator #######

class POSE_OT_finger_thumb_R_fk_to_ik(SnapBoneListOperator):
    bl_idname = "pose.finger_thumb_r_fk_to_ik"
    bl_label = "Finger thumb.R FK to IK"
    bl_description = "Snap f_thumb.R from FK to IK"

    def get_target_bones(self):
        return ["MCH-thumb.01IK.R", "MCH-thumb.02IK.R","MCH-thumb.03IK.R"]

    def get_snap_bones(self):
        return ["f_thumb.01FK.R", "f_thumb.02FK.R","f_thumb.03FK.R"]

    prop_bone: bpy.props.StringProperty(default="PROPERTIES")
    prop_name: bpy.props.StringProperty(default="f_thumb.R_IK_FK")
    prop_value: bpy.props.FloatProperty(default=1)



###################
## Rig UI Panels ##
###################

class RigUI(bpy.types.Panel):
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI'
    bl_label = "P2M Big Rig Main Properties"
    bl_idname = "VIEW3D_PT_rig_ui_" + rig_id
    bl_category = 'Item'

    @classmethod
    def poll(self, context):
        if context.mode != 'POSE':
            return False
        try:
            return (context.active_object.data.get("rig_id") == rig_id)
        except (AttributeError, KeyError, TypeError):
            return False

    def draw(self, context):
        layout = self.layout
        pose_bones = context.active_object.pose.bones
        try:
            selected_bones = set(bone.name for bone in context.selected_pose_bones)
            selected_bones.add(context.active_pose_bone.name)
        except (AttributeError, TypeError):
            return

        def is_selected(names):
            # Returns whether any of the named bones are selected.
            if isinstance(names, list) or isinstance(names, set):
                return not selected_bones.isdisjoint(names)
            elif names in selected_bones:
                return True
            return False

        num_rig_separators = [-1]

        def emit_rig_separator():
            if num_rig_separators[0] >= 0:
                layout.separator()
            num_rig_separators[0] += 1

        if is_selected({'tweak_spine', 'spine_03_fk', 'tweak_neck_01', 'torso', 'tweak_neck_02', 'spine_02_fk', 'tweak_spine_01', 'spine_01_fk', 'neck', 'tweak_spine_02', 'chest', 'spine_fk', 'hips', 'tweak_spine_03', 'head'}):
            emit_rig_separator()
            layout.prop(pose_bones['torso'], '["neck_follow"]', text='Neck Follow', slider=True)
            layout.prop(pose_bones['torso'], '["head_follow"]', text='Head Follow', slider=True)
            if is_selected({'tweak_spine', 'spine_03_fk', 'tweak_neck_01', 'torso', 'spine_02_fk', 'tweak_spine_01', 'spine_01_fk', 'chest', 'tweak_spine_02', 'spine_fk', 'hips', 'tweak_spine_03'}):
                group1 = layout.row(align=True)
                group2 = group1.split(factor=0.65, align=True)
                props = group2.operator('pose.rigify_switch_parent_x4i0wdna76224586', text='Torso Parent', icon='DOWNARROW_HLT')
                props.bone = 'torso'
                props.prop_bone = 'torso'
                props.prop_id = 'torso_parent'
                props.parent_names = '["None", "Root"]'
                props.locks = (False, False, False)
                group2.prop(pose_bones['torso'], '["torso_parent"]', text='')
                props = group1.operator('pose.rigify_switch_parent_bake_x4i0wdna76224586', text='', icon='ACTION_TWEAK')
                props.bone = 'torso'
                props.prop_bone = 'torso'
                props.prop_id = 'torso_parent'
                props.parent_names = '["None", "Root"]'
                props.locks = (False, False, False)

        if is_selected({'PROPERTIES'}):
            emit_rig_separator()
            layout.prop(pose_bones['PROPERTIES'], '["P2design_logo"]', text='P2design logo', slider=False)
            layout.prop(pose_bones['PROPERTIES'], '["Smooth"]', text='Smooth', slider=False)
            layout.prop(pose_bones['PROPERTIES'], '["Subdiv"]', text='Subdiv', slider=False)
            layout.prop(pose_bones['PROPERTIES'], '["color_1_RGB"]', text='color 1 RGB', slider=False)
            layout.prop(pose_bones['PROPERTIES'], '["color_2_RGB"]', text='color 2 RGB', slider=False)

        if is_selected({'f_thumb.pole.L','f_thumb.01FK.L','f_thumb.02FK.L','f_thumb.03FK.L','f_thumb.03IK.L'}):
            emit_rig_separator()
            layout.label(text="Finger snap")
            layout.operator("pose.finger_thumb_l_ik_to_fk", text="f_thumb.L IK->FK", icon = 'SNAP_ON')
            layout.operator("pose.finger_thumb_l_fk_to_ik", text="f_thumb.L FK->IK", icon = 'SNAP_ON')
            layout.prop(pose_bones['PROPERTIES'], '["f_thumb.L_IK_FK"]', text='f thumb L IK FK', slider=True)


        if is_selected({'f_thumb.pole.R','f_thumb.01FK.R','f_thumb.02FK.R','f_thumb.03FK.R','f_thumb.03IK.R'}):
            emit_rig_separator()
            layout.label(text="Finger snap")
            layout.operator("pose.finger_thumb_r_ik_to_fk", text="f_thumb.R IK->FK", icon = 'SNAP_ON')
            layout.operator("pose.finger_thumb_r_fk_to_ik", text="f_thumb.R FK->IK", icon = 'SNAP_ON')
            layout.prop(pose_bones['PROPERTIES'], '["f_thumb.R_IK_FK"]', text='f thumb R IK FK', slider=True)
            
        if is_selected({'f_index.pole.L','f_index.01FK.L','f_index.02FK.L','f_index.03FK.L','f_index.03IK.L'}):
            emit_rig_separator()
            layout.label(text="Finger snap")
            layout.operator("pose.finger_index_l_ik_to_fk", text="f_index.L IK->FK", icon = 'SNAP_ON')
            layout.operator("pose.finger_index_l_fk_to_ik", text="f_index.L FK->IK", icon = 'SNAP_ON')
            layout.prop(pose_bones['PROPERTIES'], '["f_index.L_IK_FK"]', text='f index L IK FK', slider=True)

        if is_selected({'f_index.pole.R','f_index.01FK.R','f_index.02FK.R','f_index.03FK.R','f_index.03IK.R'}):
            emit_rig_separator()
            layout.label(text="Finger snap")
            layout.operator("pose.finger_index_r_ik_to_fk", text="f_index.R IK->FK", icon = 'SNAP_ON')
            layout.operator("pose.finger_index_r_fk_to_ik", text="f_index.R FK->IK", icon = 'SNAP_ON')
            layout.prop(pose_bones['PROPERTIES'], '["f_index.R_IK_FK"]', text='f index R IK FK', slider=True)

        if is_selected({'f_middle.pole.L','f_middle.01FK.L','f_middle.02FK.L','f_middle.03FK.L','f_middle.03IK.L'}):
            emit_rig_separator()
            layout.label(text="Finger snap")
            layout.operator("pose.finger_middle_l_ik_to_fk", text="f_middle.L IK->FK", icon = 'SNAP_ON')
            layout.operator("pose.finger_middle_l_fk_to_ik", text="f_middle.L FK->IK", icon = 'SNAP_ON')
            layout.prop(pose_bones['PROPERTIES'], '["f_middle.L_IK_FK"]', text='f middle L IK FK', slider=True)


        if is_selected({'f_middle.pole.R','f_middle.01FK.R','f_middle.02FK.R','f_middle.03FK.R','f_middle.03IK.R'}):
            emit_rig_separator()
            layout.label(text="Finger snap")
            layout.operator("pose.finger_middle_r_ik_to_fk", text="f_middle.R IK->FK", icon = 'SNAP_ON')
            layout.operator("pose.finger_middle_r_fk_to_ik", text="f_middle.R FK->IK", icon = 'SNAP_ON')
            layout.prop(pose_bones['PROPERTIES'], '["f_middle.R_IK_FK"]', text='f middle R IK FK', slider=True)
        

        if is_selected({'f_ring.pole.L','f_ring.01FK.L','f_ring.02FK.L','f_ring.03FK.L','f_ring.03IK.L'}):
            emit_rig_separator()
            layout.label(text="Finger snap")
            layout.operator("pose.finger_ring_l_ik_to_fk", text="f_ring.L IK->FK", icon = 'SNAP_ON')
            layout.operator("pose.finger_ring_l_fk_to_ik", text="f_ring.L FK->IK", icon = 'SNAP_ON')
            layout.prop(pose_bones['PROPERTIES'], '["f_ring.L_IK_FK"]', text='f ring L IK FK', slider=True)


        if is_selected({'f_ring.pole.R','f_ring.01FK.R','f_ring.02FK.R','f_ring.03FK.R','f_ring.03IK.R'}):
            emit_rig_separator()
            layout.label(text="Finger snap")
            layout.operator("pose.finger_ring_r_ik_to_fk", text="f_ring.R IK->FK", icon = 'SNAP_ON')
            layout.operator("pose.finger_ring_r_fk_to_ik", text="f_ring.R FK->IK", icon = 'SNAP_ON')
            layout.prop(pose_bones['PROPERTIES'], '["f_ring.R_IK_FK"]', text='f ring R IK FK', slider=True)

        if is_selected({'f_pinky.pole.L','f_pinky.01FK.L','f_pinky.02FK.L','f_pinky.03FK.L','f_pinky.03IK.L'}):
            emit_rig_separator()
            layout.label(text="Finger snap")
            layout.operator("pose.finger_pinky_l_ik_to_fk", text="f_pinky.L IK->FK", icon = 'SNAP_ON')
            layout.operator("pose.finger_pinky_l_fk_to_ik", text="f_pinky.L FK->IK", icon = 'SNAP_ON')
            layout.prop(pose_bones['PROPERTIES'], '["f_pinky.L_IK_FK"]', text='f pinky L IK FK', slider=True)

        if is_selected({'f_pinky.pole.R','f_pinky.01FK.R','f_pinky.02FK.R','f_pinky.03FK.R','f_pinky.03IK.R'}):
            emit_rig_separator()
            layout.label(text="Finger snap")
            layout.operator("pose.finger_pinky_r_ik_to_fk", text="f_pinky.R IK->FK", icon = 'SNAP_ON')
            layout.operator("pose.finger_pinky_r_fk_to_ik", text="f_pinky.R FK->IK", icon = 'SNAP_ON')
            layout.prop(pose_bones['PROPERTIES'], '["f_pinky.R_IK_FK"]', text='f pinky R IK FK', slider=True)

        if is_selected({'forearm_tweak.L.001', 'upper_arm_ik_target.L', 'upper_arm_tweak.L.002', 'forearm_tweak.L.002', 'forearm_tweak.L', 'upper_arm_ik.L', 'upper_arm_tweak.L.001', 'hand_fk.L', 'upper_arm_tweak.L', 'forearm_tweak.L.003', 'hand_tweak.L', 'hand_ik.L', 'upper_arm_tweak.L.003', 'upper_arm_fk.L', 'forearm_fk.L', 'upper_arm_parent.L', 'VIS_upper_arm_ik_pole.L'}):
            emit_rig_separator()
            if is_selected({'upper_arm_fk.L', 'forearm_fk.L', 'hand_fk.L', 'upper_arm_parent.L'}):
                layout.prop(pose_bones['upper_arm_parent.L'], '["FK_limb_follow"]', text='FK Limb Follow', slider=True)
            layout.prop(pose_bones['upper_arm_parent.L'], '["IK_FK"]', text='IK-FK (hand.L)', slider=True)
            group1 = layout.column(align=True)
            props = group1.operator('pose.rigify_generic_snap_x4i0wdna76224586', text='FK->IK (hand.L)', icon='SNAP_ON')
            props.output_bones = '["upper_arm_fk.L", "forearm_fk.L", "hand_fk.L"]'
            props.input_bones = '["upper_arm_ik.L", "MCH-forearm_ik.L", "MCH-upper_arm_ik_target.L"]'
            props.ctrl_bones = '["upper_arm_ik.L", "upper_arm_ik_target.L", "hand_ik.L"]'
            group2 = group1.row(align=True)
            props = group2.operator('pose.rigify_generic_snap_bake_x4i0wdna76224586', text='Action', icon='ACTION_TWEAK')
            props.output_bones = '["upper_arm_fk.L", "forearm_fk.L", "hand_fk.L"]'
            props.input_bones = '["upper_arm_ik.L", "MCH-forearm_ik.L", "MCH-upper_arm_ik_target.L"]'
            props.ctrl_bones = '["upper_arm_ik.L", "upper_arm_ik_target.L", "hand_ik.L"]'
            props = group2.operator('pose.rigify_clear_keyframes_x4i0wdna76224586', text='Clear', icon='CANCEL')
            props.bones = '["upper_arm_fk.L", "forearm_fk.L", "hand_fk.L"]'
            group1 = layout.column(align=True)
            props = group1.operator('pose.rigify_limb_ik2fk_x4i0wdna76224586', text='IK->FK (hand.L)', icon='SNAP_ON')
            props.prop_bone = 'upper_arm_parent.L'
            props.fk_bones = '["upper_arm_fk.L", "forearm_fk.L", "hand_fk.L"]'
            props.ik_bones = '["upper_arm_ik.L", "MCH-forearm_ik.L", "MCH-upper_arm_ik_target.L"]'
            props.ctrl_bones = '["upper_arm_ik.L", "upper_arm_ik_target.L", "hand_ik.L"]'
            props.tail_bones = '[]'
            props.extra_ctrls = '[]'
            group2 = group1.row(align=True)
            props = group2.operator('pose.rigify_limb_ik2fk_bake_x4i0wdna76224586', text='Action', icon='ACTION_TWEAK')
            props.prop_bone = 'upper_arm_parent.L'
            props.fk_bones = '["upper_arm_fk.L", "forearm_fk.L", "hand_fk.L"]'
            props.ik_bones = '["upper_arm_ik.L", "MCH-forearm_ik.L", "MCH-upper_arm_ik_target.L"]'
            props.ctrl_bones = '["upper_arm_ik.L", "upper_arm_ik_target.L", "hand_ik.L"]'
            props.tail_bones = '[]'
            props.extra_ctrls = '[]'
            props = group2.operator('pose.rigify_clear_keyframes_x4i0wdna76224586', text='Clear', icon='CANCEL')
            props.bones = '["upper_arm_ik.L", "upper_arm_ik_target.L", "hand_ik.L"]'
            if is_selected({'upper_arm_ik.L', 'hand_ik.L', 'upper_arm_ik_target.L', 'upper_arm_parent.L'}):
                layout.prop(pose_bones['upper_arm_parent.L'], '["IK_Stretch"]', text='IK Stretch', slider=True)
                group1 = layout.row(align=True)
                group2 = group1.split(factor=0.65, align=True)
                props = group2.operator('pose.rigify_limb_toggle_pole_x4i0wdna76224586', icon='FORCE_MAGNETIC')
                props.prop_bone = 'upper_arm_parent.L'
                props.ik_bones = '["upper_arm_ik.L", "MCH-forearm_ik.L", "MCH-upper_arm_ik_target.L"]'
                props.ctrl_bones = '["upper_arm_ik.L", "upper_arm_ik_target.L", "hand_ik.L"]'
                props.extra_ctrls = '[]'
                group2.prop(pose_bones['upper_arm_parent.L'], '["pole_vector"]', text=('On' if pose_bones['upper_arm_parent.L']['pole_vector'] else 'Off'), toggle=True)
                props = group1.operator('pose.rigify_limb_toggle_pole_bake_x4i0wdna76224586', text='', icon='ACTION_TWEAK')
                props.prop_bone = 'upper_arm_parent.L'
                props.ik_bones = '["upper_arm_ik.L", "MCH-forearm_ik.L", "MCH-upper_arm_ik_target.L"]'
                props.ctrl_bones = '["upper_arm_ik.L", "upper_arm_ik_target.L", "hand_ik.L"]'
                props.extra_ctrls = '[]'
                group1 = layout.row(align=True)
                group2 = group1.split(factor=0.65, align=True)
                props = group2.operator('pose.rigify_switch_parent_x4i0wdna76224586', text='IK Parent', icon='DOWNARROW_HLT')
                props.bone = 'hand_ik.L'
                props.prop_bone = 'upper_arm_parent.L'
                props.prop_id = 'IK_parent'
                props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.L"]'
                props.locks = (False, False, False)
                group2.prop(pose_bones['upper_arm_parent.L'], '["IK_parent"]', text='')
                props = group1.operator('pose.rigify_switch_parent_bake_x4i0wdna76224586', text='', icon='ACTION_TWEAK')
                props.bone = 'hand_ik.L'
                props.prop_bone = 'upper_arm_parent.L'
                props.prop_id = 'IK_parent'
                props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.L"]'
                props.locks = (False, False, False)
                group1 = layout.row(align=True)
                group2 = group1.split(factor=0.65, align=True)
                props = group2.operator('pose.rigify_switch_parent_x4i0wdna76224586', text='Pole Parent', icon='DOWNARROW_HLT')
                props.bone = 'upper_arm_ik_target.L'
                props.prop_bone = 'upper_arm_parent.L'
                props.prop_id = 'pole_parent'
                props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.L", "hand_ik.L"]'
                props.locks = (False, True, True)
                group2.prop(pose_bones['upper_arm_parent.L'], '["pole_parent"]', text='')
                props = group1.operator('pose.rigify_switch_parent_bake_x4i0wdna76224586', text='', icon='ACTION_TWEAK')
                props.bone = 'upper_arm_ik_target.L'
                props.prop_bone = 'upper_arm_parent.L'
                props.prop_id = 'pole_parent'
                props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.L", "hand_ik.L"]'
                props.locks = (False, True, True)
            if is_selected({'upper_arm_tweak.L.001'}):
                layout.prop(pose_bones['upper_arm_tweak.L.001'], '["rubber_tweak"]', text='Rubber Tweak (upper_arm.L)', slider=True)
            if is_selected({'upper_arm_tweak.L.002'}):
                layout.prop(pose_bones['upper_arm_tweak.L.002'], '["rubber_tweak"]', text='Rubber Tweak (upper_arm.L)', slider=True)
            if is_selected({'upper_arm_tweak.L.003'}):
                layout.prop(pose_bones['upper_arm_tweak.L.003'], '["rubber_tweak"]', text='Rubber Tweak (upper_arm.L)', slider=True)
            if is_selected({'forearm_tweak.L'}):
                layout.prop(pose_bones['forearm_tweak.L'], '["rubber_tweak"]', text='Rubber Tweak (forearm.L)', slider=True)
            if is_selected({'forearm_tweak.L.001'}):
                layout.prop(pose_bones['forearm_tweak.L.001'], '["rubber_tweak"]', text='Rubber Tweak (forearm.L)', slider=True)
            if is_selected({'forearm_tweak.L.002'}):
                layout.prop(pose_bones['forearm_tweak.L.002'], '["rubber_tweak"]', text='Rubber Tweak (forearm.L)', slider=True)
            if is_selected({'forearm_tweak.L.003'}):
                layout.prop(pose_bones['forearm_tweak.L.003'], '["rubber_tweak"]', text='Rubber Tweak (forearm.L)', slider=True)

        if is_selected({'hand_fk.R', 'VIS_upper_arm_ik_pole.R', 'forearm_tweak.R.003', 'forearm_tweak.R', 'upper_arm_fk.R', 'forearm_tweak.R.002', 'upper_arm_ik.R', 'upper_arm_parent.R', 'upper_arm_tweak.R.002', 'upper_arm_tweak.R.001', 'forearm_fk.R', 'upper_arm_ik_target.R', 'forearm_tweak.R.001', 'upper_arm_tweak.R.003', 'hand_tweak.R', 'hand_ik.R', 'upper_arm_tweak.R'}):
            emit_rig_separator()
            if is_selected({'hand_fk.R', 'forearm_fk.R', 'upper_arm_parent.R', 'upper_arm_fk.R'}):
                layout.prop(pose_bones['upper_arm_parent.R'], '["FK_limb_follow"]', text='FK Limb Follow', slider=True)
            layout.prop(pose_bones['upper_arm_parent.R'], '["IK_FK"]', text='IK-FK (hand.R)', slider=True)
            group1 = layout.column(align=True)
            props = group1.operator('pose.rigify_generic_snap_x4i0wdna76224586', text='FK->IK (hand.R)', icon='SNAP_ON')
            props.output_bones = '["upper_arm_fk.R", "forearm_fk.R", "hand_fk.R"]'
            props.input_bones = '["upper_arm_ik.R", "MCH-forearm_ik.R", "MCH-upper_arm_ik_target.R"]'
            props.ctrl_bones = '["upper_arm_ik.R", "upper_arm_ik_target.R", "hand_ik.R"]'
            group2 = group1.row(align=True)
            props = group2.operator('pose.rigify_generic_snap_bake_x4i0wdna76224586', text='Action', icon='ACTION_TWEAK')
            props.output_bones = '["upper_arm_fk.R", "forearm_fk.R", "hand_fk.R"]'
            props.input_bones = '["upper_arm_ik.R", "MCH-forearm_ik.R", "MCH-upper_arm_ik_target.R"]'
            props.ctrl_bones = '["upper_arm_ik.R", "upper_arm_ik_target.R", "hand_ik.R"]'
            props = group2.operator('pose.rigify_clear_keyframes_x4i0wdna76224586', text='Clear', icon='CANCEL')
            props.bones = '["upper_arm_fk.R", "forearm_fk.R", "hand_fk.R"]'
            group1 = layout.column(align=True)
            props = group1.operator('pose.rigify_limb_ik2fk_x4i0wdna76224586', text='IK->FK (hand.R)', icon='SNAP_ON')
            props.prop_bone = 'upper_arm_parent.R'
            props.fk_bones = '["upper_arm_fk.R", "forearm_fk.R", "hand_fk.R"]'
            props.ik_bones = '["upper_arm_ik.R", "MCH-forearm_ik.R", "MCH-upper_arm_ik_target.R"]'
            props.ctrl_bones = '["upper_arm_ik.R", "upper_arm_ik_target.R", "hand_ik.R"]'
            props.tail_bones = '[]'
            props.extra_ctrls = '[]'
            group2 = group1.row(align=True)
            props = group2.operator('pose.rigify_limb_ik2fk_bake_x4i0wdna76224586', text='Action', icon='ACTION_TWEAK')
            props.prop_bone = 'upper_arm_parent.R'
            props.fk_bones = '["upper_arm_fk.R", "forearm_fk.R", "hand_fk.R"]'
            props.ik_bones = '["upper_arm_ik.R", "MCH-forearm_ik.R", "MCH-upper_arm_ik_target.R"]'
            props.ctrl_bones = '["upper_arm_ik.R", "upper_arm_ik_target.R", "hand_ik.R"]'
            props.tail_bones = '[]'
            props.extra_ctrls = '[]'
            props = group2.operator('pose.rigify_clear_keyframes_x4i0wdna76224586', text='Clear', icon='CANCEL')
            props.bones = '["upper_arm_ik.R", "upper_arm_ik_target.R", "hand_ik.R"]'
            if is_selected({'upper_arm_ik.R', 'upper_arm_parent.R', 'upper_arm_ik_target.R', 'hand_ik.R'}):
                layout.prop(pose_bones['upper_arm_parent.R'], '["IK_Stretch"]', text='IK Stretch', slider=True)
                group1 = layout.row(align=True)
                group2 = group1.split(factor=0.65, align=True)
                props = group2.operator('pose.rigify_limb_toggle_pole_x4i0wdna76224586', icon='FORCE_MAGNETIC')
                props.prop_bone = 'upper_arm_parent.R'
                props.ik_bones = '["upper_arm_ik.R", "MCH-forearm_ik.R", "MCH-upper_arm_ik_target.R"]'
                props.ctrl_bones = '["upper_arm_ik.R", "upper_arm_ik_target.R", "hand_ik.R"]'
                props.extra_ctrls = '[]'
                group2.prop(pose_bones['upper_arm_parent.R'], '["pole_vector"]', text=('On' if pose_bones['upper_arm_parent.R']['pole_vector'] else 'Off'), toggle=True)
                props = group1.operator('pose.rigify_limb_toggle_pole_bake_x4i0wdna76224586', text='', icon='ACTION_TWEAK')
                props.prop_bone = 'upper_arm_parent.R'
                props.ik_bones = '["upper_arm_ik.R", "MCH-forearm_ik.R", "MCH-upper_arm_ik_target.R"]'
                props.ctrl_bones = '["upper_arm_ik.R", "upper_arm_ik_target.R", "hand_ik.R"]'
                props.extra_ctrls = '[]'
                group1 = layout.row(align=True)
                group2 = group1.split(factor=0.65, align=True)
                props = group2.operator('pose.rigify_switch_parent_x4i0wdna76224586', text='IK Parent', icon='DOWNARROW_HLT')
                props.bone = 'hand_ik.R'
                props.prop_bone = 'upper_arm_parent.R'
                props.prop_id = 'IK_parent'
                props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.R"]'
                props.locks = (False, False, False)
                group2.prop(pose_bones['upper_arm_parent.R'], '["IK_parent"]', text='')
                props = group1.operator('pose.rigify_switch_parent_bake_x4i0wdna76224586', text='', icon='ACTION_TWEAK')
                props.bone = 'hand_ik.R'
                props.prop_bone = 'upper_arm_parent.R'
                props.prop_id = 'IK_parent'
                props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.R"]'
                props.locks = (False, False, False)
                group1 = layout.row(align=True)
                group2 = group1.split(factor=0.65, align=True)
                props = group2.operator('pose.rigify_switch_parent_x4i0wdna76224586', text='Pole Parent', icon='DOWNARROW_HLT')
                props.bone = 'upper_arm_ik_target.R'
                props.prop_bone = 'upper_arm_parent.R'
                props.prop_id = 'pole_parent'
                props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.R", "hand_ik.R"]'
                props.locks = (False, True, True)
                group2.prop(pose_bones['upper_arm_parent.R'], '["pole_parent"]', text='')
                props = group1.operator('pose.rigify_switch_parent_bake_x4i0wdna76224586', text='', icon='ACTION_TWEAK')
                props.bone = 'upper_arm_ik_target.R'
                props.prop_bone = 'upper_arm_parent.R'
                props.prop_id = 'pole_parent'
                props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.R", "hand_ik.R"]'
                props.locks = (False, True, True)
            if is_selected({'upper_arm_tweak.R.001'}):
                layout.prop(pose_bones['upper_arm_tweak.R.001'], '["rubber_tweak"]', text='Rubber Tweak (upper_arm.R)', slider=True)
            if is_selected({'upper_arm_tweak.R.002'}):
                layout.prop(pose_bones['upper_arm_tweak.R.002'], '["rubber_tweak"]', text='Rubber Tweak (upper_arm.R)', slider=True)
            if is_selected({'upper_arm_tweak.R.003'}):
                layout.prop(pose_bones['upper_arm_tweak.R.003'], '["rubber_tweak"]', text='Rubber Tweak (upper_arm.R)', slider=True)
            if is_selected({'forearm_tweak.R'}):
                layout.prop(pose_bones['forearm_tweak.R'], '["rubber_tweak"]', text='Rubber Tweak (forearm.R)', slider=True)
            if is_selected({'forearm_tweak.R.001'}):
                layout.prop(pose_bones['forearm_tweak.R.001'], '["rubber_tweak"]', text='Rubber Tweak (forearm.R)', slider=True)
            if is_selected({'forearm_tweak.R.002'}):
                layout.prop(pose_bones['forearm_tweak.R.002'], '["rubber_tweak"]', text='Rubber Tweak (forearm.R)', slider=True)
            if is_selected({'forearm_tweak.R.003'}):
                layout.prop(pose_bones['forearm_tweak.R.003'], '["rubber_tweak"]', text='Rubber Tweak (forearm.R)', slider=True)

        if is_selected({'thigh_fk.L', 'shin_tweak.L.003', 'thigh_tweak.L', 'toe_fk.L', 'thigh_tweak.L.003', 'foot_ik.L', 'shin_tweak.L', 'foot_spin_ik.L', 'thigh_parent.L', 'shin_tweak.L.001', 'foot_heel_ik.L', 'shin_tweak.L.002', 'thigh_ik.L', 'thigh_tweak.L.002', 'toe_ik.L', 'shin_fk.L', 'foot_tweak.L', 'thigh_tweak.L.001', 'thigh_ik_target.L', 'VIS_thigh_ik_pole.L', 'foot_fk.L'}):
            emit_rig_separator()
            if is_selected({'thigh_fk.L', 'toe_fk.L', 'thigh_parent.L', 'shin_fk.L', 'foot_fk.L'}):
                layout.prop(pose_bones['thigh_parent.L'], '["FK_limb_follow"]', text='FK Limb Follow', slider=True)
            layout.prop(pose_bones['thigh_parent.L'], '["IK_FK"]', text='IK-FK (foot.L)', slider=True)
            group1 = layout.column(align=True)
            props = group1.operator('pose.rigify_generic_snap_x4i0wdna76224586', text='FK->IK (foot.L)', icon='SNAP_ON')
            props.output_bones = '["thigh_fk.L", "shin_fk.L", "foot_fk.L", "toe_fk.L"]'
            props.input_bones = '["thigh_ik.L", "MCH-shin_ik.L", "MCH-thigh_ik_target.L", "toe_ik.L"]'
            props.ctrl_bones = '["thigh_ik.L", "thigh_ik_target.L", "foot_ik.L", "toe_ik.L", "foot_heel_ik.L", "foot_spin_ik.L"]'
            group2 = group1.row(align=True)
            props = group2.operator('pose.rigify_generic_snap_bake_x4i0wdna76224586', text='Action', icon='ACTION_TWEAK')
            props.output_bones = '["thigh_fk.L", "shin_fk.L", "foot_fk.L", "toe_fk.L"]'
            props.input_bones = '["thigh_ik.L", "MCH-shin_ik.L", "MCH-thigh_ik_target.L", "toe_ik.L"]'
            props.ctrl_bones = '["thigh_ik.L", "thigh_ik_target.L", "foot_ik.L", "toe_ik.L", "foot_heel_ik.L", "foot_spin_ik.L"]'
            props = group2.operator('pose.rigify_clear_keyframes_x4i0wdna76224586', text='Clear', icon='CANCEL')
            props.bones = '["thigh_fk.L", "shin_fk.L", "foot_fk.L", "toe_fk.L"]'
            group1 = layout.column(align=True)
            props = group1.operator('pose.rigify_limb_ik2fk_x4i0wdna76224586', text='IK->FK (foot.L)', icon='SNAP_ON')
            props.prop_bone = 'thigh_parent.L'
            props.fk_bones = '["thigh_fk.L", "shin_fk.L", "foot_fk.L", "toe_fk.L"]'
            props.ik_bones = '["thigh_ik.L", "MCH-shin_ik.L", "MCH-thigh_ik_target.L"]'
            props.ctrl_bones = '["thigh_ik.L", "thigh_ik_target.L", "foot_ik.L"]'
            props.tail_bones = '["toe_ik.L"]'
            props.extra_ctrls = '["foot_heel_ik.L", "foot_spin_ik.L"]'
            group2 = group1.row(align=True)
            props = group2.operator('pose.rigify_limb_ik2fk_bake_x4i0wdna76224586', text='Action', icon='ACTION_TWEAK')
            props.prop_bone = 'thigh_parent.L'
            props.fk_bones = '["thigh_fk.L", "shin_fk.L", "foot_fk.L", "toe_fk.L"]'
            props.ik_bones = '["thigh_ik.L", "MCH-shin_ik.L", "MCH-thigh_ik_target.L"]'
            props.ctrl_bones = '["thigh_ik.L", "thigh_ik_target.L", "foot_ik.L"]'
            props.tail_bones = '["toe_ik.L"]'
            props.extra_ctrls = '["foot_heel_ik.L", "foot_spin_ik.L"]'
            props = group2.operator('pose.rigify_clear_keyframes_x4i0wdna76224586', text='Clear', icon='CANCEL')
            props.bones = '["thigh_ik.L", "thigh_ik_target.L", "foot_ik.L", "toe_ik.L", "foot_heel_ik.L", "foot_spin_ik.L"]'
            group1 = layout.row(align=True)
            props = group1.operator('pose.rigify_leg_roll_ik2fk_x4i0wdna76224586', text='IK->FK With Roll (foot.L)', icon='SNAP_ON')
            props.prop_bone = 'thigh_parent.L'
            props.fk_bones = '["thigh_fk.L", "shin_fk.L", "foot_fk.L", "toe_fk.L"]'
            props.ik_bones = '["thigh_ik.L", "MCH-shin_ik.L", "MCH-thigh_ik_target.L"]'
            props.ctrl_bones = '["thigh_ik.L", "thigh_ik_target.L", "foot_ik.L"]'
            props.tail_bones = '["toe_ik.L"]'
            props.extra_ctrls = '["foot_heel_ik.L", "foot_spin_ik.L"]'
            props.heel_control = 'foot_heel_ik.L'
            props = group1.operator('pose.rigify_leg_roll_ik2fk_bake_x4i0wdna76224586', text='', icon='ACTION_TWEAK')
            props.prop_bone = 'thigh_parent.L'
            props.fk_bones = '["thigh_fk.L", "shin_fk.L", "foot_fk.L", "toe_fk.L"]'
            props.ik_bones = '["thigh_ik.L", "MCH-shin_ik.L", "MCH-thigh_ik_target.L"]'
            props.ctrl_bones = '["thigh_ik.L", "thigh_ik_target.L", "foot_ik.L"]'
            props.tail_bones = '["toe_ik.L"]'
            props.extra_ctrls = '["foot_heel_ik.L", "foot_spin_ik.L"]'
            props.heel_control = 'foot_heel_ik.L'
            if is_selected({'foot_spin_ik.L', 'thigh_ik.L', 'foot_heel_ik.L', 'thigh_ik_target.L', 'toe_ik.L', 'thigh_parent.L', 'foot_ik.L'}):
                layout.prop(pose_bones['thigh_parent.L'], '["IK_Stretch"]', text='IK Stretch', slider=True)
                group1 = layout.row(align=True)
                group2 = group1.split(factor=0.65, align=True)
                props = group2.operator('pose.rigify_limb_toggle_pole_x4i0wdna76224586', icon='FORCE_MAGNETIC')
                props.prop_bone = 'thigh_parent.L'
                props.ik_bones = '["thigh_ik.L", "MCH-shin_ik.L", "MCH-thigh_ik_target.L"]'
                props.ctrl_bones = '["thigh_ik.L", "thigh_ik_target.L", "foot_ik.L"]'
                props.extra_ctrls = '["foot_heel_ik.L", "foot_spin_ik.L"]'
                group2.prop(pose_bones['thigh_parent.L'], '["pole_vector"]', text=('On' if pose_bones['thigh_parent.L']['pole_vector'] else 'Off'), toggle=True)
                props = group1.operator('pose.rigify_limb_toggle_pole_bake_x4i0wdna76224586', text='', icon='ACTION_TWEAK')
                props.prop_bone = 'thigh_parent.L'
                props.ik_bones = '["thigh_ik.L", "MCH-shin_ik.L", "MCH-thigh_ik_target.L"]'
                props.ctrl_bones = '["thigh_ik.L", "thigh_ik_target.L", "foot_ik.L"]'
                props.extra_ctrls = '["foot_heel_ik.L", "foot_spin_ik.L"]'
                layout.prop(pose_bones['foot_heel_ik.L'], '["Toe_Roll"]', text='Roll On Toe', slider=True)
                group1 = layout.row(align=True)
                group2 = group1.split(factor=0.65, align=True)
                props = group2.operator('pose.rigify_switch_parent_x4i0wdna76224586', text='IK Parent', icon='DOWNARROW_HLT')
                props.bone = 'foot_ik.L'
                props.prop_bone = 'thigh_parent.L'
                props.prop_id = 'IK_parent'
                props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head"]'
                props.locks = (False, False, False)
                group2.prop(pose_bones['thigh_parent.L'], '["IK_parent"]', text='')
                props = group1.operator('pose.rigify_switch_parent_bake_x4i0wdna76224586', text='', icon='ACTION_TWEAK')
                props.bone = 'foot_ik.L'
                props.prop_bone = 'thigh_parent.L'
                props.prop_id = 'IK_parent'
                props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head"]'
                props.locks = (False, False, False)
                group1 = layout.row(align=True)
                group2 = group1.split(factor=0.65, align=True)
                props = group2.operator('pose.rigify_switch_parent_x4i0wdna76224586', text='Pole Parent', icon='DOWNARROW_HLT')
                props.bone = 'thigh_ik_target.L'
                props.prop_bone = 'thigh_parent.L'
                props.prop_id = 'pole_parent'
                props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "foot_ik.L"]'
                props.locks = (False, True, True)
                group2.prop(pose_bones['thigh_parent.L'], '["pole_parent"]', text='')
                props = group1.operator('pose.rigify_switch_parent_bake_x4i0wdna76224586', text='', icon='ACTION_TWEAK')
                props.bone = 'thigh_ik_target.L'
                props.prop_bone = 'thigh_parent.L'
                props.prop_id = 'pole_parent'
                props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "foot_ik.L"]'
                props.locks = (False, True, True)
            if is_selected({'thigh_tweak.L.001'}):
                layout.prop(pose_bones['thigh_tweak.L.001'], '["rubber_tweak"]', text='Rubber Tweak (thigh.L)', slider=True)
            if is_selected({'thigh_tweak.L.002'}):
                layout.prop(pose_bones['thigh_tweak.L.002'], '["rubber_tweak"]', text='Rubber Tweak (thigh.L)', slider=True)
            if is_selected({'thigh_tweak.L.003'}):
                layout.prop(pose_bones['thigh_tweak.L.003'], '["rubber_tweak"]', text='Rubber Tweak (thigh.L)', slider=True)
            if is_selected({'shin_tweak.L'}):
                layout.prop(pose_bones['shin_tweak.L'], '["rubber_tweak"]', text='Rubber Tweak (shin.L)', slider=True)
            if is_selected({'shin_tweak.L.001'}):
                layout.prop(pose_bones['shin_tweak.L.001'], '["rubber_tweak"]', text='Rubber Tweak (shin.L)', slider=True)
            if is_selected({'shin_tweak.L.002'}):
                layout.prop(pose_bones['shin_tweak.L.002'], '["rubber_tweak"]', text='Rubber Tweak (shin.L)', slider=True)
            if is_selected({'shin_tweak.L.003'}):
                layout.prop(pose_bones['shin_tweak.L.003'], '["rubber_tweak"]', text='Rubber Tweak (shin.L)', slider=True)

        if is_selected({'thigh_ik_target.R', 'thigh_fk.R', 'toe_ik.R', 'VIS_thigh_ik_pole.R', 'thigh_ik.R', 'foot_heel_ik.R', 'shin_fk.R', 'thigh_parent.R', 'shin_tweak.R.001', 'foot_tweak.R', 'shin_tweak.R.002', 'shin_tweak.R', 'thigh_tweak.R.001', 'foot_fk.R', 'foot_spin_ik.R', 'foot_ik.R', 'shin_tweak.R.003', 'thigh_tweak.R.002', 'toe_fk.R', 'thigh_tweak.R.003', 'thigh_tweak.R'}):
            emit_rig_separator()
            if is_selected({'foot_fk.R', 'shin_fk.R', 'thigh_fk.R', 'thigh_parent.R', 'toe_fk.R'}):
                layout.prop(pose_bones['thigh_parent.R'], '["FK_limb_follow"]', text='FK Limb Follow', slider=True)
            layout.prop(pose_bones['thigh_parent.R'], '["IK_FK"]', text='IK-FK (foot.R)', slider=True)
            group1 = layout.column(align=True)
            props = group1.operator('pose.rigify_generic_snap_x4i0wdna76224586', text='FK->IK (foot.R)', icon='SNAP_ON')
            props.output_bones = '["thigh_fk.R", "shin_fk.R", "foot_fk.R", "toe_fk.R"]'
            props.input_bones = '["thigh_ik.R", "MCH-shin_ik.R", "MCH-thigh_ik_target.R", "toe_ik.R"]'
            props.ctrl_bones = '["thigh_ik.R", "thigh_ik_target.R", "foot_ik.R", "toe_ik.R", "foot_heel_ik.R", "foot_spin_ik.R"]'
            group2 = group1.row(align=True)
            props = group2.operator('pose.rigify_generic_snap_bake_x4i0wdna76224586', text='Action', icon='ACTION_TWEAK')
            props.output_bones = '["thigh_fk.R", "shin_fk.R", "foot_fk.R", "toe_fk.R"]'
            props.input_bones = '["thigh_ik.R", "MCH-shin_ik.R", "MCH-thigh_ik_target.R", "toe_ik.R"]'
            props.ctrl_bones = '["thigh_ik.R", "thigh_ik_target.R", "foot_ik.R", "toe_ik.R", "foot_heel_ik.R", "foot_spin_ik.R"]'
            props = group2.operator('pose.rigify_clear_keyframes_x4i0wdna76224586', text='Clear', icon='CANCEL')
            props.bones = '["thigh_fk.R", "shin_fk.R", "foot_fk.R", "toe_fk.R"]'
            group1 = layout.column(align=True)
            props = group1.operator('pose.rigify_limb_ik2fk_x4i0wdna76224586', text='IK->FK (foot.R)', icon='SNAP_ON')
            props.prop_bone = 'thigh_parent.R'
            props.fk_bones = '["thigh_fk.R", "shin_fk.R", "foot_fk.R", "toe_fk.R"]'
            props.ik_bones = '["thigh_ik.R", "MCH-shin_ik.R", "MCH-thigh_ik_target.R"]'
            props.ctrl_bones = '["thigh_ik.R", "thigh_ik_target.R", "foot_ik.R"]'
            props.tail_bones = '["toe_ik.R"]'
            props.extra_ctrls = '["foot_heel_ik.R", "foot_spin_ik.R"]'
            group2 = group1.row(align=True)
            props = group2.operator('pose.rigify_limb_ik2fk_bake_x4i0wdna76224586', text='Action', icon='ACTION_TWEAK')
            props.prop_bone = 'thigh_parent.R'
            props.fk_bones = '["thigh_fk.R", "shin_fk.R", "foot_fk.R", "toe_fk.R"]'
            props.ik_bones = '["thigh_ik.R", "MCH-shin_ik.R", "MCH-thigh_ik_target.R"]'
            props.ctrl_bones = '["thigh_ik.R", "thigh_ik_target.R", "foot_ik.R"]'
            props.tail_bones = '["toe_ik.R"]'
            props.extra_ctrls = '["foot_heel_ik.R", "foot_spin_ik.R"]'
            props = group2.operator('pose.rigify_clear_keyframes_x4i0wdna76224586', text='Clear', icon='CANCEL')
            props.bones = '["thigh_ik.R", "thigh_ik_target.R", "foot_ik.R", "toe_ik.R", "foot_heel_ik.R", "foot_spin_ik.R"]'
            group1 = layout.row(align=True)
            props = group1.operator('pose.rigify_leg_roll_ik2fk_x4i0wdna76224586', text='IK->FK With Roll (foot.R)', icon='SNAP_ON')
            props.prop_bone = 'thigh_parent.R'
            props.fk_bones = '["thigh_fk.R", "shin_fk.R", "foot_fk.R", "toe_fk.R"]'
            props.ik_bones = '["thigh_ik.R", "MCH-shin_ik.R", "MCH-thigh_ik_target.R"]'
            props.ctrl_bones = '["thigh_ik.R", "thigh_ik_target.R", "foot_ik.R"]'
            props.tail_bones = '["toe_ik.R"]'
            props.extra_ctrls = '["foot_heel_ik.R", "foot_spin_ik.R"]'
            props.heel_control = 'foot_heel_ik.R'
            props = group1.operator('pose.rigify_leg_roll_ik2fk_bake_x4i0wdna76224586', text='', icon='ACTION_TWEAK')
            props.prop_bone = 'thigh_parent.R'
            props.fk_bones = '["thigh_fk.R", "shin_fk.R", "foot_fk.R", "toe_fk.R"]'
            props.ik_bones = '["thigh_ik.R", "MCH-shin_ik.R", "MCH-thigh_ik_target.R"]'
            props.ctrl_bones = '["thigh_ik.R", "thigh_ik_target.R", "foot_ik.R"]'
            props.tail_bones = '["toe_ik.R"]'
            props.extra_ctrls = '["foot_heel_ik.R", "foot_spin_ik.R"]'
            props.heel_control = 'foot_heel_ik.R'
            if is_selected({'thigh_ik_target.R', 'thigh_parent.R', 'toe_ik.R', 'foot_ik.R', 'foot_spin_ik.R', 'thigh_ik.R', 'foot_heel_ik.R'}):
                layout.prop(pose_bones['thigh_parent.R'], '["IK_Stretch"]', text='IK Stretch', slider=True)
                group1 = layout.row(align=True)
                group2 = group1.split(factor=0.65, align=True)
                props = group2.operator('pose.rigify_limb_toggle_pole_x4i0wdna76224586', icon='FORCE_MAGNETIC')
                props.prop_bone = 'thigh_parent.R'
                props.ik_bones = '["thigh_ik.R", "MCH-shin_ik.R", "MCH-thigh_ik_target.R"]'
                props.ctrl_bones = '["thigh_ik.R", "thigh_ik_target.R", "foot_ik.R"]'
                props.extra_ctrls = '["foot_heel_ik.R", "foot_spin_ik.R"]'
                group2.prop(pose_bones['thigh_parent.R'], '["pole_vector"]', text=('On' if pose_bones['thigh_parent.R']['pole_vector'] else 'Off'), toggle=True)
                props = group1.operator('pose.rigify_limb_toggle_pole_bake_x4i0wdna76224586', text='', icon='ACTION_TWEAK')
                props.prop_bone = 'thigh_parent.R'
                props.ik_bones = '["thigh_ik.R", "MCH-shin_ik.R", "MCH-thigh_ik_target.R"]'
                props.ctrl_bones = '["thigh_ik.R", "thigh_ik_target.R", "foot_ik.R"]'
                props.extra_ctrls = '["foot_heel_ik.R", "foot_spin_ik.R"]'
                layout.prop(pose_bones['foot_heel_ik.R'], '["Toe_Roll"]', text='Roll On Toe', slider=True)
                group1 = layout.row(align=True)
                group2 = group1.split(factor=0.65, align=True)
                props = group2.operator('pose.rigify_switch_parent_x4i0wdna76224586', text='IK Parent', icon='DOWNARROW_HLT')
                props.bone = 'foot_ik.R'
                props.prop_bone = 'thigh_parent.R'
                props.prop_id = 'IK_parent'
                props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head"]'
                props.locks = (False, False, False)
                group2.prop(pose_bones['thigh_parent.R'], '["IK_parent"]', text='')
                props = group1.operator('pose.rigify_switch_parent_bake_x4i0wdna76224586', text='', icon='ACTION_TWEAK')
                props.bone = 'foot_ik.R'
                props.prop_bone = 'thigh_parent.R'
                props.prop_id = 'IK_parent'
                props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head"]'
                props.locks = (False, False, False)
                group1 = layout.row(align=True)
                group2 = group1.split(factor=0.65, align=True)
                props = group2.operator('pose.rigify_switch_parent_x4i0wdna76224586', text='Pole Parent', icon='DOWNARROW_HLT')
                props.bone = 'thigh_ik_target.R'
                props.prop_bone = 'thigh_parent.R'
                props.prop_id = 'pole_parent'
                props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "foot_ik.R"]'
                props.locks = (False, True, True)
                group2.prop(pose_bones['thigh_parent.R'], '["pole_parent"]', text='')
                props = group1.operator('pose.rigify_switch_parent_bake_x4i0wdna76224586', text='', icon='ACTION_TWEAK')
                props.bone = 'thigh_ik_target.R'
                props.prop_bone = 'thigh_parent.R'
                props.prop_id = 'pole_parent'
                props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "foot_ik.R"]'
                props.locks = (False, True, True)
            if is_selected({'thigh_tweak.R.001'}):
                layout.prop(pose_bones['thigh_tweak.R.001'], '["rubber_tweak"]', text='Rubber Tweak (thigh.R)', slider=True)
            if is_selected({'thigh_tweak.R.002'}):
                layout.prop(pose_bones['thigh_tweak.R.002'], '["rubber_tweak"]', text='Rubber Tweak (thigh.R)', slider=True)
            if is_selected({'thigh_tweak.R.003'}):
                layout.prop(pose_bones['thigh_tweak.R.003'], '["rubber_tweak"]', text='Rubber Tweak (thigh.R)', slider=True)
            if is_selected({'shin_tweak.R'}):
                layout.prop(pose_bones['shin_tweak.R'], '["rubber_tweak"]', text='Rubber Tweak (shin.R)', slider=True)
            if is_selected({'shin_tweak.R.001'}):
                layout.prop(pose_bones['shin_tweak.R.001'], '["rubber_tweak"]', text='Rubber Tweak (shin.R)', slider=True)
            if is_selected({'shin_tweak.R.002'}):
                layout.prop(pose_bones['shin_tweak.R.002'], '["rubber_tweak"]', text='Rubber Tweak (shin.R)', slider=True)
            if is_selected({'shin_tweak.R.003'}):
                layout.prop(pose_bones['shin_tweak.R.003'], '["rubber_tweak"]', text='Rubber Tweak (shin.R)', slider=True)

        if is_selected({'Prop_Slot.L'}):
            emit_rig_separator()
            group1 = layout.row(align=True)
            group2 = group1.split(factor=0.65, align=True)
            props = group2.operator('pose.rigify_switch_parent_x4i0wdna76224586', text='Parent (Prop_Slot.L)', icon='DOWNARROW_HLT')
            props.bone = 'Prop_Slot.L'
            props.prop_bone = 'Prop_Slot.L'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.L", "hand_ik.L", "hand.L"]'
            props.locks = (False, False, False)
            group2.prop(pose_bones['Prop_Slot.L'], '["parent_switch"]', text='')
            props = group1.operator('pose.rigify_switch_parent_bake_x4i0wdna76224586', text='', icon='ACTION_TWEAK')
            props.bone = 'Prop_Slot.L'
            props.prop_bone = 'Prop_Slot.L'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.L", "hand_ik.L", "hand.L"]'
            props.locks = (False, False, False)

        if is_selected({'finger_IK_parent.L'}):
            emit_rig_separator()
            group1 = layout.row(align=True)
            group2 = group1.split(factor=0.65, align=True)
            props = group2.operator('pose.rigify_switch_parent_x4i0wdna76224586', text='Parent (finger_IK_parent.L)', icon='DOWNARROW_HLT')
            props.bone = 'finger_IK_parent.L'
            props.prop_bone = 'finger_IK_parent.L'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.L", "hand_ik.L", "hand.L"]'
            props.locks = (False, False, False)
            group2.prop(pose_bones['finger_IK_parent.L'], '["parent_switch"]', text='')
            props = group1.operator('pose.rigify_switch_parent_bake_x4i0wdna76224586', text='', icon='ACTION_TWEAK')
            props.bone = 'finger_IK_parent.L'
            props.prop_bone = 'finger_IK_parent.L'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.L", "hand_ik.L", "hand.L"]'
            props.locks = (False, False, False)

        if is_selected({'f_index.03IK.L'}):
            emit_rig_separator()
            group1 = layout.row(align=True)
            group2 = group1.split(factor=0.65, align=True)
            props = group2.operator('pose.rigify_switch_parent_x4i0wdna76224586', text='Parent (f_index.03IK.L)', icon='DOWNARROW_HLT')
            props.bone = 'f_index.03IK.L'
            props.prop_bone = 'f_index.03IK.L'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.L", "hand_ik.L", "hand.L", "finger_IK_parent.L"]'
            props.locks = (False, False, False)
            group2.prop(pose_bones['f_index.03IK.L'], '["parent_switch"]', text='')
            props = group1.operator('pose.rigify_switch_parent_bake_x4i0wdna76224586', text='', icon='ACTION_TWEAK')
            props.bone = 'f_index.03IK.L'
            props.prop_bone = 'f_index.03IK.L'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.L", "hand_ik.L", "hand.L", "finger_IK_parent.L"]'
            props.locks = (False, False, False)

        if is_selected({'f_index.pole.L'}):
            emit_rig_separator()
            group1 = layout.row(align=True)
            group2 = group1.split(factor=0.65, align=True)
            props = group2.operator('pose.rigify_switch_parent_x4i0wdna76224586', text='Parent (f_index.pole.L)', icon='DOWNARROW_HLT')
            props.bone = 'f_index.pole.L'
            props.prop_bone = 'f_index.pole.L'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.L", "hand_ik.L", "hand.L", "f_index.03IK.L"]'
            props.locks = (False, False, False)
            group2.prop(pose_bones['f_index.pole.L'], '["parent_switch"]', text='')
            props = group1.operator('pose.rigify_switch_parent_bake_x4i0wdna76224586', text='', icon='ACTION_TWEAK')
            props.bone = 'f_index.pole.L'
            props.prop_bone = 'f_index.pole.L'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.L", "hand_ik.L", "hand.L", "f_index.03IK.L"]'
            props.locks = (False, False, False)

        if is_selected({'f_middle.03IK.L'}):
            emit_rig_separator()
            group1 = layout.row(align=True)
            group2 = group1.split(factor=0.65, align=True)
            props = group2.operator('pose.rigify_switch_parent_x4i0wdna76224586', text='Parent (f_middle.03IK.L)', icon='DOWNARROW_HLT')
            props.bone = 'f_middle.03IK.L'
            props.prop_bone = 'f_middle.03IK.L'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.L", "hand_ik.L", "hand.L", "finger_IK_parent.L"]'
            props.locks = (False, False, False)
            group2.prop(pose_bones['f_middle.03IK.L'], '["parent_switch"]', text='')
            props = group1.operator('pose.rigify_switch_parent_bake_x4i0wdna76224586', text='', icon='ACTION_TWEAK')
            props.bone = 'f_middle.03IK.L'
            props.prop_bone = 'f_middle.03IK.L'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.L", "hand_ik.L", "hand.L", "finger_IK_parent.L"]'
            props.locks = (False, False, False)

        if is_selected({'f_middle.pole.L'}):
            emit_rig_separator()
            group1 = layout.row(align=True)
            group2 = group1.split(factor=0.65, align=True)
            props = group2.operator('pose.rigify_switch_parent_x4i0wdna76224586', text='Parent (f_middle.pole.L)', icon='DOWNARROW_HLT')
            props.bone = 'f_middle.pole.L'
            props.prop_bone = 'f_middle.pole.L'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.L", "hand_ik.L", "hand.L", "f_middle.03IK.L"]'
            props.locks = (False, False, False)
            group2.prop(pose_bones['f_middle.pole.L'], '["parent_switch"]', text='')
            props = group1.operator('pose.rigify_switch_parent_bake_x4i0wdna76224586', text='', icon='ACTION_TWEAK')
            props.bone = 'f_middle.pole.L'
            props.prop_bone = 'f_middle.pole.L'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.L", "hand_ik.L", "hand.L", "f_middle.03IK.L"]'
            props.locks = (False, False, False)

        if is_selected({'f_pinky.03IK.L'}):
            emit_rig_separator()
            group1 = layout.row(align=True)
            group2 = group1.split(factor=0.65, align=True)
            props = group2.operator('pose.rigify_switch_parent_x4i0wdna76224586', text='Parent (f_pinky.03IK.L)', icon='DOWNARROW_HLT')
            props.bone = 'f_pinky.03IK.L'
            props.prop_bone = 'f_pinky.03IK.L'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.L", "hand_ik.L", "hand.L", "finger_IK_parent.L"]'
            props.locks = (False, False, False)
            group2.prop(pose_bones['f_pinky.03IK.L'], '["parent_switch"]', text='')
            props = group1.operator('pose.rigify_switch_parent_bake_x4i0wdna76224586', text='', icon='ACTION_TWEAK')
            props.bone = 'f_pinky.03IK.L'
            props.prop_bone = 'f_pinky.03IK.L'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.L", "hand_ik.L", "hand.L", "finger_IK_parent.L"]'
            props.locks = (False, False, False)

        if is_selected({'f_pinky.pole.L'}):
            emit_rig_separator()
            group1 = layout.row(align=True)
            group2 = group1.split(factor=0.65, align=True)
            props = group2.operator('pose.rigify_switch_parent_x4i0wdna76224586', text='Parent (f_pinky.pole.L)', icon='DOWNARROW_HLT')
            props.bone = 'f_pinky.pole.L'
            props.prop_bone = 'f_pinky.pole.L'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.L", "hand_ik.L", "hand.L", "f_pinky.03IK.L"]'
            props.locks = (False, False, False)
            group2.prop(pose_bones['f_pinky.pole.L'], '["parent_switch"]', text='')
            props = group1.operator('pose.rigify_switch_parent_bake_x4i0wdna76224586', text='', icon='ACTION_TWEAK')
            props.bone = 'f_pinky.pole.L'
            props.prop_bone = 'f_pinky.pole.L'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.L", "hand_ik.L", "hand.L", "f_pinky.03IK.L"]'
            props.locks = (False, False, False)

        if is_selected({'f_ring.03IK.L'}):
            emit_rig_separator()
            group1 = layout.row(align=True)
            group2 = group1.split(factor=0.65, align=True)
            props = group2.operator('pose.rigify_switch_parent_x4i0wdna76224586', text='Parent (f_ring.03IK.L)', icon='DOWNARROW_HLT')
            props.bone = 'f_ring.03IK.L'
            props.prop_bone = 'f_ring.03IK.L'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.L", "hand_ik.L", "hand.L", "finger_IK_parent.L"]'
            props.locks = (False, False, False)
            group2.prop(pose_bones['f_ring.03IK.L'], '["parent_switch"]', text='')
            props = group1.operator('pose.rigify_switch_parent_bake_x4i0wdna76224586', text='', icon='ACTION_TWEAK')
            props.bone = 'f_ring.03IK.L'
            props.prop_bone = 'f_ring.03IK.L'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.L", "hand_ik.L", "hand.L", "finger_IK_parent.L"]'
            props.locks = (False, False, False)

        if is_selected({'f_ring.pole.L'}):
            emit_rig_separator()
            group1 = layout.row(align=True)
            group2 = group1.split(factor=0.65, align=True)
            props = group2.operator('pose.rigify_switch_parent_x4i0wdna76224586', text='Parent (f_ring.pole.L)', icon='DOWNARROW_HLT')
            props.bone = 'f_ring.pole.L'
            props.prop_bone = 'f_ring.pole.L'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.L", "hand_ik.L", "hand.L", "f_ring.03IK.L"]'
            props.locks = (False, False, False)
            group2.prop(pose_bones['f_ring.pole.L'], '["parent_switch"]', text='')
            props = group1.operator('pose.rigify_switch_parent_bake_x4i0wdna76224586', text='', icon='ACTION_TWEAK')
            props.bone = 'f_ring.pole.L'
            props.prop_bone = 'f_ring.pole.L'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.L", "hand_ik.L", "hand.L", "f_ring.03IK.L"]'
            props.locks = (False, False, False)

        if is_selected({'f_thumb.03IK.L'}):
            emit_rig_separator()
            group1 = layout.row(align=True)
            group2 = group1.split(factor=0.65, align=True)
            props = group2.operator('pose.rigify_switch_parent_x4i0wdna76224586', text='Parent (f_thumb.03IK.L)', icon='DOWNARROW_HLT')
            props.bone = 'f_thumb.03IK.L'
            props.prop_bone = 'f_thumb.03IK.L'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.L", "hand_ik.L", "hand.L", "finger_IK_parent.L"]'
            props.locks = (False, False, False)
            group2.prop(pose_bones['f_thumb.03IK.L'], '["parent_switch"]', text='')
            props = group1.operator('pose.rigify_switch_parent_bake_x4i0wdna76224586', text='', icon='ACTION_TWEAK')
            props.bone = 'f_thumb.03IK.L'
            props.prop_bone = 'f_thumb.03IK.L'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.L", "hand_ik.L", "hand.L", "finger_IK_parent.L"]'
            props.locks = (False, False, False)

        if is_selected({'f_thumb.pole.L'}):
            emit_rig_separator()
            group1 = layout.row(align=True)
            group2 = group1.split(factor=0.65, align=True)
            props = group2.operator('pose.rigify_switch_parent_x4i0wdna76224586', text='Parent (f_thumb.pole.L)', icon='DOWNARROW_HLT')
            props.bone = 'f_thumb.pole.L'
            props.prop_bone = 'f_thumb.pole.L'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.L", "hand_ik.L", "hand.L", "f_thumb.03IK.L"]'
            props.locks = (False, False, False)
            group2.prop(pose_bones['f_thumb.pole.L'], '["parent_switch"]', text='')
            props = group1.operator('pose.rigify_switch_parent_bake_x4i0wdna76224586', text='', icon='ACTION_TWEAK')
            props.bone = 'f_thumb.pole.L'
            props.prop_bone = 'f_thumb.pole.L'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.L", "hand_ik.L", "hand.L", "f_thumb.03IK.L"]'
            props.locks = (False, False, False)

        if is_selected({'Prop_Slot.R'}):
            emit_rig_separator()
            group1 = layout.row(align=True)
            group2 = group1.split(factor=0.65, align=True)
            props = group2.operator('pose.rigify_switch_parent_x4i0wdna76224586', text='Parent (Prop_Slot.R)', icon='DOWNARROW_HLT')
            props.bone = 'Prop_Slot.R'
            props.prop_bone = 'Prop_Slot.R'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.R", "hand_ik.R", "hand.R"]'
            props.locks = (False, False, False)
            group2.prop(pose_bones['Prop_Slot.R'], '["parent_switch"]', text='')
            props = group1.operator('pose.rigify_switch_parent_bake_x4i0wdna76224586', text='', icon='ACTION_TWEAK')
            props.bone = 'Prop_Slot.R'
            props.prop_bone = 'Prop_Slot.R'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.R", "hand_ik.R", "hand.R"]'
            props.locks = (False, False, False)

        if is_selected({'finger_IK_parent.R'}):
            emit_rig_separator()
            group1 = layout.row(align=True)
            group2 = group1.split(factor=0.65, align=True)
            props = group2.operator('pose.rigify_switch_parent_x4i0wdna76224586', text='Parent (finger_IK_parent.R)', icon='DOWNARROW_HLT')
            props.bone = 'finger_IK_parent.R'
            props.prop_bone = 'finger_IK_parent.R'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.R", "hand_ik.R", "hand.R"]'
            props.locks = (False, False, False)
            group2.prop(pose_bones['finger_IK_parent.R'], '["parent_switch"]', text='')
            props = group1.operator('pose.rigify_switch_parent_bake_x4i0wdna76224586', text='', icon='ACTION_TWEAK')
            props.bone = 'finger_IK_parent.R'
            props.prop_bone = 'finger_IK_parent.R'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.R", "hand_ik.R", "hand.R"]'
            props.locks = (False, False, False)

        if is_selected({'f_index.03IK.R'}):
            emit_rig_separator()
            group1 = layout.row(align=True)
            group2 = group1.split(factor=0.65, align=True)
            props = group2.operator('pose.rigify_switch_parent_x4i0wdna76224586', text='Parent (f_index.03IK.R)', icon='DOWNARROW_HLT')
            props.bone = 'f_index.03IK.R'
            props.prop_bone = 'f_index.03IK.R'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.R", "hand_ik.R", "hand.R", "finger_IK_parent.R"]'
            props.locks = (False, False, False)
            group2.prop(pose_bones['f_index.03IK.R'], '["parent_switch"]', text='')
            props = group1.operator('pose.rigify_switch_parent_bake_x4i0wdna76224586', text='', icon='ACTION_TWEAK')
            props.bone = 'f_index.03IK.R'
            props.prop_bone = 'f_index.03IK.R'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.R", "hand_ik.R", "hand.R", "finger_IK_parent.R"]'
            props.locks = (False, False, False)

        if is_selected({'f_index.pole.R'}):
            emit_rig_separator()
            group1 = layout.row(align=True)
            group2 = group1.split(factor=0.65, align=True)
            props = group2.operator('pose.rigify_switch_parent_x4i0wdna76224586', text='Parent (f_index.pole.R)', icon='DOWNARROW_HLT')
            props.bone = 'f_index.pole.R'
            props.prop_bone = 'f_index.pole.R'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.R", "hand_ik.R", "hand.R", "f_index.03IK.R"]'
            props.locks = (False, False, False)
            group2.prop(pose_bones['f_index.pole.R'], '["parent_switch"]', text='')
            props = group1.operator('pose.rigify_switch_parent_bake_x4i0wdna76224586', text='', icon='ACTION_TWEAK')
            props.bone = 'f_index.pole.R'
            props.prop_bone = 'f_index.pole.R'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.R", "hand_ik.R", "hand.R", "f_index.03IK.R"]'
            props.locks = (False, False, False)

        if is_selected({'f_middle.03IK.R'}):
            emit_rig_separator()
            group1 = layout.row(align=True)
            group2 = group1.split(factor=0.65, align=True)
            props = group2.operator('pose.rigify_switch_parent_x4i0wdna76224586', text='Parent (f_middle.03IK.R)', icon='DOWNARROW_HLT')
            props.bone = 'f_middle.03IK.R'
            props.prop_bone = 'f_middle.03IK.R'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.R", "hand_ik.R", "hand.R", "finger_IK_parent.R"]'
            props.locks = (False, False, False)
            group2.prop(pose_bones['f_middle.03IK.R'], '["parent_switch"]', text='')
            props = group1.operator('pose.rigify_switch_parent_bake_x4i0wdna76224586', text='', icon='ACTION_TWEAK')
            props.bone = 'f_middle.03IK.R'
            props.prop_bone = 'f_middle.03IK.R'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.R", "hand_ik.R", "hand.R", "finger_IK_parent.R"]'
            props.locks = (False, False, False)

        if is_selected({'f_middle.pole.R'}):
            emit_rig_separator()
            group1 = layout.row(align=True)
            group2 = group1.split(factor=0.65, align=True)
            props = group2.operator('pose.rigify_switch_parent_x4i0wdna76224586', text='Parent (f_middle.pole.R)', icon='DOWNARROW_HLT')
            props.bone = 'f_middle.pole.R'
            props.prop_bone = 'f_middle.pole.R'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.R", "hand_ik.R", "hand.R", "f_middle.03IK.R"]'
            props.locks = (False, False, False)
            group2.prop(pose_bones['f_middle.pole.R'], '["parent_switch"]', text='')
            props = group1.operator('pose.rigify_switch_parent_bake_x4i0wdna76224586', text='', icon='ACTION_TWEAK')
            props.bone = 'f_middle.pole.R'
            props.prop_bone = 'f_middle.pole.R'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.R", "hand_ik.R", "hand.R", "f_middle.03IK.R"]'
            props.locks = (False, False, False)

        if is_selected({'f_pinky.03IK.R'}):
            emit_rig_separator()
            group1 = layout.row(align=True)
            group2 = group1.split(factor=0.65, align=True)
            props = group2.operator('pose.rigify_switch_parent_x4i0wdna76224586', text='Parent (f_pinky.03IK.R)', icon='DOWNARROW_HLT')
            props.bone = 'f_pinky.03IK.R'
            props.prop_bone = 'f_pinky.03IK.R'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.R", "hand_ik.R", "hand.R", "finger_IK_parent.R"]'
            props.locks = (False, False, False)
            group2.prop(pose_bones['f_pinky.03IK.R'], '["parent_switch"]', text='')
            props = group1.operator('pose.rigify_switch_parent_bake_x4i0wdna76224586', text='', icon='ACTION_TWEAK')
            props.bone = 'f_pinky.03IK.R'
            props.prop_bone = 'f_pinky.03IK.R'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.R", "hand_ik.R", "hand.R", "finger_IK_parent.R"]'
            props.locks = (False, False, False)

        if is_selected({'f_pinky.pole.R'}):
            emit_rig_separator()
            group1 = layout.row(align=True)
            group2 = group1.split(factor=0.65, align=True)
            props = group2.operator('pose.rigify_switch_parent_x4i0wdna76224586', text='Parent (f_pinky.pole.R)', icon='DOWNARROW_HLT')
            props.bone = 'f_pinky.pole.R'
            props.prop_bone = 'f_pinky.pole.R'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.R", "hand_ik.R", "hand.R", "f_pinky.03IK.R"]'
            props.locks = (False, False, False)
            group2.prop(pose_bones['f_pinky.pole.R'], '["parent_switch"]', text='')
            props = group1.operator('pose.rigify_switch_parent_bake_x4i0wdna76224586', text='', icon='ACTION_TWEAK')
            props.bone = 'f_pinky.pole.R'
            props.prop_bone = 'f_pinky.pole.R'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.R", "hand_ik.R", "hand.R", "f_pinky.03IK.R"]'
            props.locks = (False, False, False)

        if is_selected({'f_ring.03IK.R'}):
            emit_rig_separator()
            group1 = layout.row(align=True)
            group2 = group1.split(factor=0.65, align=True)
            props = group2.operator('pose.rigify_switch_parent_x4i0wdna76224586', text='Parent (f_ring.03IK.R)', icon='DOWNARROW_HLT')
            props.bone = 'f_ring.03IK.R'
            props.prop_bone = 'f_ring.03IK.R'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.R", "hand_ik.R", "hand.R", "finger_IK_parent.R"]'
            props.locks = (False, False, False)
            group2.prop(pose_bones['f_ring.03IK.R'], '["parent_switch"]', text='')
            props = group1.operator('pose.rigify_switch_parent_bake_x4i0wdna76224586', text='', icon='ACTION_TWEAK')
            props.bone = 'f_ring.03IK.R'
            props.prop_bone = 'f_ring.03IK.R'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.R", "hand_ik.R", "hand.R", "finger_IK_parent.R"]'
            props.locks = (False, False, False)

        if is_selected({'f_ring.pole.R'}):
            emit_rig_separator()
            group1 = layout.row(align=True)
            group2 = group1.split(factor=0.65, align=True)
            props = group2.operator('pose.rigify_switch_parent_x4i0wdna76224586', text='Parent (f_ring.pole.R)', icon='DOWNARROW_HLT')
            props.bone = 'f_ring.pole.R'
            props.prop_bone = 'f_ring.pole.R'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.R", "hand_ik.R", "hand.R", "f_ring.03IK.R"]'
            props.locks = (False, False, False)
            group2.prop(pose_bones['f_ring.pole.R'], '["parent_switch"]', text='')
            props = group1.operator('pose.rigify_switch_parent_bake_x4i0wdna76224586', text='', icon='ACTION_TWEAK')
            props.bone = 'f_ring.pole.R'
            props.prop_bone = 'f_ring.pole.R'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.R", "hand_ik.R", "hand.R", "f_ring.03IK.R"]'
            props.locks = (False, False, False)

        if is_selected({'f_thumb.03IK.R'}):
            emit_rig_separator()
            group1 = layout.row(align=True)
            group2 = group1.split(factor=0.65, align=True)
            props = group2.operator('pose.rigify_switch_parent_x4i0wdna76224586', text='Parent (f_thumb.03IK.R)', icon='DOWNARROW_HLT')
            props.bone = 'f_thumb.03IK.R'
            props.prop_bone = 'f_thumb.03IK.R'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.R", "hand_ik.R", "hand.R", "finger_IK_parent.R"]'
            props.locks = (False, False, False)
            group2.prop(pose_bones['f_thumb.03IK.R'], '["parent_switch"]', text='')
            props = group1.operator('pose.rigify_switch_parent_bake_x4i0wdna76224586', text='', icon='ACTION_TWEAK')
            props.bone = 'f_thumb.03IK.R'
            props.prop_bone = 'f_thumb.03IK.R'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.R", "hand_ik.R", "hand.R", "finger_IK_parent.R"]'
            props.locks = (False, False, False)

        if is_selected({'f_thumb.pole.R'}):
            emit_rig_separator()
            group1 = layout.row(align=True)
            group2 = group1.split(factor=0.65, align=True)
            props = group2.operator('pose.rigify_switch_parent_x4i0wdna76224586', text='Parent (f_thumb.pole.R)', icon='DOWNARROW_HLT')
            props.bone = 'f_thumb.pole.R'
            props.prop_bone = 'f_thumb.pole.R'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.R", "hand_ik.R", "hand.R", "f_thumb.03IK.R"]'
            props.locks = (False, False, False)
            group2.prop(pose_bones['f_thumb.pole.R'], '["parent_switch"]', text='')
            props = group1.operator('pose.rigify_switch_parent_bake_x4i0wdna76224586', text='', icon='ACTION_TWEAK')
            props.bone = 'f_thumb.pole.R'
            props.prop_bone = 'f_thumb.pole.R'
            props.prop_id = 'parent_switch'
            props.parent_names = '["None", "Root", "Torso", "Hips", "Chest", "Head", "shoulder.tweak.R", "hand_ik.R", "hand.R", "f_thumb.03IK.R"]'
            props.locks = (False, False, False)

class RigBakeSettings(bpy.types.Panel):
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI'
    bl_label = "Rig Bake Settings"
    bl_idname = "VIEW3D_PT_rig_bake_settings_" + rig_id
    bl_category = 'Item'

    @classmethod
    def poll(self, context):
        return RigUI.poll(context) and find_action(context.active_object) is not None

    def draw(self, context):
        RigifyBakeKeyframesMixin.draw_common_bake_ui(context, self.layout)

### Displaying bone collections (Blender 4.0 and above)
class P2M_Big_RigLayers(bpy.types.Panel):
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI'
    bl_label = "P2M Big Rig Layers"
    bl_idname = "VIEW3D_PT_rig_layers_" + rig_id
    bl_category = 'Item'

    @classmethod
    def poll(self, context):
        try:
            return (context.active_object.data.get("rig_id") == rig_id)
        except (AttributeError, KeyError, TypeError):
            return False


    def draw(self, context):
        layout = self.layout
        col = layout.column()
        collection = context.active_object.data.collections_all

        ### We can create rows exposing the visibility of our bone collection and give them a name 
        
        row = col.row(align = True)  
        row.prop(collection["Face"], 'is_visible', toggle=True, text='Face')
        row.prop(collection["Face"], 'is_solo', toggle=True, text="",icon='SOLO_OFF', invert_checkbox=False, emboss=True)
        row = col.row(align = True)  
        row.prop(collection["Torso"], 'is_visible', toggle=True, text='Torso')
        row.prop(collection["Torso"], 'is_solo', toggle=True, text="",icon='SOLO_OFF', invert_checkbox=False, emboss=True)
        row = col.row(align = True)  
        row.prop(collection["Torso (Tweak)"], 'is_visible', toggle=True, text='(Tweak)')
        row.prop(collection["Torso (Tweak)"], 'is_solo', toggle=True, text="",icon='SOLO_OFF', invert_checkbox=False, emboss=True)
        
        row = col.row(align = True) 
        row = col.row(align = True) 
        row = col.row(align = True)  
        row.prop(collection["Belly"], 'is_visible', toggle=True, text='Belly')
        row.prop(collection["Belly"], 'is_solo', toggle=True, text="",icon='SOLO_OFF', invert_checkbox=False, emboss=True)
        row = col.row(align = True)  
        row.prop(collection["Belly (Tweak)"], 'is_visible', toggle=True, text='(Tweak)')
        row.prop(collection["Belly (Tweak)"], 'is_solo', toggle=True, text="",icon='SOLO_OFF', invert_checkbox=False, emboss=True)
        
        row = col.row(align = True)
        row = col.row(align = True)
        row = col.row(align = True)  
        row.prop(collection["Arm.L (IK)"], 'is_visible', toggle=True, text='Arm.L (IK)')
        row.prop(collection["Arm.L (IK)"], 'is_solo', toggle=True, text="",icon='SOLO_OFF', invert_checkbox=False, emboss=True)
        row.prop(collection["Arm.R (IK)"], 'is_visible', toggle=True, text='Arm.R (IK)')
        row.prop(collection["Arm.R (IK)"], 'is_solo', toggle=True, text="",icon='SOLO_OFF', invert_checkbox=False, emboss=True)
        row = col.row(align = True)  
        row.prop(collection["Arm.L (FK)"], 'is_visible', toggle=True, text='(FK)')
        row.prop(collection["Arm.L (FK)"], 'is_solo', toggle=True, text="",icon='SOLO_OFF', invert_checkbox=False, emboss=True)
        row.prop(collection["Arm.R (FK)"], 'is_visible', toggle=True, text='(FK)')
        row.prop(collection["Arm.R (FK)"], 'is_solo', toggle=True, text="",icon='SOLO_OFF', invert_checkbox=False, emboss=True)
        row = col.row(align = True)  
        row.prop(collection["Arm.L (Tweak)"], 'is_visible', toggle=True, text='(Tweak)')
        row.prop(collection["Arm.L (Tweak)"], 'is_solo', toggle=True, text="",icon='SOLO_OFF', invert_checkbox=False, emboss=True)
        row.prop(collection["Arm.R (Tweak)"], 'is_visible', toggle=True, text='(Tweak)')
        row.prop(collection["Arm.R (Tweak)"], 'is_solo', toggle=True, text="",icon='SOLO_OFF', invert_checkbox=False, emboss=True)
        
        row = col.row(align = True)
        row = col.row(align = True)  
        row = col.row(align = True)  
        row.prop(collection["Fingers.L"], 'is_visible', toggle=True, text='Fingers.L')
        row.prop(collection["Fingers.L"], 'is_solo', toggle=True, text="",icon='SOLO_OFF', invert_checkbox=False, emboss=True)
        row.prop(collection["Fingers.R"], 'is_visible', toggle=True, text='Fingers.R')
        row.prop(collection["Fingers.R"], 'is_solo', toggle=True, text="",icon='SOLO_OFF', invert_checkbox=False, emboss=True)
        row = col.row(align = True)  
        row.prop(collection["Fingers.L (IK)"], 'is_visible', toggle=True, text='(IK)')
        row.prop(collection["Fingers.L (IK)"], 'is_solo', toggle=True, text="",icon='SOLO_OFF', invert_checkbox=False, emboss=True)
        row.prop(collection["Fingers.R (IK)"], 'is_visible', toggle=True, text='(IK)')
        row.prop(collection["Fingers.R (IK)"], 'is_solo', toggle=True, text="",icon='SOLO_OFF', invert_checkbox=False, emboss=True)
        row = col.row(align = True)  
        row.prop(collection["Fingers.L (Tweak)"], 'is_visible', toggle=True, text='(Tweak)')
        row.prop(collection["Fingers.L (Tweak)"], 'is_solo', toggle=True, text="",icon='SOLO_OFF', invert_checkbox=False, emboss=True)
        row.prop(collection["Finger.R (Tweak)"], 'is_visible', toggle=True, text='(Tweak)')
        row.prop(collection["Finger.R (Tweak)"], 'is_solo', toggle=True, text="",icon='SOLO_OFF', invert_checkbox=False, emboss=True)
        
        
        row = col.row(align = True)
        row = col.row(align = True)  
        row = col.row(align = True)  
        row.prop(collection["Leg.L (IK)"], 'is_visible', toggle=True, text='Leg.L (IK)')
        row.prop(collection["Leg.L (IK)"], 'is_solo', toggle=True, text="",icon='SOLO_OFF', invert_checkbox=False, emboss=True)
        row.prop(collection["Leg.R (IK)"], 'is_visible', toggle=True, text='Leg.R (IK)')
        row.prop(collection["Leg.R (IK)"], 'is_solo', toggle=True, text="",icon='SOLO_OFF', invert_checkbox=False, emboss=True)
        row = col.row(align = True)  
        row.prop(collection["Leg.L (FK)"], 'is_visible', toggle=True, text='(FK)')
        row.prop(collection["Leg.L (FK)"], 'is_solo', toggle=True, text="",icon='SOLO_OFF', invert_checkbox=False, emboss=True)
        row.prop(collection["Leg.R (FK)"], 'is_visible', toggle=True, text='(FK)')
        row.prop(collection["Leg.R (FK)"], 'is_solo', toggle=True, text="",icon='SOLO_OFF', invert_checkbox=False, emboss=True)
        row = col.row(align = True)  
        row.prop(collection["Leg.L (Tweak)"], 'is_visible', toggle=True, text='(Tweak)')
        row.prop(collection["Leg.L (Tweak)"], 'is_solo', toggle=True, text="",icon='SOLO_OFF', invert_checkbox=False, emboss=True)
        row.prop(collection["Leg.R (Tweak)"], 'is_visible', toggle=True, text='(Tweak)')
        row.prop(collection["Leg.R (Tweak)"], 'is_solo', toggle=True, text="",icon='SOLO_OFF', invert_checkbox=False, emboss=True)

        row = col.row(align = True)
        row = col.row(align = True)
        row = col.row(align = True)  
        row.prop(collection["Corrective"], 'is_visible', toggle=True, text='Corrective')
        row.prop(collection["Corrective"], 'is_solo', toggle=True, text="",icon='SOLO_OFF', invert_checkbox=False, emboss=True)
        
        row = col.row(align = True)
        row = col.row(align = True)
        row = col.row(align = True)  
        row.prop(collection["Props"], 'is_visible', toggle=True, text='Props Slot')
        row.prop(collection["Props"], 'is_solo', toggle=True, text="",icon='SOLO_OFF', invert_checkbox=False, emboss=True)
       
        row = col.row(align = True) 
        row = col.row(align = True)
        row = col.row(align = True) 
        row.prop(collection["Root"], 'is_visible', toggle=True, text='Root')
        row.prop(collection["Root"], 'is_solo', toggle=True, text="",icon='SOLO_OFF', invert_checkbox=False, emboss=True)
        row = col.row(align = True)  
        row.prop(collection["PROPERTIES"], 'is_visible', toggle=True, text='Properties')     
        
################### Visibility and modifiers ####################
### Custom properties panel 
### THIS IS ONLY THE PARENT PANEL FOR ALL THE SUB PANELS
class P2MBig_PT_customprops(bpy.types.Panel):
    bl_category = 'Item'
    bl_label = "P2MBig Character"
    bl_idname = "P2MBig_PT_customprops"
    bl_space_type = 'VIEW_3D'
    bl_region_type = 'UI'
    bl_options = {'DEFAULT_CLOSED'}

    @classmethod
    def poll(self, context):
        try:
            return (context.active_object.data.get("rig_id") == rig_id)
        except (AttributeError, KeyError, TypeError):
            return False
        
    def draw(self, context):
        layout = self.layout


######################################################################################


#Exposing modifiers
class P2MBig_PT_vis_props(bpy.types.Panel):
    bl_label = "Visibility Properties" 
    bl_idname = "P2MBig_PT_vis_props"
    bl_space_type = 'VIEW_3D'
    bl_parent_id = "P2MBig_PT_customprops"
    bl_region_type = 'UI'
    bl_options = {'DEFAULT_CLOSED'}
    
    def draw(self, context):
        layout = self.layout
        
                # Get the active object and ensure it's an armature
        armature = bpy.context.active_object

        my_object = None

        if armature and armature.type == 'ARMATURE':
            # Get direct children (not recursive)
            children = armature.children
            if children:
                # If there's exactly one child, or you want the first one
                my_object = children[1]
                print("Assigned child to my_object:", my_object.name)
            else:
                print("No direct child found under the active armature.")
        else:
            print("Active object is not an armature.")
               
        # character modifiers use modifier names (if you change modifier names it you will have to change these too). Here we have an edxample with mask modifiers.
        # creating a variable to source the modifier (name_of_the_variable = my_object.modifiers["exact_modifier_name"])
        mask_arms = my_object.modifiers["Mask_Arms"]
        mask_legs = my_object.modifiers["Mask_Legs"]
        mask_torso = my_object.modifiers["Mask_Torso"]
        mask_half_L = my_object.modifiers["Mask_Half.L"]
        mask_half_R = my_object.modifiers["Mask_Half.R"]
        
        
        # start panel layout
        layout = self.layout
        layout.use_property_split = False
        layout.use_property_decorate = False 
        
        # character modifiers
        # if I didn't break it down above the code would be: (kinda messy)
        # row.prop(bpy.data.objects['my_object'].modifiers["MASK-TORSO"], 'show_viewport', text="", toggle = True, icon='HIDE_ON', emboss=False)  
        # here we use Blender's eye/hide icon
        box = layout.box()
        col = box.column(align=True)
        row = col.row() 
        row.label(text='Torso', translate=False)   
        row.prop(mask_torso, 'show_viewport', text="", icon='HIDE_ON', invert_checkbox=True, emboss=False)
        
        row = col.row() 
        row.label(text='Arms', translate=False)   
        row.prop(mask_arms, 'show_viewport', text="", icon='HIDE_ON', invert_checkbox=True, emboss=False)
        
        row = col.row(align = True)  
        row.label(text='Legs', translate=False)             
        row.prop(mask_legs, 'show_viewport', text="", icon='HIDE_ON', invert_checkbox=True, emboss=False) 
        
        row = col.row(align = True)  
        row.label(text='Left', translate=False)             
        row.prop(mask_half_L, 'show_viewport', text="", icon='HIDE_ON', invert_checkbox=True, emboss=False)     
        
        row = col.row(align = True)  
        row.label(text='Right', translate=False)             
        row.prop(mask_half_R, 'show_viewport', text="", icon='HIDE_ON', invert_checkbox=True, emboss=False)  
        
   # EACH SUB PANEL IS A CLASS

class P2MBig_PT_render_props(bpy.types.Panel):
    bl_label = "Render Properties" 
    bl_idname = "P2MBig_PT_render_props"
    bl_space_type = 'VIEW_3D'
    bl_parent_id = "P2MBig_PT_customprops" # this is the same for all it points to the main panel name
    bl_region_type = 'UI'
    bl_options = {'DEFAULT_CLOSED'} # you may not want this? delete if not
    
    def draw(self, context):
        
        # get armature and "PROPERTIES" bone 
        arm = context.active_object
        bone = arm.pose.bones["PROPERTIES"]
        
        # basic layout setup
        layout = self.layout
        # just in we case want to change the split makes make a variable
        split_size = 0.7
        
        # then we start making rows, the first one is in a box      
        box = layout.box()
        col = box.column(align=True)
        row = col.row() 			 
        split = row.split(align=True, factor=split_size)
        row = split.row(align=True) 	   
        # this is just the label you could call it anything
        row.label(text='P2design_logo', translate=False)
        row = split.row(align=True)
        # this is the property > format is important
        row.prop(bone, '["P2design_logo"]', text = "", slider=True) 
        
        
        
        row = col.row()              
        split = row.split(align=True, factor=split_size)
        row = split.row(align=True)        
        # this is just the label you could call it anything
        row.label(text='Smooth', translate=False)
        row = split.row(align=True)
        # this is the property > format is important
        row.prop(bone, '["Smooth"]', text = "")  

        row = col.row()              
        split = row.split(align=True, factor=split_size)
        row = split.row(align=True)        
        # this is just the label you could call it anything
        row.label(text='Subdiv', translate=False)
        row = split.row(align=True)
        # this is the property > format is important
        row.prop(bone, '["Subdiv"]', text = "")

        row = col.row()              
        split = row.split(align=True, factor=split_size)
        row = split.row(align=True)        
        # this is just the label you could call it anything
        row.label(text='color_1_RGB', translate=False)
        row = split.row(align=True)
        # this is the property > format is important
        row.prop(bone, '["color_1_RGB"]', text = "")

        row = col.row()              
        split = row.split(align=True, factor=split_size)
        row = split.row(align=True)        
        # this is just the label you could call it anything
        row.label(text='color_2_RGB', translate=False)
        row = split.row(align=True)
        # this is the property > format is important
        row.prop(bone, '["color_2_RGB"]', text = "")



        
# -----------------------------------------------------------
# Registration
classes = (
    RigBakeSettings,
    P2M_Big_RigLayers,
    RigUI,
    POSE_OT_finger_index_L_ik_to_fk,
    POSE_OT_finger_index_L_fk_to_ik,
    POSE_OT_finger_index_R_ik_to_fk,
    POSE_OT_finger_index_R_fk_to_ik,
    POSE_OT_finger_middle_L_ik_to_fk,
    POSE_OT_finger_middle_L_fk_to_ik,
    POSE_OT_finger_middle_R_ik_to_fk,
    POSE_OT_finger_middle_R_fk_to_ik,
    POSE_OT_finger_ring_L_ik_to_fk,
    POSE_OT_finger_ring_L_fk_to_ik,
    POSE_OT_finger_ring_R_ik_to_fk,
    POSE_OT_finger_ring_R_fk_to_ik,
    POSE_OT_finger_pinky_L_ik_to_fk,
    POSE_OT_finger_pinky_L_fk_to_ik,
    POSE_OT_finger_pinky_R_ik_to_fk,
    POSE_OT_finger_pinky_R_fk_to_ik,
    POSE_OT_finger_thumb_L_ik_to_fk,
    POSE_OT_finger_thumb_L_fk_to_ik,
    POSE_OT_finger_thumb_R_ik_to_fk,
    POSE_OT_finger_thumb_R_fk_to_ik,
    RIGIFY_OT_get_frame_range,
    POSE_OT_rigify_generic_snap,
    POSE_OT_rigify_generic_snap_bake,
    POSE_OT_rigify_clear_keyframes,
    POSE_OT_rigify_limb_ik2fk,
    POSE_OT_rigify_limb_ik2fk_bake,
    POSE_OT_rigify_limb_toggle_pole,
    POSE_OT_rigify_limb_toggle_pole_bake,
    POSE_OT_rigify_leg_roll_ik2fk,
    POSE_OT_rigify_leg_roll_ik2fk_bake,
    POSE_OT_rigify_switch_parent,
    POSE_OT_rigify_switch_parent_bake,
    P2MBig_PT_customprops,
    P2MBig_PT_vis_props,
    P2MBig_PT_render_props,
)

def register():
    for cls in classes:
        bpy.utils.register_class(cls)

def unregister():
    for cls in reversed(classes):
        bpy.utils.unregister_class(cls)

if __name__ == "__main__":
    register()
