init -10 python:
    import os
    import platform
    import subprocess

    # Centralized UI/display constants
    PERSONA_PREVIEW_MAX_CHARS = 25

    class PersonaManager:
        """
        Manages up to six user-defined persona slots and the active selection.
        Stores data in Ren'Py persistent storage and allows import/export to text files.
        """
        NEW_DIR = "persona"
        LEGACY_DIR = "personna"  # fallback for existing files
        NUM_SLOTS = 6

        @classmethod
        def _ensure_initialized(cls):
            if not hasattr(persistent, "persona_slots") or not isinstance(persistent.persona_slots, list):
                # Migrate legacy persistent if present
                legacy = getattr(persistent, "personna_slots", None)
                if isinstance(legacy, list):
                    persistent.persona_slots = legacy[:cls.NUM_SLOTS] + [""] * max(0, cls.NUM_SLOTS - len(legacy))
                else:
                    persistent.persona_slots = [""] * cls.NUM_SLOTS
            # Ensure correct size
            if len(persistent.persona_slots) != cls.NUM_SLOTS:
                current = list(persistent.persona_slots)
                persistent.persona_slots = [""] * cls.NUM_SLOTS
                for i in range(min(len(current), cls.NUM_SLOTS)):
                    persistent.persona_slots[i] = current[i] or ""
            # Active index
            if not hasattr(persistent, "active_persona_index") or not isinstance(persistent.active_persona_index, int):
                # migrate legacy index
                legacy_idx = getattr(persistent, "active_personna_index", -1)
                persistent.active_persona_index = legacy_idx if isinstance(legacy_idx, int) else -1

        @classmethod
        def get_slots(cls):
            cls._ensure_initialized()
            return list(persistent.persona_slots)

        @classmethod
        def set_slot(cls, index, text):
            cls._ensure_initialized()
            if 0 <= index < cls.NUM_SLOTS:
                persistent.persona_slots[index] = text or ""

        @classmethod
        def clear_slot(cls, index):
            cls.set_slot(index, "")

        @classmethod
        def get_active_index(cls):
            cls._ensure_initialized()
            return persistent.active_persona_index

        @classmethod
        def set_active_index(cls, index):
            cls._ensure_initialized()
            if 0 <= index < cls.NUM_SLOTS:
                persistent.active_persona_index = index
            else:
                persistent.active_persona_index = -1

        @classmethod
        def get_active_persona_text(cls):
            cls._ensure_initialized()
            idx = persistent.active_persona_index
            if 0 <= idx < cls.NUM_SLOTS:
                text = persistent.persona_slots[idx] or ""
                return text.strip()
            return ""

        @classmethod
        def _get_persona_dir(cls):
            base_game = os.path.join(config.basedir, "game")
            new_path = os.path.join(base_game, cls.NEW_DIR)
            legacy_path = os.path.join(base_game, cls.LEGACY_DIR)
            try:
                if os.path.exists(legacy_path) and not os.path.exists(new_path):
                    # If legacy exists but new doesn't, use legacy to avoid breaking existing files
                    os.makedirs(legacy_path, exist_ok=True)
                    return legacy_path
                os.makedirs(new_path, exist_ok=True)
            except Exception:
                pass
            return new_path if os.path.exists(new_path) else legacy_path

        @classmethod
        def export_slot(cls, index):
            cls._ensure_initialized()
            if not (0 <= index < cls.NUM_SLOTS):
                renpy.notify("Invalid slot index")
                return False
            folder = cls._get_persona_dir()
            filename = os.path.join(folder, f"slot_{index+1}.txt")
            try:
                with open(filename, "w", encoding="utf-8") as f:
                    f.write(persistent.persona_slots[index] or "")
                renpy.notify(f"Exported to {os.path.basename(filename)}")
                return True
            except Exception as e:
                renpy.notify(f"Export failed: {e}")
                return False

        @classmethod
        def import_slot(cls, index):
            cls._ensure_initialized()
            if not (0 <= index < cls.NUM_SLOTS):
                renpy.notify("Invalid slot index")
                return False
            folder = cls._get_persona_dir()
            filename = os.path.join(folder, f"slot_{index+1}.txt")
            if not os.path.exists(filename):
                renpy.notify(f"File not found: slot_{index+1}.txt")
                return False
            try:
                with open(filename, "r", encoding="utf-8") as f:
                    text = f.read()
                persistent.persona_slots[index] = text
                renpy.notify(f"Imported from {os.path.basename(filename)}")
                return True
            except Exception as e:
                renpy.notify(f"Import failed: {e}")
                return False

        @classmethod
        def open_persona_folder(cls):
            try:
                path = cls._get_persona_dir()
                if platform.system() == "Windows":
                    os.startfile(path)
                elif platform.system() == "Darwin":
                    subprocess.Popen(["open", path])
                else:
                    subprocess.Popen(["xdg-open", path])
                renpy.notify("Opening persona folder...")
            except Exception as e:
                renpy.notify(f"Failed to open folder: {e}")

        @classmethod
        def _escape_renpy_text(cls, text):
            """Escape Ren'Py text tag characters so raw text displays safely."""
            return text.replace("{", "{{").replace("[", "[[")

        @classmethod
        def get_slot_previews(cls, max_chars=None):
            """Return a list of short one-line previews for each slot."""
            if max_chars is None:
                max_chars = PERSONA_PREVIEW_MAX_CHARS
            slots = cls.get_slots()
            previews = []
            for s in slots:
                if not s:
                    previews.append("")
                    continue
                line = s.strip().splitlines()[0] if s.strip() else ""
                if len(line) > max_chars:
                    line = line[:max_chars] + "…"
                previews.append(cls._escape_renpy_text(line))
            return previews

# Convenience functions for screen actions
init python:
    def set_persona_slot_text(slot_index, new_text):
        PersonaManager.set_slot(slot_index, new_text)
        return None

    def set_active_persona(slot_index):
        PersonaManager.set_active_index(slot_index)
        renpy.notify(f"Active persona set to Slot {slot_index+1}")
        return None

    def clear_persona_slot(slot_index):
        PersonaManager.clear_slot(slot_index)
        renpy.notify(f"Cleared Slot {slot_index+1}")
        return None

    def export_persona_slot(slot_index):
        PersonaManager.export_slot(slot_index)
        return None

    def import_persona_slot(slot_index):
        PersonaManager.import_slot(slot_index)
        return None

    def open_persona_folder():
        PersonaManager.open_persona_folder()
        return None

    def persona_debounced_save():
        try:
            idx = renpy.get_screen_variable("persona_selected_idx")
            text = renpy.get_screen_variable("persona_editor_text") or ""
            from store import PersonaManager as _PM
            slots = _PM.get_slots()
            if isinstance(idx, int) and 0 <= idx < len(slots):
                if slots[idx] != text:
                    _PM.set_slot(idx, text)
                    # Refresh screen previews/slots only when content actually changed
                    renpy.set_screen_variable("slots", _PM.get_slots())
                    renpy.set_screen_variable("previews", _PM.get_slot_previews())
        except Exception:
            pass
        return None

    def refresh_persona_after_import(slot_index):
        try:
            from store import PersonaManager as _PM
            slots = _PM.get_slots()
            if isinstance(slot_index, int) and 0 <= slot_index < len(slots):
                renpy.set_screen_variable("persona_editor_text", slots[slot_index])
            renpy.set_screen_variable("slots", slots)
            renpy.set_screen_variable("previews", _PM.get_slot_previews())
        except Exception:
            pass
        return None 