# conversation.rpy
# -------------------------------------------------------------------------
# Defines two main AI-driven labels:
#   1) ai_conversation(characters, custom_system_prompt)
#   2) generate_ai_scene(characters, location_context, scene_prompt)
#
# These labels orchestrate the flow for AI interactions, relying on
# components defined in ai_*.rpy files for specific logic like
# validation, sprite handling, prompt generation, and background tasks.
# -------------------------------------------------------------------------

# Default variables for interrupt feature
default current_scene_queue = []  # Will be converted to SceneQueue at runtime
default interrupt_requested = False
default scene_interrupted = False
default scene_was_interrupted = False
default in_ai_conversation = False
default rollback_debug = False

# Default variable to store the last typed input
default last_typed_input = ""

# Optional: Allow advancing time during Q&A (UI-controlled)
default enable_qa_time_advance = False
default time_advance_requested = False

# Default variables for save/load sprite positioning fix
# These persist across saves and allow us to restore character positions correctly
default current_conversation_characters = []  # Store characters for save/load in Q&A
default current_scene_characters = []  # Store scene characters for save/load in scenes

# Default variables for location switching during Q&A
default qa_current_location = None  # Current location during Q&A session
default qa_location_history = []  # Stack of locations for undo/regenerate
default pre_question_location = None  # Location before user's question
default dynamic_location_changed = False  # Flag to track if location was changed dynamically
default last_dynamic_location_time = 0  # Timestamp of last dynamic location change
default initial_prompt_location = None  # Location when system prompt was built
default qa_session_start_location = None  # Location where Q&A session began (for sleepover eligibility)
default pending_roster_changes = None

init -10 python:
    import re
    if not hasattr(store, '_norm'):
        def _norm(s):
            return re.sub(r"[^a-z0-9]", "", s.lower())
        store._norm = _norm

init python:
    from store import PromptManager, parse_gpt_lines_to_list, SpriteManager
    import re
    import threading

    # Rollback debug helper (enable with: $ rollback_debug = True in console)
    def rb_log(*parts):
        try:
            if config.developer or getattr(store, 'rollback_debug', False):
                print("[RB]", *parts)
        except Exception:
            pass

    # Component classes (Validator, SpriteManager, DialogueProcessor, PromptManager)
    # and helper functions are now defined in separate ai_*.rpy files.

    def _restore_sprite_state(char_config_map, positions_map, shown_characters, sprite_state_to_restore):
        """Clears current sprites and restores them to a specific state without logging."""
        SpriteManager.clear_all_sprites()
        shown_characters.clear()
        for tag, emotion in sprite_state_to_restore.items():
            if tag in char_config_map:
                sprite_line = f"[sprite: {tag} {emotion}]"
                SpriteManager.apply_sprite_change(
                    sprite_line, char_config_map, positions_map,
                    shown_characters, list(char_config_map.values()),
                    log_to_history=False # Explicitly disable history logging
                )
        renpy.with_statement(dissolve)

    def _query_ai_and_process_response(system_prompt, dialogue_history, user_question, post_story_instructions, char_config_map):
        """Handles the background AI query, validation, and processing."""
        # Prepare allowed characters for validation
        allowed_characters = set(char_config_map.keys()) | {"narrator"}

        # Launch background task for AI query
        global async_conversation_response
        async_conversation_response = None
        renpy.invoke_in_thread(conversation_query_background, system_prompt, dialogue_history, user_question, post_story_instructions, allowed_characters)

        # Show loading indicator and wait for response
        renpy.show_screen("conversation_loading_indicator")
        while async_conversation_response is None:
            renpy.pause(0.05)
        renpy.hide_screen("conversation_loading_indicator")

        response = async_conversation_response

        # Validate and potentially correct the AI response
        validation_errors = AIOutputValidator.validate_ai_output(response, allowed_characters)
        if validation_errors:
            corrected_response = AIOutputValidator.get_corrected_output(
                system_prompt, dialogue_history, response, validation_errors
            )
            # Use corrected response if it's not an error message
            if "Error:" not in corrected_response:
                response = corrected_response

        return parse_gpt_lines_to_list(response)


# === CHARACTER CONVERSATION SYSTEM ===
label ai_conversation(characters=None, custom_system_prompt=None, interrupt_context=None):
    # Always clear scene queue before starting a conversation
    $ current_scene_queue = []
    $ in_ai_conversation = True
    # Initialize conversation state variables
    $ show_ai_quit_button = False
    $ show_regenerate_button = False
    $ regenerate_requested = False
    $ undo_requested = False
    $ last_user_question = ""
    $ last_ai_response_start_index = -1
    $ last_ai_response_line_count = 0
    $ last_question_index = -1 # Track where the user question starts in history

    python:
        # Verify and normalize characters list.
        if not characters:
            raise Exception("No characters provided to ai_conversation.")
        if not isinstance(characters, list):
            characters = [characters]
            
        # Store characters in default variable for save/load
        store.current_conversation_characters = characters

        # Build a mapping from character tag to configuration.
        char_config_map = {c.tag: c for c in characters}

        # State tracking for undo/regenerate
        global last_sprite_state # Relies on global sprite state
        pre_response_sprite_state = {} # Sprite state before AI response
        pre_question_sprite_state = {} # Sprite state before user asked question
        pre_response_characters = [] # Characters in conversation before AI response
        pre_response_shown_characters = set() # Shown characters before AI response
        
        # Initialize location tracking for Q&A session
        # Check if a dynamic location change happened recently (within last 60 seconds)
        dynamic_change_recent = False
        if hasattr(store, 'dynamic_location_changed') and store.dynamic_location_changed:
            if hasattr(store, 'last_dynamic_location_time'):
                time_since_change = renpy.time.time() - store.last_dynamic_location_time
                if time_since_change < 60:  # Within last minute
                    dynamic_change_recent = True
                    if config.developer:
                        print(f"[Q&A Start] Dynamic location change detected {time_since_change:.1f}s ago")
        
        # Smart sync: Don't overwrite qa_current_location if it was recently changed dynamically
        if not dynamic_change_recent:
            # Normal sync from current_location_id if no recent dynamic changes
            if 'current_location_id' in globals() and current_location_id:
                qa_current_location = current_location_id
                if config.developer:
                    print(f"[Q&A Start] Syncing qa_current_location to: {qa_current_location} (from current_location_id: {current_location_id})")
            elif qa_current_location is None:
                # Fallback if current_location_id is not set
                qa_current_location = "unknown_location"
                if config.developer:
                    print(f"[Q&A Start] Warning: No current_location_id found, qa_current_location set to: {qa_current_location}")
        else:
            # Keep the dynamically changed location
            if config.developer:
                print(f"[Q&A Start] Keeping dynamically changed location: {qa_current_location}")
                print(f"  current_location_id: {current_location_id if 'current_location_id' in globals() else 'not set'}")
                if qa_current_location != current_location_id:
                    print(f"  WARNING: Variables out of sync! This shouldn't happen after our fix.")
        
        # Clear the dynamic change flag after processing
        store.dynamic_location_changed = False
        
        # Store the location where Q&A session started (for sleepover eligibility)
        store.qa_session_start_location = qa_current_location
        
        qa_location_history = []
        
        # Initialize shown_characters based on current sprite state
        # This preserves which characters are visible from scene generation
        shown_characters = set()
        if 'last_sprite_state' in globals() and last_sprite_state:
            # Populate shown_characters with tags that have sprites currently shown
            for tag in last_sprite_state:
                if tag in char_config_map:
                    shown_characters.add(tag)
        
        last_question_index = -1

        # Import DialogueProcessor now that all modules should be initialized
        from store import DialogueProcessor

        # Get positions map for characters
        positions_map = SpriteManager.get_positions_map(characters)

        # Generate the system prompt for the conversation using PromptManager directly
        prompt_result = PromptManager.build_conversation_prompt(characters, custom_system_prompt, interrupt_context)
        
        # Handle both old format (string) and new format (tuple)
        if isinstance(prompt_result, tuple):
            system_prompt, post_story_instructions = prompt_result
        else:
            # Backward compatibility
            system_prompt = prompt_result
            post_story_instructions = ""
        
        # Remember the location when prompt was built
        initial_prompt_location = qa_current_location
        if config.developer:
            print(f"[Q&A] System prompt built with location: {initial_prompt_location}")

    # Jump to the conversation loop label
    jump ai_conversation_loop

# Main conversation loop as a label
label ai_conversation_loop:
    python:
        # Determine character management mode
        character_mode = getattr(store, 'CHARACTER_MANAGEMENT_MODE', 'off')
        sandbox_mode = getattr(store, 'sandbox_mode', False)
        
        # Show appropriate UI buttons based on mode
        show_ai_quit_button = True
        
        # Show roster button only in manual mode and sandbox
        if character_mode == "manual" and sandbox_mode:
            show_roster_button = True
        else:
            show_roster_button = False
        
        # Get location management mode
        location_mode = getattr(store, 'LOCATION_MANAGEMENT_MODE', 'off')
        
        # Show location button only in manual mode and sandbox
        if location_mode == "manual" and sandbox_mode:
            show_location_button = True
        else:
            show_location_button = False
        
        # Get user input, handling potential modal interference
        user_input = renpy.input(_("Ask something:"), length=500)
        
        # Ensure user_input is always a string
        if not isinstance(user_input, str):
            user_input = "" if (user_input is False or user_input is None) else str(user_input)
        
        # Hide UI buttons after input
        show_ai_quit_button = False
        show_roster_button = False
        show_location_button = False
        show_regenerate_button = False

        # Handle regenerate or undo actions, or process new input
        if regenerate_requested:
            regenerate_requested = False
            # Remove previous AI response from history
            remove_scene_from_history(last_ai_response_start_index, last_ai_response_line_count)
            # Restore location first if it changed (without clearing sprites)
            if pre_question_location and qa_current_location != pre_question_location:
                from store import LocationManager
                LocationManager.restore_location_for_qa(pre_question_location)
                qa_current_location = pre_question_location
                # Also restore current_location_id to keep them in sync
                current_location_id = pre_question_location
            
            # Restore full character state (not just sprite state)
            # First restore the character list and shown characters
            characters[:] = pre_response_characters
            shown_characters.clear()
            shown_characters.update(pre_response_shown_characters)
            char_config_map.clear()
            char_config_map.update({c.tag: c for c in characters})
            
            # Recalculate positions for the restored character set
            positions_map = SpriteManager.get_positions_map(characters)
            
            # Then restore sprite state with all characters properly positioned
            _restore_sprite_state(char_config_map, positions_map, shown_characters, pre_response_sprite_state)
            user_question = last_user_question

        elif undo_requested:
            undo_requested = False
            # Remove last Q&A from history
            if last_question_index >= 0:
                remove_scene_from_history(last_question_index, 1 + last_ai_response_line_count)
            # Restore location first (without clearing sprites)
            if qa_location_history:
                previous_location = qa_location_history.pop()
                if qa_current_location != previous_location:
                    from store import LocationManager
                    LocationManager.restore_location_for_qa(previous_location)
                    qa_current_location = previous_location
                    # Also restore current_location_id to keep them in sync
                    current_location_id = previous_location
            # Then restore sprite state
            _restore_sprite_state(char_config_map, positions_map, shown_characters, pre_question_sprite_state)
            # Reset and loop back to input
            last_question_index = -1
            last_ai_response_start_index = -1
            last_ai_response_line_count = 0
            # Jump back to get new input
            renpy.jump("ai_conversation_loop")

        elif sleepover_requested:
            sleepover_requested = False
            # Reset error recovery flag if user is retrying after an error
            sleepover_error_recovery = False
            # Handle sleepover request - generate AI-initiated sleepover suggestion
            # Use a minimal user question to avoid API errors
            user_question = "..."  # Minimal prompt that APIs will accept
            
            # Treat this like a question for location restore semantics so that
            # regenerate doesn't bounce back to a pre-move location.
            # Capture the current QA location as the pre-question location and
            # push it onto the location history, mirroring the normal question path.
            pre_question_location = qa_current_location
            qa_location_history.append(qa_current_location)
            # Optionally snapshot sprite state for undo parity with normal questions
            try:
                pre_question_sprite_state = last_sprite_state.copy()
            except Exception:
                pass
            
            # Get the sleepover prompt and use it as the system instruction
            sleepover_system_prompt = SleepoversSystem.get_sleepover_prompt(current_location_id, current_conversation_characters)
            system_prompt = sleepover_system_prompt  # Override system prompt
            
            # Mark that a sleepover is planned, but set a flag to indicate it's tentative
            # (in case there's an error and we need to reset)
            sleepover_planned = True
            sleepover_prompt_shown = True
            last_user_question = user_question

        elif time_advance_requested:
            time_advance_requested = False
            # Handle time advancement during conversation (morning -> afternoon)
            try:
                from store import time_system
                if hasattr(store, 'enable_qa_time_advance') and not store.enable_qa_time_advance:
                    pass
                else:
                    changed = time_system.advance_time_slot_to_afternoon()
                    # Region-aware narration
                    region_name = getattr(store, 'current_map_region', 'Ponyville')
                    if changed:
                        # Consume half-day like map 'Pass Time' does
                        try:
                            store.daily_visits_count += 1
                        except Exception:
                            pass
                        # Reset sleepover flags similar to end-of-visit cleanup
                        try:
                            if hasattr(store, 'sleepover_planned') and store.sleepover_planned:
                                store.sleepover_planned = False
                            if hasattr(store, 'sleepover_requested') and store.sleepover_requested:
                                store.sleepover_requested = False
                            if hasattr(store, 'sleepover_morning') and store.sleepover_morning:
                                store.sleepover_morning = False
                            if hasattr(store, 'sleepover_prompt_shown') and store.sleepover_prompt_shown:
                                store.sleepover_prompt_shown = False
                        except Exception:
                            pass
                        renpy.say(narrator, f"Time passes... It's now afternoon in {region_name}.")
                    else:
                        renpy.say(narrator, f"It's already afternoon in {region_name}.")
            except Exception:
                # Fail-soft and continue the conversation loop
                pass
            # Loop back for normal user input
            renpy.jump("ai_conversation_loop")

        else:
            # Normal user question:
            # Handle case where user_input might be False (cancelled input)
            if not isinstance(user_input, str):
                if user_input is False or user_input is None:
                    user_input = ""
                else:
                    user_input = str(user_input)
            
            user_input = user_input.strip()
            
            # Handle empty input - loop back for new input
            if not user_input:
                renpy.jump("ai_conversation_loop")
            
            if user_input.lower() == "quit":
                renpy.jump("end_ai_conversation")

            # Store the last typed input
            store.last_typed_input = user_input

            # Store sprite state before processing this question
            pre_question_sprite_state = last_sprite_state.copy()
            last_question_index = len(dialogue_history)
            
            # If sleepover invite already shown, reset back to normal conversation prompt
            if sleepover_planned and 'sleepover_prompt_shown' in globals() and sleepover_prompt_shown:
                prompt_result = PromptManager.build_conversation_prompt(characters, custom_system_prompt, interrupt_context)
                if isinstance(prompt_result, tuple):
                    system_prompt, post_story_instructions = prompt_result
                else:
                    system_prompt = prompt_result
                    post_story_instructions = ""
                
                # Update initial prompt location when rebuilding prompt
                initial_prompt_location = qa_current_location
                if config.developer:
                    print(f"[Q&A] Prompt rebuilt with location: {initial_prompt_location}")

            # Store location state before processing this question
            pre_question_location = qa_current_location
            qa_location_history.append(qa_current_location)

            # Check if this is a duplicate of the last player input
            formatted_input = user_input.replace('"', "\"").replace("'", "\'")
            new_entry = f'p "{formatted_input}"'
            
            # Check if the last entry in dialogue history is the same player input
            is_duplicate = False
            if dialogue_history:
                # Look for the last player entry (starts with 'p "')
                for i in range(len(dialogue_history) - 1, -1, -1):
                    if dialogue_history[i].startswith('p "'):
                        if dialogue_history[i] == new_entry:
                            is_duplicate = True
                        break  # Stop at the first player entry found
            
            # Only add to history if it's not a duplicate
            if not is_duplicate:
                dialogue_history.append(new_entry)
            else:
                # Still update the index for proper tracking
                last_question_index = len(dialogue_history) - 1
                
            user_question = user_input
            last_user_question = user_question

        # Track AI response start index in history
        last_ai_response_start_index = len(dialogue_history)

        # In dynamic mode, rebuild the system prompt each user turn using the CURRENT roster
        # so the header reflects characters who entered/exited during the last AI response.
        if character_mode == "dynamic" and sandbox_mode:
            try:
                # Sync characters from store (active roster maintained by dynamic manager)
                if hasattr(store, 'current_conversation_characters') and store.current_conversation_characters:
                    characters = list(store.current_conversation_characters)
                    char_config_map = {c.tag: c for c in characters}
                    positions_map = SpriteManager.get_positions_map(characters)

                    # Prefer stored prompt params if available, else fall back to locals
                    prompt_custom = getattr(store, 'qa_custom_system_prompt', None)
                    prompt_interrupt = getattr(store, 'qa_interrupt_context', None)
                    if prompt_custom is None:
                        prompt_custom = custom_system_prompt if 'custom_system_prompt' in locals() else None
                    if prompt_interrupt is None:
                        prompt_interrupt = interrupt_context if 'interrupt_context' in locals() else None

                    prompt_result = PromptManager.build_conversation_prompt(characters, prompt_custom, prompt_interrupt)
                    if isinstance(prompt_result, tuple):
                        system_prompt, post_story_instructions = prompt_result
                    else:
                        system_prompt = prompt_result
                        post_story_instructions = ""
            except Exception as _e:
                if config.developer:
                    print(f"[Q&A Dynamic] Prompt rebuild skipped due to: {_e}")

        # Backup full character state before generating the response
        pre_response_sprite_state = last_sprite_state.copy()
        pre_response_characters = list(characters)  # Save all characters in conversation
        pre_response_shown_characters = shown_characters.copy()  # Save which ones were shown
        
        # Handle manual roster changes if in manual mode
        character_mode = getattr(store, 'CHARACTER_MANAGEMENT_MODE', 'off')
        sandbox_mode = getattr(store, 'sandbox_mode', False)
        
        if character_mode == "manual" and sandbox_mode:
            from store import ManualCharacterManager
            
            # Check for pending roster changes from the modal
            if hasattr(store, 'pending_roster_changes') and store.pending_roster_changes:
                changes = store.pending_roster_changes
                selected_tags = changes['selected_tags']
                arrivals = changes['arrivals']
                departures = changes['departures']
                
                # Build new roster from selected tags
                all_chars = ManualCharacterManager.get_available_characters()
                new_roster = [c for c in all_chars if c.tag in selected_tags]
                
                # Apply the roster changes
                ManualCharacterManager.set_roster(new_roster)
                
                # Generate instructions for AI
                instructions = []
                if arrivals:
                    arrival_names = [c.name for c in new_roster if c.tag in arrivals]
                    if arrival_names:
                        instructions.append(f"The following characters have just joined the conversation: {', '.join(arrival_names)}.")
                
                if departures:
                    # Get names of departed characters
                    departure_names = []
                    for tag in departures:
                        for c in all_chars:
                            if c.tag == tag:
                                departure_names.append(c.name)
                                break
                    if departure_names:
                        instructions.append(f"The following characters have just left the conversation: {', '.join(departure_names)}.")
                
                # Build roster instructions (one-shot for this turn)
                roster_instructions = "\n".join(instructions) if instructions else ""

                # Update character list and maps BEFORE rebuilding prompt
                characters = new_roster
                char_config_map = {c.tag: c for c in characters}
                positions_map = SpriteManager.get_positions_map(characters)

                # Rebuild the system prompt so the header reflects the updated roster
                prompt_result = PromptManager.build_conversation_prompt(characters, custom_system_prompt, interrupt_context)
                if isinstance(prompt_result, tuple):
                    system_prompt, base_post_story = prompt_result
                else:
                    system_prompt = prompt_result
                    base_post_story = ""

                # Recompose post-story instructions freshly for this turn
                if roster_instructions:
                    post_story_instructions = (base_post_story + "\n\n" + roster_instructions) if base_post_story else roster_instructions
                else:
                    post_story_instructions = base_post_story

                # Clear pending changes
                store.pending_roster_changes = None
        
        # Check for location changes based on mode
        location_mode = getattr(store, 'LOCATION_MANAGEMENT_MODE', 'off')
        
        if location_mode == "manual":
            # In manual mode, check for manual location changes
            from store import ManualLocationManager
            location_instructions = ManualLocationManager.generate_location_instructions()
            if location_instructions:
                if post_story_instructions:
                    post_story_instructions = post_story_instructions + "\n\n" + location_instructions
                else:
                    post_story_instructions = location_instructions
        # Dynamic location instructions removed - already handled elsewhere in the prompt
        
        # No need to inject location - AI will infer from context
        # We still track location internally for game mechanics
        if config.developer and qa_current_location != initial_prompt_location:
            print(f"[Q&A] Location has changed from {initial_prompt_location} to {qa_current_location}")
            print(f"[Q&A] AI will infer location from conversation context")

        # Query AI and get processed lines with error handling for sleepover requests
        try:
            output_lines = _query_ai_and_process_response(system_prompt, dialogue_history, user_question, post_story_instructions, char_config_map)
            # Detect error responses (narrator lines starting with (Error...))
            is_error_response = False
            if output_lines and len(output_lines) > 0 and isinstance(output_lines[0], str):
                first_line = output_lines[0].strip()
                if first_line.startswith('narrator "(') and ("Error" in first_line or "error" in first_line.lower()):
                    is_error_response = True
            if is_error_response and sleepover_planned and user_question == "...":
                sleepover_planned = False
                sleepover_error_recovery = True
                sleepover_prompt_shown = False
        except Exception as e:
            if sleepover_planned and user_question == "...":
                sleepover_planned = False
                sleepover_error_recovery = True
                sleepover_prompt_shown = False
            raise e

        # Fill the queue with the response lines
        current_scene_queue = SceneQueue(output_lines)
        
        # Store these in globals for the play_qa_response label
        scene_characters = characters
        
    # Use the new label-based processing to support save/load
    call play_qa_response from _call_play_qa_response
    
    python:
        # Check if we were interrupted
        if qa_response_interrupted:
            # Response was interrupted, continue to next input
            qa_response_interrupted = False
            show_regenerate_button = False
        else:
            # Response completed normally
            # Record how many lines the AI added to history (for regenerate/undo)
            last_ai_response_line_count = len(dialogue_history) - last_ai_response_start_index
            
            # Enable regenerate/undo buttons for the next input cycle
            show_regenerate_button = True
    
    # Continue the conversation loop
    jump ai_conversation_loop

# New label for processing Q&A responses with proper save/load support
label play_qa_response:
    python:
        # Initialize tracking variables if not already set
        if 'qa_response_interrupted' not in globals():
            qa_response_interrupted = False
        if 'qa_lines_processed' not in globals():
            qa_lines_processed = 0
            
        # Get necessary variables from parent context
        # Use the saved conversation characters
        if hasattr(store, 'current_conversation_characters') and store.current_conversation_characters:
            qa_characters = store.current_conversation_characters
        elif 'scene_characters' in globals() and scene_characters:
            qa_characters = scene_characters
        elif 'characters' in locals():
            qa_characters = characters
        else:
            # This shouldn't happen
            qa_characters = []
            
        # Rebuild char_config_map from characters
        qa_char_config_map = {c.tag: c for c in qa_characters} if qa_characters else {}
        qa_shown_characters = shown_characters if 'shown_characters' in locals() else set()
        
        # Always recalculate positions to ensure correct positioning after load
        qa_positions_map = SpriteManager.get_positions_map(qa_characters)
        
        # Fix sprite positions if we're resuming from a save
        fix_sprite_positions_after_load(qa_characters, qa_char_config_map, qa_positions_map, qa_shown_characters)
        
    # Main Q&A response playback loop
    while True:
        python:
            # Check for interrupts first
            if interrupt_requested:
                qa_response_interrupted = True
                interrupt_requested = False
                # Clear remaining lines
                remaining_lines = len(current_scene_queue) if current_scene_queue else 0
                current_scene_queue = []
                # Jump out of this label
                renpy.jump("qa_response_interrupted")
                
            # Check if we have more lines to process
            if not current_scene_queue or len(current_scene_queue) == 0:
                # All lines processed successfully
                renpy.jump("qa_response_complete")
                
            upcoming_line = current_scene_queue[0] if current_scene_queue else None
            rb_log("Q/A upcoming:", (upcoming_line[:80] if upcoming_line else 'None'), "| can_rb=", renpy.can_rollback(), "| hist=", (len(dialogue_history) if 'dialogue_history' in globals() else 'N/A'), "| queue=", (len(current_scene_queue) if current_scene_queue else 0))

            # Get next line from queue
            next_line = current_scene_queue.pop(0) if current_scene_queue else None
            
            if not next_line:
                renpy.jump("qa_response_complete")
                
            # Track that we're processing a line
            qa_lines_processed += 1
            
        # Process the line (outside of Python block for proper save support)
        python:
            line = next_line.strip()
            
            # Process sprite changes
            if line.startswith("[sprite:"):
                # Normalize name-based sprite commands like "[sprite: FancyPants happy]"
                # to use the character tag if the token matches a current character's name.
                try:
                    import re
                    inner = line[len("[sprite:"):-1].strip()  # content between [sprite: ...]
                    parts = inner.split()
                    if parts:
                        token = parts[0]
                        # If token isn't a known tag, try resolving by display name
                        if token not in qa_char_config_map:
                            from store import _norm as _normalize_name
                            for _c in qa_characters:
                                if _normalize_name(token) == _normalize_name(_c.name):
                                    parts[0] = _c.tag
                                    inner = " ".join(parts)
                                    line = f"[sprite: {inner}]"
                                    break
                except Exception:
                    pass

                # Do not anchor rollback at sprite-only steps; let Back skip them
                applied = SpriteManager.apply_sprite_change(
                    line, qa_char_config_map, qa_positions_map, 
                    qa_shown_characters, qa_characters
                )
                rb_log("Q/A applied SPRITE:", line, "| can_rb=", renpy.can_rollback())
                # Sprite change - no dialogue to show
                speaker_obj = None
                text_to_say = ""
            elif line.startswith("[location:") or line.startswith("[location_change:"):
                # Process location change command
                from store import DialogueProcessor
                DialogueProcessor.process_location_change(line)
                rb_log("Q/A applied LOCATION:", line, "| can_rb=", renpy.can_rollback())
                # Location change - no dialogue to show
                speaker_obj = None
                text_to_say = ""
            elif line.startswith("[character_enter:") or line.startswith("[character_exit:") or line.startswith("[character_step_aside:"):
                # Handle dynamic character management commands during Q&A
                character_mode = getattr(store, 'CHARACTER_MANAGEMENT_MODE', 'off')
                sandbox_mode = getattr(store, 'sandbox_mode', False)
                if character_mode == "dynamic" and sandbox_mode:
                    from store import DynamicCharacterManager
                    command_type, character_tag = DynamicCharacterManager.parse_character_command(line)
                    if command_type and character_tag:
                        DynamicCharacterManager.apply_character_command(
                            command_type,
                            character_tag,
                            qa_char_config_map,
                            qa_positions_map,
                            qa_shown_characters,
                            qa_characters
                        )
                        rb_log("Q/A applied ROSTER:", line, "| can_rb=", renpy.can_rollback())
                # Character management - no dialogue to show
                speaker_obj = None
                text_to_say = ""
            else:
                # Store the line info for the say statement
                speaker_obj = None
                text_to_say = ""
                
                # First, support a tolerant "name: \"...\"" format by normalizing to tag or narrator
                handled_colon = False
                try:
                    import re
                    m = re.match(r'^(\w[\w]*)\s*:\s*"([\s\S]*)"\s*$', line)
                    if m:
                        token = m.group(1)
                        payload = m.group(2)
                        safe_payload = payload.replace('%', '%%').replace('[', '[[').replace('{', '{{')
                        if token.lower() == 'narrator':
                            speaker_obj = narrator
                            text_to_say = safe_payload
                            handled_colon = True
                        else:
                            from store import _norm as _normalize_name
                            matched_char = None
                            for c in qa_characters:
                                if token == c.tag or _normalize_name(token) == _normalize_name(c.name):
                                    matched_char = c
                                    break
                            if matched_char:
                                # Update focus before saying the line
                                SpriteManager.update_focus(matched_char.tag, qa_characters, qa_shown_characters, qa_positions_map)
                                # Get the character object if available; fallback to narrator with prefix
                                speaker_obj_candidate = getattr(store, matched_char.tag, None)
                                if speaker_obj_candidate:
                                    speaker_obj = speaker_obj_candidate
                                    text_to_say = safe_payload
                                else:
                                    safe_name = matched_char.name.replace('[', '[[').replace('{', '{{').replace('%', '%%')
                                    speaker_obj = narrator
                                    text_to_say = f"[{safe_name}] {safe_payload}"
                                handled_colon = True
                            else:
                                # Unknown token, show as narrator with the token prefix
                                safe_token = token.replace('[', '[[').replace('{', '{{').replace('%', '%%')
                                speaker_obj = narrator
                                text_to_say = f"[{safe_token}] {safe_payload}"
                                handled_colon = True
                except Exception:
                    handled_colon = False

                if not handled_colon:
                    # Check for narrator (space-formatted)
                    if line.startswith("narrator "):
                        text_to_say = line[len("narrator "):].strip().strip('"')
                        text_to_say = text_to_say.replace('%', '%%').replace('[', '[[').replace('{', '{{')
                        speaker_obj = narrator
                    else:
                        # Check for character dialogue using tag + space
                        for c in qa_characters:
                            prefix = f"{c.tag} "
                            if line.startswith(prefix):
                                text_to_say = line[len(prefix):].strip().strip('"')
                                text_to_say = text_to_say.replace('%', '%%').replace('[', '[[').replace('{', '{{')
                                
                                # Update focus before saying the line
                                SpriteManager.update_focus(c.tag, qa_characters, qa_shown_characters, qa_positions_map)
                                
                                # Get the character object
                                speaker_obj = getattr(store, c.tag, None)
                                if not speaker_obj:
                                    # Fallback if character not found
                                    safe_name = c.name.replace('[', '[[').replace('{', '{{').replace('%', '%%')
                                    text_to_say = f"[{safe_name}] {text_to_say}"
                                    speaker_obj = narrator
                                break
                
                # If line wasn't recognized and it's not a sprite command, show as error
                if speaker_obj is None and text_to_say == "":
                    safe_line = line.replace('[', '[[').replace('{', '{{').replace('%', '%%')
                    text_to_say = f"(Unexpected format: {safe_line})"
                    speaker_obj = narrator
        
        # Execute the say statement if we have dialogue to show
        if speaker_obj is not None and text_to_say:
            # Show dialogue. Rely on Ren'Py's built-in rollback for say statements.
            $ renpy.say(speaker_obj, text_to_say)

        # Small pause to allow rollback/back to take effect before the next iteration
        $ renpy.pause(0.01)
            
    # Should not reach here, but just in case
    jump qa_response_complete

# Label for handling interrupted Q&A responses
label qa_response_interrupted:
    python:
        # Reset the processed lines counter
        qa_lines_processed = 0
        qa_response_interrupted = False
    return

# Label for handling completed Q&A responses  
label qa_response_complete:
    python:
        # Record how many lines were actually added to history
        if 'last_ai_response_start_index' in globals() and 'dialogue_history' in globals():
            last_ai_response_line_count = len(dialogue_history) - last_ai_response_start_index
        
        # Reset the processed lines counter
        qa_lines_processed = 0
        qa_response_interrupted = False
    return

label end_ai_conversation:
    # Called when user explicitly quits the AI chat
    $ in_ai_conversation = False
    $ show_ai_quit_button = False
    $ show_regenerate_button = False
    
    # Handle sleepover-related cleanup if applicable
    if 'sleepover_morning' in globals() and sleepover_morning:
        $ sleepover_morning = False
        $ sleepover_planned = False
        if 'sleepover_prompt_shown' in globals():
            $ sleepover_prompt_shown = False
    elif 'sleepover_error_recovery' in globals() and sleepover_error_recovery:
        $ sleepover_planned = False
        $ sleepover_error_recovery = False
        if 'sleepover_prompt_shown' in globals():
            $ sleepover_prompt_shown = False
    return


# === SCENE GENERATION SYSTEM ===
label generate_ai_scene(characters=None, location_context="", scene_prompt=None):
    # Store parameters globally so they persist during jumps/calls within this label
    # (Necessary due to Ren'Py's call/jump behavior with screens)
    $ scene_characters = characters
    $ scene_location_context = location_context
    $ scene_prompt_func = scene_prompt
    
    # Store characters in default variable for save/load
    $ store.current_scene_characters = characters
    
    # Initialize state variables for scene generation
    $ scene_retry_requested = False
    $ original_sprite_state = {}
    $ original_scene_lines = []
    $ scene_context_combined = ""
    $ scene_completed = False
    $ scene_interrupted = False
    $ scene_was_interrupted = False
    $ original_visible_characters = set()  # Track initially visible characters
    $ current_scene_prompt_text = scene_prompt() if callable(scene_prompt) else scene_prompt
    
    # Check if this is a sandbox scene (prompt contains specific sandbox markers)
    $ is_sandbox_scene = "visual novel sandbox mode" in str(current_scene_prompt_text)
    
    python:
        # Get positions map for characters (uses SpriteManager)
        positions_map = SpriteManager.get_positions_map(scene_characters)
        
        # Store which character sprites are visible at the start (for restoration)
        global last_sprite_state
        original_visible_characters = set()
        if 'last_sprite_state' in globals():
            for tag in last_sprite_state:
                if tag in {c.tag for c in scene_characters}:
                    original_visible_characters.add(tag)

        # Build the system prompt for the AI using PromptManager directly
        prompt_result = PromptManager.build_scene_prompt(scene_characters, current_scene_prompt_text)
        
        # Handle both old format (string) and new format (tuple)
        if isinstance(prompt_result, tuple):
            scene_context_combined, scene_post_story_instructions = prompt_result
        else:
            # Backward compatibility
            scene_context_combined = prompt_result
            scene_post_story_instructions = ""

    # Main loop for generating/editing/retrying the scene
    label .generation_loop:
        # If this is a retry, restore original character sprites before generating new content
        if scene_retry_requested:
            # Uses restore_original_characters helper from ai_helpers.rpy
            $ restore_original_characters(original_sprite_state, original_visible_characters, positions_map, char_config_map, scene_characters, shown_characters)
            $ scene_retry_requested = False
        else:
            # First time through - backup original sprite state before generation
            $ original_sprite_state = last_sprite_state.copy()
            
        # Generate the scene content (uses generate_scene_content helper from ai_helpers.rpy)
        $ scene_lines, scene_start_index, scene_line_count = generate_scene_content(scene_context_combined, save_original=True, post_story_instructions=scene_post_story_instructions)
        # Add an initial rollback checkpoint at the very start of AI scene playback
        $ renpy.checkpoint()
        # Safety: prevent rollback from going BEFORE the start of this scene
        $ renpy.block_rollback()
        
        # Play the scene using the queue system
        call play_generated_scene from _call_play_generated_scene

        if scene_interrupted:
            $ scene_was_interrupted = True
            $ scene_interrupted = False
            $ scene_completed = True
        else:
            # Present choice to keep, edit, or retry (uses screen defined in screens.rpy)
            $ choice = renpy.call_screen("scene_choice_menu")

            if choice == "keep":
                # User is satisfied - complete the scene
                $ scene_completed = True

            elif choice == "retry":
                # User wants to regenerate with same prompt
                $ renpy.notify("Regenerating scene...")
                # Uses remove_scene_from_history helper from ai_helpers.rpy
                $ remove_scene_from_history(scene_start_index, scene_line_count)
                $ scene_retry_requested = True
                
                # For sandbox mode, rebuild prompt with current additional instructions
                if is_sandbox_scene:
                    $ current_scene_prompt_text = create_sandbox_prompt(scene_characters, scene_location_context, sandbox_additional_instructions)
                    
                jump .generation_loop # Jump back to regenerate

            elif choice == "edit":
                # User wants to edit the prompt
                jump .edit_branch # Jump to edit logic

            else:
                # Fallback for unexpected choice value
                $ renpy.notify(f"Unexpected choice value: {choice}")
                $ scene_completed = True
        
        # Safety check - loop should only exit if scene_completed is True
        if not scene_completed:
            jump .generation_loop
    
    # Scene is complete - return to caller
    return

# Branch for handling prompt editing
label .edit_branch:
    # For sandbox mode, show only additional instructions
    if is_sandbox_scene:
        $ edit_prompt_text = sandbox_additional_instructions if 'sandbox_additional_instructions' in globals() else ""
    else:
        $ edit_prompt_text = current_scene_prompt_text
        
    # Show prompt editor screen (defined in screens.rpy)
    $ edited_prompt = renpy.call_screen("edit_scene_prompt", current_prompt=edit_prompt_text)
    
    # Handle if user cancelled the edit
    if edited_prompt == "__CANCEL__":
        $ renpy.notify("Edit cancelled")
        # Jump back to allow keep/retry on the *existing* generated content
        jump .cancel_edit
    
    # Handle if user applied changes
    elif edited_prompt is not None:
        $ renpy.notify("Updating prompt and regenerating...")
        
        # Remove current (pre-edit) scene from history
        $ remove_scene_from_history(scene_start_index, scene_line_count)
        
        # Update the prompt text based on mode
        if is_sandbox_scene:
            # For sandbox, update additional instructions and rebuild full prompt
            $ sandbox_additional_instructions = edited_prompt
            $ rebuilt_prompt_text = create_sandbox_prompt(scene_characters, scene_location_context, sandbox_additional_instructions)
            $ prompt_result = PromptManager.build_scene_prompt(scene_characters, rebuilt_prompt_text)
        else:
            # For story mode, update the prompt directly
            $ current_scene_prompt_text = edited_prompt
            $ prompt_result = PromptManager.build_scene_prompt(scene_characters, current_scene_prompt_text)

        # Rebuild the final prompt context string
        python:
            if isinstance(prompt_result, tuple):
                scene_context_combined, _unused = prompt_result
            else:
                scene_context_combined = prompt_result
            
        # Regenerate scene with the updated prompt
        $ restore_original_characters(original_sprite_state, original_visible_characters, positions_map, char_config_map, scene_characters, shown_characters)
        $ scene_lines, scene_start_index, scene_line_count = generate_scene_content(scene_context_combined)
        # Play the regenerated scene so the player can view it before making a new decision
        call play_generated_scene from _call_play_generated_scene_after_edit

        # After playing the new scene, check if it was interrupted
        if scene_interrupted:
            # If interrupted, mark scene as completed and exit, bypassing choice menu
            $ scene_was_interrupted = True
            $ scene_interrupted = False
            $ scene_completed = True
            return
        
        # Show choice menu again for the *newly* generated scene
        $ choice = renpy.call_screen("scene_choice_menu")
        
        # Handle user's choice after editing
        if choice == "keep":
            $ scene_completed = True
            # Scene completed, return to caller
            return
        
        elif choice == "retry":
            # User wants to retry *again* with the (edited) prompt
            $ renpy.notify("Regenerating scene...")
            $ remove_scene_from_history(scene_start_index, scene_line_count)
            $ scene_retry_requested = True
            jump .generation_loop # Go back to main loop for regeneration
        
        elif choice == "edit":
            # User wants to edit *again*
            jump .edit_branch # Go back to edit logic
        
        else:
            # Fallback
            $ scene_completed = True
            # Scene completed, return to caller
            return
    
    # Fallback if edited_prompt was None (shouldn't happen with current screen logic)
    $ renpy.notify("Edit prompt returned unexpected value.")
    jump .cancel_edit # Go back to let user decide on original content

# Branch jumped to after cancelling an edit - shows choice menu again
label .cancel_edit:
    # Show choice menu again without regenerating, using the last generated content
    $ choice = renpy.call_screen("scene_choice_menu")
    
    # Process the choice after cancellation
    if choice == "keep":
        $ scene_completed = True
        return
    
    elif choice == "retry":
        $ renpy.notify("Regenerating scene...")
        $ remove_scene_from_history(scene_start_index, scene_line_count)
        $ scene_retry_requested = True
        jump .generation_loop
    
    elif choice == "edit":
        # User changed their mind and wants to edit after all
        jump .edit_branch
    
    else:
        # Fallback
        $ scene_completed = True
        return

# Legacy label for backward compatibility - Jumps to the main scene gen logic
label .regenerate_scene:
    $ scene_retry_requested = True
    $ renpy.notify("Using legacy regenerate_scene label...")
    jump .generation_loop # Jump into the main loop marked for retry

label sleepover_sleep_action:
    # Backward-compatible label name; now route to the enhanced flow
    jump enhanced_sleep_action
