﻿################################################################################
## Initialization
################################################################################

init offset = -1


################################################################################
## Styles
################################################################################

style default:
    properties gui.text_properties()
    language gui.language

style input:
    properties gui.text_properties("input", accent=True)
    adjust_spacing False

style hyperlink_text:
    properties gui.text_properties("hyperlink", accent=True)
    hover_underline True

style gui_text:
    properties gui.text_properties("interface")

style button:
    properties gui.button_properties("button")

style button_text is gui_text:
    properties gui.text_properties("button")
    yalign 0.5


style label_text is gui_text:
    properties gui.text_properties("label", accent=True)

style prompt_text is gui_text:
    properties gui.text_properties("prompt")


style bar:
    ysize gui.bar_size
    left_bar Frame("gui/bar/left.png", gui.bar_borders, tile=gui.bar_tile)
    right_bar Frame("gui/bar/right.png", gui.bar_borders, tile=gui.bar_tile)

style vbar:
    xsize gui.bar_size
    top_bar Frame("gui/bar/top.png", gui.vbar_borders, tile=gui.bar_tile)
    bottom_bar Frame("gui/bar/bottom.png", gui.vbar_borders, tile=gui.bar_tile)

style scrollbar:
    ysize gui.scrollbar_size
    base_bar Frame("gui/scrollbar/horizontal_[prefix_]bar.png", gui.scrollbar_borders, tile=gui.scrollbar_tile)
    thumb Frame("gui/scrollbar/horizontal_[prefix_]thumb.png", gui.scrollbar_borders, tile=gui.scrollbar_tile)

style vscrollbar:
    xsize gui.scrollbar_size
    base_bar Frame("gui/scrollbar/vertical_[prefix_]bar.png", gui.vscrollbar_borders, tile=gui.scrollbar_tile)
    thumb Frame("gui/scrollbar/vertical_[prefix_]thumb.png", gui.vscrollbar_borders, tile=gui.scrollbar_tile)

style slider:
    ysize gui.slider_size
    left_bar Frame("gui/slider/left_[prefix_]bar.png", gui.slider_borders, tile=gui.slider_tile)
    right_bar Frame("gui/slider/right_[prefix_]bar.png", gui.slider_borders, tile=gui.slider_tile)
    thumb "gui/slider/horizontal_[prefix_]thumb.png"
    thumb_offset -20
    right_gutter 10
    left_gutter 10

style vslider:
    xsize gui.slider_size
    base_bar Frame("gui/slider/vertical_[prefix_]bar.png", gui.vslider_borders, tile=gui.slider_tile)
    thumb "gui/slider/vertical_[prefix_]thumb.png"

style frame:
    padding gui.frame_borders.padding
    background Frame("gui/frame.png", gui.frame_borders, tile=gui.frame_tile)


init python:
    config.character_id_prefixes.append('namebox')

style window is default
style say_label is default
style say_dialogue is default
style say_thought is say_dialogue

style namebox is default
style namebox_label is say_label


style window:
    xalign 0.5
    xfill True
    yalign gui.textbox_yalign
    ysize gui.textbox_height
    background Image("gui/textbox.png", xalign=0.5, yalign=1.0)

style namebox:
    xpos gui.name_xpos-20
    xanchor gui.name_xalign
    xsize gui.namebox_width-75
    ypos gui.name_ypos+80
    ysize 7
    background Frame("gui/backloghbar.png", gui.namebox_borders, tile=gui.namebox_tile, xalign=gui.name_xalign)
    # padding gui.namebox_borders.padding

style say_label:
    outlines [ 
        (3, "#00000066", 0, 3),
        (2, "#00000066", 0, 3),
        (1, "#00000099", 0, 2),
        # (2.5, "#00000077", 0, 0),
    ]
    properties gui.text_properties("name", accent=True)
    xalign gui.name_xalign
    yalign 0.5
    xpos 145
    ypos -22
    size 43

style say_dialogue:
    outlines [ 
        (3, "#00000033", 0, 3.5),
        (2, "#00000066", 0, 3),
        (1, "#00000099", 0, 2),
        (1.25, "#00000090", 0.2, 0),
    ]
    properties gui.text_properties("dialogue")
    xpos gui.dialogue_xpos-50
    xsize 1200
    ypos gui.dialogue_ypos-15
    size 29
    adjust_spacing False

## Styles for thoughts
style say_label_thoughts:
    outlines [ 
        (3, "#00000066", 0, 3),
        (2, "#00000066", 0, 3),
        (1, "#00000099", 0, 2),
        # (2.5, "#00000077", 0, 0),
    ]
    properties gui.text_properties("name", accent=True)
    xalign gui.name_xalign
    yalign 0.5
    xpos 120
    ypos -22
    size 43
    italic True

style say_dialogue_thoughts:
    outlines [ 
        (3, "#00000033", 0, 3.5),
        (2, "#00000066", 0, 3),
        (1, "#00000099", 0, 2),
        (1.25, "#00000090", 0.2, 0),
    ]
    properties gui.text_properties("dialogue")
    xpos gui.dialogue_xpos-50
    xsize 1200
    ypos gui.dialogue_ypos-15
    size 29
    color "#ffffffbb"
    italic True
    adjust_spacing False


################################################################################
## In-game screens
################################################################################


## Say screen ##################################################################
##
## The say screen is used to display dialogue to the player. It takes two
## parameters, who and what, which are the name of the speaking character and
## the text to be displayed, respectively. (The who parameter can be None if no
## name is given.)
##
## This screen must create a text displayable with id "what", as Ren'Py uses
## this to manage text display. It can also create displayables with id "who"
## and id "window" to apply style properties.
##
## https://www.renpy.org/doc/html/screen_special.html#say


image ctc_cherry:
    ("gui/memories/cherry.png" if not preferences.afm_enable else Null())
    xoffset 0
    zoom 0.4
    alpha 1.0

    yoffset -11+5.25

    linear 0.2 yoffset -10+5.25
    linear 0.2 yoffset -9+5.25
    linear 0.2 yoffset -8+5.25
    linear 0.2 yoffset -9+5.25
    linear 0.2 yoffset -10+5.25
    linear 0.2 yoffset -11+5.25

    linear 0.2 yoffset -12+5.25
    linear 0.2 yoffset -13+5.25
    linear 0.2 yoffset -14+5.25
    linear 0.2 yoffset -13+5.25
    linear 0.2 yoffset -12+5.25
    linear 0.2 yoffset -11+5.25

    repeat



screen say(who, what):
    style_prefix "say"

    window:
        id "window"
        background Transform(style.window.background, alpha=persistent.dialogueBoxOpacity)
    
        if who is not None:

            window:
                id "namebox"
                background Transform(style.namebox.background, alpha=persistent.dialogueBoxOpacity)

                style "namebox"
                text who id "who"

        text what id "what"

    if not renpy.variant("small"):
        add SideImage() xalign 0.0 yalign 1.0


## Input screen ################################################################
##
## This screen is used to display renpy.input. The prompt parameter is used to
## pass a text prompt in.
##
## This screen must create an input displayable with id "input" to accept the
## various input parameters.
##
## https://www.renpy.org/doc/html/screen_special.html#input

screen input(prompt):
    style_prefix "input"

    window:

        vbox:
            xanchor gui.dialogue_text_xalign
            xpos gui.dialogue_xpos
            xsize gui.dialogue_width
            ypos gui.dialogue_ypos

            text prompt style "input_prompt"
            input id "input"

style input_prompt is default

style input_prompt:
    xalign gui.dialogue_text_xalign
    properties gui.text_properties("input_prompt")

style input:
    xalign gui.dialogue_text_xalign
    xmaximum gui.dialogue_width


## Choice screen ###############################################################
##
## This screen is used to display the in-game choices presented by the menu
## statement. The one parameter, items, is a list of objects, each with caption
## and action fields.
##
## https://www.renpy.org/doc/html/screen_special.html#choice

screen choice(items):
    style_prefix "choice"

    vbox:
        for i in items:
            textbutton i.caption action i.action


style choice_vbox is vbox
style choice_button is button
style choice_button_text is button_text

style choice_vbox:
    xalign 0.5
    yalign 0.5
    yanchor 0.5
    spacing gui.choice_spacing-60


style choice_button is default:
    properties gui.button_properties("choice_button")
    xmaximum 950
    xalign 0.5

style choice_button_text is default:
    properties gui.text_properties("choice_button")
    font "fonts/Nunito/NunitoSans-Bold.ttf"
    selected_outlines [(absolute(20), "#f6f6f6", absolute(-0.5), absolute(-0.5))]
    size 27



screen mid_typewriter(lines):
    zorder 999

    frame:
        xalign 0.5
        yalign 0.5
        xminimum 800
        yminimum 400
        padding (30, 30, 30, 30)
        background None

        # Text block
        text colorize_words("\n".join(lines)) style "typewriter_text":
            xalign 0.5
            yalign 0.5
            textalign 0.5
            at fade_in


init python:
    import renpy.exports as renpy
    import re

    def colorize_words(text):
        if not highlights:
            return text

        words_pattern = "|".join(re.escape(word) for word, color in highlights)
        pattern = re.compile(r'\b(' + words_pattern + r')\b')

        def repl(m):
            matched_word = m.group(0)
            color = next(color for word, color in highlights if word == matched_word)
            return '{{color={}}}{}{{/color}}'.format(color, matched_word)

        return pattern.sub(repl, text)

    def typewriter_lines(lines, visible_lines, cps=15.0, pause_time=1.0):
        visible_lines[:] = []

        for i, line in enumerate(lines):
            current = ""
            if visible_lines:
                visible_lines[-1] = current
            else:
                visible_lines.append(current)

            for ch in line:
                current += ch
                visible_lines[-1] = current
                renpy.restart_interaction()
                renpy.pause(1.0 / cps)

            visible_lines[-1] = current
            renpy.restart_interaction()

            if pause_time > 0:
                renpy.pause(pause_time)

            if i < len(lines) - 1:
                visible_lines.append("")



style typewriter_text:
    color "#ffffff"
    size 23
    outlines [(4, "#000000")]
    line_spacing 20
    kerning 2
    font "fonts/LINESeed/LINESeedSans_Rg.ttf"


screen quick_menu():

    zorder 100

    if quick_menu:

        hbox:
            style_prefix "quick"
            xalign 0.5
            yalign 0.9999
            spacing 8  # small gap between icons

            # Backlog
            imagebutton:
                idle "gui/quickmenu/qmbacklog_idle.png"
                hover "gui/quickmenu/qmbacklog_hover.png"
                action ShowMenu('history')
                tooltip "Backlog"
                at icon_small

            if not _in_replay:
                # Memories
                if has_pending_memories():
                    imagebutton:
                        idle "gui/quickmenu/qmmemoriespending_idle.png"
                        hover "gui/quickmenu/qmmemories_hover.png"
                        action ShowMenu('memories_screen')
                        tooltip "Memories (new!)"
                        at icon_small
                else:
                    imagebutton:
                        idle "gui/quickmenu/qmmemories_idle.png"
                        hover "gui/quickmenu/qmmemories_hover.png"
                        action ShowMenu('memories_screen')
                        tooltip "Memories"
                        at icon_small

            # Skip
            imagebutton:
                idle ("gui/quickmenu/qmskip_hover.png"
                    if renpy.is_skipping()
                    else "gui/quickmenu/qmskip_idle.png")
                hover "gui/quickmenu/qmskip_hover.png"
                action Skip()
                alternate Skip(fast=True, confirm=True)
                tooltip "Skip"
                at icon_small

            # Auto
            imagebutton:
                idle ("gui/quickmenu/qmauto_hover.png" if renpy.game.preferences.afm_enable
                    else "gui/quickmenu/qmauto_idle.png")
                hover "gui/quickmenu/qmauto_hover.png"
                action Preference("auto-forward", "toggle")
                tooltip "Auto"
                at icon_small


            if not _in_replay:
                # Save
                imagebutton:
                    idle "gui/quickmenu/qmsave_idle.png"
                    hover "gui/quickmenu/qmsave_hover.png"
                    action ShowMenu('save')
                    tooltip "Save"
                    at icon_small

                # Load
                imagebutton:
                    idle "gui/quickmenu/qmload_idle.png"
                    hover "gui/quickmenu/qmload_hover.png"
                    action ShowMenu('load')
                    tooltip "Load"
                    at icon_small

                # Quick Save
                imagebutton:
                    idle "gui/quickmenu/qmquicksave_idle.png"
                    hover "gui/quickmenu/qmquicksave_hover.png"
                    action QuickSave()
                    tooltip "Quick Save"
                    at icon_small

                # Quick Load
                imagebutton:
                    idle "gui/quickmenu/qmquickload_idle.png"
                    hover "gui/quickmenu/qmquickload_hover.png"
                    action QuickLoad()
                    tooltip "Quick Load"
                    at icon_small

            # Options
            imagebutton:
                idle "gui/quickmenu/qmoptions_idle.png"
                hover "gui/quickmenu/qmoptions_hover.png"
                action ShowMenu('preferences')
                tooltip "Options"
                at icon_small

        # END MEMORY
        if _in_replay:
            frame:
                style_prefix "check"
                xpos 1400
                ypos 750

                textbutton "END MEMORY" action EndReplay(confirm=True)




## This code ensures that the quick_menu screen is displayed in-game, whenever
## the player has not explicitly hidden the interface.
init python:
    config.overlay_screens.append("quick_menu")

default quick_menu = True

style quick_button is default
style quick_button_text is button_text

style quick_button:
    properties gui.button_properties("quick_button")

style quick_button_text:
    properties gui.text_properties("quick_button")
    font "fonts/Nunito/NunitoSans-Bold.ttf"
    outlines [ 
        (3, "#00000033", 0, 3.5),
        (2, "#00000066", 0, 3),
        (1, "#00000099", 0, 2),
        (1.25, "#00000090", 0.2, 0),
    ]

# Define font settings for the About tab
style about_text:
    font "fonts/Nunito/NunitoSans-Regular.ttf"  # Set the font file
    size 20
style about_title:
    font "fonts/Nunito/NunitoSans-Regular.ttf"


################################################################################
## Main and Game Menu Screens
################################################################################

## Navigation screen ###########################################################
##
## This screen is included in the main and game menus, and provides navigation
## to other menus, and to start the game.

screen navigation():

    hbox:
        if renpy.get_screen("main_menu"):
            style_prefix "navigation"
        else:
            style_prefix "navigationsub"

        if renpy.get_screen("main_menu"):
            xpos 0.5
            xanchor "center"
            yalign 0.97
            spacing 50
        else:
            xpos 0.01
            xanchor "left"
            yalign 0.6
            spacing gui.navigation_spacing
        
        
        if renpy.get_screen("main_menu"):
            textbutton _("New Game") action [Play("sound", "sfx/selectmenu.ogg"), Start()] at fade_in
            textbutton _("Continue") action [Play("sound", "sfx/selectmenu.ogg"), ShowMenu("load")] at fade_in
            textbutton _("Options") action [Play("sound", "sfx/selectmenu.ogg"), ShowMenu("preferences")] at fade_in
            textbutton _("About") action [Play("sound", "sfx/selectmenu.ogg"), ShowMenu("about")] at fade_in
            textbutton _("Exit") action Quit(confirm=not main_menu) at fade_in

    vbox:

        if renpy.get_screen("main_menu"):
            style_prefix "navigation"
        else:
            style_prefix "navigationsub"

        if renpy.get_screen("main_menu") and not renpy.android:
            xpos 0.5
            xanchor "center"
            yalign 0.95
            spacing 50
        elif renpy.get_screen("main_menu"):
            xpos 0.5
            xanchor "center"
            yalign 0.95
            spacing 5
        elif renpy.android:
            xpos 0.01
            xanchor "left"
            yalign 0.8
            spacing 5
        else:
            xpos 0.01
            xanchor "left"
            yalign 0.6
            spacing gui.navigation_spacing

        if main_menu and not renpy.get_screen("main_menu"):
            # New Game
            if main_menu:
                imagebutton:
                    idle "gui/navigation buttons/newgame1.png"
                    hover "gui/navigation buttons/newgame2.png"
                    align (0.1, 0.7)
                    xsize 200
                    ysize 90
                    action Start()
                    hovered Play("sound", "sfx/hovermenu.ogg")

            # Load button with hover effect
            if renpy.get_screen("load") and not renpy.get_screen("main_menu") and not _in_replay:
                imagebutton:
                    idle "gui/navigation buttons/load2.png"
                    hover "gui/navigation buttons/load2.png"
                    align (0.1, 0.7)
                    xsize 200
                    ysize 90
                    action [Play("sound", "sfx/menupage2.ogg"), ShowMenu("load")]
            elif not renpy.get_screen("main_menu") and not _in_replay:
                imagebutton:
                    idle "gui/navigation buttons/load1.png"
                    hover "gui/navigation buttons/load2.png"
                    align (0.1, 0.7)
                    xsize 200
                    ysize 90
                    action [Play("sound", "sfx/menupage2.ogg"), ShowMenu("load")]
                    hovered Play("sound", "sfx/hovermenu.ogg")

            # Options
            if renpy.get_screen("preferences") and not renpy.get_screen("main_menu"):
                imagebutton:
                    idle "gui/navigation buttons/options2.png"
                    hover "gui/navigation buttons/options2.png"
                    align (0.1, 0.7)
                    xsize 200
                    ysize 90
                    action [Play("sound", "sfx/menupage2.ogg"), ShowMenu("preferences")]
            elif not renpy.get_screen("main_menu"):
                imagebutton:
                    idle "gui/navigation buttons/options1.png"
                    hover "gui/navigation buttons/options2.png"
                    align (0.1, 0.7)
                    xsize 200
                    ysize 90
                    action [Play("sound", "sfx/menupage2.ogg"), ShowMenu("preferences")]
                    hovered Play("sound", "sfx/hovermenu.ogg")

            # About
            if renpy.get_screen("about") and not renpy.get_screen("main_menu"):
                imagebutton:
                    idle "gui/navigation buttons/about2.png"
                    hover "gui/navigation buttons/about2.png"
                    align (0.1, 0.7)
                    xsize 200
                    ysize 90
                    action [Play("sound", "sfx/menupage2.ogg"), ShowMenu("about")]
            elif not renpy.get_screen("main_menu"):
                imagebutton:
                    idle "gui/navigation buttons/about1.png"
                    hover "gui/navigation buttons/about2.png"
                    align (0.1, 0.7)
                    xsize 200
                    ysize 90
                    action [Play("sound", "sfx/menupage2.ogg"), ShowMenu("about")]
                    hovered Play("sound", "sfx/hovermenu.ogg")

        else:
            # History
            if renpy.get_screen("history") and not renpy.get_screen("main_menu"):
                imagebutton:
                    idle "gui/navigation buttons/backlog2.png"
                    hover "gui/navigation buttons/backlog2.png"
                    align (0.1, 0.7)
                    xsize 200
                    ysize 90
                    action [Play("sound", "sfx/menupage2.ogg"), ShowMenu("history")]
            elif not renpy.get_screen("main_menu"):
                imagebutton:
                    idle "gui/navigation buttons/backlog1.png"
                    hover "gui/navigation buttons/backlog2.png"
                    align (0.1, 0.7)
                    xsize 200
                    ysize 90
                    action [Play("sound", "sfx/menupage2.ogg"), ShowMenu("history")]
                    hovered Play("sound", "sfx/hovermenu.ogg")

            # Save
            if renpy.get_screen("save") and not renpy.get_screen("main_menu") and not _in_replay:
                imagebutton:
                    idle "gui/navigation buttons/save2.png"
                    hover "gui/navigation buttons/save2.png"
                    align (0.1, 0.7)
                    xsize 200
                    ysize 90
                    action [Play("sound", "sfx/menupage2.ogg"), ShowMenu("save")]
            elif not renpy.get_screen("main_menu") and not _in_replay:
                imagebutton:
                    idle "gui/navigation buttons/save1.png"
                    hover "gui/navigation buttons/save2.png"
                    align (0.1, 0.7)
                    xsize 200
                    ysize 90
                    action [Play("sound", "sfx/menupage2.ogg"), ShowMenu("save")]
                    hovered Play("sound", "sfx/hovermenu.ogg")

            # Load
            if renpy.get_screen("load") and not renpy.get_screen("main_menu") and not _in_replay:
                imagebutton:
                    idle "gui/navigation buttons/load2.png"
                    hover "gui/navigation buttons/load2.png"
                    align (0.1, 0.7)
                    xsize 200
                    ysize 90
                    action [Play("sound", "sfx/menupage2.ogg"), ShowMenu("load")]
            elif not renpy.get_screen("main_menu") and not _in_replay:
                imagebutton:
                    idle "gui/navigation buttons/load1.png"
                    hover "gui/navigation buttons/load2.png"
                    align (0.1, 0.7)
                    xsize 200
                    ysize 90
                    action [Play("sound", "sfx/menupage2.ogg"), ShowMenu("load")]
                    hovered Play("sound", "sfx/hovermenu.ogg")

            # Options
            if renpy.get_screen("preferences") and not renpy.get_screen("main_menu"):
                imagebutton:
                    idle "gui/navigation buttons/options2.png"
                    hover "gui/navigation buttons/options2.png"
                    align (0.1, 0.7)
                    xsize 200
                    ysize 90
                    action [Play("sound", "sfx/menupage2.ogg"), ShowMenu("preferences")]
            elif not renpy.get_screen("main_menu"):
                imagebutton:
                    idle "gui/navigation buttons/options1.png"
                    hover "gui/navigation buttons/options2.png"
                    align (0.1, 0.7)
                    xsize 200
                    ysize 90
                    action [Play("sound", "sfx/menupage2.ogg"), ShowMenu("preferences")]
                    hovered Play("sound", "sfx/hovermenu.ogg")

            # About
            if renpy.get_screen("about") and not renpy.get_screen("main_menu"):
                imagebutton:
                    idle "gui/navigation buttons/about2.png"
                    hover "gui/navigation buttons/about2.png"
                    align (0.1, 0.7)
                    xsize 200
                    ysize 90
                    action [Play("sound", "sfx/menupage2.ogg"), ShowMenu("about")]
            elif not renpy.get_screen("main_menu"):
                imagebutton:
                    idle "gui/navigation buttons/about1.png"
                    hover "gui/navigation buttons/about2.png"
                    align (0.1, 0.7)
                    xsize 200
                    ysize 90
                    action [Play("sound", "sfx/menupage2.ogg"), ShowMenu("about")]
                    hovered Play("sound", "sfx/hovermenu.ogg")

            # Title menu
            if renpy.get_screen("main_menu") and not renpy.get_screen("main_menu"):
                imagebutton:
                    idle "gui/navigation buttons/title2.png"
                    hover "gui/navigation buttons/title2.png"
                    align (0.1, 0.7)
                    xsize 200
                    ysize 90
                    action [Play("sound", "sfx/menupage2.ogg"), MainMenu()]
            elif not renpy.get_screen("main_menu"):
                imagebutton:
                    idle "gui/navigation buttons/title1.png"
                    hover "gui/navigation buttons/title2.png"
                    align (0.1, 0.7)
                    xsize 200
                    ysize 90
                    action [Play("sound", "sfx/menupage2.ogg"), MainMenu()]
                    hovered Play("sound", "sfx/hovermenu.ogg")

            # Replay
        # if _in_replay:
        #     imagebutton:
        #         idle "gui/navigation buttons/load1.png"
        #         hover "gui/navigation buttons/load2.png"
        #         align (0.1, 0.7)
        #         xsize 200
        #         ysize 90
        #         action EndReplay(confirm=True)

        # Exit
        if renpy.get_screen("quit") and not renpy.get_screen("main_menu"):
            imagebutton:
                idle "gui/navigation buttons/exit2.png"
                hover "gui/navigation buttons/exit2.png"
                align (0.1, 0.7)
                xsize 200
                ysize 90
                action Quit(confirm=not main_menu)
        elif not renpy.get_screen("main_menu"):
            imagebutton:
                idle "gui/navigation buttons/exit1.png"
                hover "gui/navigation buttons/exit2.png"
                align (0.1, 0.7)
                xsize 200
                ysize 90
                action [
                    Function(renpy.sound.set_volume, 0.35, channel="sound"),
                    Play("sound", "sfx/exitgame.ogg"),
                    Quit(confirm=not main_menu)
                ]
        
        # Return Button, Android
        if renpy.android and not renpy.get_screen("main_menu"):
            imagebutton:
                idle "gui/navigation buttons/return1.png"
                hover "gui/navigation buttons/return2.png"                
                align (0.1, 0.7)
                xsize 200
                ysize 90
                action Return()

style navigation_button is gui_button
style navigation_button_text is gui_button_text

style navigation_button:
    size_group "navigation"
    properties gui.button_properties("navigation_button")

style navigation_button_text:
    xalign 0.5
    properties gui.text_properties("navigation_button")
    color "#1a486e"
    font "fonts/handwritting.ttf"
    outlines [(absolute(2), "#ffffff")]
    hover_color "#99ccff"
    size 47

style navigationsub_button is gui_button
style navigationsub_button_text is gui_button_text

style navigationsub_button:
    size_group "navigationsub"
    properties gui.button_properties("navigationsub_button")

style navigationsub_button_text:
    xpos 0.2
    properties gui.text_properties("navigation_button")
    outlines [(absolute(3), "#4a75a0", absolute(-1.5), absolute(-1.5))]
    selected_color "#28415a"
    selected_outlines [(absolute(2), "#f6f6f6", absolute(-0.5), absolute(-0.5))]
    size 37



## Main Menu screen ############################################################
##
## Used to display the main menu when Ren'Py starts.
##
## https://www.renpy.org/doc/html/screen_special.html#main-menu

screen main_menu():

    tag menu

    # Show main background fully visible immediately
    add gui.main_menu_background

    # Add a white layer on top that fades out smoothly
    add Solid("#FFFFFF") at white_fade_out

    # Logo fades in after background
    add "gui/bloom_again.png" at fade_in_logo:
        yalign 0.80
        xalign 0.5

    frame:
        style "main_menu_frame"

    imagebutton auto "gui/discord_%s.png":
        action OpenURL("https://discord.gg/Fu944qR5hG")
        at discord
        focus_mask True

    imagebutton auto "gui/substar_%s.png":
        action OpenURL("https://subscribestar.com/bloomagain")
        at supportpage
        focus_mask True

    imagebutton auto "gui/itch_%s.png":
        action OpenURL("https://gaasha.itch.io/bloomagain")
        at itch
        focus_mask True

    use navigation

    if gui.show_name:
        vbox:
            style "main_menu_vbox"
            text "[config.name!t]":
                style "main_menu_title"
            text "[config.version]":
                style "main_menu_version"



style main_menu_frame is empty
style main_menu_vbox is vbox
style main_menu_text is gui_text
style main_menu_title is main_menu_text
style main_menu_version is main_menu_text

style main_menu_frame:
    xsize 420
    yfill True

style main_menu_vbox:
    xalign 1.0
    xoffset -30
    xmaximum 1200
    yalign 1.0
    yoffset -30

style main_menu_text:
    properties gui.text_properties("main_menu", accent=True)

style main_menu_title:
    properties gui.text_properties("title")
    text_align 0.5  # Ensures the text itself is centered within its block

style main_menu_version:
    properties gui.text_properties("version")

## Game Menu screen ############################################################
##
## This lays out the basic common structure of a game menu screen. It's called
## with the screen title, and displays the background, title, and navigation.
##
## The scroll parameter can be None, or one of "viewport" or "vpgrid".
## This screen is intended to be used with one or more children, which are
## transcluded (placed) inside it.

screen game_menu(title, scroll=None, yinitial=0.0, spacing=0):

    style_prefix "game_menu"

    if main_menu:
        add gui.main_menu_background
    else:
        add gui.game_menu_background

    frame:
        style "game_menu_outer_frame"

        hbox:

            ## Reserve space for the navigation section.
            frame:
                style "game_menu_navigation_frame"

            frame:
                style "game_menu_content_frame"

                if scroll == "viewport":

                    viewport:
                        yinitial yinitial
                        scrollbars "vertical"
                        mousewheel True
                        draggable True
                        pagekeys True

                        side_yfill True

                        vbox:
                            spacing spacing

                            transclude

                elif scroll == "vpgrid":

                    vpgrid:
                        cols 1
                        yinitial yinitial

                        scrollbars "vertical"
                        mousewheel True
                        draggable True
                        pagekeys True

                        side_yfill True

                        spacing spacing

                        transclude

                else:

                    transclude

    use navigation

    # textbutton _("Return"):
    #     style "return_button"

    #     if not gameactive:
    #         action[SetVariable("main_menu",True), ShowMenu("main_menu")]

    #     action Return()

    label title

    if main_menu:
        key "game_menu" action ShowMenu("main_menu")


style game_menu_outer_frame is empty
style game_menu_navigation_frame is empty
style game_menu_content_frame is empty
style game_menu_viewport is gui_viewport
style game_menu_side is gui_side
style game_menu_scrollbar is gui_vscrollbar

style game_menu_label is gui_label
style game_menu_label_text is gui_label_text

style return_button is navigation_button
style return_button_text is navigation_button_text

style game_menu_outer_frame:
    bottom_padding 45
    top_padding 180
    background "gui/overlay/nav_overlay.png"

style game_menu_navigation_frame:
    xsize 350
    yfill True

style game_menu_content_frame:
    left_margin 85
    right_margin 30
    top_margin 15

style game_menu_viewport:
    xsize 1380

style game_menu_vscrollbar:
    xpos -10
    unscrollable gui.unscrollable

style game_menu_side:
    spacing 15

style game_menu_label:
    xpos 75
    ysize 180

style game_menu_label_text:
    size 45
    font "fonts/poetry.ttf"
    color "#ffffff"
    outlines [
        (2.5, "#888888", 0.5, 0.5),
        (3, "#00000050", 0, 3),
        (2, "#00000050", 0, 3)
    ]
    yalign 0.5
    xpos 10
    ypos 130

style return_button:
    xpos gui.navigation_xpos
    yalign 1.0
    yoffset -45


## About screen ################################################################
##
## This screen gives credit and copyright information about the game and Ren'Py.
##
## There's nothing special about this screen, and hence it also serves as an
## example of how to make a custom screen.

screen about():

    tag menu


    use game_menu(_("About"), scroll="viewport"):

        style_prefix "about"

        vbox:
            add "gui/logo.png" xysize(805, 217) xalign 0.5

            text _("{size=20}Thank you for playing {i}{b}{size=24}Bloom Again{/size}{/b}{/i} by {b}gasha{/b}. This project, including the character models and scenes, was created using Koikatsu/Chara Studio by Illusion.{/size}")

            text _("{size=20}{b}Special thanks to{/b} Veebuggsy, Zadok, Brosephalon, Altherian, and all my 4F friends! Without your support I wouldn't have made it this far.{/size}")
            text _("{size=20}{b}In loving memory of{/b} Andrei.{/size}")
            text _("{size=20}Please consider supporting me on {a=https://subscribestar.adult/bloomagain}{color=#49acfc}{i}{b}SubscribeStar{/b}{/i}{/color}{/a} and joining my {a=https://discord.gg/Fu944qR5hG}{color=#49acfc}{i}{b}Discord server{/b}{/i}{/color}{/a}.{/size}")

            hbox:
                spacing 40
                xalign 0.5
                imagebutton:
                    idle "gui/substar_hover.png"
                    action OpenURL("https://subscribestar.adult/bloomagain")
                    at Transform(size=(120, 120))                    
                imagebutton:
                    idle "gui/discord_hover.png"
                    action OpenURL("https://discord.gg/Fu944qR5hG")
                    at Transform(size=(120, 120))                    
                imagebutton:
                    idle "gui/itch_hover.png"
                    action OpenURL("https://gaasha.itch.io/bloomagain")
                    at Transform(size=(120, 120))     

            text _("Credits:") size 24 font "fonts/handwritting.ttf" underline True bold True xalign 0.0
            text _("{size=20}{b}  • BGM/SFX:{/b} Clavier-Music, NicksNicky, and various artists on pixabay.{/size}")
            text _("{size=20}{b}  • Title Menu song:{/b} Performed by Clavier-Music.{/size}")
            text _("{size=20}{b}  • Mouse cursor:{/b} Designed by SeyfDesigner.{/size}")
            text _("{size=20}{b}  • Everything else:{/b} gasha.\n{/size}")

            if gui.about:
                text "[gui.about!t]\n"

            text _("{size=20}Made with {a=https://www.renpy.org/}{color=#49acfc}Ren'Py{/color}{/a} [renpy.version_only]. [renpy.license!t]{/size}")


style about_label is gui_label
style about_label_text is gui_label_text
style about_text is gui_text

style about_label_text:
    size gui.label_text_size

style about_text:
    color "#ffffff"
    line_spacing 5
    outlines [ 
        (1, "#00000050", 0, 3.25),
        (1, "#00000050", 0, 3)
    ]

## Load and Save screens #######################################################
##
## These screens are responsible for letting the player save the game and load
## it again. Since they share nearly everything in common, both are implemented
## in terms of a third screen, file_slots.
##
## https://www.renpy.org/doc/html/screen_special.html#save https://
## www.renpy.org/doc/html/screen_special.html#load

screen save():

    tag menu

    use file_slots(_("Save"))


screen load():

    tag menu

    use file_slots(_("Load"))


screen file_slots(title):

    default page_name_value = FilePageNameInputValue(pattern=_("Page {}"), auto=_("Automatic saves"), quick=_("Quick saves"))

    use game_menu(title):

        fixed:

            ## This ensures the input will get the enter event before any of the
            ## buttons do.
            order_reverse True

            ## The page name, which can be edited by clicking on a button.
            button:
                style "page_label"

                key_events True
                xalign 0.48
                yalign 0.05
                action page_name_value.Toggle()

                input:
                    style "page_label_text"
                    value page_name_value

            ## The grid of file slots.
            grid gui.file_slot_cols gui.file_slot_rows:
                if renpy.get_screen("save"):
                    style_prefix "slotsave"
                else:
                    style_prefix "slotload"

                if renpy.android:
                    xalign 0.5
                else:
                    xalign -0.065
                
                yalign 0.5

                spacing gui.slot_spacing

                for i in range(gui.file_slot_cols * gui.file_slot_rows):

                    $ slot = i + 1

                    button:
                        action FileAction(slot)

                        has vbox

                        add FileScreenshot(slot):
                            xpos 5
                            ypos 17


                        text FileTime(slot, format=_("{#file_time}%a, %b %d %Y, %H:%M"), empty=_("empty slot")):
                            style "slot_time_text"
                            font "fonts/Aller/Aller_Std_Bd.ttf"
                            xalign 0.5
                            ypos 40
                            size 17

                        text FileSaveName(slot):
                            style "slot_name_text"
                            font "fonts/Aller/Aller_Std_Bd.ttf"

                        key "save_delete" action FileDelete(slot)

            ## Buttons to access other pages.
            vbox:
                style_prefix "page"

                xalign 0.462
                yalign 0.97

                hbox:
                    xalign 0.5
                    
                    spacing gui.page_spacing

                    imagebutton:
                        yalign 0.5
                        idle "gui/arrows/leftarrow.png"
                        hover "gui/arrows/leftarrow_hover.png"
                        action FilePagePrevious()
                    key "save_page_prev" action FilePagePrevious()

                    if config.has_autosave:
                        textbutton _("{#auto_page}A") action FilePage("auto")

                    if config.has_quicksave:
                        textbutton _("{#quick_page}Q") action FilePage("quick")

                    ## range(1, 10) gives the numbers from 1 to 9.
                    for page in range(1, 10):
                        textbutton "[page]" action FilePage(page)

                    imagebutton:
                        yalign 0.5
                        idle "gui/arrows/rightarrow.png"
                        hover "gui/arrows/rightarrow_hover.png"
                        action FilePageNext()
                    key "save_page_next" action FilePageNext()


style page_label is gui_label
style page_label_text is gui_label_text
style page_button is gui_button
style page_button_text is gui_button_text

style slot_button is gui_button
style slot_button_text is gui_button_text
style slot_time_text is slot_button_text
style slot_name_text is slot_button_text

style page_label:
    xpadding 75
    ypadding 5

style page_label_text:
    size 42
    color "#ffffff"
    font "fonts/poetry.ttf"
    outlines [ 
        (2, "#00000050", 0, 3),
        (1, "#00000050", 0, 3)
    ]
    textalign 0.5
    layout "subtitle"

style page_button:
    properties gui.button_properties("page_button")

style page_button_text:
    size 32
    font "fonts/Nunito/NunitoSans-Bold.ttf"
    outlines [ 
        (1.37, "#203b5299", 0, 2.5),
        (1.40, "#203b5295", 0, 3.0)
    ]
    hover_color "#fadb51"
    properties gui.text_properties("page_button")

style slotsave_button:
    properties gui.button_properties("slot_button")
    color "#ffffff"
    background "gui/button/saveslot_idle_background.png"
    hover_background "gui/button/saveslot_hover_background.png"

style slotload_button:
    properties gui.button_properties("slot_button")
    color "#ffffff"
    background "gui/button/loadslot_idle_background.png"
    hover_background "gui/button/loadslot_hover_background.png"

style slot_button_text:
    color "#ffffff"
    properties gui.text_properties("slot_button")
    hover_color "#fadb51"
    outlines [ 
        (2.5, "#00000040", 0, 4),
        (1.5, "#00000025", 0, 2.5),
        (1, "#00000020", 0, 1)
    ]

## Preferences screen ##########################################################
##
## The preferences screen allows the player to configure the game to better suit
## themselves.
##
## https://www.renpy.org/doc/html/screen_special.html#preferences

screen preferences():
    key ("o", "O") action Return()
    tag menu


    use game_menu(_("Options"), scroll="viewport"):

        vbox:

            hbox:
                box_wrap True

                if renpy.variant("pc") or renpy.variant("web"):

                    vbox:
                        style_prefix "radio"
                        label _("Display")
                        textbutton _("Window") action Preference("display", "window")
                        textbutton _("Fullscreen") action Preference("display", "fullscreen")

                vbox:
                    style_prefix "check"
                    label _("Skip")
                    textbutton _("Unseen Text") action Preference("skip", "toggle")
                    textbutton _("After Choices") action Preference("after choices", "toggle")
                    textbutton _("Transitions") action InvertSelected(Preference("transitions", "toggle"))

            null height (4 * gui.pref_spacing)

            hbox:
                if renpy.android:
                    spacing 200

                style_prefix "slider"
                box_wrap True

                vbox:
                    style_prefix "slider"

                    label _("Dialogue Box Opacity")
                    bar value FieldValue(persistent, "dialogueBoxOpacity", range=1.0, style="slider")

                    label _("Text Speed")

                    bar value Preference("text speed")

                    label _("Auto-Forward Time")

                    bar value Preference("auto-forward time")

                    label _("Tool-tip duration")

                    bar value FieldValue(persistent, "notify_duration", range=1.0, step=0.5, style="slider")

                vbox:

                    if config.has_music:
                        label _("Music Volume")

                        hbox:
                            bar value Preference("music volume")

                    if config.has_sound:

                        label _("Sound Volume")

                        hbox:
                            bar value Preference("sound volume")

                            if config.sample_sound:
                                textbutton _("Test") action Play("sound", config.sample_sound)


                    if config.has_voice:
                        label _("Voice Volume")

                        hbox:
                            bar value Preference("voice volume")

                            if config.sample_voice:
                                textbutton _("Test") action Play("voice", config.sample_voice)

                    if config.has_music or config.has_sound or config.has_voice:
                        null height gui.pref_spacing

                        textbutton _("Mute All"):
                            action Preference("all mute", "toggle")
                            style "mute_all_button"



style pref_label is gui_label
style pref_label_text is gui_label_text
style pref_vbox is vbox

style radio_label is pref_label
style radio_label_text is pref_label_text
style radio_button is gui_button
style radio_button_text is gui_button_text
style radio_vbox is pref_vbox

style check_label is pref_label
style check_label_text is pref_label_text
style check_button is gui_button
style check_button_text is gui_button_text
style check_vbox is pref_vbox

style slider_label is pref_label
style slider_label_text is pref_label_text
style slider_slider is gui_slider
style slider_button is gui_button
style slider_button_text is gui_button_text
style slider_pref_vbox is pref_vbox

style mute_all_button is check_button
style mute_all_button_text is check_button_text

style pref_label:
    top_margin gui.pref_spacing
    bottom_margin 3

style pref_label_text:
    yalign 1.0
    size 32
    outlines [(absolute(2.5), "#28415a", absolute(-1.5), absolute(-1.5))]

style pref_vbox:
    xsize 338

style radio_vbox:
    spacing gui.pref_button_spacing

style radio_button:
    properties gui.button_properties("radio_button")
    foreground "gui/button/radio_[prefix_]foreground.png"

style radio_button_text:
    size 28
    color "#ffffff"
    outlines [ 
        (2, "#00000050", 0, 3),
        (1, "#00000050", 0, 3)
    ]
    font "fonts/Nunito/NunitoSans-Bold.ttf"
    hover_color "#fadb51"
    properties gui.text_properties("radio_button")


style check_vbox:
    spacing gui.pref_button_spacing

style check_button:
    properties gui.button_properties("check_button")
    foreground "gui/button/check_[prefix_]foreground.png"

style check_button_text:
    size 26
    color "#ffffff"
    outlines [ 
        (2, "#00000050", 0, 3),
        (1, "#00000050", 0, 3)
    ]

    font "fonts/Nunito/NunitoSans-Bold.ttf"
    hover_color "#fadb51"
    properties gui.text_properties("check_button")

style slider_slider:
    xsize 450
    xpos 15

style slider_button:
    properties gui.button_properties("slider_button")
    yalign 0.5
    left_margin 15

style slider_label_text:
    size 28

style slider_button_text:
    properties gui.text_properties("slider_button")

style slider_vbox:
    xsize 675


## History screen ##############################################################
##
## This is a screen that displays the dialogue history to the player. While
## there isn't anything special about this screen, it does have to access the
## dialogue history stored in _history_list.
##
## https://www.renpy.org/doc/html/history.html

init -1:
    if renpy.android:
        $ gui.history_height = None
        $ gui.history_spacing = 0

screen history():
    key ("K_HOME") action Return()
    tag menu

    predict False

    # Limit backlog to last 200 lines for performance:
    $ recent_history = _history_list[-200:]

    use game_menu(_("Backlog"), scroll=("vpgrid" if gui.history_height else "viewport"), yinitial=1.0, spacing=gui.history_spacing if not renpy.android else 0):

        style_prefix "history"

        for h in recent_history:

            vbox:
                spacing 10
                window:

                    has fixed:
                        yfit True

                    if h.who:

                        label h.who:
                            style "history_name"
                            substitute False

                            if "color" in h.who_args:
                                text_color h.who_args["color"]

                    $ what = renpy.filter_text_tags(h.what, allow=history_allow_tags)
                    text what:
                        substitute False

                add "gui/backloghbar.png" xalign 0.5

        if not recent_history:
            label _("There is no dialogue history.")
            add "gui/backloghbar.png" xalign 0.5 ypos 20


define history_allow_tags = { "alt", "noalt", "rt", "rb", "art" }

define config.keymap["game_menu"] = ["K_BACKSPACE", "K_ESCAPE", "mouseup_3"]


style history_window is empty

style history_name is gui_label
style history_name_text is gui_label_text
style history_text is gui_text

style history_label is gui_label
style history_label_text is gui_label_text

style history_window:
    xfill True
    ysize gui.history_height    

style history_name:
    xpos 100
    font "fonts/Nunito/NunitoSans-Bold.ttf"
    xanchor gui.history_name_xalign
    ypos gui.history_name_ypos
    xsize gui.history_name_width

style history_name_text:
    font "fonts/Nunito/NunitoSans-Bold.ttf"
    min_width gui.history_name_width
    textalign gui.history_name_xalign
    outlines [ 
        (2.75, "#0000002a", 0, 3),
        (1.5, "#00000044", 0, 3),
        (0.75, "#0000005e", 0, 1.5),
        (1, "#0000003a", 0, 0),
    ]
    properties gui.text_properties("name", accent=True)
    size 28

style history_text:
    xpos 120
    ypos gui.history_text_ypos
    xanchor gui.history_text_xalign
    xsize gui.history_text_width
    min_width gui.history_text_width
    textalign gui.history_text_xalign
    layout ("subtitle" if gui.history_text_xalign else "tex")
    line_spacing 10
    outlines [ 
        (1, "#00000070", 0, 2.5),
        (1, "#00000070", 0, 2.75)
    ]
    color "#d8d8d8"
    size 28

style empty_history_text:
    xpos 120
    ypos gui.history_text_ypos
    xanchor gui.history_text_xalign
    xsize gui.history_text_width
    min_width gui.history_text_width
    textalign gui.history_text_xalign
    layout ("subtitle" if gui.history_text_xalign else "tex")
    line_spacing 10
    size 28

style history_label:
    xfill True

style history_label_text:
    xalign 0.5
    ypos 25
    font "fonts/Nunito/NunitoSans-SemiBold.ttf"
    outlines [ 
        (0.5, "#d8d8d8", 0, 2),
        (0.5, "#d8d8d8", 0, 3)
    ]
    color "#0F2A41"
    size 30

## Help screen #################################################################
##
## A screen that gives information about key and mouse bindings. It uses other
## screens (keyboard_help, mouse_help, and gamepad_help) to display the actual
## help.

screen help():

    tag menu

    default device = "keyboard"

    use game_menu(_("HELP"), scroll="viewport"):

        style_prefix "help"

        vbox:
            spacing 23

            hbox:

                textbutton _("Keyboard") action SetScreenVariable("device", "keyboard")
                textbutton _("Mouse") action SetScreenVariable("device", "mouse")

                if GamepadExists():
                    textbutton _("Gamepad") action SetScreenVariable("device", "gamepad")

            if device == "keyboard":
                use keyboard_help
            elif device == "mouse":
                use mouse_help
            elif device == "gamepad":
                use gamepad_help


screen keyboard_help():

    hbox:
        label _("Enter")
        text _("Advances dialogue and activates the interface.")

    hbox:
        label _("Space")
        text _("Advances dialogue without selecting choices.")

    hbox:
        label _("Arrow Keys")
        text _("Navigate the interface.")

    hbox:
        label _("Escape")
        text _("Accesses the game menu.")

    hbox:
        label _("Ctrl")
        text _("Skips dialogue while held down.")

    hbox:
        label _("Tab")
        text _("Toggles dialogue skipping.")

    hbox:
        label _("Page Up")
        text _("Rolls back to earlier dialogue.")

    hbox:
        label _("Page Down")
        text _("Rolls forward to later dialogue.")

    hbox:
        label "H"
        text _("Hides the user interface.")

    hbox:
        label "S"
        text _("Takes a screenshot.")

    hbox:
        label "V"
        text _("Toggles assistive {a=https://www.renpy.org/l/voicing}self-voicing{/a}.")

    hbox:
        label "Shift+A"
        text _("Opens the accessibility menu.")


screen mouse_help():

    hbox:
        label _("Left Click")
        text _("Advances dialogue and activates the interface.")

    hbox:
        label _("Middle Click")
        text _("Hides the user interface.")

    hbox:
        label _("Right Click")
        text _("Accesses the game menu.")

    hbox:
        label _("Mouse Wheel Up")
        text _("Rolls back to earlier dialogue.")

    hbox:
        label _("Mouse Wheel Down")
        text _("Rolls forward to later dialogue.")


screen gamepad_help():

    hbox:
        label _("Right Trigger\nA/Bottom Button")
        text _("Advances dialogue and activates the interface.")

    hbox:
        label _("Left Trigger\nLeft Shoulder")
        text _("Rolls back to earlier dialogue.")

    hbox:
        label _("Right Shoulder")
        text _("Rolls forward to later dialogue.")

    hbox:
        label _("D-Pad, Sticks")
        text _("Navigate the interface.")

    hbox:
        label _("Start, Guide, B/Right Button")
        text _("Accesses the game menu.")

    hbox:
        label _("Y/Top Button")
        text _("Hides the user interface.")

    textbutton _("Calibrate") action GamepadCalibrate()


style help_button is gui_button
style help_button_text is gui_button_text
style help_label is gui_label
style help_label_text is gui_label_text
style help_text is gui_text

style help_button:
    properties gui.button_properties("help_button")
    xmargin 12

style help_button_text:
    properties gui.text_properties("help_button")

style help_label:
    xsize 375
    right_padding 30

style help_label_text:
    size gui.text_size
    xalign 1.0
    textalign 1.0



################################################################################
## Additional screens
################################################################################


## Confirm screen ##############################################################
##
## The confirm screen is called when Ren'Py wants to ask the player a yes or no
## question.
##
## https://www.renpy.org/doc/html/screen_special.html#confirm

screen confirm(message, yes_action, no_action):

    tag menu
    
    modal True

    if "exit" in message.lower():
        add "gui/warnings/exit_warning.png" at center, fade_in_confirm
    elif "main menu" in message.lower():
        add "gui/warnings/title_warning.png" at center, fade_in_confirm
    elif "delete" in message.lower():
        add "gui/warnings/deletesave_warning.png" at center, fade_in_confirm
    elif "overwrite" in message.lower():
        add "gui/warnings/overwritesave_warning.png" at center, fade_in_confirm
    elif "load" in message.lower():
        add "gui/warnings/load_warning.png" at center, fade_in_confirm
    elif "skip" in message.lower():
        add "gui/warnings/skip_warning.png" at center, fade_in_confirm
    elif "replay" in message.lower():
        add "gui/warnings/end_memory.png" at center, fade_in_confirm
    else:
        add "gui/warnings/exit_warning.png" at center, fade_in_confirm
     
    hbox:
        ypos 0.5
        xpos 0.418
        spacing 140

        textbutton "Yes" action [
            If("exit" in message.lower() or "quit" in message.lower(), Call("fade_quit"), yes_action),
            If("delete" in message.lower() or "overwrite" in message.lower(), ShowMenu("save")),
            If("load" in message.lower(), ShowMenu("load")),            
            If(not ("delete" in message.lower() or "overwrite" in message.lower()), [Hide('confirm'), Return(False)]),            
        ] at fade_in_confirm:

            xminimum 120
            xmaximum 120
            yminimum 50
            ymaximum 50
            text_size 28
            text_font "fonts/Nunito/NunitoSans-Bold.ttf"

        textbutton "No" action [
            If("delete" in message.lower() or "overwrite" in message.lower(), ShowMenu("save")),
            Hide('confirm'),
            Return(False)
        ] at fade_in_confirm:
            xminimum 120
            xmaximum 120
            yminimum 50
            ymaximum 50
            text_size 28
            text_font "fonts/Nunito/NunitoSans-Bold.ttf"



style confirm_frame is gui_frame
style confirm_prompt is gui_prompt
style confirm_prompt_text is gui_prompt_text
style confirm_button is gui_medium_button
style confirm_button_text is gui_medium_button_text

style confirm_frame:
    background Frame([ "gui/confirm_frame.png", "gui/frame.png"], gui.confirm_frame_borders, tile=gui.frame_tile)
    xsize 1920
    ysize 250
    xalign 0.5
    yalign 0.5

style confirm_prompt_text:
    textalign 0.5
    layout "subtitle"

style confirm_button:
    properties gui.button_properties("confirm_button")

style confirm_button_text:
    font "fonts/poetry.ttf"
    properties gui.text_properties("confirm_button")
    


## Skip indicator screen #######################################################
##
## The skip_indicator screen is displayed to indicate that skipping is in
## progress.
##
## https://www.renpy.org/doc/html/screen_special.html#skip-indicator

screen skip_indicator():

    zorder 100
    style_prefix "skip"

    frame:

        hbox:
            spacing 9

            text _("Skipping")

            text "▸" at delayed_blink(0.0, 1.0) style "skip_triangle"
            text "▸" at delayed_blink(0.2, 1.0) style "skip_triangle"
            text "▸" at delayed_blink(0.4, 1.0) style "skip_triangle"


## This transform is used to blink the arrows one after another.
transform delayed_blink(delay, cycle):
    alpha .5

    pause delay

    block:
        linear .2 alpha 1.0
        pause .2
        linear .2 alpha 0.5
        pause (cycle - .4)
        repeat


style skip_frame is empty
style skip_text is gui_text
style skip_triangle is skip_text

style skip_frame:
    ypos gui.skip_ypos
    background Frame("gui/skip.png", gui.skip_frame_borders, tile=gui.frame_tile)
    padding gui.skip_frame_borders.padding

style skip_text:
    size gui.notify_text_size

style skip_triangle:
    ## We have to use a font that has the BLACK RIGHT-POINTING SMALL TRIANGLE
    ## glyph in it.
    font "DejaVuSans.ttf"


## Notify screen ###############################################################
##
## The notify screen is used to show the player a message. (For example, when
## the game is quicksaved or a screenshot has been taken.)
##
## https://www.renpy.org/doc/html/screen_special.html#notify-screen

screen notify(message):

    zorder 100
    style_prefix "notify"

    frame at notify_appear:
        text "[message!tq]"

    timer 3.25 action Hide('notify')

screen tlnotify(romaji_text, kanji_text, message_en):

    zorder 100
    style_prefix "notify"

    frame at notify_appear:
        vbox:
            text "{font=handwritting.ttf}[romaji_text] {/font}{font=fonts/Noto Sans JP/static/NotoSansJP-Bold.ttf}[kanji_text]{/font}":
                size 17
                color "#3074b4"
                
            text "[message_en!tq]":  
                font "handwritting.ttf"
                italic True
                size 15
                color "#FFFFFF"

    timer persistent.notify_duration*10 action Hide('tlnotify')

screen memory_notify(girlname, girlcolor, eventname):
    zorder 100
    style_prefix "notify"

    frame at notify_appear:
        background None
        padding (4, 5)
        vbox:
            spacing -12

            text "  A {b}memory{/b} finds its way back to me...":
                size 18
                font "poetry.ttf"
                color "#FFFFFF"
                xalign 0.5
                ypos -50
                outlines [ 
                    (2, "#00000050", 0, 3),
                    (1, "#00000050", 0, 3)
                ]

            text "  I see you clearly, {color=%s}{b}%s{/b}{/color}" % (girlcolor, girlname):
                size 18
                font "poetry.ttf"
                color "#FFFFFF"
                xalign 0.5
                ypos -50
                outlines [ 
                    (2.5, "#00000050", 0, 3),
                    (1.5, "#00000050", 0, 3)
                ]

            text "  ~ " + eventname + " ~":
                size 16
                font "poetry.ttf"
                color "#00ce9a"
                xalign 0.5
                ypos -50
                outlines [ 
                    (2, "#00000050", 0, 3),
                    (1, "#00000050", 0, 3)
                ]

    timer persistent.notify_duration*12 action Hide('memory_notify')

init python:
    def remember(girlname, girlcolor, eventname):
        if not _in_replay:
            renpy.music.set_volume(0.5, channel="sfxmemory")            
            renpy.play("sfx/menupage2.ogg", channel="sfxmemory")            
            renpy.show_screen("memory_notify", girlname=girlname, girlcolor=girlcolor, eventname=eventname)

transform notify_appear:
    on show:
        alpha 0
        linear 1.0 alpha 1.0
    on hide:
        linear .5 alpha 0.0


style notify_frame is empty
style notify_text is gui_text

style notify_frame:
    ypos gui.notify_ypos

    background Frame("gui/notify.png", gui.notify_frame_borders, tile=gui.frame_tile)
    padding gui.notify_frame_borders.padding

style notify_text:
    properties gui.text_properties("notify")
    size 15
    color "#ffffff"
    font "fonts/handwritting.ttf"
    outlines [ 
        (2, "#00000050", 0, 3),
        (1, "#00000050", 0, 3)
    ]



## NVL screen ##################################################################
##
## This screen is used for NVL-mode dialogue and menus.
##
## https://www.renpy.org/doc/html/screen_special.html#nvl

screen nvl(dialogue, items=None):

    if nvl_mode == "phone":
        use PhoneDialogue(dialogue, items)
    else:
        use poetry_nvl(dialogue, items)

        add SideImage() xalign 0 yalign 1.0


screen nvl_dialogue(dialogue):

    for d in dialogue:

        window:
            id d.window_id

            fixed:
                yfit gui.nvl_height is None

                if d.who is not None:

                    text d.who:
                        id d.who_id

                text d.what:
                    id d.what_id


## This controls the maximum number of NVL-mode entries that can be displayed at
## once.
define config.nvl_list_length = gui.nvl_list_length

style nvl_window is default
style nvl_entry is default

style nvl_label is say_label
style nvl_dialogue is say_dialogue

style nvl_button is button
style nvl_button_text is button_text

style nvl_window:
    xfill True
    yfill True
    background "gui/nvl.png"
    padding gui.nvl_borders.padding

style nvl_entry:
    xfill True
    ysize gui.nvl_height

style nvl_label:
    xpos gui.nvl_name_xpos
    size 50
    xanchor gui.nvl_name_xalign
    ypos gui.nvl_name_ypos
    yanchor 0.0
    xsize gui.nvl_name_width
    min_width gui.nvl_name_width
    text_align gui.nvl_name_xalign

style nvl_dialogue:
    xpos 0
    xanchor gui.nvl_text_xalign
    ypos gui.nvl_text_ypos-10
    xsize gui.nvl_text_width
    min_width gui.nvl_text_width
    text_align gui.nvl_text_xalign
    antialias True
    size 17
    outlines [ (0, "#00000000", 0, 0) ]
    font my_fontgroup
    layout ("subtitle" if gui.nvl_text_xalign else "tex")

style nvl_thought:
    xpos gui.nvl_thought_xpos
    xanchor gui.nvl_thought_xalign
    ypos gui.nvl_thought_ypos
    xsize gui.nvl_thought_width
    min_width gui.nvl_thought_width
    text_align gui.nvl_thought_xalign
    layout ("subtitle" if gui.nvl_text_xalign else "tex")

style nvl_button:
    properties gui.button_properties("nvl_button")
    xpos gui.nvl_button_xpos
    xanchor gui.nvl_button_xalign

style nvl_button_text:
    properties gui.button_text_properties("nvl_button")


screen poetry_nvl(dialogue, items=None):

    window:
        style "poetry_nvl_window"

        has vbox:
            spacing gui.nvl_spacing

        if gui.nvl_height:

            vpgrid:
                cols 1
                yinitial 1.0

                use poetry_nvl_dialogue(dialogue)

        else:

            use poetry_nvl_dialogue(dialogue)

        for i in items:
            textbutton i.caption:
                action i.action
                style "poetry_nvl_button"

    add SideImage() xalign 0.0 yalign 1.0

screen poetry_nvl_dialogue(dialogue):

    for d in dialogue:

        window:
            id d.window_id

            fixed:
                yfit gui.nvl_height is None

                if d.who is not None:

                    text d.who:
                        id d.who_id

                text d.what:
                    id d.what_id


style poetry_nvl_window is default
style poetry_nvl_entry is default

style poetry_nvl_label is say_label
style poetry_nvl_dialogue is say_dialogue

style poetry_nvl_button is button
style poetry_nvl_button_text is button_text

style poetry_nvl_window:
    xfill True
    yfill True
    font "fonts/poetry.ttf"
    background "gui/nvl.png"
    padding gui.nvl_borders.padding

style poetry_nvl_entry:
    xfill True
    ysize gui.nvl_height

style poetry_nvl_label:
    xpos gui.nvl_name_xpos
    xanchor gui.nvl_name_xalign
    ypos gui.nvl_name_ypos
    yanchor 0.0
    xsize gui.nvl_name_width
    min_width gui.nvl_name_width
    textalign gui.nvl_name_xalign

style poetry_nvl_dialogue:
    xpos 0.5
    xanchor 0.5
    ypos gui.nvl_text_ypos+90
    xsize gui.nvl_text_width+100
    min_width gui.nvl_text_width
    textalign 0.5
    font "fonts/poetry.ttf"
    size 30
    layout ("subtitle" if gui.nvl_text_xalign else "tex")

style poetry_nvl_thought:
    xpos gui.nvl_thought_xpos
    xanchor gui.nvl_thought_xalign
    ypos gui.nvl_thought_ypos
    xsize gui.nvl_thought_width
    min_width gui.nvl_thought_width
    textalign gui.nvl_thought_xalign
    layout ("subtitle" if gui.nvl_text_xalign else "tex")

style poetry_nvl_button:
    properties gui.button_properties("nvl_button")
    xpos gui.nvl_button_xpos
    xanchor gui.nvl_button_xalign

style poetry_nvl_button_text:
    properties gui.text_properties("nvl_button")


## Bubble screen ###############################################################
##
## The bubble screen is used to display dialogue to the player when using speech
## bubbles. The bubble screen takes the same parameters as the say screen, must
## create a displayable with the id of "what", and can create displayables with
## the "namebox", "who", and "window" ids.
##
## https://www.renpy.org/doc/html/bubble.html#bubble-screen

screen bubble(who, what):
    style_prefix "bubble"

    window:
        id "window"

        if who is not None:

            window:
                id "namebox"
                style "bubble_namebox"

                text who:
                    id "who"

        text what:
            id "what"

style bubble_window is empty
style bubble_namebox is empty
style bubble_who is default
style bubble_what is default

style bubble_window:
    xpadding 30
    top_padding 5
    bottom_padding 5

style bubble_namebox:
    xalign 0.5

style bubble_who:
    xalign 0.5
    textalign 0.5
    color "#000"

style bubble_what:
    align (0.5, 0.5)
    text_align 0.5
    layout "subtitle"
    color "#000"

define bubble.frame = Frame("gui/bubble.png", 55, 55, 55, 95)
define bubble.thoughtframe = Frame("gui/thoughtbubble.png", 55, 55, 55, 55)

define bubble.properties = {
    "bottom_left" : {
        "window_background" : Transform(bubble.frame, xzoom=1, yzoom=1),
        "window_bottom_padding" : 27,
    },

    "bottom_right" : {
        "window_background" : Transform(bubble.frame, xzoom=-1, yzoom=1),
        "window_bottom_padding" : 27,
    },

    "top_left" : {
        "window_background" : Transform(bubble.frame, xzoom=1, yzoom=-1),
        "window_top_padding" : 27,
    },

    "top_right" : {
        "window_background" : Transform(bubble.frame, xzoom=-1, yzoom=-1),
        "window_top_padding" : 27,
    },

    "thought" : {
        "window_background" : bubble.thoughtframe,
    }
}

define bubble.expand_area = {
    "bottom_left" : (0, 0, 0, 22),
    "bottom_right" : (0, 0, 0, 22),
    "top_left" : (0, 22, 0, 0),
    "top_right" : (0, 22, 0, 0),
    "thought" : (0, 0, 0, 0),
}



################################################################################
## Mobile Variants
################################################################################

style pref_vbox:
    variant "medium"
    xsize 675

## Since a mouse may not be present, we replace the quick menu with a version
## that uses fewer and bigger buttons that are easier to touch.
screen quick_menu():
    variant "touch"

    zorder 100

    if quick_menu:

        hbox:
            style_prefix "quick"

            xalign 0.5
            yalign 0.9999
            spacing -40

            imagebutton:
                idle "gui/phone/hide.png"
                hover "gui/phone/hide.png"
                action HideInterface()
                at Transform(xzoom=1.0, yzoom=1.0)
                ypos 15
            null width 100

            textbutton _("Back") action Rollback()
            textbutton _("Backlog") action ShowMenu('history')
            
            if has_pending_memories():
                textbutton _("Memories") action ShowMenu('memories') text_color "#00ce9a"
            else:
                textbutton _("Memories") action ShowMenu('memories')
            
            textbutton _("Skip") action Skip() alternate Skip(fast=True, confirm=True)
            textbutton _("Auto") action Preference("auto-forward", "toggle")
            textbutton _("Menu") action ShowMenu()
        if _in_replay:
            frame:
                style_prefix "check"
                xpos 1400
                ypos 750

                textbutton "END MEMORY" action EndReplay(confirm=True)

style window:
    variant "small"
    background "gui/phone/textbox.png"

style radio_button:
    variant "small"
    foreground "gui/phone/button/radio_[prefix_]foreground.png"

style check_button:
    variant "small"
    foreground "gui/phone/button/check_[prefix_]foreground.png"

style nvl_window:
    variant "small"
    background "gui/phone/nvl.png"

# style main_menu_frame:
#     variant "small"
#     background "gui/bloom_again.png"

style game_menu_outer_frame:
    variant "small"
    background "gui/phone/overlay/nav_overlay.png"

style game_menu_navigation_frame:
    variant "small"
    xsize 510

style game_menu_content_frame:
    variant "small"
    top_margin 0
    left_margin -75

style pref_vbox:
    variant "small"
    xsize 600

style bar:
    variant "small"
    ysize gui.bar_size
    left_bar Frame("gui/phone/bar/left.png", gui.bar_borders, tile=gui.bar_tile)
    right_bar Frame("gui/phone/bar/right.png", gui.bar_borders, tile=gui.bar_tile)

style vbar:
    variant "small"
    xsize gui.bar_size
    top_bar Frame("gui/phone/bar/top.png", gui.vbar_borders, tile=gui.bar_tile)
    bottom_bar Frame("gui/phone/bar/bottom.png", gui.vbar_borders, tile=gui.bar_tile)

style scrollbar:
    variant "small"
    ysize gui.scrollbar_size
    base_bar Frame("gui/phone/scrollbar/horizontal_[prefix_]bar.png", gui.scrollbar_borders, tile=gui.scrollbar_tile)
    thumb Frame("gui/phone/scrollbar/horizontal_[prefix_]thumb.png", gui.scrollbar_borders, tile=gui.scrollbar_tile)

style vscrollbar:
    variant "small"
    xsize gui.scrollbar_size
    base_bar Frame("gui/phone/scrollbar/vertical_[prefix_]bar.png", gui.vscrollbar_borders, tile=gui.scrollbar_tile)
    thumb Frame("gui/phone/scrollbar/vertical_[prefix_]thumb.png", gui.vscrollbar_borders, tile=gui.scrollbar_tile)

style slider:
    variant "small"
    ysize 36
    left_bar Frame("gui/phone/slider/left_[prefix_]bar.png", gui.slider_borders, tile=gui.slider_tile)
    right_bar Frame("gui/phone/slider/right_[prefix_]bar.png", gui.slider_borders, tile=gui.slider_tile)
    thumb "gui/phone/slider/horizontal_[prefix_]thumb.png"
    thumb_offset -20
    right_gutter 10
    left_gutter 10

style vslider:
    variant "small"
    xsize gui.slider_size
    base_bar Frame("gui/phone/slider/vertical_[prefix_]bar.png", gui.vslider_borders, tile=gui.slider_tile)
    thumb "gui/phone/slider/vertical_[prefix_]thumb.png"

style slider_vbox:
    variant "small"
    xsize None

style slider_slider:
    variant "small"
    xsize 450

style window:
    variant "small"
    xalign 0.5
    xfill True
    yalign gui.textbox_yalign
    ysize gui.textbox_height
    xsize 1100
    background Image("gui/textbox.png", xalign=0.5, yalign=1.0)

style namebox:
    variant "small"
    xpos gui.name_xpos
    xanchor gui.name_xalign
    xsize gui.namebox_width-65
    ypos gui.name_ypos+175
    ysize 7
    background Frame("gui/backloghbar.png", gui.namebox_borders, tile=gui.namebox_tile, xalign=gui.name_xalign)
    padding gui.namebox_borders.padding

style say_label:
    variant "small"
    outlines [ 
        (3, "#00000066", 0, 3),
        (2, "#00000066", 0, 3),
        (1, "#00000099", 0, 2),
        # (2.5, "#00000077", 0, 0),
    ]
    properties gui.text_properties("name", accent=True)
    xalign gui.name_xalign
    yalign 0.5
    xpos 140
    ypos -27
    size 43

style say_dialogue:
    variant "small"
    outlines [ 
        (3, "#00000033", 0, 3.5),
        (2, "#00000066", 0, 3),
        (1, "#00000099", 0, 2),
        (1.25, "#00000090", 0.2, 0),
    ]
    properties gui.text_properties("dialogue")
    xpos gui.dialogue_xpos-193
    xsize 1200
    ypos gui.dialogue_ypos+80
    size 35
    adjust_spacing False

## Styles for thoughts
style say_label_thoughts:
    variant "small"
    outlines [ 
        (3, "#00000066", 0, 3),
        (2, "#00000066", 0, 3),
        (1, "#00000099", 0, 2),
        # (2.5, "#00000077", 0, 0),
    ]
    properties gui.text_properties("name", accent=True)
    xalign gui.name_xalign
    yalign 0.5
    xpos 115
    ypos -27
    size 43
    italic True

style say_dialogue_thoughts:
    variant "small"
    outlines [ 
        (3, "#00000033", 0, 3.5),
        (2, "#00000066", 0, 3),
        (1, "#00000099", 0, 2),
        (1.25, "#00000090", 0.2, 0),
    ]
    properties gui.text_properties("dialogue")
    xpos gui.dialogue_xpos-193
    xsize 1200
    ypos gui.dialogue_ypos+80
    size 29
    color "#ffffffbb"
    italic True
    adjust_spacing False

###################################
############# KEYBINDS ############
###################################

screen keybinds():
    key ("m","M") action ShowMenu("memories_screen")
    key ("K_HOME") action ShowMenu("history")

init python:
    config.overlay_screens.append("keybinds")