<?php

class Venues extends CI_Controller 
{

	public function __construct()
	{
		parent::__construct();
		$this->load->model('users_model', 'users');
		$this->load->model('promoters_model', 'promoters');
		$this->load->model('venues_model', 'venues');
		$this->load->model('contacts_model', 'contacts');
		$this->load->model('logs_model', 'logs');
		$this->load->database('default');
	}
	
	// --------------------------------------------------------------------
	
	/**
	 * index
	 *
	 * @access	public
	 */
	function index()
	{
		redirect('venues/listing');
	}
	
	// --------------------------------------------------------------------
	
	/**
	 * listing
	 *
	 * @access	public
	 */
	function listing()
	{
		// prepare data for the template
		$data['page_title'] = 'Venues';
		$data['browser_title'] = $this->config->item('site_title') . ' - ' . $data['page_title'];
		
		// log this
		$this->logs->add($data['page_title']);
		
		// get all venues
		$data['venues'] = $this->venues->get_venues();
		
		// check users session
		$this->users->check_session(array('Admin','Super Admin','Admin2','Event','Finance'));
		
		$this->load->view('venues_listing', $data);
	}
	// --------------------------------------------------------------------
	
	/**
	 * view
	 *
	 * @access	public
	 */
	function view($venue_id = FALSE)
	{
		if (! $venue_id)
		{
			show_404();
		}
		
		// prepare data for the template
		$data['page_title'] = 'View Venue';
		$data['browser_title'] = $this->config->item('site_title') . ' - ' . $data['page_title'];
			
		// log this
		$this->logs->add($data['page_title']);
		
		// check user session
		$this->users->check_session(array('Admin','Super Admin','Admin2', 'Seller','Event','Reservation','Reservation-Admin','Finance'));
	
		// get venues
		$data['venues'] = $this->venues->get($venue_id);
		
		// get contacts
		$data['contacts'] = $this->contacts->get_contacts('venues', $venue_id);
		
		// display the page
		$this->load->view('venues_view', $data);
	}	
	// --------------------------------------------------------------------
	
	/**
	 * add
	 *
	 * @access	public
	 */
	function add()
	{
		// prepare data for the template
		$data['page_title'] = 'Add Venue';
		$data['browser_title'] = $this->config->item('site_title') . ' - ' . $data['page_title'];
			
		// log this
		$this->logs->add($data['page_title']);
		
		// check user session
		$this->users->check_session(array('Admin','Super Admin','Admin2','Event'));
			
		$config = array(
			array(
				'field'   => 'venue_username',
				'label'   => 'Username',
				'rules'   => 'trim|required|strip_tags|min_length[2]|max_length[50]|callback__username_check'
			),
			array(
				'field'   => 'venue_password',
				'label'   => 'Password',
				'rules'   => 'trim|required|strip_tags|min_length[2]|max_length[50]'
			),		
			array(
				'field'   => 'venue_name',
				'label'   => 'Venue Name',
				'rules'   => 'trim|required|strip_tags|min_length[2]|max_length[255]|callback__venuename_check'
			),
			array(
				'field'   => 'venue_address1',
				'label'   => 'Address 1',
				'rules'   => 'trim|strip_tags|min_length[1]|max_length[255]'
			),
			array(
				'field'   => 'venue_address2',
				'label'   => 'Address 2',
				'rules'   => 'trim|strip_tags|min_length[1]|max_length[255]'
			),
			array(
				'field'   => 'venue_city',
				'label'   => 'City',
				'rules'   => 'trim|strip_tags|min_length[1]|max_length[50]'
			),
			array(
				'field'   => 'venue_zip',
				'label'   => 'Zip Code',
				'rules'   => 'trim|strip_tags|min_length[1]|max_length[9]'
			),
			array(
				'field'   => 'venue_country',
				'label'   => 'Country',
				'rules'   => 'trim|strip_tags|min_length[1]|max_length[50]'
			),
			array(
				'field'   => 'venue_website',
				'label'   => 'Website',
				'rules'   => 'trim|strip_tags|min_length[2]|max_length[255]'
			),
			array(
				'field'   => 'venue_direction',
				'label'   => 'Direction',
				'rules'   => 'trim|strip_tags|min_length[2]|max_length[255]'
			),
		);
		
		$this->form_validation->set_rules($config); 
		
		$this->form_validation->set_error_delimiters('<small class="error_field">', '</small>');
		
		if ($this->form_validation->run() == FALSE)
		{
		
			// get contacts
			$data['contacts'] = $this->contacts->get_contacts_dropdown();
					
			// display the page
			$this->load->view('venues_add', $data);
			
		}
		else
		{
			$venues_id = $this->venues->add();
			$this->session->set_flashdata('message', assemble_message('Venue has been successfully added.  Please add the contact person.'));
			redirect('contacts/add/venues/'.$venues_id);
		}
	}
	
	// --------------------------------------------------------------------
	
	/**
	 * _venuename_check
	 *
	 * @access	private
	 */
	function _venuename_check($str)
	{	
		$venuename_exists = $this->venues->is_venuename_exist($str);
		if ($venuename_exists)
		{
			$this->form_validation->set_message('_venuename_check', "The Venue $str already exists.");
			return FALSE;
		}
		else
		{
			return TRUE;
		}
	}
	
	// --------------------------------------------------------------------
	
	/**
	 * edit
	 *
	 * @access	public
	 */
	function edit($venue_id = FALSE)
	{
		if (! $venue_id)
		{
			show_404();
		}
		
		// prepare data for the template
		$data['page_title'] = 'Edit Venue';
		$data['browser_title'] = $this->config->item('site_title') . ' - ' . $data['page_title'];
			
		// log this
		$this->logs->add($data['page_title']);
		
		// check user session
		$this->users->check_session(array('Admin','Super Admin','Admin2','Event'));
			
		$config = array(
			array(
				'field'   => 'venue_username',
				'label'   => 'Username',
				'rules'   => 'trim|required|strip_tags|min_length[2]|max_length[50]|callback__username_check_update'
			),
			array(
				'field'   => 'venue_password',
				'label'   => 'Password',
				'rules'   => 'trim|strip_tags|min_length[2]|max_length[50]'
			),
			array(
				'field'   => 'venue_name',
				'label'   => 'Venue Name',
				'rules'   => 'trim|required|strip_tags|min_length[2]|max_length[255]'
			),
			array(
				'field'   => 'venue_address1',
				'label'   => 'Address 1',
				'rules'   => 'trim|strip_tags|min_length[1]|max_length[255]'
			),
			array(
				'field'   => 'venue_address2',
				'label'   => 'Address 2',
				'rules'   => 'trim|strip_tags|min_length[1]|max_length[255]'
			),
			array(
				'field'   => 'venue_city',
				'label'   => 'City',
				'rules'   => 'trim|strip_tags|min_length[1]|max_length[50]'
			),
			array(
				'field'   => 'venue_zip',
				'label'   => 'Zip Code',
				'rules'   => 'trim|strip_tags|min_length[1]|max_length[9]'
			),
			array(
				'field'   => 'venue_country',
				'label'   => 'Country',
				'rules'   => 'trim|strip_tags|min_length[1]|max_length[50]'
			),
			array(
				'field'   => 'venue_website',
				'label'   => 'Website',
				'rules'   => 'trim|strip_tags|min_length[2]|max_length[255]'
			),
			array(
				'field'   => 'venue_direction',
				'label'   => 'Direction',
				'rules'   => 'trim|strip_tags|min_length[2]|max_length[255]'
			),
			array(
				'field'   => 'venue_active',
				'label'   => 'Status',
				'rules'   => 'trim|strip_tags|exact_length[1]'
			),
		);
		
		$this->form_validation->set_rules($config); 
		
		$this->form_validation->set_error_delimiters('<small class="error_field">', '</small>');
		
		if ($this->form_validation->run() == FALSE)
		{
		
			// get contacts
			//$data['contacts'] = $this->contacts->get_contacts_dropdown();
			
			// get data
			$data['venue'] = $this->venues->get($venue_id);
			
			// get all contacts
			$data['contacts'] = $this->contacts->get_contacts('venues', $venue_id);
	
			// display the page
			$this->load->view('venues_edit', $data);
		}
		else
		{
			$this->venues->edit($venue_id);
			$this->session->set_flashdata('message', assemble_message('Venue has been successfully modified'));
			//redirect('venues/listing');
			redirect('venues/view/'.$venue_id);
		}
	}
	
	// --------------------------------------------------------------------
	
	/**
	 * delete
	 *
	 * @access	public
	 */
	function delete($venue_id = FALSE)
	{	
		// log this
		$this->logs->add('Delete Venue');
		
		// check user session
		$this->users->check_session(array('Admin','Super Admin','Admin2','Event'));
		
		$this->venues->delete($venue_id);
		$this->session->set_flashdata('message', assemble_message('Venue has been successfully deleted'));
		redirect('venues/listing/');
	}
	
	function _username_check($str)
	{	
		$username_exists = $this->venues->is_username_exist($str);
		if ($username_exists)
		{
			$this->form_validation->set_message('_username_check', "Username $str is already taken");
			return FALSE;
		}
		else
		{
			return TRUE;
		}
	}	
	
	function _username_check_update($str)
	{	
		$username_exists = $this->venues->is_username_exist_update($str);
		if ($username_exists)
		{
			$this->form_validation->set_message('_username_check_update', "Username $str is already taken");
			return FALSE;
		}
		else
		{
			return TRUE;
		}
	}
	
}

/* End of file venues.php */
/* Location: ./application/controllers/venues.php */	
