/**
 * @file	pmpr201.cpp
 * @brief	PC-PR201nNX̓̎s܂
 */

#include "compiler.h"
#include "pmpr201.h"
#include <codecnv/codecnv.h>

static unsigned short jis_to_sjis(unsigned short jis)
{
	UINT8 j1 = (UINT8)(jis >> 8);
	UINT8 j2 = (UINT8)(jis & 0xFF);

	/* JIS X 0208 ̗L͈̓`FbN */
	if (j1 < 0x21 || j1 > 0x7E || j2 < 0x21 || j2 > 0x7E) {
		return 0;
	}

	/*
	 *   s1 = (j1 + 1)/2 + 0x70;  s1 >= 0xA0 Ȃ s1 += 0x40;
	 *   s2 = j2 + 0x1F;          j1 Ȃ s2 += 0x5E;
	 *   s2 >= 0x7F Ȃ s2++;
	 */
	UINT8 s1 = (UINT8)(((j1 + 1) >> 1) + 0x70);
	if (s1 >= 0xA0) {
		s1 = (UINT8)(s1 + 0x40);
	}

	UINT8 s2 = (UINT8)(j2 + 0x1F);
	if ((j1 & 1) == 0) {               /* j1 i悪j */
		s2 = (UINT8)(s2 + 0x5E);
	}
	if (s2 >= 0x7F) {
		s2 = (UINT8)(s2 + 1);
	}

	return (UINT16)((UINT16)s1 << 8 | s2);
}

static COLORREF ColorCodeToColorRef(UINT8 colorCode)
{
	switch (colorCode) {
	case 0:
		return RGB(0, 0, 0);
	case 1:
		return RGB(0, 0, 255);
	case 2:
		return RGB(255, 0, 0);
	case 3:
		return RGB(255, 0, 255);
	case 4:
		return RGB(0, 255, 0);
	case 5:
		return RGB(0, 255, 255);
	case 6:
		return RGB(255, 255, 0);
	case 7:
		return RGB(255, 255, 255);
	}
	return RGB(0, 0, 0);
}

typedef enum {
	COMMANDFUNC_RESULT_OK = 0, // R}hs
	COMMANDFUNC_RESULT_COMPLETELINE = 1, // sKv
	COMMANDFUNC_RESULT_COMPLETEPAGE = 2, // y[WKv
	COMMANDFUNC_RESULT_OVERFLOWLINE = 3, // Ō̃R}ȟŉsKv
	COMMANDFUNC_RESULT_OVERFLOWPAGE = 4, // Ō̃R}h̑Oŉy[WKv
	COMMANDFUNC_RESULT_RENDERLINE = 5, // s`Kvis͂ɖ߂邾j
} COMMANDFUNC_RESULT;

typedef COMMANDFUNC_RESULT (*PFNPRINTCMD_COMMANDFUNC)(void* param, const PRINTCMD_DATA& data, bool render);

static COMMANDFUNC_RESULT pmpr201_CommandVFU(void* param, const PRINTCMD_DATA& data, bool render) {
	CPrintPR201* owner = (CPrintPR201*)param;

	UINT16* vfuData = (UINT16*)(&(data.data[0]));
	int vfuDataLen = data.data.size() / 2;

	//if (vfuDataLen >= 2) {
	//	// VFU
	//	int topPos = -1;
	//	int bottomPos = -1;
	//	for (int i = 0; i < vfuDataLen; i++) {
	//		if (vfuData[i] & 0x40) {
	//			if (vfuData[i] & 0x01) {
	//				if (topPos == -1) {
	//					topPos = i;
	//				}
	//				else {
	//					bottomPos = i;
	//					break;
	//				}
	//			}
	//		}
	//	}
	//	if (topPos != -1 && bottomPos != -1) {
	//		float paperLengthInch = (float)(bottomPos - topPos) / 6;
	//		float marginTop = ((float)owner->m_heightPixel / owner->m_dpiY - paperLengthInch) / 2;
	//		if (marginTop < 0) marginTop = 0;
	//		owner->m_state.topMargin = marginTop;
	//	}
	//	else {
	//		owner->m_state.topMargin = 0;
	//	}
	//}
	//else {
	//	owner->m_state.topMargin = 0;
	//}

	return COMMANDFUNC_RESULT_OK;
}

static COMMANDFUNC_RESULT pmpr201_CommandCR(void* param, const PRINTCMD_DATA& data, bool render) {
	CPrintPR201* owner = (CPrintPR201*)param;
	owner->m_state.posX = 0;
	if (owner->CheckOverflowPage(0)) {
		owner->m_state.posY = 0;
		return owner->CheckOverflowPage(0) ? COMMANDFUNC_RESULT_COMPLETEPAGE : COMMANDFUNC_RESULT_OVERFLOWPAGE;
	}
	return COMMANDFUNC_RESULT_RENDERLINE;
}

static COMMANDFUNC_RESULT pmpr201_CommandLF(void* param, const PRINTCMD_DATA& data, bool render) {
	CPrintPR201* owner = (CPrintPR201*)param;
	if (owner->m_state.actualLineHeight == 0) {
		owner->m_state.actualLineHeight = owner->CalcLineHeight(); // sɉȂꍇ͌ݐݒ̍sƂ
	}
	owner->m_state.posY += owner->m_state.actualLineHeight;
	owner->m_state.actualLineHeight = 0;
	if (owner->CheckOverflowPage(0)) {
		owner->m_state.posY -= owner->m_heightPixel;
		return COMMANDFUNC_RESULT_COMPLETEPAGE;
	}
	return COMMANDFUNC_RESULT_COMPLETELINE;
}

static COMMANDFUNC_RESULT pmpr201_CommandHT(void* param, const PRINTCMD_DATA& data, bool render) {
	CPrintPR201* owner = (CPrintPR201*)param;
	// TODO: 
	return COMMANDFUNC_RESULT_OK;
}

static COMMANDFUNC_RESULT pmpr201_CommandVT(void* param, const PRINTCMD_DATA& data, bool render) {
	CPrintPR201* owner = (CPrintPR201*)param;
	float vtHeight = owner->CalcVFULineHeight() * 6; // TODO: Kv 6sɃ^u
	owner->m_state.posY = floor((owner->m_state.posY + vtHeight) / vtHeight) * vtHeight;
	if (owner->CheckOverflowPage(0)) {
		owner->m_state.posY = 0;
		return COMMANDFUNC_RESULT_COMPLETEPAGE;
	}
	return COMMANDFUNC_RESULT_COMPLETELINE;
}

static COMMANDFUNC_RESULT pmpr201_CommandFF(void* param, const PRINTCMD_DATA& data, bool render) {
	CPrintPR201* owner = (CPrintPR201*)param;
	owner->m_state.posY = 0;
	return COMMANDFUNC_RESULT_COMPLETEPAGE;
}

static COMMANDFUNC_RESULT pmpr201_CommandSO(void* param, const PRINTCMD_DATA& data, bool render) {
	CPrintPR201* owner = (CPrintPR201*)param;
	if (owner->m_state.codemode == PRINT_PR201_CODEMODE_8BIT) {
		owner->m_state.charScaleX = 2;
	}
	else {
		owner->m_state.charMode = PRINT_PR201_CHARMODE_7BIT_KATAKANA;
	}
	return COMMANDFUNC_RESULT_OK;
}

static COMMANDFUNC_RESULT pmpr201_CommandSI(void* param, const PRINTCMD_DATA& data, bool render) {
	CPrintPR201* owner = (CPrintPR201*)param;
	if (owner->m_state.codemode == PRINT_PR201_CODEMODE_8BIT) {
		owner->m_state.charScaleX = 1;
	}
	else {
		owner->m_state.charMode = PRINT_PR201_CHARMODE_7BIT_ASCII;
	}
	return COMMANDFUNC_RESULT_OK;
}

static COMMANDFUNC_RESULT pmpr201_CommandDC2(void* param, const PRINTCMD_DATA& data, bool render) {
	CPrintPR201* owner = (CPrintPR201*)param;
	if (owner->m_state.codemode == PRINT_PR201_CODEMODE_7BIT) {
		owner->m_state.charScaleX = 2;
	}
	return COMMANDFUNC_RESULT_OK;
}

static COMMANDFUNC_RESULT pmpr201_CommandDC4(void* param, const PRINTCMD_DATA& data, bool render) {
	CPrintPR201* owner = (CPrintPR201*)param;
	if (owner->m_state.codemode == PRINT_PR201_CODEMODE_7BIT) {
		owner->m_state.charScaleX = 1;
	}
	return COMMANDFUNC_RESULT_OK;
}

static COMMANDFUNC_RESULT pmpr201_CommandCAN(void* param, const PRINTCMD_DATA& data, bool render) {
	CPrintPR201* owner = (CPrintPR201*)param;
	// TODO: 
	return COMMANDFUNC_RESULT_OK;
}

static COMMANDFUNC_RESULT pmpr201_CommandDC1(void* param, const PRINTCMD_DATA& data, bool render) {
	CPrintPR201* owner = (CPrintPR201*)param;
	owner->m_state.isSelect = true;
	return COMMANDFUNC_RESULT_OK;
}

static COMMANDFUNC_RESULT pmpr201_CommandDC3(void* param, const PRINTCMD_DATA& data, bool render) {
	CPrintPR201* owner = (CPrintPR201*)param;
	owner->m_state.isSelect = false;
	return COMMANDFUNC_RESULT_OK;
}

static COMMANDFUNC_RESULT pmpr201_CommandUS(void* param, const PRINTCMD_DATA& data, bool render) {
	CPrintPR201* owner = (CPrintPR201*)param;
	int val = data.data[0];
	if (val < 16) {
		// VFUs TODO:
	}
	else {
		// nss
		int c = val - 16;
		if (c >= 1) {
			for (int i = 0; i < c; i++) {
				pmpr201_CommandLF(param, data, render);
			}
		}
		if (owner->CheckOverflowPage(0)) {
			owner->m_state.posY -= owner->m_heightPixel;
			return COMMANDFUNC_RESULT_COMPLETEPAGE;
		}
	}
	return COMMANDFUNC_RESULT_OK;
}



static COMMANDFUNC_RESULT pmpr201_CommandESCPrintMode(void* param, const PRINTCMD_DATA& data, bool render) {
	CPrintPR201* owner = (CPrintPR201*)param;
	owner->m_state.mode = (PRINT_PR201_PRINTMODE)data.cmd->cmd[1];
	owner->m_state.isKanji = owner->m_state.mode == PRINT_PR201_PRINTMODE_K || owner->m_state.mode == PRINT_PR201_PRINTMODE_t;
	if (render) owner->UpdateFont();
	return COMMANDFUNC_RESULT_OK;
}

static COMMANDFUNC_RESULT pmpr201_CommandESCHSPMode(void* param, const PRINTCMD_DATA& data, bool render) {
	CPrintPR201* owner = (CPrintPR201*)param;
	owner->m_state.hspMode = (PRINT_PR201_HSPMODE)data.data[0];
	return COMMANDFUNC_RESULT_OK;
}

static COMMANDFUNC_RESULT pmpr201_CommandESCCharMode(void* param, const PRINTCMD_DATA& data, bool render) {
	CPrintPR201* owner = (CPrintPR201*)param;
	owner->m_state.charMode = (PRINT_PR201_CHARMODE)data.cmd->cmd[1];
	return COMMANDFUNC_RESULT_OK;
}

static COMMANDFUNC_RESULT pmpr201_CommandESCScriptMode(void* param, const PRINTCMD_DATA& data, bool render) {
	CPrintPR201* owner = (CPrintPR201*)param;
	owner->m_state.scriptMode = (PRINT_PR201_SCRIPTMODE)data.data[0];
	return COMMANDFUNC_RESULT_OK;
}

static COMMANDFUNC_RESULT pmpr201_CommandESCDownloadCharMode(void* param, const PRINTCMD_DATA& data, bool render) {
	CPrintPR201* owner = (CPrintPR201*)param;
	owner->m_state.downloadCharMode = (data.cmd->cmd[1] == '+');
	return COMMANDFUNC_RESULT_OK;
}

static COMMANDFUNC_RESULT pmpr201_CommandESCe(void* param, const PRINTCMD_DATA& data, bool render) {
	CPrintPR201* owner = (CPrintPR201*)param;
	int scaleX = data.data[0] - '0';
	int scaleY = data.data[1] - '0';
	if (1 <= scaleX && scaleX <= 9) owner->m_state.charScaleX = scaleX;
	if (1 <= scaleY && scaleY <= 9) owner->m_state.charScaleY = scaleY;
	return COMMANDFUNC_RESULT_OK;
}

static COMMANDFUNC_RESULT pmpr201_CommandESCR(void* param, const PRINTCMD_DATA& data, bool render) {
	CPrintPR201* owner = (CPrintPR201*)param;
	int scaleX = data.data[0] - '0';
	int scaleY = data.data[1] - '0';
	if (1 <= scaleX && scaleX <= 9) owner->m_state.charScaleX = scaleX;
	if (1 <= scaleY && scaleY <= 9) owner->m_state.charScaleY = scaleY;
	return COMMANDFUNC_RESULT_OK;
}

static COMMANDFUNC_RESULT pmpr201_CommandESCBoldMode(void* param, const PRINTCMD_DATA& data, bool render) {
	CPrintPR201* owner = (CPrintPR201*)param;
	if (data.cmd->cmd[1] == '!') {
		owner->m_state.bold = true;
	}
	else if (data.cmd->cmd[1] == '\x22') { // "
		owner->m_state.bold = false;
	}
	if (render) owner->UpdateFont();
	return COMMANDFUNC_RESULT_OK;
}

static COMMANDFUNC_RESULT pmpr201_CommandESCLineSelect(void* param, const PRINTCMD_DATA& data, bool render) {
	CPrintPR201* owner = (CPrintPR201*)param;
	if (data.data[0] == '1') {
		owner->m_state.lineselect = 1;
	}
	else if (data.data[0] == '2') {
		owner->m_state.lineselect = 2;
	}
	return COMMANDFUNC_RESULT_OK;
}

static COMMANDFUNC_RESULT pmpr201_CommandESCLineEnable(void* param, const PRINTCMD_DATA& data, bool render) {
	CPrintPR201* owner = (CPrintPR201*)param;
	if (data.cmd->cmd[1] == 'X') {
		owner->m_state.lineenable = true;
	}
	else if (data.cmd->cmd[1] == 'Y') {
		owner->m_state.lineenable = false;
	}
	return COMMANDFUNC_RESULT_OK;
}

static COMMANDFUNC_RESULT pmpr201_CommandESCDotSpace(void* param, const PRINTCMD_DATA& data, bool render) {
	CPrintPR201* owner = (CPrintPR201*)param;
	owner->m_state.posX += owner->CalcDotPitchX() * data.cmd->cmd[1]; // * owner->m_state.charScaleX;
	if (owner->CheckOverflowLine(0)) {
		pmpr201_CommandLF(param, data, render);
		if (owner->CheckOverflowPage(0)) {
			owner->m_state.posY = 0;
			return owner->CheckOverflowPage(0) ? COMMANDFUNC_RESULT_COMPLETEPAGE : COMMANDFUNC_RESULT_OVERFLOWPAGE;
		}
		return COMMANDFUNC_RESULT_OVERFLOWLINE;
	}
	else {
		if (owner->CheckOverflowPage(0)) {
			owner->m_state.posY = 0;
			return owner->CheckOverflowPage(0) ? COMMANDFUNC_RESULT_COMPLETEPAGE : COMMANDFUNC_RESULT_OVERFLOWPAGE;
		}
	}
	return COMMANDFUNC_RESULT_OK;
}

static COMMANDFUNC_RESULT pmpr201_CommandESCGraph(void* param, const PRINTCMD_DATA& data, bool render) {
	CPrintPR201* owner = (CPrintPR201*)param;
	float pitchX = owner->CalcDotPitchX();
	float pitchY = owner->CalcDotPitchY();
	if (owner->CheckOverflowPage(max(owner->m_state.actualLineHeight, pitchY * 24))) {
		owner->m_state.posY = 0;
		return owner->CheckOverflowPage(max(owner->m_state.actualLineHeight, pitchY * 24)) ? COMMANDFUNC_RESULT_COMPLETEPAGE : COMMANDFUNC_RESULT_OVERFLOWPAGE;
	}
	const int length = (data.data[0] - '0') * 1000 + (data.data[1] - '0') * 100 + (data.data[2] - '0') * 10 + (data.data[3] - '0');
	if (owner->CheckOverflowLine(pitchX * length)) {
		owner->m_state.posX = 0;
		return COMMANDFUNC_RESULT_OVERFLOWLINE;
	}

	// FKvȂ̂Ńobt@ɂ߂ĂsIƂɕ`
	if (render) {
		int posx = (int)floor(owner->m_state.posX / pitchX + 0.5);
		int extstep = (int)floor(owner->m_state.charScaleX + 0.5);
		bool rep = false;
		const UINT8* buf = &(data.data[4]);
		switch (data.cmd->cmd[1]) {
		case 'V':
			rep = true;
		case 'S':
			for (int i = 0; i < length; i++) {
				int cx = posx + i * extstep;
				if (cx + extstep - 1 >= owner->m_colorbuf_w) break;
				UINT8 bitdata = buf[0];
				for (int cy = 0; cy < 16; cy += 2) {
					if (bitdata & 1) {
						int idx = cy * owner->m_colorbuf_w + cx;
						owner->m_colorbuf[idx] &= owner->m_state.color;
						if (extstep == 2) {
							owner->m_colorbuf[idx + 1] &= owner->m_state.color;
						}
					}
					bitdata >>= 1;
				}
				if (!rep) buf++;
			}
			break;

		case 'W':
			rep = true;
		case 'I':
			for (int i = 0; i < length; i++) {
				int cx = posx + i * extstep;
				if (cx + extstep - 1 >= owner->m_colorbuf_w) break;
				UINT16 bitdata = ((UINT16)buf[1] << 8) | buf[0];
				for (int cy = 0; cy < 16; cy++) {
					if (bitdata & 1) {
						int idx = cy * owner->m_colorbuf_w + cx;
						owner->m_colorbuf[idx] &= owner->m_state.color;
						if (extstep == 2) {
							owner->m_colorbuf[idx + 1] &= owner->m_state.color;
						}
					}
					bitdata >>= 1;
				}
				if (!rep) buf += 2;
			}
			break;

		case 'U':
			rep = true;
		case 'J':
			for (int i = 0; i < length; i++) {
				int cx = posx + i * extstep;
				if (cx + extstep - 1 >= owner->m_colorbuf_w) break;
				UINT32 bitdata = ((UINT32)buf[2] << 16) | ((UINT32)buf[1] << 8) | buf[0];
				for (int cy = 0; cy < 24; cy++) {
					if (bitdata & 1) {
						int idx = cy * owner->m_colorbuf_w + cx;
						owner->m_colorbuf[idx] &= owner->m_state.color;
						if (extstep == 2) {
							owner->m_colorbuf[idx + 1] &= owner->m_state.color;
						}
					}
					bitdata >>= 1;
				}
				if (!rep) buf += 3;
			}
			break;
		}
	}
	owner->m_state.posX += pitchX * length;

	owner->m_state.hasGraphic = true;
	owner->m_state.graphicPosY = owner->m_state.posY; // OtBbN`YWL

	return COMMANDFUNC_RESULT_OK;
}

static COMMANDFUNC_RESULT pmpr201_CommandESCF(void* param, const PRINTCMD_DATA& data, bool render) {
	CPrintPR201* owner = (CPrintPR201*)param;
	const int length = (data.data[0] - '0') * 1000 + (data.data[1] - '0') * 100 + (data.data[2] - '0') * 10 + (data.data[3] - '0');
	owner->m_state.posX = length * owner->CalcDotPitchX();
	return COMMANDFUNC_RESULT_OK;
}

static COMMANDFUNC_RESULT pmpr201_CommandESCCopyMode(void* param, const PRINTCMD_DATA& data, bool render) {
	CPrintPR201* owner = (CPrintPR201*)param;
	owner->m_state.copymode = true;
	return COMMANDFUNC_RESULT_OK;
}

static COMMANDFUNC_RESULT pmpr201_CommandESCNativeMode(void* param, const PRINTCMD_DATA& data, bool render) {
	CPrintPR201* owner = (CPrintPR201*)param;
	owner->m_state.copymode = false;
	return COMMANDFUNC_RESULT_OK;
}

static COMMANDFUNC_RESULT pmpr201_CommandESCLeftMargin(void* param, const PRINTCMD_DATA& data, bool render) {
	CPrintPR201* owner = (CPrintPR201*)param;
	owner->m_state.copymode = false;
	float charWidthInInch = owner->CalcCurrentLetterWidth() / owner->m_dpiX;
	const int margin = (data.data[0] - '0') * 100 + (data.data[1] - '0') * 10 + (data.data[2] - '0');
	owner->m_state.leftMargin = margin * charWidthInInch;

	return COMMANDFUNC_RESULT_OK;
}

static COMMANDFUNC_RESULT pmpr201_CommandESCRightMargin(void* param, const PRINTCMD_DATA& data, bool render) {
	CPrintPR201* owner = (CPrintPR201*)param;
	owner->m_state.copymode = false;
	float charWidthInInch = owner->CalcCurrentLetterWidth() / owner->m_dpiX;
	const int margin = (data.data[0] - '0') * 100 + (data.data[1] - '0') * 10 + (data.data[2] - '0');
	owner->m_state.rightMargin = margin * charWidthInInch;

	return COMMANDFUNC_RESULT_OK;
}

static COMMANDFUNC_RESULT pmpr201_CommandESCA(void* param, const PRINTCMD_DATA& data, bool render) {
	CPrintPR201* owner = (CPrintPR201*)param;
	owner->m_state.lpi = 6;
	return COMMANDFUNC_RESULT_OK;
}

static COMMANDFUNC_RESULT pmpr201_CommandESCB(void* param, const PRINTCMD_DATA& data, bool render) {
	CPrintPR201* owner = (CPrintPR201*)param;
	owner->m_state.lpi = 8;
	return COMMANDFUNC_RESULT_OK;
}

static COMMANDFUNC_RESULT pmpr201_CommandESCT(void* param, const PRINTCMD_DATA& data, bool render) {
	CPrintPR201* owner = (CPrintPR201*)param;
	const int value = (data.data[0] - '0') * 10 + (data.data[1] - '0');
	owner->m_state.lpi = 120.0f / value;
	return COMMANDFUNC_RESULT_OK;
}

static COMMANDFUNC_RESULT pmpr201_CommandESCC(void* param, const PRINTCMD_DATA& data, bool render) {
	CPrintPR201* owner = (CPrintPR201*)param;
	owner->m_state.color = (data.data[0] - '0') & 0x7;
	if (render) {
		owner->UpdateLinePen();
		SetTextColor(owner->m_hdc, ColorCodeToColorRef(owner->m_state.color));
	}
	return COMMANDFUNC_RESULT_OK;
}


static COMMANDFUNC_RESULT pmpr201_CommandFS04L(void* param, const PRINTCMD_DATA& data, bool render) {
	CPrintPR201* owner = (CPrintPR201*)param;
	owner->m_state.linep1 = data.data[0];
	owner->m_state.linep2 = data.data[1] - '0';
	owner->m_state.linep3 = data.data[2] - '0';
	if (render) owner->UpdateLinePen();
	return COMMANDFUNC_RESULT_OK;
}

static COMMANDFUNC_RESULT pmpr201_CommandFSw(void* param, const PRINTCMD_DATA& data, bool render) {
	CPrintPR201* owner = (CPrintPR201*)param;
	owner->m_state.dotsp_left = data.data[0];
	owner->m_state.dotsp_right = data.data[2];
	return COMMANDFUNC_RESULT_OK;
}

static COMMANDFUNC_RESULT pmpr201_CommandESCc1(void* param, const PRINTCMD_DATA& data, bool render) {
	CPrintPR201* owner = (CPrintPR201*)param;
	owner->m_state.SetDefault();
	return COMMANDFUNC_RESULT_OK;
}

static COMMANDFUNC_RESULT pmpr201_PutChar(void* param, const PRINTCMD_DATA& data, bool render) {
	CPrintPR201* owner = (CPrintPR201*)param;
	float charWidth = owner->CalcCurrentLetterWidth();
	TCHAR buf[3] = { 0 };
	UINT8 th[3] = { 0 };
	if (owner->m_state.isKanji) {
		if (data.data[0] == 0) {
			// 1oCg
			th[0] = data.data[1];
		}
		else {
			UINT16 sjis = jis_to_sjis(data.data[0] << 8 | data.data[1]);
			th[0] = sjis >> 8;
			th[1] = sjis & 0xff;
			charWidth *= 2;
		}
	}
	else {
		th[0] = data.data[0];
	}
	UINT16 thw[2];
	codecnv_sjistoucs2(thw, 1, (const char*)th, 2);
	buf[0] = (TCHAR)thw[0];
	buf[1] = (TCHAR)thw[1];

	float pitchX = owner->CalcDotPitchX();
	float pitchY = owner->CalcDotPitchY();
	owner->m_state.posX += owner->m_state.dotsp_left * pitchX;

	float lineHeight = owner->CalcActualLineHeight();
	if (owner->CheckOverflowPage(lineHeight)) {
		owner->m_state.posY = 0;
		return owner->CheckOverflowPage(lineHeight) ? COMMANDFUNC_RESULT_COMPLETEPAGE : COMMANDFUNC_RESULT_OVERFLOWPAGE;
	}
	float scaleX = owner->m_state.charScaleX;
	float scaleY = owner->m_state.charScaleY;
	if (owner->m_state.mode == PRINT_PR201_PRINTMODE_Q) { // RfX
		scaleX *= 0.6;
	}
	else if (owner->m_state.mode == PRINT_PR201_PRINTMODE_E) { // G[g
		scaleX *= 0.8;
	}
	else if (owner->m_state.mode == PRINT_PR201_PRINTMODE_P) { // v|[Vi XXX; {͎̕
		scaleX *= 0.9;
	}
	charWidth *= scaleX;
	if (owner->CheckOverflowLine(charWidth)) {
		owner->m_state.posX = 0;
		return COMMANDFUNC_RESULT_OVERFLOWLINE;
	}
	if (owner->m_state.isSelect) {
		// F͂ŕ`
		if (render) {
			int yOfsEx = owner->m_state.mode == PRINT_PR201_PRINTMODE_t ? (owner->m_dpiY / owner->m_state.lpi) : 0;
			if (scaleX != 1 || scaleY != 1) {
				XFORM xf = { 0 };
				xf.eM11 = scaleX;  // X{
				xf.eM22 = scaleY;  // Y{
				xf.eM12 = xf.eM21 = 0.0f;
				xf.eDx = 0.0f;
				xf.eDy = 0.0f;

				SetWorldTransform(owner->m_hdc, &xf);
				TextOut(owner->m_hdc, (owner->m_offsetXPixel + owner->m_state.leftMargin * owner->m_dpiX + owner->m_state.posX) / xf.eM11, (owner->m_offsetYPixel + owner->m_state.topMargin * owner->m_dpiY + owner->m_state.posY + yOfsEx) / xf.eM22, buf, 1);
				ModifyWorldTransform(owner->m_hdc, nullptr, MWT_IDENTITY);
			}
			else {
				TextOut(owner->m_hdc, owner->m_offsetXPixel + owner->m_state.leftMargin * owner->m_dpiX + owner->m_state.posX, owner->m_offsetYPixel + owner->m_state.topMargin * owner->m_dpiY + owner->m_state.posY + yOfsEx, buf, 1);
			}
			if (owner->m_state.lineenable) {
				float lineBeginX = owner->m_state.posX - owner->m_state.dotsp_left * pitchX;
				float lineEndX = owner->m_state.posX + charWidth + owner->m_state.dotsp_right * pitchX;
				const float dotPitch = 1.0f / 160;
				int dotsize = (float)pitchX;
				dotsize *= owner->m_state.linep3 / 2;
				HPEN hOldPen = (HPEN)SelectObject(owner->m_hdc, owner->m_gdiobj.penline);
				int ypos = 0;
				if (owner->m_state.lineselect == 1) {
					// 
					ypos = owner->m_offsetYPixel + owner->m_state.topMargin * owner->m_dpiY + owner->m_dpiY / owner->m_state.lpi - dotsize / 2;
				}
				else if (owner->m_state.lineselect == 2) {
					// 
					ypos = owner->m_offsetYPixel + owner->m_state.topMargin * owner->m_dpiY + dotsize / 2;
				}
				MoveToEx(owner->m_hdc, owner->m_offsetXPixel + owner->m_state.leftMargin * owner->m_dpiX + lineBeginX, ypos, NULL);
				LineTo(owner->m_hdc, owner->m_offsetXPixel + owner->m_state.leftMargin * owner->m_dpiX + lineEndX, ypos);
				SelectObject(owner->m_hdc, hOldPen);
			}
		}

		owner->m_state.actualLineHeight = lineHeight;
	}
	owner->m_state.posX += charWidth;
	owner->m_state.posY += owner->m_state.dotsp_right * pitchX;
	return COMMANDFUNC_RESULT_OK;
}

static PRINTCMD_DEFINE s_commandTablePR201[] = {
	// {R[h
	PRINTCMD_DEFINE_FIXEDLEN("\x01", 0, NULL), // SOH
	PRINTCMD_DEFINE_FIXEDLEN("\x02", 0, NULL), // STX
	PRINTCMD_DEFINE_FIXEDLEN("\x03", 0, NULL), // ETX
	PRINTCMD_DEFINE_FIXEDLEN("\x04", 0, NULL), // EOT
	PRINTCMD_DEFINE_FIXEDLEN("\x05", 0, NULL), // ENQ
	PRINTCMD_DEFINE_FIXEDLEN("\x06", 0, NULL), // ACK
	PRINTCMD_DEFINE_FIXEDLEN("\x07", 0, NULL), // BEL
	PRINTCMD_DEFINE_FIXEDLEN("\x08", 0, NULL), // BS
	PRINTCMD_DEFINE_FIXEDLEN("\x09", 0, pmpr201_CommandHT), // HT
	PRINTCMD_DEFINE_FIXEDLEN("\x0a", 0, pmpr201_CommandLF), // LF
	PRINTCMD_DEFINE_FIXEDLEN("\x0b", 0, pmpr201_CommandVT), // VT
	PRINTCMD_DEFINE_FIXEDLEN("\x0c", 0, pmpr201_CommandFF), // FF
	PRINTCMD_DEFINE_FIXEDLEN("\x0d", 0, pmpr201_CommandCR), // CR
	PRINTCMD_DEFINE_FIXEDLEN("\x0e", 0, pmpr201_CommandSO), // SO
	PRINTCMD_DEFINE_FIXEDLEN("\x0f", 0, pmpr201_CommandSI), // SI
	PRINTCMD_DEFINE_FIXEDLEN("\x10", 0, NULL), // DLE
	PRINTCMD_DEFINE_FIXEDLEN("\x11", 0, pmpr201_CommandDC1), // DC1
	PRINTCMD_DEFINE_FIXEDLEN("\x12", 0, pmpr201_CommandDC2), // DC2
	PRINTCMD_DEFINE_FIXEDLEN("\x13", 0, pmpr201_CommandDC3), // DC3
	PRINTCMD_DEFINE_FIXEDLEN("\x14", 0, pmpr201_CommandDC4), // DC4
	PRINTCMD_DEFINE_FIXEDLEN("\x15", 0, NULL), // NAK
	PRINTCMD_DEFINE_FIXEDLEN("\x16", 0, NULL), // SYN
	PRINTCMD_DEFINE_FIXEDLEN("\x17", 0, NULL), // ETB
	PRINTCMD_DEFINE_FIXEDLEN("\x18", 0, pmpr201_CommandCAN), // CAN
	PRINTCMD_DEFINE_FIXEDLEN("\x19", 0, NULL), // EM
	PRINTCMD_DEFINE_FIXEDLEN("\x1a", 0, NULL), // SUB
	//PRINTCMD_DEFINE_FIXEDLEN("\x1b", 0, NULL), // ESC
	//PRINTCMD_DEFINE_FIXEDLEN("\x1c", 0, NULL), // FS
	PRINTCMD_DEFINE_TERMINATOR("\x1d", '\x1e', pmpr201_CommandVFU), // GS VFUݒ@RSŉ
	PRINTCMD_DEFINE_FIXEDLEN("\x1e", 0, NULL), // RS
	PRINTCMD_DEFINE_FIXEDLEN("\x1f", 1, pmpr201_CommandUS), // US

	// gR[h ESC
	PRINTCMD_DEFINE_FIXEDLEN("\x1b""N", 0, pmpr201_CommandESCPrintMode), // HSpCJ[h
	PRINTCMD_DEFINE_FIXEDLEN("\x1b""H", 0, pmpr201_CommandESCPrintMode), // HDpCJ[h
	PRINTCMD_DEFINE_FIXEDLEN("\x1b""Q", 0, pmpr201_CommandESCPrintMode), // RfX[h
	PRINTCMD_DEFINE_FIXEDLEN("\x1b""E", 0, pmpr201_CommandESCPrintMode), // G[g[h
	PRINTCMD_DEFINE_FIXEDLEN("\x1b""P", 0, pmpr201_CommandESCPrintMode), // v|[Vi[h
	PRINTCMD_DEFINE_FIXEDLEN("\x1b""K", 0, pmpr201_CommandESCPrintMode), // ij[h
	PRINTCMD_DEFINE_FIXEDLEN("\x1b""t", 0, pmpr201_CommandESCPrintMode), // icj[h

	PRINTCMD_DEFINE_FIXEDLEN("\x1b""n", 1, pmpr201_CommandESCHSPMode), // HSpCJ[h

	PRINTCMD_DEFINE_FIXEDLEN("\x1b""$", 0, pmpr201_CommandESCCharMode), // ̓^Ji[hݒ(8bit) p[hݒ(7bit)
	PRINTCMD_DEFINE_FIXEDLEN("\x1b""&", 0, pmpr201_CommandESCCharMode), // Ђ炪ȃ[hݒ
	PRINTCMD_DEFINE_FIXEDLEN("\x1b""#", 0, pmpr201_CommandESCCharMode), // CGOtBbN[hݒ(7bit)

	PRINTCMD_DEFINE_FIXEDLEN("\x1b""s", 1, pmpr201_CommandESCScriptMode), // XNvg[hݒ

	PRINTCMD_DEFINE_FIXEDLEN("\x1b""+", 3 + 72, NULL), // O24x24o^
	PRINTCMD_DEFINE_FIXEDLEN("\x1b""+", 3 + 32, NULL), // O16x16o^
	PRINTCMD_DEFINE_FIXEDLEN_V("\x1b""l", 6, NULL), // _E[h̓o^@GȂ̂VariableLengthCallbackgp
	PRINTCMD_DEFINE_FIXEDLEN("\x1b""l+", 0, pmpr201_CommandESCDownloadCharMode), // _E[h
	PRINTCMD_DEFINE_FIXEDLEN("\x1b""l-", 0, pmpr201_CommandESCDownloadCharMode), // v^
	PRINTCMD_DEFINE_FIXEDLEN("\x1b""l0", 0, NULL), // _E[hNA

	PRINTCMD_DEFINE_FIXEDLEN("\x1b""e", 2, pmpr201_CommandESCe), // XP[ݒ
	PRINTCMD_DEFINE_FIXEDLEN_V("\x1b""R", 4, pmpr201_CommandESCR), // LN^s[g@GȂ̂VariableLengthCallbackgp

	PRINTCMD_DEFINE_FIXEDLEN("\x1b""!", 0, pmpr201_CommandESCBoldMode), // [hݒ
	PRINTCMD_DEFINE_FIXEDLEN("\x1b""\x22", 0, pmpr201_CommandESCBoldMode), // [h

	PRINTCMD_DEFINE_FIXEDLEN("\x1b""_", 1, pmpr201_CommandESCLineSelect), // I[o[CEA_[CI
	PRINTCMD_DEFINE_FIXEDLEN("\x1b""X", 0, pmpr201_CommandESCLineEnable), // CL
	PRINTCMD_DEFINE_FIXEDLEN("\x1b""Y", 0, pmpr201_CommandESCLineEnable), // C

	PRINTCMD_DEFINE_FIXEDLEN("\x1b""d", 1, NULL), // htg[hij

	PRINTCMD_DEFINE_FIXEDLEN("\x1b""m", 1, pmpr201_CommandESCDotSpace), // htg[hij

	PRINTCMD_DEFINE_FIXEDLEN("\x1b""\x00", 0, pmpr201_CommandESCDotSpace), // hbgXy[X 0dot
	PRINTCMD_DEFINE_FIXEDLEN("\x1b""\x01", 0, pmpr201_CommandESCDotSpace), // hbgXy[X 1dot
	PRINTCMD_DEFINE_FIXEDLEN("\x1b""\x02", 0, pmpr201_CommandESCDotSpace), // hbgXy[X 2dot
	PRINTCMD_DEFINE_FIXEDLEN("\x1b""\x03", 0, pmpr201_CommandESCDotSpace), // hbgXy[X 3dot
	PRINTCMD_DEFINE_FIXEDLEN("\x1b""\x04", 0, pmpr201_CommandESCDotSpace), // hbgXy[X 4dot
	PRINTCMD_DEFINE_FIXEDLEN("\x1b""\x05", 0, pmpr201_CommandESCDotSpace), // hbgXy[X 5dot
	PRINTCMD_DEFINE_FIXEDLEN("\x1b""\x06", 0, pmpr201_CommandESCDotSpace), // hbgXy[X 6dot
	PRINTCMD_DEFINE_FIXEDLEN("\x1b""\x07", 0, pmpr201_CommandESCDotSpace), // hbgXy[X 7dot
	PRINTCMD_DEFINE_FIXEDLEN("\x1b""\x08", 0, pmpr201_CommandESCDotSpace), // hbgXy[X 8dot

	PRINTCMD_DEFINE_LENFIELD("\x1b""S", 4, 1, pmpr201_CommandESCGraph), // 8bit OtBbN
	PRINTCMD_DEFINE_LENFIELD("\x1b""I", 4, 2, pmpr201_CommandESCGraph), // 16bit OtBbN
	PRINTCMD_DEFINE_LENFIELD("\x1b""J", 4, 3, pmpr201_CommandESCGraph), // 24bit OtBbN
	PRINTCMD_DEFINE_FIXEDLEN("\x1b""V", 5, pmpr201_CommandESCGraph), // 8bit OtBbN s[g
	PRINTCMD_DEFINE_FIXEDLEN("\x1b""W", 6, pmpr201_CommandESCGraph), // 16bit OtBbN s[g
	PRINTCMD_DEFINE_FIXEDLEN("\x1b""U", 7, pmpr201_CommandESCGraph), // 24bit OtBbN s[g

	PRINTCMD_DEFINE_FIXEDLEN("\x1b""F", 4, pmpr201_CommandESCF), // hbgAhX

	PRINTCMD_DEFINE_FIXEDLEN("\x1b""D", 0, pmpr201_CommandESCCopyMode), // Rs[[hɐݒ
	PRINTCMD_DEFINE_FIXEDLEN("\x1b""M", 0, pmpr201_CommandESCNativeMode), // lCeBu[hɐݒ

	PRINTCMD_DEFINE_FIXEDLEN("\x1b"">", 0, NULL), // Е󎚃[h
	PRINTCMD_DEFINE_FIXEDLEN("\x1b""]", 0, NULL), // 󎚃[h

	PRINTCMD_DEFINE_TERMINATOR("\x1b""(", '.', NULL), // ^uݒ
	PRINTCMD_DEFINE_TERMINATOR("\x1b"")", '.', NULL), // ^uNA
	PRINTCMD_DEFINE_FIXEDLEN("\x1b""2", 0, NULL), // ^uSNA

	PRINTCMD_DEFINE_TERMINATOR("\x1b""v", '.', NULL), // ȈVFU
		
	PRINTCMD_DEFINE_FIXEDLEN("\x1b""L", 3, pmpr201_CommandESCLeftMargin), // tg}[W
	PRINTCMD_DEFINE_FIXEDLEN("\x1b""/", 3, pmpr201_CommandESCRightMargin), // Cg}[W

	PRINTCMD_DEFINE_FIXEDLEN("\x1b""h", 1, NULL), // pc
	PRINTCMD_DEFINE_FIXEDLEN("\x1b""q", 0, NULL), // pgc
	PRINTCMD_DEFINE_FIXEDLEN("\x1b""h1", 0, NULL), // pc
	PRINTCMD_DEFINE_FIXEDLEN("\x1b""h1", 0, NULL), // pc
	PRINTCMD_DEFINE_FIXEDLEN("\x1b""h1", 0, NULL), // pc
	PRINTCMD_DEFINE_FIXEDLEN("\x1b""h1", 0, NULL), // pc

	PRINTCMD_DEFINE_FIXEDLEN("\x1b""A", 0, pmpr201_CommandESCA), // 1/6C`s[h
	PRINTCMD_DEFINE_FIXEDLEN("\x1b""B", 0, pmpr201_CommandESCB), // 1/8C`s[h
	PRINTCMD_DEFINE_FIXEDLEN("\x1b""T", 2, pmpr201_CommandESCT), // n/120C`s[h

	PRINTCMD_DEFINE_FIXEDLEN("\x1b""f", 0, NULL), // s[h
	PRINTCMD_DEFINE_FIXEDLEN("\x1b""r", 0, NULL), // ts[h

	PRINTCMD_DEFINE_FIXEDLEN("\x1b""a", 0, NULL), // SroSz
	PRINTCMD_DEFINE_FIXEDLEN("\x1b""b", 0, NULL), // Sro

	PRINTCMD_DEFINE_FIXEDLEN("\x1b""f", 4, NULL), // zbp؂ւ
		
	PRINTCMD_DEFINE_FIXEDLEN("\x1b""O", 1, NULL), // ANKtHg

	PRINTCMD_DEFINE_FIXEDLEN("\x1b""C", 1, pmpr201_CommandESCC), // J[؂ւ

	PRINTCMD_DEFINE_FIXEDLEN("\x1b""c1", 0, pmpr201_CommandESCc1), // Zbg

	// gR[h FS
	PRINTCMD_DEFINE_FIXEDLEN("\x1c""04L", 3, pmpr201_CommandFS04L), // Cw
	PRINTCMD_DEFINE_FIXEDLEN("\x1c""w", 4, pmpr201_CommandFSw), // ŒhbgXy[X

	PRINTCMD_DEFINE_TERMINATOR("\x1c""m", '.', NULL), // CӔ{

	PRINTCMD_DEFINE_FIXEDLEN("\x1c""P", 0, NULL), // k̑g
	PRINTCMD_DEFINE_FIXEDLEN("\x1c""A", 0, NULL), // TCY10.5pt@3/20
	PRINTCMD_DEFINE_FIXEDLEN("\x1c""B", 0, NULL), // TCY10.5pt@1/5
	PRINTCMD_DEFINE_FIXEDLEN("\x1c""C", 0, NULL), // TCY9.5pt@1/6
	PRINTCMD_DEFINE_FIXEDLEN("\x1c""D", 0, NULL), // TCY9.5pt@2/15
	PRINTCMD_DEFINE_FIXEDLEN("\x1c""F", 0, NULL), // TCY7pt@1/10
	PRINTCMD_DEFINE_FIXEDLEN("\x1c""G", 0, NULL), // TCY12pt@1/6
	PRINTCMD_DEFINE_TERMINATOR("\x1c""p", '.', NULL), // 
	PRINTCMD_DEFINE_FIXEDLEN("\x1c""04S", 3, NULL), // TCY
	PRINTCMD_DEFINE_FIXEDLEN("\x1c""06F", 5, NULL), // tHgI
	PRINTCMD_DEFINE_TERMINATOR("\x1c""c", '.', NULL), // C

};

// Xe[^XɂĒςϒR}h
static UINT32 pmpr201_VariableLengthCallback(void* param, const PRINTCMD_DEFINE& cmddef, const std::vector<UINT8>& curBuffer) {
	CPrintPR201* owner = (CPrintPR201*)param;

	if (cmddef.cmd[0] == '\x1b') {
		if (cmddef.cmd[1] == 'l') {
			// _E[h̓o^
			switch (owner->m_state.mode) {
			case PRINT_PR201_PRINTMODE_N:
			case PRINT_PR201_PRINTMODE_H:
				return 54 - 4; // Ƀf[^4byteǂł̂ł̕
			case PRINT_PR201_PRINTMODE_Q:
				return 42 - 4; // Ƀf[^4byteǂł̂ł̕
			case PRINT_PR201_PRINTMODE_E:
				return 45 - 4; // Ƀf[^4byteǂł̂ł̕
				break;
			case PRINT_PR201_PRINTMODE_P:
			{
				const int w3 = (((int)curBuffer[4] - '0') * 10 + ((int)curBuffer[5] - '0')) * 3;
				if (w3 < 0 || 48 < w3) return PRINTCMD_CALLBACK_RESULT_INVALID; // f[^TCY͍ő48byte
				return w3; // v|[Vȉꍇ̓f[^Ö̗悪4bytêłOK
			}
			}
		}
		else if (cmddef.cmd[1] == 'R') {
			// LN^s[g
			return owner->m_state.isKanji ? 1 : 0; // [h1byte
		}
	}
	return 0;
}

// o^OR}h̏
static PRINTCMD_CALLBACK_RESULT pmpr201_CommandParseCallback(void* param, const std::vector<UINT8>& curBuffer) {
	CPrintPR201* owner = (CPrintPR201*)param;

	// 0xff͖Ƃ
	if (curBuffer[0] == 0xff) return PRINTCMD_CALLBACK_RESULT_INVALID;

	// ʏ1byte [hȂ2byte
	if (owner->m_state.isKanji) {
		if (curBuffer.size() == 1) return PRINTCMD_CALLBACK_RESULT_CONTINUE;
	}

	return PRINTCMD_CALLBACK_RESULT_COMPLETE;
}

/**
 * RXgN^
 */
CPrintPR201::CPrintPR201()
	: CPrintBase()
	, m_state()
	, m_renderstate()
	, m_colorbuf(NULL)
	, m_colorbuf_w(0)
	, m_colorbuf_h(0)
	, m_cmdIndex(0)
{
	m_parser = new PrinterCommandParser(s_commandTablePR201, NELEMENTS(s_commandTablePR201), pmpr201_CommandParseCallback, pmpr201_VariableLengthCallback, this);
	m_state.SetDefault();
	m_renderstate.SetDefault();
}

/**
 * fXgN^
 */
CPrintPR201::~CPrintPR201()
{
	EndPrint();

	delete m_parser;
	if (m_colorbuf) {
		delete[] m_colorbuf;
		m_colorbuf = NULL;
		m_colorbuf_w = 0;
		m_colorbuf_h = 0;
	}
}

void CPrintPR201::StartPrint(HDC hdc, int offsetXPixel, int offsetYPixel, int widthPixel, int heightPixel, float dpiX, float dpiY, float dotscale, bool rectdot)
{
	m_state.SetDefault();
	m_cmdIndex = 0;
	m_lastNewPage = false;

	CPrintBase::StartPrint(hdc, offsetXPixel, offsetYPixel, widthPixel, heightPixel, dpiX, dpiY, dotscale, rectdot);

	const float dotPitch = CalcDotPitchX();
	m_colorbuf_w = (int)ceil(widthPixel / dotPitch);
	m_colorbuf_h = 24;
	m_colorbuf = new UINT8[m_colorbuf_w * m_colorbuf_h];
	memset(m_colorbuf, 0xff, m_colorbuf_w * m_colorbuf_h);

	// GDIIuWFNgp
	memset(&m_gdiobj, 0, sizeof(m_gdiobj));

	const int fontPx = MulDiv(12, m_dpiY, 72); // 12pt
	LOGFONT lf = { 0 };
	lf.lfHeight = -fontPx;
	lf.lfWeight = FW_NORMAL;
	lf.lfCharSet = DEFAULT_CHARSET;
	lf.lfPitchAndFamily = FIXED_PITCH | FF_MODERN;
	lstrcpyW(lf.lfFaceName, _T("MS Mincho"));
	m_gdiobj.fontbase = CreateFontIndirect(&lf);

	lf.lfEscapement = 900;
	lf.lfOrientation = 900;
	m_gdiobj.fontrot90 = CreateFontIndirect(&lf);

	lf.lfEscapement = 0;
	lf.lfOrientation = 0;
	lf.lfWeight = FW_BOLD;
	m_gdiobj.fontbold = CreateFontIndirect(&lf);
	lf.lfEscapement = 900;
	lf.lfOrientation = 900;
	m_gdiobj.fontboldrot90 = CreateFontIndirect(&lf);

	m_gdiobj.oldfont = nullptr;
	if (m_gdiobj.fontbase) m_gdiobj.oldfont = (HFONT)SelectObject(m_hdc, m_gdiobj.fontbase);

	UpdateFont();

	UpdateLinePen();

	SetTextColor(m_hdc, ColorCodeToColorRef(m_state.color));

	m_gdiobj.brsDot[0] = (HBRUSH)GetStockObject(BLACK_BRUSH);
	for (int i = 1; i < _countof(m_gdiobj.brsDot); i++) {
		if (!m_gdiobj.brsDot[i]) {
			m_gdiobj.brsDot[i] = CreateSolidBrush(ColorCodeToColorRef(i));
		}
	}

	SetGraphicsMode(m_hdc, GM_ADVANCED);

	SetBkMode(m_hdc, TRANSPARENT);
}

void CPrintPR201::EndPrint()
{
	// `c蕪o͂
	std::vector<PRINTCMD_DATA>& cmdList = m_parser->GetParsedCommandList();
	Render(cmdList.size()); // f[^`悷
	RenderGraphic(); // OtBbN`悷
	m_cmdIndex = 0;
	cmdList.clear();
	
	CPrintBase::EndPrint();

	if (m_colorbuf) {
		delete[] m_colorbuf;
		m_colorbuf = NULL;
		m_colorbuf_w = 0;
		m_colorbuf_h = 0;
	}

	// GDIIuWFNg폜
	if (m_gdiobj.oldfont) SelectObject(m_hdc, m_gdiobj.oldfont);
	if (m_gdiobj.fontbase) DeleteObject(m_gdiobj.fontbase);
	if (m_gdiobj.fontrot90) DeleteObject(m_gdiobj.fontrot90);
	if (m_gdiobj.fontbold) DeleteObject(m_gdiobj.fontbold);
	if (m_gdiobj.fontboldrot90) DeleteObject(m_gdiobj.fontboldrot90);
	if (m_gdiobj.penline) DeleteObject(m_gdiobj.penline);
	m_gdiobj.fontbase = NULL;
	m_gdiobj.fontrot90 = NULL;
	m_gdiobj.fontbold = NULL;
	m_gdiobj.fontboldrot90 = NULL;
	m_gdiobj.penline = NULL;
	m_gdiobj.brsDot[0] = NULL;
	for (int i = 1; i < _countof(m_gdiobj.brsDot); i++) {
		if (m_gdiobj.brsDot[i]) {
			DeleteObject(m_gdiobj.brsDot[i]);
			m_gdiobj.brsDot[i] = NULL;
		}
	}
}

bool CPrintPR201::Write(UINT8 data)
{
	return m_parser->PushByte(data);
}

void CPrintPR201::RenderGraphic()
{
	if (m_state.hasGraphic) {
		float pitchx = CalcDotPitchX();
		float pitchy = CalcDotPitchY();
		if (m_state.copymode) {
			//pitchy *= (float)24 / 16 * 120 / 160 * 160 / 24 / 6; // Ȃ̂ق
		}
		int r = pitchx / 2 * m_dotscale;
		int rx = (float)pitchx / 2 + 1;
		int ry = (float)pitchy / 2 + 1;
		if (r == 0) r = 1;
		HBRUSH hBrush = m_gdiobj.brsDot[m_state.color];
		HPEN hPen = (HPEN)GetStockObject(NULL_PEN);
		HGDIOBJ oldPen = SelectObject(m_hdc, hPen);
		HGDIOBJ oldBrush = SelectObject(m_hdc, hBrush);
		int curColor = m_state.color;
		for (int y = 0; y < m_colorbuf_h; y++) {
			int cy = m_offsetYPixel + m_state.topMargin * m_dpiY + (int)(m_state.graphicPosY + y * pitchy);
			for (int x = 0; x < m_colorbuf_w; x++) {
				int cx = m_offsetXPixel + m_state.leftMargin * m_dpiX + (int)(m_state.posX + x * pitchx);
				int idx = y * m_colorbuf_w + x;
				if ((m_colorbuf[idx] & 0x7) != 0x7) {
					if ((m_colorbuf[idx] & 0x7) != curColor) {
						curColor = (m_colorbuf[idx] & 0x7);
						SelectObject(m_hdc, m_gdiobj.brsDot[curColor]);
					}
					if (m_rectdot) {
						Rectangle(m_hdc, cx - rx, cy - ry, cx + rx, cy + ry);
					}
					else {
						Ellipse(m_hdc, cx - r, cy - r, cx + r, cy + r);
					}
				}
			}
		}
		SelectObject(m_hdc, oldBrush);
		SelectObject(m_hdc, oldPen);
		memset(m_colorbuf, 0xff, m_colorbuf_w * m_colorbuf_h); // OtBbN󎚃obt@𔒎ɖ߂
		m_state.hasGraphic = false;
	}
}

void CPrintPR201::Render(int count)
{
	float lastPosX = m_state.posX;
	float lastPosY = m_state.posY;

	// m_statem_renderstate֎OvZf[^
	m_renderstate.actualLineHeight = m_state.actualLineHeight;

	// Õ_ȌԂɖ߂
	m_state = m_renderstate;

	// `s
	std::vector<PRINTCMD_DATA>& cmdList = m_parser->GetParsedCommandList();
	for (int i = 0; i < count; i++) {
		PFNPRINTCMD_COMMANDFUNC cmdfunc;
		if ((PFNPRINTCMD_COMMANDFUNC)cmdList[i].cmd) {
			cmdfunc = (PFNPRINTCMD_COMMANDFUNC)cmdList[i].cmd->userdata;
		}
		else {
			cmdfunc = pmpr201_PutChar;
		}
		if (cmdfunc) {
			COMMANDFUNC_RESULT cmdResult = cmdfunc(this, cmdList[i], true);
			if (cmdResult == COMMANDFUNC_RESULT_OVERFLOWLINE || cmdResult == COMMANDFUNC_RESULT_COMPLETELINE || 
				cmdResult == COMMANDFUNC_RESULT_OVERFLOWPAGE || cmdResult == COMMANDFUNC_RESULT_COMPLETEPAGE) {
				// OtBbN󎚂Ε`
				RenderGraphic();
			}
		}
	}

	// sNA
	m_state.actualLineHeight = 0; 

	// W͌̒l̕Mp
	// ȂƉ̔qm_statem_renderstateHƂɖ[v肷̂Ŋ댯
	m_state.posX = lastPosX;
	m_state.posY = lastPosY;

	// _ȌԂL
	m_renderstate = m_state;
}

bool CPrintPR201::CheckOverflowLine(float addCharWidth)
{
	// return m_state.posX + addCharWidth > m_widthPixel;
	return m_state.posX + addCharWidth > (m_state.rightMargin - m_state.leftMargin) * m_dpiX;
}
bool CPrintPR201::CheckOverflowPage(float addLineHeight)
{
	return m_state.posY + addLineHeight > m_heightPixel;
}

PRINT_COMMAND_RESULT CPrintPR201::DoCommand()
{
	std::vector<PRINTCMD_DATA>& cmdList = m_parser->GetParsedCommandList();

	int cmdListLen = cmdList.size();
	if (cmdListLen <= m_cmdIndex) return PRINT_COMMAND_RESULT_OK;

	for (; m_cmdIndex < cmdListLen; m_cmdIndex++) {
		if (m_cmdIndex < 0) {
			// BUG!!!!!
			m_cmdIndex = 0;
			cmdList.clear();
			return PRINT_COMMAND_RESULT_OK;
		}
		PFNPRINTCMD_COMMANDFUNC cmdfunc;
		if ((PFNPRINTCMD_COMMANDFUNC)cmdList[m_cmdIndex].cmd) {
			cmdfunc = (PFNPRINTCMD_COMMANDFUNC)cmdList[m_cmdIndex].cmd->userdata;
		}
		else {
			cmdfunc = pmpr201_PutChar;
		}
		if (cmdfunc) {
			COMMANDFUNC_RESULT cmdResult = cmdfunc(this, cmdList[m_cmdIndex], false);
			if (cmdResult == COMMANDFUNC_RESULT_OVERFLOWLINE || cmdResult == COMMANDFUNC_RESULT_COMPLETELINE) {
				// ss
				if (cmdResult == COMMANDFUNC_RESULT_COMPLETELINE) {
					m_cmdIndex++; // ݂̃R}h͊Ă̂1i߂iłȂꍇŌ̃R}h͖sȂ̂Ŏcj
				}
				Render(m_cmdIndex); // f[^`悷
				cmdList.erase(cmdList.begin(), cmdList.begin() + m_cmdIndex);
				m_cmdIndex = -1; // [v+1̂-1
				cmdListLen = cmdList.size();
				m_lastNewPage = false;
			}
			else if (cmdResult == COMMANDFUNC_RESULT_OVERFLOWPAGE || cmdResult == COMMANDFUNC_RESULT_COMPLETEPAGE) {
				// y[Ws
				if (cmdResult == COMMANDFUNC_RESULT_COMPLETEPAGE || m_lastNewPage) {
					m_cmdIndex++; // ݂̃R}h͊Ă̂1i߂iłȂꍇŌ̃R}h͖sȂ̂Ŏcj
				}
				Render(m_cmdIndex); // f[^`悷
				cmdList.erase(cmdList.begin(), cmdList.begin() + m_cmdIndex);
				m_cmdIndex = 0;
				m_lastNewPage = true;
				return PRINT_COMMAND_RESULT_COMPLETEPAGE;
			}
			else if (cmdResult == COMMANDFUNC_RESULT_RENDERLINE) {
				// s̕`s
				m_cmdIndex++; // ݂̃R}h͊Ă̂1i߂iłȂꍇŌ̃R}h͖sȂ̂Ŏcj
				Render(m_cmdIndex); // f[^`悷
				cmdList.erase(cmdList.begin(), cmdList.begin() + m_cmdIndex);
				m_cmdIndex = -1; // [v+1̂-1
				cmdListLen = cmdList.size();
				m_lastNewPage = false;
			}
			else {
				m_lastNewPage = false;
			}
		}
	}
	return PRINT_COMMAND_RESULT_OK;
}

bool CPrintPR201::HasRenderingCommand()
{
	std::vector<PRINTCMD_DATA>& cmdList = m_parser->GetParsedCommandList();

	int cmdListLen = cmdList.size();
	if (cmdListLen <= m_cmdIndex) return false;

	for (int i = m_cmdIndex; i < cmdListLen; i++) {
		if ((PFNPRINTCMD_COMMANDFUNC)cmdList[m_cmdIndex].cmd) {
			PFNPRINTCMD_COMMANDFUNC cmdfunc = (PFNPRINTCMD_COMMANDFUNC)cmdList[m_cmdIndex].cmd->userdata;
			if (cmdfunc == pmpr201_CommandESCGraph) {
				// OtBbN`
				return true;
			}
		}
		else {
			// `
			return true;
		}
	}
	return false;
}

void CPrintPR201::UpdateFont()
{
	if (m_state.mode == PRINT_PR201_PRINTMODE_t) {
		if (m_gdiobj.fontrot90 && m_gdiobj.fontboldrot90) SelectObject(m_hdc, m_state.bold ? m_gdiobj.fontboldrot90 : m_gdiobj.fontrot90);
	}
	else {
		if (m_gdiobj.fontbase && m_gdiobj.fontbold) SelectObject(m_hdc, m_state.bold ? m_gdiobj.fontbold : m_gdiobj.fontbase);
	}
}

void CPrintPR201::UpdateLinePen()
{
	if (m_gdiobj.penline == NULL || m_state.linep1 != m_gdiobj.lastlinep1 || m_state.linep2 != m_gdiobj.lastlinep2 || m_state.linep3 != m_gdiobj.lastlinep3 || m_state.linecolor != m_gdiobj.lastlinecolor) {
		if (m_gdiobj.penline) {
			DeleteObject(m_gdiobj.penline);
			m_gdiobj.penline = NULL;
		}
		const float dotPitch = 1.0f / 160;
		int dotsize = (float)m_dpiX * dotPitch;
		dotsize *= m_state.linep3 / 2;
		if (dotsize <= 0) dotsize = 1;
		m_gdiobj.penline = CreatePen(PS_SOLID, dotsize, ColorCodeToColorRef(m_state.color));

		m_gdiobj.lastlinep1 = m_state.linep1;
		m_gdiobj.lastlinep2 = m_state.linep2;
		m_gdiobj.lastlinep3 = m_state.linep3;
		m_gdiobj.lastlinecolor = m_state.color;
	}
}
