//=================================================================================================
/**
 * @file wazawasure_Base.cpp
 * @brief ZYʂ̊{NX
 * @author obata_toshihiro
 * @date 2011.08.28
 */
//=================================================================================================
#include <gflib.h>
#include <pmlib.h>
#include "wazawasure_Base.h"

#include <message/msg_waza_wasure.h>


namespace app { 
  namespace wazawasure {


    Base::Base( 
        gfl::heap::HeapBase* parent_heap,
        gfl::heap::HEAPID app_heap_id, u32 app_heap_size, 
        gfl::heap::HeapBase* parent_device_heap, 
        gfl::heap::HEAPID device_heap_id, u32 device_heap_size,
        gfl::grp::g2d::LytSys* layout_system,
        const pml::pokepara::PokemonParam* poke_param ) :
      m_heap( NULL ),
      m_device_heap( NULL ),
      m_nw_allocator( NULL ),
      m_nw_device_allocator( NULL ),
      m_layout_system( layout_system ),
      m_pokeparam( poke_param ),
      m_finish_flag( false )
    {
      this->CreateHeap( 
          parent_heap, app_heap_id, app_heap_size, 
          parent_device_heap, device_heap_id, device_heap_size );
      this->CreateNwAllocator();
    }

    void Base::CreateHeap( 
        gfl::heap::HeapBase* parent_heap, 
        gfl::heap::HEAPID app_heap_id, u32 app_heap_size, 
        gfl::heap::HeapBase* parent_device_heap, 
        gfl::heap::HEAPID device_heap_id, u32 device_heap_size )
    {
      GFL_ASSERT( !m_heap );
      if( !m_heap ) {
        m_heap = gfl::heap::Manager::CreateHeap( 
            parent_heap, app_heap_id, app_heap_size, gfl::heap::HEAP_TYPE_EXP );
      }

      GFL_ASSERT( !m_device_heap );
      if( !m_device_heap ) {
        m_device_heap = gfl::heap::Manager::CreateHeap( 
            parent_device_heap, device_heap_id, device_heap_size, gfl::heap::HEAP_TYPE_EXP );
      }
    }

    void Base::CreateNwAllocator( void )
    {
      GFL_ASSERT( !m_nw_allocator );
      if( !m_nw_allocator ) {
        m_nw_allocator = GFL_NEW( m_heap ) gfl::heap::NwAllocator( m_heap );
      }

      GFL_ASSERT( !m_nw_device_allocator );
      if( !m_nw_device_allocator ) {
        m_nw_device_allocator = GFL_NEW( m_heap ) gfl::heap::NwAllocator( m_device_heap );
      }
    }






    Base::~Base()
    {
      this->DeleteNwAllocator();
      this->DeleteHeap();
    }

    void Base::DeleteNwAllocator( void )
    {
      if( m_nw_allocator ) {
        GFL_DELETE m_nw_allocator;
        m_nw_allocator = NULL;
      }
      if( m_nw_device_allocator ) {
        GFL_DELETE m_nw_device_allocator;
        m_nw_device_allocator = NULL;
      }
    }

    void Base::DeleteHeap( void )
    {
      if( m_heap ) {
        gfl::heap::Manager::DeleteHeap( m_heap );
        m_heap = NULL;
      } 
      if( m_device_heap ) {
        gfl::heap::Manager::DeleteHeap( m_device_heap );
        m_device_heap = NULL;
      }
    }





    bool Base::IsFinished( void ) const
    {
      return m_finish_flag;
    }

    void Base::SetFinishFlag( void )
    {
      m_finish_flag = true;
    }

    void Base::ResetFinishFlag( void )
    {
      m_finish_flag = false;
    }




    gfl::heap::HeapBase* Base::GetHeap( void ) const
    {
      return m_heap;
    }

    gfl::heap::HeapBase* Base::GetDeviceHeap( void ) const
    {
      return m_device_heap;
    }

    gfl::heap::NwAllocator* Base::GetNwAllocator( void ) const
    {
      return m_nw_allocator;
    }

    gfl::heap::NwAllocator* Base::GetNwDeviceAllocator( void ) const
    {
      return m_nw_device_allocator;
    }

    gfl::grp::GraphicsSystem* Base::GetGraphicsSystem( void ) const
    {
      gamesystem::GameManager* game_manager = GFL_SINGLETON_INSTANCE( gamesystem::GameManager );
      return game_manager->GetGraphicsSystem();
    }

    gfl::grp::g2d::LytSys* Base::GetLayoutSystem( void ) const
    {
      return m_layout_system;
    }

    gfl::ui::DeviceManager* Base::GetUiDeviceManager( void ) const
    {
      gamesystem::GameManager* game_manager = GFL_SINGLETON_INSTANCE( gamesystem::GameManager );
      return game_manager->GetUiDeviceManager();
    }

    const pml::pokepara::PokemonParam* Base::GetPokeParam( void ) const
    {
      return m_pokeparam;
    }








    const u32 Base::TEMP_STRBUF_LENGTH = 256;
    gfl::str::StrBuf* Base::m_temp_strbuf = NULL; 
    gfl::str::MsgData* Base::m_message_data = NULL;

    void Base::Initialize( gfl::heap::HeapBase* heap )
    {
      if( m_temp_strbuf == NULL ) {
        m_temp_strbuf = GFL_NEW( heap ) gfl::str::StrBuf( TEMP_STRBUF_LENGTH, heap );
      }
      if( m_message_data == NULL ) {
        m_message_data = GFL_NEW( heap ) 
          gfl::str::MsgData( 
              L"rom:/message.garc", GARC_message_waza_wasure_DAT, 
              heap, gfl::str::MsgData::LOAD_FULL );
      }
    }

    void Base::Finalize( void )
    {
      if( m_temp_strbuf ) {
        GFL_DELETE m_temp_strbuf;
        m_temp_strbuf = NULL;
      }
      if( m_message_data ) {
        GFL_DELETE m_message_data;
        m_message_data = NULL;
      }
    }



    void Base::SetPaneVisibility( 
        gfl::grp::g2d::LytWk* layout_work, gfl::grp::g2d::LytPaneIndex pane_index, bool visibility )
    {
      gfl::grp::g2d::LytPane* pane;
      pane = layout_work->GetPane( pane_index ); 
      GFL_ASSERT( pane );
      if( pane ) {
        pane->SetVisible( visibility );
      }
    }


    void Base::SetPaneVisible(
        gfl::grp::g2d::LytWk* layout_work, gfl::grp::g2d::LytPaneIndex pane_index )
    {
      SetPaneVisibility( layout_work, pane_index, true );
    }

    void Base::SetPaneInvisible(
        gfl::grp::g2d::LytWk* layout_work, gfl::grp::g2d::LytPaneIndex pane_index )
    {
      SetPaneVisibility( layout_work, pane_index, false );
    }


    void Base::SetTextboxPaneSex( 
        gfl::grp::g2d::LytWk* layout_work, 
        gfl::grp::g2d::LytPaneIndex pane_index, 
        pml::Sex sex )
    {
      u32 str_id = 0;
      switch( sex ) {
      case pml::SEX_MALE:
        str_id = STR_WW_MALE;
        break;
      case pml::SEX_FEMALE:
        str_id = STR_WW_FEMALE;
        break;
      default:
        str_id = STR_WW_UNKNOWN;
        break;
      }
      this->SetTextboxPaneMessage( layout_work, pane_index, str_id );
    }


    void Base::SetTextboxPaneMessage( 
        gfl::grp::g2d::LytWk* layout_work, 
        gfl::grp::g2d::LytPaneIndex pane_index, u32 str_id )
    {
      GFL_ASSERT( m_temp_strbuf );
      m_message_data->GetString( str_id, *m_temp_strbuf );
      this->SetTextboxPaneString( layout_work, pane_index, m_temp_strbuf );
    }


    void Base::SetTextboxPaneString( 
        gfl::grp::g2d::LytWk* layout_work, 
        gfl::grp::g2d::LytPaneIndex pane_index, const gfl::str::StrBuf* strbuf )
    {
      gfl::grp::g2d::LytTextBox* text_box;
      text_box = layout_work->GetTextBoxPane( pane_index ); 
      GFL_ASSERT( text_box );
      if( text_box ) {
        const wchar_t* string = strbuf->GetPtr();
        text_box->SetString( string );
      }
    }


    void Base::SetTextboxPaneNumber( 
        gfl::grp::g2d::LytWk* layout_work, 
        gfl::grp::g2d::LytPaneIndex pane_index, u32 value, u32 keta )
    {
      print::StrNumber( m_temp_strbuf, value, keta, print::NUM_DISP_SPACE, print::NUM_CODE_HANKAKU );
      this->SetTextboxPaneString( layout_work, pane_index, m_temp_strbuf );
    }


    void Base::SetTextboxPaneWazaName( 
        gfl::grp::g2d::LytWk* layout_work, 
        gfl::grp::g2d::LytPaneIndex pane_index, 
        WazaNo wazano )
    {
      pml::System::WazaDataSystem()->GetWazaName( wazano, m_temp_strbuf );
      this->SetTextboxPaneString( layout_work, pane_index, m_temp_strbuf );
    }


    void Base::SetTextboxPaneMonsName( 
        gfl::grp::g2d::LytWk* layout_work, 
        gfl::grp::g2d::LytPaneIndex pane_index, 
        MonsNo monsno )
    {
      pml::System::PersonalSystem()->GetMonsName( monsno, m_temp_strbuf );
      this->SetTextboxPaneString( layout_work, pane_index, m_temp_strbuf );
    }



    void Base::StartAnimation( gfl::grp::g2d::LytWk* layout_work, u32 animation_index )
    {
      if( layout_work->IsBindAnime( animation_index ) == false ) {
        layout_work->BindAnime( animation_index );
      }
      layout_work->SetAutoAnime( animation_index, true );
      layout_work->SetAnimeFrame( animation_index, 0 );
    }


    void Base::StopAnimation( gfl::grp::g2d::LytWk* layout_work, u32 animation_index )
    {
      if( layout_work->IsBindAnime( animation_index ) ) {
        layout_work->UnBindAnime( animation_index );
      }
    }


    void Base::PauseAnimation( 
        gfl::grp::g2d::LytWk* layout_work, u32 animation_index, bool pause )
    {
      if( layout_work->IsBindAnime( animation_index ) ) {
        layout_work->SetAutoAnime( animation_index, pause );
      }
    }


    void Base::SetAnimationFrame( 
        gfl::grp::g2d::LytWk* layout_work, u32 animation_index, u32 frame )
    {
      if( layout_work->IsBindAnime( animation_index ) ) {
        layout_work->SetAnimeFrame( animation_index, frame );
      }
    }



  } // namespace wazawasure 
} // namespace app
