//=======================================================================================
/**
 * @file PokeList.cpp
 * @brief 퓬( |PXg̓ )
 * @author obata_toshihiro
 * @date 2011.08.24
 */
//=======================================================================================
#include <gflib.h>
#include <battle/btl_common.h>
#include <app/tool/app_tool_ButtonManager.h>
#include <app/tool/app_tool_CursorManager.h>
#include <app/tool/app_tool_Gauge.h>
#include "AppDrawUtil.h"
#include "PokeListBase.h"
#include "PokeList.h"

#include <arc/arc_def.h>
#include <arc/btl_app.gaix>
#include <message/msg_btl_app.h>
#include "poke_list.h"
#include "poke_list_pane.h"


namespace btl {
  namespace app {


    // q[vTCY
    const u32 PokeList::APP_HEAPSIZE = 0x1000;
    const u32 PokeList::DEVICE_HEAPSIZE = 0x200000;

    // ľ
    const u32 PokeList::LV_PLACE_COUNT = 3;
    const u32 PokeList::HP_PLACE_COUNT = 3;


    // CAEg[ÑZbgAbvf[^
    const gfl::grp::g2d::LytWkSetUp PokeList::LAYOUT_WORK_SETUP_DATA = 
    {
      gfl::math::VEC3( 0, LAYOUT_DOWN_OFS, 0 ),  // pos
      gfl::math::VEC2( 1, 1 ),                   // scale
      gfl::math::VEC3( 0, 0, 0 ),                // rotate
      1,                                         // priority
    };


    // Aj[Vf[^e[u
    const gfl::grp::g2d::LytArcIndex PokeList::ANIMATION_DATA_TABLE[ PokeList::ANIMATION_DATA_NUM ] = 
    {
      /* ANIMATION_DATA_DUMMY_HP_GAUGE */ LYTRES_POKE_LIST_POKE_LIST_DUMMY_HP_GAUGE_BCLAN,
      /* ANIMATION_DATA_DUMMY_PLATE_NORMAL */ LYTRES_POKE_LIST_POKE_LIST_DUMMY_PLATE_NORMAL_BCLAN,
      /* ANIMATION_DATA_DUMMY_PLATE_PLAYER */ LYTRES_POKE_LIST_POKE_LIST_DUMMY_PLATE_PLAYER_BCLAN,
      /* ANIMATION_DATA_DUMMY_PLATE_PARTNER */ LYTRES_POKE_LIST_POKE_LIST_DUMMY_PLATE_PARTNER_BCLAN,
      /* ANIMATION_DATA_DUMMY_PLATE_HOLD */ LYTRES_POKE_LIST_POKE_LIST_DUMMY_PLATE_HOLD_BCLAN,
      /* ANIMATION_DATA_DUMMY_PLATE_RELEASE */ LYTRES_POKE_LIST_POKE_LIST_DUMMY_PLATE_RELEASE_BCLAN,
      /* ANIMATION_DATA_DUMMY_PLATE_SELECT */ LYTRES_POKE_LIST_POKE_LIST_DUMMY_PLATE_SELECT_BCLAN,
      /* ANIMATION_DATA_RETURN_HOLD */ LYTRES_POKE_LIST_POKE_LIST_RETURN_HOLD_BCLAN,
      /* ANIMATION_DATA_RETURN_RELEASE */ LYTRES_POKE_LIST_POKE_LIST_RETURN_RELEASE_BCLAN,
      /* ANIMATION_DATA_RETURN_SELECT */ LYTRES_POKE_LIST_POKE_LIST_RETURN_SELECT_BCLAN,
      /* ANIMATION_DATA_RETURN_PASSIVE */ LYTRES_POKE_LIST_POKE_LIST_RETURN_PASSIVE_BCLAN,
      /* ANIMATION_DATA_RETURN_ACTIVE */ LYTRES_POKE_LIST_POKE_LIST_RETURN_ACTIVE_BCLAN,
      /* ANIMATION_DATA_POKE0_ALIVE */ LYTRES_POKE_LIST_POKE_LIST_POKE0_ALIVE_BCLAN,
      /* ANIMATION_DATA_POKE0_DEAD */ LYTRES_POKE_LIST_POKE_LIST_POKE0_DEAD_BCLAN,
      /* ANIMATION_DATA_POKE0_EMPTY */ LYTRES_POKE_LIST_POKE_LIST_POKE0_EMPTY_BCLAN,
      /* ANIMATION_DATA_POKE0_IN */ LYTRES_POKE_LIST_POKE_LIST_POKE0_IN_BCLAN,
      /* ANIMATION_DATA_POKE0_OUT */ LYTRES_POKE_LIST_POKE_LIST_POKE0_OUT_BCLAN,
      /* ANIMATION_DATA_POKE1_ALIVE */ LYTRES_POKE_LIST_POKE_LIST_POKE1_ALIVE_BCLAN,
      /* ANIMATION_DATA_POKE1_DEAD */ LYTRES_POKE_LIST_POKE_LIST_POKE1_DEAD_BCLAN,
      /* ANIMATION_DATA_POKE1_EMPTY */ LYTRES_POKE_LIST_POKE_LIST_POKE1_EMPTY_BCLAN,
      /* ANIMATION_DATA_POKE1_IN */ LYTRES_POKE_LIST_POKE_LIST_POKE1_IN_BCLAN,
      /* ANIMATION_DATA_POKE1_OUT */ LYTRES_POKE_LIST_POKE_LIST_POKE1_OUT_BCLAN,
      /* ANIMATION_DATA_POKE2_ALIVE */ LYTRES_POKE_LIST_POKE_LIST_POKE2_ALIVE_BCLAN,
      /* ANIMATION_DATA_POKE2_DEAD */ LYTRES_POKE_LIST_POKE_LIST_POKE2_DEAD_BCLAN,
      /* ANIMATION_DATA_POKE2_EMPTY */ LYTRES_POKE_LIST_POKE_LIST_POKE2_EMPTY_BCLAN,
      /* ANIMATION_DATA_POKE2_IN */ LYTRES_POKE_LIST_POKE_LIST_POKE2_IN_BCLAN,
      /* ANIMATION_DATA_POKE2_OUT */ LYTRES_POKE_LIST_POKE_LIST_POKE2_OUT_BCLAN,
      /* ANIMATION_DATA_POKE3_ALIVE */ LYTRES_POKE_LIST_POKE_LIST_POKE3_ALIVE_BCLAN,
      /* ANIMATION_DATA_POKE3_DEAD */ LYTRES_POKE_LIST_POKE_LIST_POKE3_DEAD_BCLAN,
      /* ANIMATION_DATA_POKE3_EMPTY */ LYTRES_POKE_LIST_POKE_LIST_POKE3_EMPTY_BCLAN,
      /* ANIMATION_DATA_POKE3_IN */ LYTRES_POKE_LIST_POKE_LIST_POKE3_IN_BCLAN,
      /* ANIMATION_DATA_POKE3_OUT */ LYTRES_POKE_LIST_POKE_LIST_POKE3_OUT_BCLAN,
      /* ANIMATION_DATA_POKE4_ALIVE */ LYTRES_POKE_LIST_POKE_LIST_POKE4_ALIVE_BCLAN,
      /* ANIMATION_DATA_POKE4_DEAD */ LYTRES_POKE_LIST_POKE_LIST_POKE4_DEAD_BCLAN,
      /* ANIMATION_DATA_POKE4_EMPTY */ LYTRES_POKE_LIST_POKE_LIST_POKE4_EMPTY_BCLAN,
      /* ANIMATION_DATA_POKE4_IN */ LYTRES_POKE_LIST_POKE_LIST_POKE4_IN_BCLAN,
      /* ANIMATION_DATA_POKE4_OUT */ LYTRES_POKE_LIST_POKE_LIST_POKE4_OUT_BCLAN,
      /* ANIMATION_DATA_POKE5_ALIVE */ LYTRES_POKE_LIST_POKE_LIST_POKE5_ALIVE_BCLAN,
      /* ANIMATION_DATA_POKE5_DEAD */ LYTRES_POKE_LIST_POKE_LIST_POKE5_DEAD_BCLAN,
      /* ANIMATION_DATA_POKE5_EMPTY */ LYTRES_POKE_LIST_POKE_LIST_POKE5_EMPTY_BCLAN,
      /* ANIMATION_DATA_POKE5_IN */ LYTRES_POKE_LIST_POKE_LIST_POKE5_IN_BCLAN,
      /* ANIMATION_DATA_POKE5_OUT */ LYTRES_POKE_LIST_POKE_LIST_POKE5_OUT_BCLAN,
      /* ANIMATION_DATA_CURSOR_FLUCTUATE */ LYTRES_POKE_LIST_POKE_LIST_CURSOR_FLUCTUATE_BCLAN,
      /* ANIMATION_DATA_CURSOR_ON_POKE0 */ LYTRES_POKE_LIST_POKE_LIST_CURSOR_ON_POKE0_BCLAN,
      /* ANIMATION_DATA_CURSOR_ON_POKE1 */ LYTRES_POKE_LIST_POKE_LIST_CURSOR_ON_POKE1_BCLAN,
      /* ANIMATION_DATA_CURSOR_ON_POKE2 */ LYTRES_POKE_LIST_POKE_LIST_CURSOR_ON_POKE2_BCLAN,
      /* ANIMATION_DATA_CURSOR_ON_POKE3 */ LYTRES_POKE_LIST_POKE_LIST_CURSOR_ON_POKE3_BCLAN,
      /* ANIMATION_DATA_CURSOR_ON_POKE4 */ LYTRES_POKE_LIST_POKE_LIST_CURSOR_ON_POKE4_BCLAN,
      /* ANIMATION_DATA_CURSOR_ON_POKE5 */ LYTRES_POKE_LIST_POKE_LIST_CURSOR_ON_POKE5_BCLAN,
      /* ANIMATION_DATA_CURSOR_ON_RETURN */ LYTRES_POKE_LIST_POKE_LIST_CURSOR_ON_RETURN_BCLAN,
    };


    // LAj[VƂ̐ݒ
    const PokeList::ShareAnimationDesc PokeList::SHARE_ANIMATION_DESC[ PokeList::SHARE_ANIMATION_NUM ] = 
    {
      /* SHARE_ANIMATION_POKE_HP_GAUGE */
      {
        ANIMATION_POKE0_HP_GAUGE, 
        ANIMATION_DATA_DUMMY_HP_GAUGE,
        POKE_BUTTON_NUM,
        PANENAME_POKE_LIST_GROUP_HP_GAUGE,
        gfl::grp::g2d::LYTSYS_PANEINDEX_ERROR,  // O[v̑SẴyCɓo^
      },
      /* SHARE_ANIMATION_POKE_PLATE_NORMAL */
      {
        ANIMATION_POKE0_PLATE_NORMAL, 
        ANIMATION_DATA_DUMMY_PLATE_NORMAL,
        POKE_BUTTON_NUM,
        PANENAME_POKE_LIST_GROUP_PLATE,
        gfl::grp::g2d::LYTSYS_PANEINDEX_ERROR,  // O[v̑SẴyCɓo^
      },
      /* SHARE_ANIMATION_POKE_PLATE_PLAYER */
      {
        ANIMATION_POKE0_PLATE_PLAYER, 
        ANIMATION_DATA_DUMMY_PLATE_PLAYER,
        POKE_BUTTON_NUM,
        PANENAME_POKE_LIST_GROUP_PLATE,
        gfl::grp::g2d::LYTSYS_PANEINDEX_ERROR,  // O[v̑SẴyCɓo^
      },
      /* SHARE_ANIMATION_POKE_PLATE_PARTNER */
      {
        ANIMATION_POKE0_PLATE_PARTNER, 
        ANIMATION_DATA_DUMMY_PLATE_PARTNER,
        POKE_BUTTON_NUM,
        PANENAME_POKE_LIST_GROUP_PLATE,
        gfl::grp::g2d::LYTSYS_PANEINDEX_ERROR,  // O[v̑SẴyCɓo^
      },
      /* SHARE_ANIMATION_POKE_PLATE_HOLD */
      {
        ANIMATION_POKE0_PLATE_HOLD, 
        ANIMATION_DATA_DUMMY_PLATE_HOLD,
        POKE_BUTTON_NUM,
        PANENAME_POKE_LIST_GROUP_PLATE,
        gfl::grp::g2d::LYTSYS_PANEINDEX_ERROR,  // O[v̑SẴyCɓo^
      },
      /* SHARE_ANIMATION_POKE_PLATE_RELEASE */
      {
        ANIMATION_POKE0_PLATE_RELEASE, 
        ANIMATION_DATA_DUMMY_PLATE_RELEASE,
        POKE_BUTTON_NUM,
        PANENAME_POKE_LIST_GROUP_PLATE,
        gfl::grp::g2d::LYTSYS_PANEINDEX_ERROR,  // O[v̑SẴyCɓo^
      },
      /* SHARE_ANIMATION_POKE_PLATE_SELECT */
      {
        ANIMATION_POKE0_PLATE_SELECT, 
        ANIMATION_DATA_DUMMY_PLATE_SELECT,
        POKE_BUTTON_NUM,
        PANENAME_POKE_LIST_GROUP_PLATE,
        gfl::grp::g2d::LYTSYS_PANEINDEX_ERROR,  // O[v̑SẴyCɓo^
      },
    };


    // J[\ړf[^
    const ::app::tool::CursorManager::MOVE_DATA PokeList::CURSOR_MOVE_DATA[ PokeList::BUTTON_NUM ] = 
    {
      /*   id            up             down           left           right  */
      { BUTTON_POKE_0, { BUTTON_POKE_0, BUTTON_POKE_2, BUTTON_POKE_0, BUTTON_POKE_1 } },
      { BUTTON_POKE_1, { BUTTON_POKE_1, BUTTON_POKE_3, BUTTON_POKE_0, BUTTON_POKE_1 } },
      { BUTTON_POKE_2, { BUTTON_POKE_0, BUTTON_POKE_4, BUTTON_POKE_2, BUTTON_POKE_3 } },
      { BUTTON_POKE_3, { BUTTON_POKE_1, BUTTON_POKE_5, BUTTON_POKE_2, BUTTON_POKE_3 } },
      { BUTTON_POKE_4, { BUTTON_POKE_2, BUTTON_RETURN, BUTTON_POKE_4, BUTTON_POKE_5 } },
      { BUTTON_POKE_5, { BUTTON_POKE_3, BUTTON_RETURN, BUTTON_POKE_4, BUTTON_POKE_5 } },
      { BUTTON_RETURN, { BUTTON_POKE_5, BUTTON_RETURN, BUTTON_RETURN, BUTTON_RETURN } },
    };



    // {^Ƃ̐ݒ
    const PokeList::ButtonDesc PokeList::BUTTON_DESC[ PokeList::BUTTON_NUM ] = 
    {
      /* BUTTON_POKE_0 */ 
      {
        PokeListBase::RESULT_SELECT_PLAYER_POKE,
        PANENAME_POKE_LIST_PANE_POKE0_PLATE,
        static_cast<AnimationIndex>( ANIMATION_POKE0_PLATE_HOLD + 0 ),
        static_cast<AnimationIndex>( ANIMATION_POKE0_PLATE_RELEASE + 0 ),
        static_cast<AnimationIndex>( ANIMATION_POKE0_PLATE_SELECT + 0 ),
        ANIMATION_NULL,
        ANIMATION_NULL,
        ANIMATION_CURSOR_ON_POKE0,
      },
      /* BUTTON_POKE_1 */ 
      {
        PokeListBase::RESULT_SELECT_PLAYER_POKE,
        PANENAME_POKE_LIST_PANE_POKE1_PLATE,
        static_cast<AnimationIndex>( ANIMATION_POKE0_PLATE_HOLD + 1 ),
        static_cast<AnimationIndex>( ANIMATION_POKE0_PLATE_RELEASE + 1 ),
        static_cast<AnimationIndex>( ANIMATION_POKE0_PLATE_SELECT + 1 ),
        ANIMATION_NULL,
        ANIMATION_NULL,
        ANIMATION_CURSOR_ON_POKE1,
      },
      /* BUTTON_POKE_2 */ 
      {
        PokeListBase::RESULT_SELECT_PLAYER_POKE,
        PANENAME_POKE_LIST_PANE_POKE2_PLATE,
        static_cast<AnimationIndex>( ANIMATION_POKE0_PLATE_HOLD + 2 ),
        static_cast<AnimationIndex>( ANIMATION_POKE0_PLATE_RELEASE + 2 ),
        static_cast<AnimationIndex>( ANIMATION_POKE0_PLATE_SELECT + 2 ),
        ANIMATION_NULL,
        ANIMATION_NULL,
        ANIMATION_CURSOR_ON_POKE2,
      },
      /* BUTTON_POKE_3 */ 
      {
        PokeListBase::RESULT_SELECT_PLAYER_POKE,
        PANENAME_POKE_LIST_PANE_POKE3_PLATE,
        static_cast<AnimationIndex>( ANIMATION_POKE0_PLATE_HOLD + 3 ),
        static_cast<AnimationIndex>( ANIMATION_POKE0_PLATE_RELEASE + 3 ),
        static_cast<AnimationIndex>( ANIMATION_POKE0_PLATE_SELECT + 3 ),
        ANIMATION_NULL,
        ANIMATION_NULL,
        ANIMATION_CURSOR_ON_POKE3,
      },
      /* BUTTON_POKE_4 */ 
      {
        PokeListBase::RESULT_SELECT_PLAYER_POKE,
        PANENAME_POKE_LIST_PANE_POKE4_PLATE,
        static_cast<AnimationIndex>( ANIMATION_POKE0_PLATE_HOLD + 4 ),
        static_cast<AnimationIndex>( ANIMATION_POKE0_PLATE_RELEASE + 4 ),
        static_cast<AnimationIndex>( ANIMATION_POKE0_PLATE_SELECT + 4 ),
        ANIMATION_NULL,
        ANIMATION_NULL,
        ANIMATION_CURSOR_ON_POKE4,
      },
      /* BUTTON_POKE_5 */ 
      {
        PokeListBase::RESULT_SELECT_PLAYER_POKE,
        PANENAME_POKE_LIST_PANE_POKE5_PLATE,
        static_cast<AnimationIndex>( ANIMATION_POKE0_PLATE_HOLD + 5 ),
        static_cast<AnimationIndex>( ANIMATION_POKE0_PLATE_RELEASE + 5 ),
        static_cast<AnimationIndex>( ANIMATION_POKE0_PLATE_SELECT + 5 ),
        ANIMATION_NULL,
        ANIMATION_NULL,
        ANIMATION_CURSOR_ON_POKE5,
      },
      /* BUTTON_RETURN */ 
      {
        PokeListBase::RESULT_RETURN,
        PANENAME_POKE_LIST_PANE_RETURN_BUTTON,
        ANIMATION_RETURN_HOLD,
        ANIMATION_RETURN_RELEASE,
        ANIMATION_RETURN_SELECT,
        ANIMATION_RETURN_ACTIVE,
        ANIMATION_RETURN_PASSIVE,
        ANIMATION_CURSOR_ON_RETURN,
      },
    };



    // |P{^Ƃ̐ݒ
    const PokeList::PokeButtonDesc PokeList::POKE_BUTTON_DESC[ POKE_BUTTON_NUM ] = 
    {
      {
        { PokeListBase::PARTY_PLAYER, 0 },
        PANENAME_POKE_LIST_PANE_POKE0_NAME,
        PANENAME_POKE_LIST_PANE_POKE0_SEX,
        PANENAME_POKE_LIST_PANE_POKE0_LV_VALUE,
        PANENAME_POKE_LIST_PANE_POKE0_HP_MAX,
        PANENAME_POKE_LIST_PANE_POKE0_HP_VALUE,
        PANENAME_POKE_LIST_PANE_POKE0_ITEM,
        ANIMATION_POKE0_IN,
        ANIMATION_POKE0_OUT,
        static_cast<AnimationIndex>( ANIMATION_POKE0_HP_GAUGE + 0 ),
        static_cast<AnimationIndex>( ANIMATION_POKE0_PLATE_NORMAL + 0 ),
        static_cast<AnimationIndex>( ANIMATION_POKE0_PLATE_PLAYER + 0 ),
        static_cast<AnimationIndex>( ANIMATION_POKE0_PLATE_PARTNER + 0 ),
        ANIMATION_POKE0_ALIVE,
        ANIMATION_POKE0_DEAD,
        ANIMATION_POKE0_EMPTY,
      },
      {
        { PokeListBase::PARTY_PLAYER, 1 },
        PANENAME_POKE_LIST_PANE_POKE1_NAME,
        PANENAME_POKE_LIST_PANE_POKE1_SEX,
        PANENAME_POKE_LIST_PANE_POKE1_LV_VALUE,
        PANENAME_POKE_LIST_PANE_POKE1_HP_MAX,
        PANENAME_POKE_LIST_PANE_POKE1_HP_VALUE,
        PANENAME_POKE_LIST_PANE_POKE1_ITEM,
        ANIMATION_POKE1_IN,
        ANIMATION_POKE1_OUT,
        static_cast<AnimationIndex>( ANIMATION_POKE0_HP_GAUGE + 1 ),
        static_cast<AnimationIndex>( ANIMATION_POKE0_PLATE_NORMAL + 1 ),
        static_cast<AnimationIndex>( ANIMATION_POKE0_PLATE_PLAYER + 1 ),
        static_cast<AnimationIndex>( ANIMATION_POKE0_PLATE_PARTNER + 1 ),
        ANIMATION_POKE1_ALIVE,
        ANIMATION_POKE1_DEAD,
        ANIMATION_POKE1_EMPTY,
      },
      {
        { PokeListBase::PARTY_PLAYER, 2 },
        PANENAME_POKE_LIST_PANE_POKE2_NAME,
        PANENAME_POKE_LIST_PANE_POKE2_SEX,
        PANENAME_POKE_LIST_PANE_POKE2_LV_VALUE,
        PANENAME_POKE_LIST_PANE_POKE2_HP_MAX,
        PANENAME_POKE_LIST_PANE_POKE2_HP_VALUE,
        PANENAME_POKE_LIST_PANE_POKE2_ITEM,
        ANIMATION_POKE2_IN,
        ANIMATION_POKE2_OUT,
        static_cast<AnimationIndex>( ANIMATION_POKE0_HP_GAUGE + 2 ),
        static_cast<AnimationIndex>( ANIMATION_POKE0_PLATE_NORMAL + 2 ),
        static_cast<AnimationIndex>( ANIMATION_POKE0_PLATE_PLAYER + 2 ),
        static_cast<AnimationIndex>( ANIMATION_POKE0_PLATE_PARTNER + 2 ),
        ANIMATION_POKE2_ALIVE,
        ANIMATION_POKE2_DEAD,
        ANIMATION_POKE2_EMPTY,
      },
      {
        { PokeListBase::PARTY_PLAYER, 3 },
        PANENAME_POKE_LIST_PANE_POKE3_NAME,
        PANENAME_POKE_LIST_PANE_POKE3_SEX,
        PANENAME_POKE_LIST_PANE_POKE3_LV_VALUE,
        PANENAME_POKE_LIST_PANE_POKE3_HP_MAX,
        PANENAME_POKE_LIST_PANE_POKE3_HP_VALUE,
        PANENAME_POKE_LIST_PANE_POKE3_ITEM,
        ANIMATION_POKE3_IN,
        ANIMATION_POKE3_OUT,
        static_cast<AnimationIndex>( ANIMATION_POKE0_HP_GAUGE + 3 ),
        static_cast<AnimationIndex>( ANIMATION_POKE0_PLATE_NORMAL + 3 ),
        static_cast<AnimationIndex>( ANIMATION_POKE0_PLATE_PLAYER + 3 ),
        static_cast<AnimationIndex>( ANIMATION_POKE0_PLATE_PARTNER + 3 ),
        ANIMATION_POKE3_ALIVE,
        ANIMATION_POKE3_DEAD,
        ANIMATION_POKE3_EMPTY,
      },
      {
        { PokeListBase::PARTY_PLAYER, 4 },
        PANENAME_POKE_LIST_PANE_POKE4_NAME,
        PANENAME_POKE_LIST_PANE_POKE4_SEX,
        PANENAME_POKE_LIST_PANE_POKE4_LV_VALUE,
        PANENAME_POKE_LIST_PANE_POKE4_HP_MAX,
        PANENAME_POKE_LIST_PANE_POKE4_HP_VALUE,
        PANENAME_POKE_LIST_PANE_POKE4_ITEM,
        ANIMATION_POKE4_IN,
        ANIMATION_POKE4_OUT,
        static_cast<AnimationIndex>( ANIMATION_POKE0_HP_GAUGE + 4 ),
        static_cast<AnimationIndex>( ANIMATION_POKE0_PLATE_NORMAL + 4 ),
        static_cast<AnimationIndex>( ANIMATION_POKE0_PLATE_PLAYER + 4 ),
        static_cast<AnimationIndex>( ANIMATION_POKE0_PLATE_PARTNER + 4 ),
        ANIMATION_POKE4_ALIVE,
        ANIMATION_POKE4_DEAD,
        ANIMATION_POKE4_EMPTY,
      },
      {
        { PokeListBase::PARTY_PLAYER, 5 },
        PANENAME_POKE_LIST_PANE_POKE5_NAME,
        PANENAME_POKE_LIST_PANE_POKE5_SEX,
        PANENAME_POKE_LIST_PANE_POKE5_LV_VALUE,
        PANENAME_POKE_LIST_PANE_POKE5_HP_MAX,
        PANENAME_POKE_LIST_PANE_POKE5_HP_VALUE,
        PANENAME_POKE_LIST_PANE_POKE5_ITEM,
        ANIMATION_POKE5_IN,
        ANIMATION_POKE5_OUT,
        static_cast<AnimationIndex>( ANIMATION_POKE0_HP_GAUGE + 5 ),
        static_cast<AnimationIndex>( ANIMATION_POKE0_PLATE_NORMAL + 5 ),
        static_cast<AnimationIndex>( ANIMATION_POKE0_PLATE_PLAYER + 5 ),
        static_cast<AnimationIndex>( ANIMATION_POKE0_PLATE_PARTNER + 5 ),
        ANIMATION_POKE5_ALIVE,
        ANIMATION_POKE5_DEAD,
        ANIMATION_POKE5_EMPTY,
      },
    };





    // {^Ƃ̐ݒ擾
    // @note ̊֐͔hNXŃI[o[Ch, hNXƎ̃f[^œ삷悤ɂB
    const PokeList::ButtonDesc& PokeList::GetButtonDesc( ButtonIndex button_index ) const
    {
      GFL_ASSERT( button_index < BUTTON_NUM );
      return BUTTON_DESC[ button_index ];
    }




    // |P{^̐ݒ
    // @note ̊֐͉z֐BhNXł, I[o[ChƎ̃f[^ŏB
    void PokeList::InitialziePokeButtonDesc( void )
    {
      for( u8 i=0; i<POKE_BUTTON_NUM; i++ ) {
        m_poke_button_desc[i] = POKE_BUTTON_DESC[i];
      }
    } 

    // |P{^Ƃ̐ݒ擾
    // @note ̊֐͔hNXŃI[o[Ch, hNXƎ̃f[^œ삷悤ɂB
    const PokeList::PokeButtonDesc& PokeList::GetPokeButtonDesc( ButtonIndex button_index ) const
    {
      GFL_ASSERT( button_index < POKE_BUTTON_NUM );
      return m_poke_button_desc[ button_index ];
    }



    // w肵|P\Ă{^𒲂ׂ
    PokeList::ButtonIndex PokeList::GetButtonIndex( PartyId party_id, u8 poke_order ) const
    {
      for( u8 i=0; i<POKE_BUTTON_NUM; i++ ) {
        ButtonIndex btn_index = static_cast<ButtonIndex>( i );
        const PokeButtonDesc& desc = this->GetPokeButtonDesc( btn_index );
        if( ( desc.link.party_id == party_id ) && ( desc.link.poke_order == poke_order ) ) {
          return btn_index;
        }
      }
      GFL_ASSERT(0);
      return BUTTON_POKE_0;
    }









    //---------------------------------------------------------------------------------
    // RXgN^
    //---------------------------------------------------------------------------------
    PokeList::PokeList( 
        gfl::heap::HeapBase* heap, gfl::heap::HeapBase* device_heap, 
        BattleViewSystem* btlv_system ) : 
      PokeListBase( heap, APP_HEAPSIZE, device_heap, DEVICE_HEAPSIZE, btlv_system ),
      m_layout_work( NULL ),
      m_layout_resource_id( 0 ),
      m_button_manager( NULL ),
      m_cursor_manager( NULL ),
      m_seq( SEQ_INPUT_WAIT ),
      m_next_seq( SEQ_INPUT_WAIT ),
      m_wait_animation_index( ANIMATION_NULL ),
      m_exchange_button(),
      m_poke_button_desc()
    {
      this->InitializeGraphicsSystem();
      this->SetupFont();
      this->CreateLayoutWork();
      this->CreateShareAnimation();
      this->CreateCursorManager();  // J[\ړ̓[hԂōقȂ߃RRŐ
      this->InitialziePokeButtonDesc(); 

#if 1 // q[vcʃ`FbN
      gfl::heap::HeapBase* app_heap = this->GetHeap();
      gfl::heap::HeapBase* dev_heap = this->GetDeviceHeap();
      GFL_PRINT( "HEAPID_BATTLE_POKELIST: 0x%x/0x%x, 0x%x/0x%x\n", 
          app_heap->GetTotalFreeSize(), app_heap->GetTotalSize(),
          dev_heap->GetTotalFreeSize(), dev_heap->GetTotalSize() );
#endif
    }



    // OtBbNXVXe
    void PokeList::InitializeGraphicsSystem( void )
    {
      gfl::heap::NwAllocator* nw_device_allocator = this->GetNwDeviceAllocator();
      gfl::grp::GraphicsSystem* graphics_system = this->GetGraphicsSystem();
      graphics_system->InitializeApplicationMemory( nw_device_allocator );
    }


    // tHg̏
    void PokeList::SetupFont( void )
    {
      gfl::grp::g2d::LytSys::SetDefaultFont( SystemFont_GetFont(), "cbf_std.bcfnt" );
    }


    // CAEg[N𐶐
    void PokeList::CreateLayoutWork( void )
    {
      gfl::heap::HeapBase* heap = this->GetHeap();
      gfl::heap::NwAllocator* nw_device_allocator = this->GetNwDeviceAllocator();
      gfl::grp::g2d::LytSys* layout_system = this->GetLayoutSystem();

      gfl::fs::ArcFile arc_file( ARCID_BATTLE_APP, heap );
      m_layout_resource_id = layout_system->LoadLytRes( 
          nw_device_allocator, &arc_file, GARC_btl_app_poke_list_LYTDAT );
      layout_system->TransferAllTexture( m_layout_resource_id, gfl::grp::MEMORY_AREA_FCRAM );

      m_layout_work = layout_system->CreateLytWk(
          m_layout_resource_id,
          LYTRES_POKE_LIST_POKE_LIST_BCLYT,
          ANIMATION_DATA_TABLE, ANIMATION_DATA_NUM, ANIMATION_NUM,
          &LAYOUT_WORK_SETUP_DATA,
          false );  // Aj[VoChȂ
    }


    // LAj[V𐶐
    void PokeList::CreateShareAnimation( void )
    {
      for( u32 i=0; i<SHARE_ANIMATION_NUM; i++ ) {
        m_layout_work->CreateShareAnime( 
            SHARE_ANIMATION_DESC[i].start_animation_index,
            ANIMATION_DATA_TABLE[ SHARE_ANIMATION_DESC[i].animation_data_index ],
            SHARE_ANIMATION_DESC[i].max_share_num,
            SHARE_ANIMATION_DESC[i].share_group_index,
            SHARE_ANIMATION_DESC[i].share_pane_index );
      }
    }


    // J[\ǗIuWFNg𐶐
    void PokeList::CreateCursorManager( void )
    {
      gfl::heap::HeapBase* heap = this->GetHeap();
      m_cursor_manager = GFL_NEW( heap ) 
        ::app::tool::CursorManager( CURSOR_MOVE_DATA, BUTTON_NUM, heap, 0 );
    }










    //---------------------------------------------------------------------------------
    // fXgN^
    //---------------------------------------------------------------------------------
    PokeList::~PokeList()
    {
      this->DeleteCursorManager();
      this->DeleteButtonManager();
      this->DeleteLayoutWork();
      this->FinalizeGraphicsSystem();
    }


    // J[\ǗIuWFNgj
    void PokeList::DeleteCursorManager( void )
    {
      GFL_DELETE m_cursor_manager;
    }


    // {^}l[Wj
    void PokeList::DeleteButtonManager( void )
    {
      if( m_button_manager ) {
        GFL_DELETE m_button_manager;
      }
    }


    // CAEg[Nj
    void PokeList::DeleteLayoutWork( void ) 
    {
      gfl::grp::g2d::LytSys* layout_system = this->GetLayoutSystem();
      layout_system->RemoveLytWk( m_layout_work );
      layout_system->ReleaseLytRes( m_layout_resource_id );
    }


    // OtBbNXVXȅI
    void PokeList::FinalizeGraphicsSystem( void )
    {
      gfl::grp::GraphicsSystem* graphics_system = this->GetGraphicsSystem();
      graphics_system->FinalizeApplicationMemory();
    }












    //---------------------------------------------------------------------------------
    // ݒύX
    //---------------------------------------------------------------------------------
    void PokeList::ChangeSetting( const Setting& new_setting )
    { 
      this->DeleteButtonManager();
      this->CreateButtonManager();

      this->SetSetting( new_setting ); 
      this->SetupButtonManager();
      this->SetupMessage();
      this->SetupReturnButtonActivity();

      for( u8 i=0; i<POKE_BUTTON_NUM; i++ ) {
        ButtonIndex button_index = static_cast<ButtonIndex>( i );
        const PokeButtonDesc& button_desc = this->GetPokeButtonDesc( button_index );
        this->SetupPokeButton( button_index, button_desc );
      }
    } 


    // {^}l[W𐶐
    void PokeList::CreateButtonManager( void )
    {
      const gfl::ui::DeviceManager* ui_device_manager = this->GetUiDeviceManager(); 
      gfl::heap::HeapBase* heap = this->GetHeap();
      m_button_manager = GFL_NEW( heap ) 
        ::app::tool::ButtonManager( heap, ui_device_manager, BUTTON_NUM, ANIMATION_NULL );
    }


    // {^}l[WZbgAbv
    void PokeList::SetupButtonManager( void )
    {
      for( u32 i=0; i<BUTTON_NUM; i++ ) {
        ButtonIndex button_index = static_cast<ButtonIndex>(i);
        const ButtonDesc& desc = this->GetButtonDesc( button_index );
        m_button_manager->AddButton( 
            this->GetHeap(), 
            i,  // ID ̓{^̃CfbNX
            m_layout_work,
            desc.pane_index,
            desc.hold_animation,
            desc.release_animation,
            desc.select_animation,
            desc.active_animation,
            desc.passive_animation,
            NULL );  // R[obNȂ
      }
    }





    // \郁bZ[W̃ZbgAbv
    void PokeList::SetupMessage( void )
    {
      PokeListBase::MessageKind kind = this->GetMessageKind();
      u32 string_id = STR_SELECT_POKE;
      switch( kind ) {
      case MESSAGE_KIND_POKE_SELECT: string_id = STR_SELECT_POKE; break;
      case MESSAGE_KIND_POKE_CHANGE: string_id = STR_SELECT_POKE_FIGHT; break;
      case MESSAGE_KIND_ITEM_USE: string_id = STR_SELECT_POKE_ITEM; break;
      }
      this->SetTextboxPaneMessage( 
          m_layout_work, PANENAME_POKE_LIST_PANE_MESSAGE_STRING, string_id );
    }




    //uǂv{^L̃ZbgAbv
    void PokeList::SetupReturnButtonActivity( void )
    {
      if( this->IsReturnEnable() ) {
        m_button_manager->SetButtonActive( BUTTON_RETURN );
      }
      else {
        m_button_manager->SetButtonPassive( BUTTON_RETURN );
      }
    }




    // w肵|P{^̃ZbgAbv
    void PokeList::SetupPokeButton( ButtonIndex button_index, const PokeButtonDesc& button_desc )
    {
      GFL_ASSERT( button_index < POKE_BUTTON_NUM  );
      if( button_index < POKE_BUTTON_NUM  ) {
        this->SetupPokeButton_Params( button_index, button_desc );
        bool selectable_kind = this->SetupPokeButton_Kind( button_index, button_desc );
        bool selectable_state = this->SetupPokeButton_State( button_index, button_desc );
        if( selectable_kind && selectable_state ) {
          m_button_manager->SetButtonActive( button_index );
        }
        else {
          m_button_manager->SetButtonPassive( button_index );
        }
      }
    } 


    // {^̎ނZbgAbv
    // ZbgAbv̌, I\ǂԂ
    bool PokeList::SetupPokeButton_Kind( 
        ButtonIndex button_index, const PokeButtonDesc& button_desc )
    {
      if( ( this->GetMode() == PokeListBase::MODE_MULTI_L ) ||
          ( this->GetMode() == PokeListBase::MODE_MULTI_R ) ) 
      {
        if( button_desc.link.party_id == PokeListBase::PARTY_PLAYER ) {
          this->SetPokeButtonDisplay_Player( button_index, button_desc );
          return true;
        }
        else {
          this->SetPokeButtonDisplay_Partner( button_index, button_desc );
          return false;
        }
      }
      else
      {
        this->SetPokeButtonDisplay_Normal( button_index, button_desc );
        return true;
      }
    } 

    // ʏ펞̃v[gɂ
    void PokeList::SetPokeButtonDisplay_Normal(
        ButtonIndex button_index, const PokeButtonDesc& button_desc )
    {
      this->StartAnimation( m_layout_work, button_desc.plate_normal_animation );
      this->StopAnimation( m_layout_work, button_desc.plate_player_animation );
      this->StopAnimation( m_layout_work, button_desc.plate_partner_animation );
    }

    // }`og̃vC[pv[gɂ
    void PokeList::SetPokeButtonDisplay_Player( 
        ButtonIndex button_index, const PokeButtonDesc& button_desc )
    {
      this->StopAnimation( m_layout_work, button_desc.plate_normal_animation );
      this->StartAnimation( m_layout_work, button_desc.plate_player_animation );
      this->StopAnimation( m_layout_work, button_desc.plate_partner_animation );
    }

    // }`og̃p[gi[pv[gɂ
    void PokeList::SetPokeButtonDisplay_Partner( 
        ButtonIndex button_index, const PokeButtonDesc& button_desc )
    {
      this->StopAnimation( m_layout_work, button_desc.plate_normal_animation );
      this->StopAnimation( m_layout_work, button_desc.plate_player_animation );
      this->StartAnimation( m_layout_work, button_desc.plate_partner_animation );
    }




    // {^̏ԂZbgAbv
    // ZbgAbv̌, I\ǂԂ
    bool PokeList::SetupPokeButton_State( 
        ButtonIndex button_index, const PokeButtonDesc& button_desc )
    {
      const BTL_PARTY* party = this->GetParty( button_desc.link.party_id );
      u8 poke_index = this->GetSortedPokeIndex( button_desc.link.party_id, button_desc.link.poke_order );
      u8 member_count = party->GetMemberCount();
      if( member_count <= poke_index ) {
        this->SetPokeButtonDisplay_Empty( button_index, button_desc );
        return false;  // Is
      }
      else {
        const BTL_POKEPARAM* poke_param = party->GetMemberDataConst( poke_index );
        if( poke_param->IsDead() ) {
          this->SetPokeButtonDisplay_Dead( button_index, button_desc );
          return false;  // Is
        }
        else {
          this->SetPokeButtonDisplay_Alive( button_index, button_desc );
          return true;  // I\
        }
      }
    } 

    // 퓬\Ԃ̕\ɂ
    void PokeList::SetPokeButtonDisplay_Alive( 
        ButtonIndex button_index, const PokeButtonDesc& button_desc )
    {
      this->StartAnimation( m_layout_work, button_desc.plate_alive_animation );
      this->StopAnimation( m_layout_work, button_desc.plate_dead_animation );
      this->StopAnimation( m_layout_work, button_desc.plate_empty_animation );
    }

    // 퓬s\Ԃ̕\ɂ
    void PokeList::SetPokeButtonDisplay_Dead(
        ButtonIndex button_index, const PokeButtonDesc& button_desc )
    {
      this->StopAnimation( m_layout_work, button_desc.plate_alive_animation );
      this->StartAnimation( m_layout_work, button_desc.plate_dead_animation );
      this->StopAnimation( m_layout_work, button_desc.plate_empty_animation );
    }

    // ̕\ɂ
    void PokeList::SetPokeButtonDisplay_Empty( 
        ButtonIndex button_index, const PokeButtonDesc& button_desc )
    {
      this->StopAnimation( m_layout_work, button_desc.plate_alive_animation );
      this->StopAnimation( m_layout_work, button_desc.plate_dead_animation );
      this->StartAnimation( m_layout_work, button_desc.plate_empty_animation );
    }






    // |P̃p[^𔽉f
    void PokeList::SetupPokeButton_Params( 
        ButtonIndex button_index, const PokeButtonDesc& button_desc )
    {
      const BTL_PARTY* party = this->GetParty( button_desc.link.party_id );
      u8 poke_index = this->GetSortedPokeIndex( button_desc.link.party_id, button_desc.link.poke_order );
      u8 member_count = party->GetMemberCount();
      const BTL_POKEPARAM* poke_param = party->GetMemberDataConst( poke_index );
      if( poke_index < member_count ) {
        // jbNl[, 
        this->SetPokeButtonDisplay_Name( button_index, button_desc, poke_param );
        this->SetPokeButtonDisplay_Sex( button_index, button_desc, poke_param );
        // x, HP
        u32 level = poke_param->GetValue( BTL_POKEPARAM::BPP_LEVEL );
        u32 max_hp = poke_param->GetValue( BTL_POKEPARAM::BPP_MAX_HP );
        u32 now_hp = poke_param->GetValue( BTL_POKEPARAM::BPP_HP );
        // ACe
        u32 item = poke_param->GetItem();
        this->SetPokeButtonDisplay_Level( button_index, button_desc, level );
        this->SetPokeButtonDisplay_Hp( button_index, button_desc, max_hp, now_hp );
        if( ( item != ITEM_DUMMY_DATA ) && ( item < ITEM_DATA_MAX ) ) {
          this->SetPokeButtonDisplay_HaveItem( button_index, button_desc );
        }
        else {
          this->SetPokeButtonDisplay_NotHaveItem( button_index, button_desc );
        }
      }
    }


    // OZbg
    void PokeList::SetPokeButtonDisplay_Name( 
        ButtonIndex button_index, const PokeButtonDesc& button_desc, 
        const BTL_POKEPARAM* btl_param )
    {
      const pml::pokepara::PokemonParam* poke_param = btl_param->GetSrcData();
      gfl::str::StrBuf* strbuf = this->GetTempStrBuf();
      poke_param->GetNickName( strbuf );
      this->SetTextboxPaneString( m_layout_work, button_desc.name_pane, strbuf );
    }


    // ʂZbg
    void PokeList::SetPokeButtonDisplay_Sex(
        ButtonIndex button_index, const PokeButtonDesc& button_desc, 
        const BTL_POKEPARAM* btl_param )
    {
      pml::Sex sex = static_cast<pml::Sex>( btl_param->GetValue( BTL_POKEPARAM::BPP_SEX ) );
      this->SetTextboxPaneSex( m_layout_work, button_desc.sex_pane, sex ); 
    }


    // x̒lZbg
    void PokeList::SetPokeButtonDisplay_Level( 
        ButtonIndex button_index, const PokeButtonDesc& button_desc, u32 level )
    {
      this->SetTextboxPaneNumber( 
          m_layout_work, button_desc.lv_value_pane, level, LV_PLACE_COUNT );
    }


    // HP̒lZbg
    void PokeList::SetPokeButtonDisplay_Hp( 
        ButtonIndex button_index, const PokeButtonDesc& button_desc, u32 max_hp, u32 now_hp )
    {
      u32 gauge_frame = ::app::tool::Gauge::GetDottoNum( now_hp, max_hp, 100 );  // ő100t[
      this->SetTextboxPaneNumber( m_layout_work, button_desc.hp_max_pane, max_hp, HP_PLACE_COUNT );
      this->SetTextboxPaneNumber( m_layout_work, button_desc.hp_value_pane, now_hp, HP_PLACE_COUNT );
      this->StartAnimation( m_layout_work, button_desc.hp_gauge_animation );
      this->PauseAnimation( m_layout_work, button_desc.hp_gauge_animation, false );
      this->SetAnimationFrame( m_layout_work, button_desc.hp_gauge_animation, gauge_frame );
    } 


    // ACeĂԂ̕\ɂ
    void PokeList::SetPokeButtonDisplay_HaveItem( 
        ButtonIndex button_index, const PokeButtonDesc& button_desc )
    {
      this->SetPaneVisible( m_layout_work, button_desc.item_icon_pane );
    } 

    // ACeĂȂԂ̕\ɂ
    void PokeList::SetPokeButtonDisplay_NotHaveItem(
        ButtonIndex button_index, const PokeButtonDesc& button_desc )
    {
      this->SetPaneInvisible( m_layout_work, button_desc.item_icon_pane );
    }







    




    //---------------------------------------------------------------------------------
    // ŏĊJ( ݒ͈p )
    //---------------------------------------------------------------------------------
    void PokeList::Restart( void )
    {
      this->ResetFinishFlag();
      this->ResetResult();
      m_seq = SEQ_INPUT_WAIT;
      m_next_seq = SEQ_INPUT_WAIT;
      m_wait_animation_index = ANIMATION_NULL;
      this->InitializeCursorPos();
    }


    // J[\ʒu
    void PokeList::InitializeCursorPos( void )
    {
      ButtonIndex cursor_pos = static_cast<ButtonIndex>( this->GetInitCursorPos() );
      const ButtonDesc& init_button_desc = this->GetButtonDesc( cursor_pos );
      m_cursor_manager->SetPos( cursor_pos );
      this->StartAnimation( m_layout_work, init_button_desc.cursor_animation );
      this->StartAnimation( m_layout_work, ANIMATION_CURSOR_FLUCTUATE );
    }





    //---------------------------------------------------------------------------------
    // J[\w肵|Pɍ킹
    //---------------------------------------------------------------------------------
    void PokeList::SetCursorPos( PartyId party_id, u8 poke_order )
    {
      ButtonIndex old_cursor_pos = this->GetCurrentCursorPos();
      ButtonIndex new_cursor_pos = this->GetButtonIndex( party_id, poke_order );
      m_cursor_manager->SetPos( new_cursor_pos );
      this->StartCursorMoveAnimation( old_cursor_pos, new_cursor_pos );
    }





    //---------------------------------------------------------------------------------
    // I[g`tOZbg
    //---------------------------------------------------------------------------------
    void PokeList::SetAutoDrawFlag( bool flag )
    {
      if( m_layout_work ) {
        m_layout_work->SetAutoDrawFlag( flag );
      }
    }







    //---------------------------------------------------------------------------------
    // ւAj[VJn
    //---------------------------------------------------------------------------------
    void PokeList::StartExAnimation( PartyId party_id, u8 poke_order1, u8 poke_order2 )
    {
      GFL_ASSERT( poke_order1 < POKE_BUTTON_NUM );
      GFL_ASSERT( poke_order2 < POKE_BUTTON_NUM );
      GFL_ASSERT( this->IsExAnimationFinished() );

      if( this->IsExAnimationFinished() ) {
        m_next_seq = m_seq;
        m_seq = SEQ_EXOUT_ANIME_START;
        m_exchange_button[0] = this->GetButtonIndex( party_id, poke_order1 );
        m_exchange_button[1] = this->GetButtonIndex( party_id, poke_order2 );
      }
    } 

    //---------------------------------------------------------------------------------
    // ւAj[VIǂ𒲂ׂ
    //---------------------------------------------------------------------------------
    bool PokeList::IsExAnimationFinished( void ) const
    {
      return ( 
          ( m_seq != SEQ_EXOUT_ANIME_START ) && 
          ( m_seq != SEQ_EXOUT_ANIME_WAIT ) && 
          ( m_seq != SEQ_EXIN_ANIME_START ) && 
          ( m_seq != SEQ_EXIN_ANIME_WAIT ) );
    }






    //---------------------------------------------------------------------------------
    // XV
    //---------------------------------------------------------------------------------
    void PokeList::Update( void )
    {
      m_button_manager->Update();
      this->UpdateSequence(); 
    }


    // V[PXXV
    void PokeList::UpdateSequence( void )
    {
      const gfl::ui::Button* ui_button = 
        this->GetUiDeviceManager()->GetButton( gfl::ui::CTR_DeviceManager::BUTTON_STANDARD );

      switch( m_seq )
      {
      case SEQ_INPUT_WAIT:
        if( ui_button->IsTrigger( gfl::ui::BUTTON_X ) ) {
          this->StartExAnimation( PokeListBase::PARTY_PLAYER, 0, 1 );
        }
        else if( ui_button->IsTrigger( gfl::ui::BUTTON_Y ) ) {
          this->StartExAnimation( PokeListBase::PARTY_PLAYER, 0, 2 );
        }
        else
        if( this->IsButtonSelected() ) {
          ButtonIndex cursor_pos = this->GetSelectedCursorPosByButton();
          if( this->IsSelectEnable( cursor_pos ) ) {
            this->SetResultByCursorPos( cursor_pos );
            this->StartSelectAnimation( cursor_pos );
            m_seq = SEQ_ANIME_WAIT;
            m_next_seq = SEQ_FINISH;
          }
        }
        else if( this->IsButtonHolding() ) {
          // do nothing
        }
        else if( ui_button->IsTrigger( gfl::ui::BUTTON_A ) ) {
          ButtonIndex cursor_pos = this->GetCurrentCursorPos();
          if( this->IsSelectEnable( cursor_pos ) ) {
            this->SetResultByCursorPos( cursor_pos );
            this->StartSelectAnimation( cursor_pos );
            m_seq = SEQ_ANIME_WAIT;
            m_next_seq = SEQ_FINISH;
          }
        }
        else if( this->UpdateCursorMove() ) {  // J[\ړ
          ButtonIndex old_cursor_pos = this->GetPrevCursorPos();
          ButtonIndex new_cursor_pos = this->GetCurrentCursorPos();
          this->StartCursorMoveAnimation( old_cursor_pos, new_cursor_pos );
        }
        break;


      case SEQ_ANIME_WAIT:
        if( this->IsSelectAnimationFinished() ) {
          m_seq = m_next_seq;
        }
        break;


      case SEQ_EXOUT_ANIME_START:
        this->StartExchangeOutAnimation();
        m_seq = SEQ_EXOUT_ANIME_WAIT;
        break;


      case SEQ_EXOUT_ANIME_WAIT:
        if( this->IsExchangeOutAnimationFinished() ) {
          this->StopExchangeOutAnimation();
          this->ExchangePokeButtons();
          m_seq = SEQ_EXIN_ANIME_START;
        }
        break;


      case SEQ_EXIN_ANIME_START:
        this->StartExchangeInAnimation();
        m_seq = SEQ_EXIN_ANIME_WAIT;
        break;


      case SEQ_EXIN_ANIME_WAIT:
        if( this->IsExchangeInAnimationFinished() ) {
          this->StopExchangeInAnimation();
          m_seq = m_next_seq;
        }
        break;


      default:
        GFL_ASSERT(0);
        // fall through

      case SEQ_FINISH:
        GFL_ASSERT( this->GetResult() != RESULT_NULL );
        this->SetFinishFlag();
        break; 
      }
    }






    // {^Ă邩ǂ𒲂ׂ
    bool PokeList::IsButtonHolding( void ) const
    {
      return m_button_manager->IsButtonHolding();
    }

    // {^Iꂽǂ𒲂ׂ
    bool PokeList::IsButtonSelected( void ) const
    {
      return m_button_manager->IsButtonSelected();
    }

    // Iꂽ{^Ɋ֘AtĂJ[\ʒu擾
    PokeList::ButtonIndex PokeList::GetSelectedCursorPosByButton( void ) const
    {
      return static_cast<ButtonIndex>( m_button_manager->GetSelectedButtonId() );
    }




    // J[\̓XV
    bool PokeList::UpdateCursorMove( void )
    {
      ::app::tool::CursorManager::MOVE_RET move_result = m_cursor_manager->MainTrigger();
      return( move_result == ::app::tool::CursorManager::RET_MOVE ); // ړǂ
    } 


    // ݂̃J[\ʒu擾
    PokeList::ButtonIndex PokeList::GetCurrentCursorPos( void ) const
    {
      return static_cast<ButtonIndex>( m_cursor_manager->GetID() );
    } 


    // ÕJ[\ʒu擾
    PokeList::ButtonIndex PokeList::GetPrevCursorPos( void ) const
    {
      return static_cast<ButtonIndex>( m_cursor_manager->GetOldID() );
    }


    // J[\ړۂ̏
    void PokeList::StartCursorMoveAnimation( 
        PokeList::ButtonIndex old_cursor_pos, PokeList::ButtonIndex new_cursor_pos )
    {
      const ButtonDesc& old_button_desc = this->GetButtonDesc( old_cursor_pos );
      const ButtonDesc& new_button_desc = this->GetButtonDesc( new_cursor_pos );
      this->StopAnimation( m_layout_work, old_button_desc.cursor_animation );
      this->StartAnimation( m_layout_work, new_button_desc.cursor_animation );
    }






    // I\ǂ𒲂ׂ
    bool PokeList::IsSelectEnable( ButtonIndex cursor_pos ) const
    {
      return m_button_manager->IsButtonActive( cursor_pos );
    }




    // J[\ʒu, 쌋ʂZbg
    void PokeList::SetResultByCursorPos( ButtonIndex cursor_pos )
    {
      if( cursor_pos != BUTTON_RETURN ) {
        const PokeButtonDesc& poke_button_desc = this->GetPokeButtonDesc( cursor_pos );
        this->SetSelectedPokeOrder( poke_button_desc.link.poke_order );  // Ԗڂ̃|PI񂾂H
      } 

      const ButtonDesc& button_desc = this->GetButtonDesc( cursor_pos );
      this->SetResult( button_desc.result );
    } 


    // ȊIAj[VJn
    void PokeList::StartSelectAnimation( ButtonIndex cursor_pos )
    {
      const ButtonDesc& desc = this->GetButtonDesc( cursor_pos );
      m_wait_animation_index = desc.select_animation;
      this->StartAnimation( m_layout_work, m_wait_animation_index );
    }


    // ȊIAj[VIǂ𒲂ׂ
    bool PokeList::IsSelectAnimationFinished( void ) const
    {
      if( m_wait_animation_index < ANIMATION_NUM ) {
        return m_layout_work->IsAnimeEnd( m_wait_animation_index );
      }
      return true;  // Aj[Vo^ȂIƂɂ
    }






    // ւ( OUT )Aj[VJn
    void PokeList::StartExchangeOutAnimation( void )
    {
      for( u8 i=0; i<2; i++ ) {
        const PokeButtonDesc& desc = this->GetPokeButtonDesc( m_exchange_button[i] );
        this->StartAnimation( m_layout_work, desc.out_animation );
      }
    } 
    
    // ւ( OUT )Aj[V~
    void PokeList::StopExchangeOutAnimation( void )
    {
      for( u8 i=0; i<2; i++ ) {
        const PokeButtonDesc& desc = this->GetPokeButtonDesc( m_exchange_button[i] );
        this->StopAnimation( m_layout_work, desc.out_animation );
      }
    } 

    // ւ( OUT )Aj[VIǂ𒲂ׂ
    bool PokeList::IsExchangeOutAnimationFinished( void ) const 
    {
      for( u8 i=0; i<2; i++ ) {
        const PokeButtonDesc& desc = this->GetPokeButtonDesc( m_exchange_button[i] );
        if( m_layout_work->IsAnimeEnd( desc.out_animation ) == false ) {
          return false;
        }
      }
      return true;
    }





    // ւ( IN )Aj[VJn
    void PokeList::StartExchangeInAnimation( void )
    {
      for( u8 i=0; i<2; i++ ) {
        const PokeButtonDesc& desc = this->GetPokeButtonDesc( m_exchange_button[i] );
        this->StartAnimation( m_layout_work, desc.in_animation );
      }
    } 

    // ւ( IN )Aj[V~
    void PokeList::StopExchangeInAnimation( void )
    {
      for( u8 i=0; i<2; i++ ) {
        const PokeButtonDesc& desc = this->GetPokeButtonDesc( m_exchange_button[i] );
        this->StopAnimation( m_layout_work, desc.in_animation );
      }
    } 

    // ւ( IN )Aj[VIǂ𒲂ׂ
    bool PokeList::IsExchangeInAnimationFinished( void ) const 
    {
      for( u8 i=0; i<2; i++ ) {
        const PokeButtonDesc& desc = this->GetPokeButtonDesc( m_exchange_button[i] );
        if( m_layout_work->IsAnimeEnd( desc.in_animation ) == false ) {
          return false;
        }
      }
      return true;
    }


    // |P{^ւ
    void PokeList::ExchangePokeButtons( void )
    {
      // f[^XV
      {
        const PokeButtonDesc& desc0 = this->GetPokeButtonDesc( m_exchange_button[0] );
        const PokeButtonDesc& desc1 = this->GetPokeButtonDesc( m_exchange_button[1] );
        if( desc0.link.party_id == desc1.link.party_id ) { // p[eBԂł̓ւ̏ꍇ, |PԂ̕яω
          this->ExchangeSortedPokeIndex( 
              desc0.link.party_id, desc0.link.poke_order, desc1.link.poke_order );
        }
        else { // قȂp[eBԂł̓ւ̏ꍇ, {^̃N񂪕ω
          this->ExchangePokeButtonLinks( m_exchange_button[0], m_exchange_button[1] );
        }
      }

      // \XV
      const PokeButtonDesc& desc0 = this->GetPokeButtonDesc( m_exchange_button[0] );
      const PokeButtonDesc& desc1 = this->GetPokeButtonDesc( m_exchange_button[1] );
      this->SetupPokeButton( m_exchange_button[0], desc0 );
      this->SetupPokeButton( m_exchange_button[1], desc1 );

    }

    // |P{^̃Nւ
    void PokeList::ExchangePokeButtonLinks( ButtonIndex button1, ButtonIndex button2 )
    {
      GFL_ASSERT( ( button1 < POKE_BUTTON_NUM ) && ( button2 < POKE_BUTTON_NUM ) );
      if( ( button1 < POKE_BUTTON_NUM ) && ( button2 < POKE_BUTTON_NUM ) ) {
        PokeButtonLink temp = m_poke_button_desc[ button1 ].link;
        m_poke_button_desc[ button1 ].link = m_poke_button_desc[ button2 ].link;
        m_poke_button_desc[ button2 ].link = temp;
      }
    }



  }  // namespace app
}  // namespace btl
