//=================================================================================================
/**
 * @brief BGMǗNX
 * @file  BGMManager.h
 * @author obata
 * @date   2011.06.29
 */
//================================================================================================= 
#include <gflib.h>
#include <system/thread_priority.h>
#include <sound/PokemonSoundSystem.h>
#include <sound/BGMManager.h>
#include "BGMPrepareThread.h"

// fobOo̓XCb`
#define DEBUG_PRINT_ON


namespace snd {



  //-----------------------------------------------------------------------------------------------
  // eԂɑ΂, NGXg̉se[u
  //-----------------------------------------------------------------------------------------------
  // RequestAcceptTable[ VXe ][ NGXg ]  true ̎,
  // VXe ̃NGXg邱Ƃo.
  const u8 BGMManager::RequestAcceptTable[ BGMManager::STATE_NUM ][ BGMManager::REQUEST_NUM ] = 
  {
    /*                       NONE    F_IN    F_OUT   PUSH    POP     CHANGE  S_BY    F_PLAY */
    /* STOP             */{  false,  false,  false,  false,   true,   true,   true,   true,  },
    /* PLAY             */{  false,   true,   true,   true,   true,   true,   true,   true,  },
    /* WAIT             */{  false,   true,  false,  false,  false,  false,  false,   true,  },
    /* FADE_OUT         */{  false,  false,  false,  false,  false,  false,  false,  false,  },
    /* FALDE_IN         */{  false,  false,  false,  false,  false,  false,  false,  false,  },
    /* PUSH             */{  false,  false,  false,  false,  false,  false,  false,  false,  },
    /* POP_out          */{  false,  false,  false,  false,  false,  false,  false,  false,  },
    /* POP_in           */{  false,  false,  false,  false,  false,  false,  false,  false,  },
    /* CHANGE_out       */{  false,  false,  false,   true,  false,   true,  false,  false,  },
    /* CHANGE_load      */{  false,  false,  false,   true,  false,   true,  false,  false,  },
    /* CHANGE_in        */{  false,  false,  false,  false,  false,   true,  false,  false,  },
    /* CHANGE_PUSH_out  */{  false,  false,  false,  false,  false,  false,  false,  false,  },
    /* CHANGE_PUSH_load */{  false,  false,  false,  false,  false,  false,  false,  false,  },
    /* STAND_BY_out     */{  false,  false,  false,  false,  false,  false,  false,  false,  },
    /* STAND_BY_load    */{  false,  false,  false,  false,  false,  false,  false,  false,  },
  };


  //-----------------------------------------------------------------------------------------------
  // VKNGXgɂfbhbN e[u
  //-----------------------------------------------------------------------------------------------
  //  DeadRockTable[ ŏINGXg ][ VKNGXg ]  false ̎, 
  //  VKNGXg󗝂邱Ƃo, fbhbNԂƂȂ.
  const u8 BGMManager::DeadRockTable[ BGMManager::REQUEST_NUM ][ BGMManager::REQUEST_NUM ] = 
  {
    /*             NONE    F_IN    F_OUT   PUSH    POP     CHANGE  S_BY    F_PLAY */
    /* NONE   */{  false,  false,  false,  false,  false,  false,  false,  false,  }, 
    /* F_IN   */{  false,   true,   true,   true,   true,   true,   true,   true,  },
    /* F_OUT  */{  false,   true,  false,  false,  false,  false,  false,   true,  },
    /* PUSH   */{  false,  false,  false,  false,   true,   true,   true,   true,  },
    /* POP    */{  false,   true,   true,   true,   true,   true,   true,   true,  },
    /* CHANGE */{  false,   true,   true,   true,   true,   true,   true,   true,  },
    /* S_BY   */{  false,   true,  false,  false,  false,  false,  false,   true,  },
    /* F_PLAY */{  false,   true,   true,   true,   true,   true,   true,   true,  },
  };


  // TEhACeID̖l
  const u32 BGMManager::SOUND_ITEM_ID_NULL = 0;

  // NGXgL[̑傫
  const u8 BGMManager::REQUEST_QUEUE_SIZE = 32;






  //-----------------------------------------------------------------------------------------------
  // RXgN^
  //-----------------------------------------------------------------------------------------------
  BGMManager::BGMManager( gfl::heap::HeapBase* heap, snd::PokemonSoundSystem* sound_system ) : 
    m_sound_system( sound_system ),
    m_request_queue( heap, REQUEST_QUEUE_SIZE ),
    state( STATE_STOP ),
    currentBGM( 0 ),
    requestBGM( 0 ),
    loadBGM( 0 ),
    loadSeq( 0 ),
    fadeInFrame( 0 ),
    fadeOutFrame( 0 ),
    request( REQUEST_NONE ),
    m_heap_for_work( heap ),
    m_bgm_prepare_thread( NULL )
  {
    this->Initialize();
  }

  
  // Ԃ
  void BGMManager::Initialize( void )
  {
    state        = STATE_STOP;
    request      = REQUEST_NONE;
    currentBGM   = SOUND_ITEM_ID_NULL;
    requestBGM   = SOUND_ITEM_ID_NULL;
    loadBGM      = SOUND_ITEM_ID_NULL;
    loadSeq      = 0;
    fadeInFrame  = 0;
    fadeOutFrame = 0;
    m_request_queue.Clear();
  } 



  //-----------------------------------------------------------------------------------------------
  // fXgN^
  //-----------------------------------------------------------------------------------------------
  BGMManager::~BGMManager()
  {
  }



  //-----------------------------------------------------------------------------------------------
  // Zbg
  //-----------------------------------------------------------------------------------------------
  void BGMManager::Reset( void )
  {
    // @todo Đ~
    // @todo ޔԂZbg
    this->Initialize();
  }






  //-----------------------------------------------------------------------------------------------
  // ( or  )̃NGXgĂ邩ǂ
  //-----------------------------------------------------------------------------------------------
  bool BGMManager::HaveRequest( void ) const
  {
    return ( QueueHaveRequest() || ( request != REQUEST_NONE ) );
  }


  // L[ɃNGXg邩ǂ𔻒肷
  bool BGMManager::QueueHaveRequest( void ) const
  {
    return !m_request_queue.IsEmpty();
  }





  //-----------------------------------------------------------------------------------------------
  // NGXgL[ɓo^( FADE_IN )
  //-----------------------------------------------------------------------------------------------
  void BGMManager::RegisterRequest_FADE_IN( u32 fade_in_frame )
  { 
    RequestData request_data;
    request_data.request_type = REQUEST_FADE_IN;
    request_data.sound_item_id = SOUND_ITEM_ID_NULL;
    request_data.fade_out_frame = 0;
    request_data.fade_in_frame = fade_in_frame;
    this->RegisterNewRequest( request_data ); 
  }
  //-----------------------------------------------------------------------------------------------
  // NGXgL[ɓo^( FADE_OUT )
  //-----------------------------------------------------------------------------------------------
  void BGMManager::RegisterRequest_FADE_OUT( u32 fade_out_frame )
  { 
    RequestData request_data;
    request_data.request_type = REQUEST_FADE_OUT;
    request_data.sound_item_id = SOUND_ITEM_ID_NULL;
    request_data.fade_out_frame = fade_out_frame;
    request_data.fade_in_frame = 0;
    this->RegisterNewRequest( request_data ); 
  }
  //-----------------------------------------------------------------------------------------------
  // NGXgL[ɓo^( PUSH )
  //-----------------------------------------------------------------------------------------------
  void BGMManager::RegisterRequest_PUSH( u32 fade_out_frame )
  { 
    RequestData request_data;
    request_data.request_type = REQUEST_PUSH;
    request_data.sound_item_id = SOUND_ITEM_ID_NULL;
    request_data.fade_out_frame = fade_out_frame;
    request_data.fade_in_frame = 0;
    this->RegisterNewRequest( request_data ); 
  }
  //-----------------------------------------------------------------------------------------------
  // NGXgL[ɓo^( POP )
  //-----------------------------------------------------------------------------------------------
  void BGMManager::RegisterRequest_POP( u32 fade_out_frame, u32 fade_in_frame )
  { 
    RequestData request_data;
    request_data.request_type = REQUEST_POP;
    request_data.sound_item_id = SOUND_ITEM_ID_NULL;
    request_data.fade_out_frame = fade_out_frame;
    request_data.fade_in_frame = fade_in_frame;
    this->RegisterNewRequest( request_data ); 
  }
  //-----------------------------------------------------------------------------------------------
  // NGXgL[ɓo^( CAHNGE )
  //-----------------------------------------------------------------------------------------------
  void BGMManager::RegisterRequest_CHANGE( u32 sound_item_id, u32 fade_out_frame, u32 fade_in_frame )
  { 
    RequestData request_data;
    request_data.request_type = REQUEST_CHANGE;
    request_data.sound_item_id = sound_item_id;
    request_data.fade_out_frame = fade_out_frame;
    request_data.fade_in_frame = fade_in_frame;
    this->RegisterNewRequest( request_data ); 
  }
  //-----------------------------------------------------------------------------------------------
  // NGXgL[ɓo^( STAND_BY )
  //-----------------------------------------------------------------------------------------------
  void BGMManager::RegisterRequest_STAND_BY( u32 sound_item_id, u32 fade_out_frame )
  { 
    RequestData request_data;
    request_data.request_type = REQUEST_STAND_BY;
    request_data.sound_item_id = sound_item_id;
    request_data.fade_out_frame = fade_out_frame;
    request_data.fade_in_frame = 0;
    this->RegisterNewRequest( request_data ); 
  }
  //-----------------------------------------------------------------------------------------------
  // NGXgL[ɓo^( FORCE_PLAY )
  //-----------------------------------------------------------------------------------------------
  void BGMManager::RegisterRequest_FORCE_PLAY( u32 sound_item_id )
  { 
    RequestData request_data;
    request_data.request_type = REQUEST_FORCE_PLAY;
    request_data.sound_item_id = sound_item_id;
    request_data.fade_out_frame = 0;
    request_data.fade_in_frame = 0;
    this->RegisterNewRequest( request_data ); 
  }


  // VKNGXgNGXgL[ɓo^
  void BGMManager::RegisterNewRequest( const RequestData& requestData )
  {
    // fbhbN
    if( this->CanRegisterRequest( requestData.request_type ) == false ) {
      GFL_PRINT( "FIELD-SOUND-QUEUE: Can't register request(%d)\n", requestData.request_type );
      this->DebugPrint_AllInfo();
      GFL_ASSERT(0);
      return;
    }

    // NGXgL[ I[o[t[
    if( m_request_queue.IsFull() ) {
      GFL_PRINT( "FIELD-SOUND: Request queue overflow\n" );
      GFL_ASSERT(0);
      return;
    }

    // o^
    m_request_queue.Enqueue( requestData );

#ifdef DEBUG_PRINT_ON
    GFL_PRINT( "FIELD-SOUND-QUEUE: Regist new request\n" );
    DebugPrint_RequestQueue();
#endif
  }


  // NGXgL[ɃNGXgo^ł邩ǂ𔻒肷
  bool BGMManager::CanRegisterRequest( RequestType request ) const
  { 
    // L[ɃNGXg݂
    if( !m_request_queue.IsEmpty() ) {
      u32 request_count = m_request_queue.GetDataNum();
      const RequestData& tail_request = m_request_queue.GetData( request_count - 1 );
      return DeadRockTable[ tail_request.request_type ][ request ];
    }

    // Ō̃NGXg
    if( this->request != REQUEST_NONE ) {
      return DeadRockTable[ this->request ][ request ];
    }

    // NGXg݂Ȃ, ԂɈˑ
    return CanAcceptRequest( request );
  }









  //-----------------------------------------------------------------------------------------------
  // t[̍XV
  //-----------------------------------------------------------------------------------------------
  void BGMManager::Update( void )
  {
    // NGXgL[烊NGXgo
    this->RequestQueueCheck();

    // NGXgɑ΂锽
    switch( state ) {
    case STATE_STOP:  this->RequestCheck_STOP();  break;
    case STATE_PLAY:  this->RequestCheck_PLAY();  break;
    case STATE_WAIT:  this->RequestCheck_WAIT();  break;
    case STATE_FADE_OUT:  this->RequestCheck_FADE_OUT();  break;
    case STATE_FADE_IN:  this->RequestCheck_FADE_IN();  break;
    case STATE_PUSH:  this->RequestCheck_PUSH();  break;
    case STATE_POP_out:  this->RequestCheck_POP_out();  break;
    case STATE_POP_in:  this->RequestCheck_POP_in();  break;
    case STATE_CHANGE_out:  this->RequestCheck_CHANGE_out();  break;
    case STATE_CHANGE_load:  this->RequestCheck_CHANGE_load();  break;
    case STATE_CHANGE_in:  this->RequestCheck_CHANGE_in();  break;
    case STATE_CHANGE_PUSH_out:  this->RequestCheck_CHANGE_PUSH_out();  break;
    case STATE_CHANGE_PUSH_load:  this->RequestCheck_CHANGE_PUSH_load();  break;
    case STATE_STAND_BY_out:  this->RequestCheck_STAND_BY_out();  break;
    case STATE_STAND_BY_load:  this->RequestCheck_STAND_BY_load();  break;
    }

    // 
    switch( state ) {
    case STATE_STOP:  this->Main_STOP();  break;
    case STATE_PLAY:  this->Main_PLAY();  break;
    case STATE_WAIT:  this->Main_WAIT();  break;
    case STATE_FADE_OUT:  this->Main_FADE_OUT();  break;
    case STATE_FADE_IN:  this->Main_FADE_IN();  break;
    case STATE_PUSH:  this->Main_PUSH();  break;
    case STATE_POP_out:  this->Main_POP_out();  break;
    case STATE_POP_in:  this->Main_POP_in();  break;
    case STATE_CHANGE_out:  this->Main_CHANGE_out();  break;
    case STATE_CHANGE_load:  this->Main_CHANGE_load();  break;
    case STATE_CHANGE_in:  this->Main_CHANGE_in();  break;
    case STATE_CHANGE_PUSH_out:  this->Main_CHANGE_PUSH_out();  break;
    case STATE_CHANGE_PUSH_load:  this->Main_CHANGE_PUSH_load();  break;
    case STATE_STAND_BY_out:  this->Main_STAND_BY_out();  break;
    case STATE_STAND_BY_load:  this->Main_STAND_BY_load();  break;
    }
  } 



  // NGXgL[̐擪NGXgo
  void BGMManager::RequestQueueCheck( void )
  {
    if( this->QueueHaveRequest() ) {
      const RequestData headRequestData = m_request_queue.GetData(0);
      if( CanAcceptRequest( headRequestData.request_type ) ) {
        ThrowRequest( headRequestData );
        m_request_queue.Dequeue();
      }
    }
  }



  // NGXg̔s
  void BGMManager::ThrowRequest( const RequestData& request_data )
  {
    switch( request_data.request_type ) {
    case REQUEST_NONE:
      break;
    case REQUEST_FADE_IN:  
      ThrowRequest_FADE_IN( request_data.fade_in_frame );
      break;
    case REQUEST_FADE_OUT:
      ThrowRequest_FADE_OUT( request_data.fade_out_frame );
      break;
    case REQUEST_PUSH:
      ThrowRequest_PUSH( request_data.fade_out_frame );
      break;
    case REQUEST_POP:
      ThrowRequest_POP( request_data.fade_out_frame, request_data.fade_in_frame );
      break;
    case REQUEST_CHANGE:
      ThrowRequest_CHANGE( request_data.sound_item_id, request_data.fade_out_frame, request_data.fade_in_frame );
      break;
    case REQUEST_STAND_BY:
      ThrowRequest_STAND_BY( request_data.sound_item_id, request_data.fade_out_frame );
      break;
    case REQUEST_FORCE_PLAY:
      ThrowRequest_FORCE_PLAY( request_data.sound_item_id );
      break;
    default:
      GFL_ASSERT(0);
      break;
    }
  }


  // NGXg̔s ( REQUEST_FADE_IN )
  void BGMManager::ThrowRequest_FADE_IN( u32 fadeInFrame )
  {
    if( CanAcceptRequest( REQUEST_FADE_IN ) == false ) {
      GFL_PRINT( "FIELD-SOUND: Can't Accept BGM fade in request\n" );
      GFL_ASSERT(0);
      return;
    }
    this->request = REQUEST_FADE_IN;
    this->fadeInFrame = fadeInFrame;

#ifdef DEBUG_PRINT_ON
    GFL_PRINT( "FIELD-SOUND: accept request(FADE_IN)\n" );
#endif
  }


  // NGXg̔s ( REQUEST_FADE_OUT )
  void BGMManager::ThrowRequest_FADE_OUT( u32 fadeOutFrame )
  {
    if( CanAcceptRequest(REQUEST_FADE_OUT) == false ) {
      GFL_PRINT( "FIELD-SOUND: Can't Accept BGM fade out request\n" );
      GFL_ASSERT(0);
      return;
    }
    this->request = REQUEST_FADE_OUT;
    this->fadeOutFrame = fadeOutFrame;

#ifdef DEBUG_PRINT_ON
    GFL_PRINT( "FIELD-SOUND: accept request(FADE_OUT)\n" );
#endif
  }


  // NGXg̔s ( REQUEST_PUSH )
  void BGMManager::ThrowRequest_PUSH( u32 fadeOutFrame )
  {
    if( CanAcceptRequest(REQUEST_PUSH) == false ) {
      GFL_PRINT( "FIELD-SOUND: Can't Accept BGM push request\n" );
      GFL_ASSERT(0);
      return;
    }
    this->request = REQUEST_PUSH;
    this->fadeOutFrame = fadeOutFrame;

#ifdef DEBUG_PRINT_ON
    GFL_PRINT( "FIELD-SOUND: accept request(PUSH)\n" );
#endif
  }


  // NGXg̔s ( REQUEST_POP )
  void BGMManager::ThrowRequest_POP( u32 fadeOutFrame, u32 fadeInFrame )
  {
    if( CanAcceptRequest(REQUEST_POP) == false ) {
      GFL_PRINT( "FIELD-SOUND: Can't Accept BGM pop request\n" );
      GFL_ASSERT(0);
      return;
    }
    this->request = REQUEST_POP;
    this->fadeInFrame = fadeInFrame;
    this->fadeOutFrame = fadeOutFrame;

#ifdef DEBUG_PRINT_ON
    GFL_PRINT( "FIELD-SOUND: accept request(POP)\n" );
#endif
  }


  // NGXg̔s ( REQUEST_CHANGE )
  void BGMManager::ThrowRequest_CHANGE( u32 sound_item_id, u32 fadeOutFrame, u32 fadeInFrame )
  {
    // NGXg󗝂łȂ
    if( CanAcceptRequest(REQUEST_CHANGE) == false ) {
      GFL_PRINT( "FIELD-SOUND: Can't Accept BGM change request\n" );
      GFL_ASSERT(0);
      return;
    } 

    // NGXg
    this->request = REQUEST_CHANGE;
    this->requestBGM = sound_item_id;
    this->fadeInFrame = fadeInFrame;
    this->fadeOutFrame = fadeOutFrame;

#ifdef DEBUG_PRINT_ON
    GFL_PRINT( "FIELD-SOUND: accept request(CHANGE)\n" );
#endif
  }


  // NGXg̔s ( REQUEST_STAND_BY )
  void BGMManager::ThrowRequest_STAND_BY( u32 sound_item_id, u32 fadeOutFrame )
  {
    if( CanAcceptRequest(REQUEST_STAND_BY) == false ) {
      GFL_PRINT( "FIELD-SOUND: Can't Accept BGM stand-by request\n" );
      GFL_ASSERT(0);
      return;
    }

    // NGXg
    this->request = REQUEST_STAND_BY;
    this->requestBGM = sound_item_id;
    this->fadeOutFrame = fadeOutFrame;

#ifdef DEBUG_PRINT_ON
    GFL_PRINT( "FIELD-SOUND: accept request(STAND_BY)\n" );
#endif
  }


  // NGXg̔s ( REQUEST_FORCE_PLAY )
  void BGMManager::ThrowRequest_FORCE_PLAY( u32 sound_item_id )
  {
    if( CanAcceptRequest(REQUEST_FORCE_PLAY) == false ) {
      GFL_PRINT( "FIELD-SOUND: Can't Accept BGM force play request\n" );
      GFL_ASSERT(0);
      return;
    }

    // NGXg
    this->request = REQUEST_FORCE_PLAY;
    this->requestBGM = sound_item_id;

#ifdef DEBUG_PRINT_ON
    GFL_PRINT( "FIELD-SOUND: accept request(FORCE_PLAY)\n" );
#endif
  }



  // NGXg󗝂ł邩ǂ𔻒肷
  bool BGMManager::CanAcceptRequest( RequestType request ) const
  {
    return RequestAcceptTable[ this->state ][ request ];
  }




  //  STOP , e탊NGXgɑ΂铮
  void BGMManager::RequestCheck_STOP( void )
  {
    switch( this->request )
    {
    case REQUEST_NONE:  
      break;
    case REQUEST_POP:
      PopBGM();
      FadeInBGM();
      ChangeState( STATE_POP_in );
      break;
    case REQUEST_CHANGE:  
      ChangeState( STATE_CHANGE_out );
      break;
    case REQUEST_STAND_BY:  
      ChangeState( STATE_STAND_BY_out );
      break;
    case REQUEST_FORCE_PLAY:
      ForcePlayBGM();
      ChangeState( STATE_PLAY );
      FinishRequest();
      break;
    default:  
      GFL_ASSERT(0);
    }
  }

  //  PLAY , e탊NGXgɑ΂铮
  void BGMManager::RequestCheck_PLAY( void )
  {
    switch( this->request )
    {
    case REQUEST_NONE: 
      break;
    case REQUEST_FADE_IN:
      // Đ̃NGXg͖
      CancelRequest( );
      break;
    case REQUEST_FADE_OUT: 
      FadeOutBGM( );
      ChangeState( STATE_FADE_OUT );
      break;
    case REQUEST_PUSH: 
      FadeOutBGM( );
      ChangeState( STATE_PUSH );
      break;
    case REQUEST_POP: 
      FadeOutBGM( );
      ChangeState( STATE_POP_out );
      break;
    case REQUEST_CHANGE: 
      // Đ̋Ȃ̃NGXg͖
      if( this->requestBGM == this->currentBGM ) {
        CancelRequest();
      }
      else {
        FadeOutBGM();
        ChangeState( STATE_CHANGE_out );
      }
      break;
    case REQUEST_STAND_BY: 
      // Đ̋Ȃ̃NGXg͖
      if( this->requestBGM == this->currentBGM ) {
        CancelRequest( );
      }
      else {
        FadeOutBGM( );
        ChangeState( STATE_STAND_BY_out );
      }
      break;
    case REQUEST_FORCE_PLAY:
      // Đ̋Ȃ̃NGXg͖
      if( this->requestBGM == this->currentBGM ) {
        CancelRequest(  );
      }
      else {
        ForcePlayBGM(  );
        ChangeState( STATE_PLAY );
        FinishRequest(  );
      }
      break;
    default:
      GFL_ASSERT(0);
    }
  }

  //  WAIT , e탊NGXgɑ΂铮
  void BGMManager::RequestCheck_WAIT( void )
  {
    switch( this->request )
    {
    case REQUEST_NONE: 
      break;
    case REQUEST_FADE_IN: 
      FadeInBGM( );
      ChangeState( STATE_FADE_IN );
      break;
    case REQUEST_FORCE_PLAY: 
      ForcePlayBGM( );
      ChangeState( STATE_PLAY );
      FinishRequest( );
      break;
    default:
      GFL_ASSERT(0);
    }
  }

  //  FADE_OUT , e탊NGXgɑ΂铮
  void BGMManager::RequestCheck_FADE_OUT( void )
  {
    switch( this->request )
    {
    case REQUEST_NONE: 
      break;
    case REQUEST_FADE_OUT: 
      break;
    default:
      GFL_ASSERT(0);
    }
  }

  //  FADE_IN , e탊NGXgɑ΂铮
  void BGMManager::RequestCheck_FADE_IN( void )
  {
    switch( this->request )
    {
    case REQUEST_NONE: 
      break;
    case REQUEST_FADE_IN: 
      break;
    default:
      GFL_ASSERT(0);
    }
  }

  //  PUSH , e탊NGXgɑ΂铮
  void BGMManager::RequestCheck_PUSH( void )
  {
    switch( this->request )
    {
    case REQUEST_NONE: 
      break;
    case REQUEST_PUSH:
      break;
    default:
      GFL_ASSERT(0);
    }
  }

  //  POP_out , e탊NGXgɑ΂铮
  void BGMManager::RequestCheck_POP_out( void )
  {
    switch( this->request )
    {
    case REQUEST_NONE: 
      break;
    case REQUEST_POP:
      break;
    default:
      GFL_ASSERT(0);
    }
  }

  //  POP_in , e탊NGXgɑ΂铮
  void BGMManager::RequestCheck_POP_in( void )
  {
    switch( this->request )
    {
    case REQUEST_NONE: 
      break;
    case REQUEST_POP:
      break;
    default:
      GFL_ASSERT(0);
    }
  }

  //  CHANGE_out , e탊NGXgɑ΂铮
  void BGMManager::RequestCheck_CHANGE_out( void )
  {
    switch( this->request )
    {
    case REQUEST_NONE: 
      break;
    case REQUEST_PUSH: 
      ChangeState( STATE_CHANGE_PUSH_out);
      break;
    case REQUEST_CHANGE:
      break;
    default:
      GFL_ASSERT(0);
    }
  }

  //  CHANGE_load , e탊NGXgɑ΂铮
  void BGMManager::RequestCheck_CHANGE_load( void )
  {
    switch( this->request )
    {
    case REQUEST_NONE: 
      break;
    case REQUEST_PUSH: 
      ChangeState( STATE_CHANGE_PUSH_load );
      break;
    case REQUEST_CHANGE:
      break;
    default:
      GFL_ASSERT(0);
    }
  }

  //  CHANGE_in , e탊NGXgɑ΂铮
  void BGMManager::RequestCheck_CHANGE_in( void )
  {
    switch( this->request )
    {
    case REQUEST_NONE: 
      break;
    case REQUEST_CHANGE:
      break;
    default:
      GFL_ASSERT(0);
    }
  }

  //  CHANGE_PUSH_out , e탊NGXgɑ΂铮
  void BGMManager::RequestCheck_CHANGE_PUSH_out ( void )
  {
    switch( this->request )
    {
    case REQUEST_NONE: 
      break;
    case REQUEST_PUSH:
      break;
    default:
      GFL_ASSERT(0);
    }
  }

  //  CHANGE_PUSH_load , e탊NGXgɑ΂铮
  void BGMManager::RequestCheck_CHANGE_PUSH_load( void )
  {
    switch( this->request )
    {
    case REQUEST_NONE: 
      break;
    case REQUEST_PUSH:
      break;
    default:
      GFL_ASSERT(0);
    }
  }

  //  STAND_BY_out , e탊NGXgɑ΂铮
  void BGMManager::RequestCheck_STAND_BY_out( void )
  {
    switch( this->request )
    {
    case REQUEST_NONE: 
      break;
    case REQUEST_STAND_BY:
      break;
    default:
      GFL_ASSERT(0);
    }
  }

  //  STAND_BY_load , e탊NGXgɑ΂铮
  void BGMManager::RequestCheck_STAND_BY_load( void )
  {
    switch( this->request )
    {
    case REQUEST_NONE: 
      break;
    case REQUEST_STAND_BY:
      break;
    default:
      GFL_ASSERT(0);
    }
  }

  //---------------------------------------------------------------------------------
  /**
   * @brief  STOP ̓
   */
  //---------------------------------------------------------------------------------
  void BGMManager::Main_STOP( void )
  { 
  }

  //---------------------------------------------------------------------------------
  /**
   * @brief  PLAY ̓
   */
  //---------------------------------------------------------------------------------
  void BGMManager::Main_PLAY( void )
  {
  }

  //---------------------------------------------------------------------------------
  /**
   * @brief  WAIT ̓
   */
  //---------------------------------------------------------------------------------
  void BGMManager::Main_WAIT( void )
  {
  }

  //---------------------------------------------------------------------------------
  /**
   * @brief  FADE_OUT ̓
   */
  //---------------------------------------------------------------------------------
  void BGMManager::Main_FADE_OUT( void )
  {
    if( m_sound_system->IsBGMFade() == false )
    {
      ChangeState( STATE_WAIT );
      FinishRequest( );
    }
  }

  //---------------------------------------------------------------------------------
  /**
   * @brief  FADE_IN ̓
   */
  //---------------------------------------------------------------------------------
  void BGMManager::Main_FADE_IN( void )
  {
    if( m_sound_system->IsBGMFade() == false )
    {
      ChangeState( STATE_PLAY );
      FinishRequest( );
    }
  }

  //---------------------------------------------------------------------------------
  /**
   * @brief  PUSH ̓
   */
  //---------------------------------------------------------------------------------
  void BGMManager::Main_PUSH( void )
  {
    if( m_sound_system->IsBGMFade() == false )
    {
      PushBGM( );
      ChangeState( STATE_STOP );
      FinishRequest( );
    }
  }

  //---------------------------------------------------------------------------------
  /**
   * @brief  POP_out ̓
   */
  //---------------------------------------------------------------------------------
  void BGMManager::Main_POP_out( void )
  {
    if( m_sound_system->IsBGMFade() == false )
    {
      PopBGM( );
      FadeInBGM( );
      ChangeState( STATE_POP_in );
    }
  }

  //---------------------------------------------------------------------------------
  /**
   * @brief  POP_in ̓
   */
  //---------------------------------------------------------------------------------
  void BGMManager::Main_POP_in( void )
  {
    if( m_sound_system->IsBGMFade() == false )
    {
      ChangeState( STATE_PLAY );
      FinishRequest( );
    }
  }

  //---------------------------------------------------------------------------------
  /**
   * @brief  CHANGE_out ̓
   */
  //---------------------------------------------------------------------------------
  void BGMManager::Main_CHANGE_out( void )
  {
    if( m_sound_system->IsBGMFade() == false )
    {
      this->StartAsyncPrepareBGM();
      //DivLoadBGM_start( );
      ChangeState( STATE_CHANGE_load );
    }
  }

  //---------------------------------------------------------------------------------
  /**
   * @brief  CHANGE_load ̓
   */
  //---------------------------------------------------------------------------------
  void BGMManager::Main_CHANGE_load( void )
  {
    bool loadFinished;

    // [hs
    //loadFinished = DivLoadBGM_load( );
    loadFinished = this->IsAsyncPrepareBGMFinished();

    // [h(ĐJn)
    if( loadFinished ) { 
      this->EndAsyncPrepareBGM();
      if( (this->requestBGM != SOUND_ITEM_ID_NULL) &&
          (this->requestBGM != this->currentBGM) ) { 
        // ă[hJn
        m_sound_system->StopBGM(0);
        ChangeState( STATE_CHANGE_out );
      }
      else { 
        // tF[hCJn
        FadeInBGM( );
        ChangeState( STATE_CHANGE_in );
      }
    }
  }

  //---------------------------------------------------------------------------------
  /**
   * @brief  CHANGE_in ̓
   */
  //---------------------------------------------------------------------------------
  void BGMManager::Main_CHANGE_in( void )
  {
    // tF[hC
    if( m_sound_system->IsBGMFade() == false )
    {
      // ĐJnȍ~ BGMύXNGXg󂯂ꍇ
      if( (this->requestBGM != SOUND_ITEM_ID_NULL) &&
          (this->requestBGM != this->currentBGM) )
      {
        FadeOutBGM(  ); // tF[hAEgJn
        ChangeState( STATE_CHANGE_out );
      }
      else
      {
        ChangeState( STATE_PLAY );
        FinishRequest( );
      }
    }
  }

  //---------------------------------------------------------------------------------
  /**
   * @brief  CHANGE_PUSH_out ̓
   */
  //---------------------------------------------------------------------------------
  void BGMManager::Main_CHANGE_PUSH_out( void )
  {
    if( m_sound_system->IsBGMFade() == false ) {
      this->StartAsyncPrepareBGM();
      //DivLoadBGM_start( );
      ChangeState( STATE_CHANGE_PUSH_load );
    }
  }

  //---------------------------------------------------------------------------------
  /**
   * @brief  CHANGE_PUSH_load ̓
   */
  //---------------------------------------------------------------------------------
  void BGMManager::Main_CHANGE_PUSH_load( void )
  {
    bool loadFinished;

    // [hs
    //loadFinished = DivLoadBGM_load( );
    loadFinished = this->IsAsyncPrepareBGMFinished();

    // [h(ĐJn)
    if( loadFinished ) { 
      this->EndAsyncPrepareBGM();
      m_sound_system->PauseBGM(0);
      PushBGM();
      ChangeState( STATE_STOP );
      FinishRequest( ); 
    }
  }

  //---------------------------------------------------------------------------------
  /**
   * @brief  STAND_BY_out ̓
   */
  //---------------------------------------------------------------------------------
  void BGMManager::Main_STAND_BY_out( void )
  {
    if( m_sound_system->IsBGMFade() == false ) { 
      ChangeState( STATE_STAND_BY_load );
    }
  }

  //---------------------------------------------------------------------------------
  /**
   * @brief  STAND_BY_load ̓
   */
  //---------------------------------------------------------------------------------
  void BGMManager::Main_STAND_BY_load( void )
  {
    ForcePlayBGM();
    m_sound_system->PauseBGM(0);
    ChangeState( STATE_WAIT );
    FinishRequest( );
  } 

  //---------------------------------------------------------------------------------
  /**
   * @brief ԂύX
   *
   * @param fieldSound 
   * @param nextState  Jڐ̏
   */
  //---------------------------------------------------------------------------------
  void BGMManager::ChangeState( SystemState nextState )
  {
#ifdef DEBUG_PRINT_ON
    switch( this->state ) {
    case STATE_STOP:              GFL_PRINT( "FIELD-SOUND: change state (STOP ==> " );  break;
    case STATE_PLAY:              GFL_PRINT( "FIELD-SOUND: change state (PLAY ==> " );  break;
    case STATE_WAIT:              GFL_PRINT( "FIELD-SOUND: change state (WAIT ==> " );  break;
    case STATE_FADE_OUT:          GFL_PRINT( "FIELD-SOUND: change state (FADE OUT ==> " );  break;
    case STATE_FADE_IN:           GFL_PRINT( "FIELD-SOUND: change state (FADE IN ==> " );  break;
    case STATE_PUSH:              GFL_PRINT( "FIELD-SOUND: change state (PUSH ==> " );  break;
    case STATE_POP_out:           GFL_PRINT( "FIELD-SOUND: change state (POP out ==> " );  break;
    case STATE_POP_in:            GFL_PRINT( "FIELD-SOUND: change state (POP in ==> " );  break;
    case STATE_CHANGE_out:        GFL_PRINT( "FIELD-SOUND: change state (CHANGE out ==> " );  break;
    case STATE_CHANGE_load:       GFL_PRINT( "FIELD-SOUND: change state (CHANGE load ==> " );  break;
    case STATE_CHANGE_in:         GFL_PRINT( "FIELD-SOUND: change state (CHANGE in ==> " );  break;
    case STATE_CHANGE_PUSH_out:   GFL_PRINT( "FIELD-SOUND: change state (CHANGE PUSH out ==> " );  break;
    case STATE_CHANGE_PUSH_load:  GFL_PRINT( "FIELD-SOUND: change state (CHANGE PUSH load ==> " );  break;
    case STATE_STAND_BY_out:      GFL_PRINT( "FIELD-SOUND: change state (STAND BY out ==> " );  break;
    case STATE_STAND_BY_load:     GFL_PRINT( "FIELD-SOUND: change state (STAND BY load ==> " );  break;
    default: GFL_ASSERT(0);
    }
    switch( nextState ) {
    case STATE_STOP:              GFL_PRINT( "STOP)\n" );  break;
    case STATE_PLAY:              GFL_PRINT( "PLAY)\n" );  break;
    case STATE_WAIT:              GFL_PRINT( "WAIT)\n" );  break;
    case STATE_FADE_OUT:          GFL_PRINT( "FADE OUT)\n" );  break;
    case STATE_FADE_IN:           GFL_PRINT( "FADE IN)\n" );  break;
    case STATE_PUSH:              GFL_PRINT( "PUSH)\n" );  break;
    case STATE_POP_out:           GFL_PRINT( "POP out)\n" );  break;
    case STATE_POP_in:            GFL_PRINT( "POP in)\n" );  break;
    case STATE_CHANGE_out:        GFL_PRINT( "CHANGE out)\n" );  break;
    case STATE_CHANGE_load:       GFL_PRINT( "CHANGE load)\n" );  break;
    case STATE_CHANGE_in:         GFL_PRINT( "CHANGE in)\n" );  break;
    case STATE_CHANGE_PUSH_out:   GFL_PRINT( "CHANGE PUSH out)\n" );  break;
    case STATE_CHANGE_PUSH_load:  GFL_PRINT( "CHANGE PUSH load)\n" );  break;
    case STATE_STAND_BY_out:      GFL_PRINT( "STAND BY out)\n" );  break;
    case STATE_STAND_BY_load:     GFL_PRINT( "STAND BY load)\n" );  break;
    default: GFL_ASSERT(0);
    }
#endif

    // XV
    this->state = nextState; 
  }

  //---------------------------------------------------------------------------------
  /**
   * @brief NGXg󗝏ԂZbg
   *
   * @param fieldSound
   */
  //---------------------------------------------------------------------------------
  void BGMManager::ResetRequest( void )
  {
    this->request = REQUEST_NONE;
    this->requestBGM = SOUND_ITEM_ID_NULL;
    this->fadeInFrame = 0;
    this->fadeOutFrame = 0;
  }

  //---------------------------------------------------------------------------------
  /**
   * @brief 󂯎NGXgLZ
   *
   * @param fieldSound
   */
  //---------------------------------------------------------------------------------
  void BGMManager::CancelRequest( void )
  {
    // NGXgĂȂ
    if( this->request == REQUEST_NONE ) {
      GFL_PRINT( "FIELD-SOUND: Don't have BGM request\n" );
      GFL_ASSERT(0);
      return;
    }

    // NGXg󗝏ԂZbg
    ResetRequest( );

#ifdef DEBUG_PRINT_ON
    GFL_PRINT( "FIELD-SOUND: Cancel request\n" );
#endif
  }

  //---------------------------------------------------------------------------------
  /**
   * @brief NGXgɑ΂铮̊
   *
   * @param fieldSound
   */
  //---------------------------------------------------------------------------------
  void BGMManager::FinishRequest( void )
  {
    // NGXgĂȂ
    if( this->request == REQUEST_NONE ) {
      GFL_PRINT( "FIELD-SOUND: Don't have BGM request\n" );
      GFL_ASSERT(0);
      return;
    }

#ifdef DEBUG_PRINT_ON
    switch( this->request ) {
    case REQUEST_FADE_IN:     GFL_PRINT( "FIELD-SOUND: request(FADE IN) finished!!\n" );  break;
    case REQUEST_FADE_OUT:    GFL_PRINT( "FIELD-SOUND: request(FADE OUT) finished!!\n" );  break;
    case REQUEST_PUSH:        GFL_PRINT( "FIELD-SOUND: request(PUSH) finished!!\n" );  break;
    case REQUEST_POP:         GFL_PRINT( "FIELD-SOUND: request(POP) finished!!\n" );  break;
    case REQUEST_CHANGE:      GFL_PRINT( "FIELD-SOUND: request(CHANGE) finished!!\n" );  break;
    case REQUEST_STAND_BY:    GFL_PRINT( "FIELD-SOUND: request(STAND BY) finished!!\n" );  break;
    case REQUEST_FORCE_PLAY:  GFL_PRINT( "FIELD-SOUND: request(FORCE_PLAY) finished!!\n" );  break;
    }
#endif

    // NGXg󗝏ԂZbg
    ResetRequest();
  }

  //---------------------------------------------------------------------------------
  /**
   * @brief BGM̃tF[hCJn
   *
   * @param fieldSound
   */
  //---------------------------------------------------------------------------------
  void BGMManager::FadeInBGM( void )
  {
    // BGMĂȂ
    if( this->currentBGM == SOUND_ITEM_ID_NULL ) {
      GFL_PRINT( "FIELD-SOUND: Can't fade in BGM (Don't playing BGM)\n" );
      GFL_ASSERT(0);
      return;
    } 

    m_sound_system->PauseReleaseBGM( this->fadeInFrame );

#ifdef DEBUG_PRINT_ON
    GFL_PRINT( "FIELD-SOUND: BGM fade in start\n" );
#endif
  }

  //---------------------------------------------------------------------------------
  /**
   * @brief BGM̃tF[hAEgJn
   *
   * @param fieldSound
   */
  //---------------------------------------------------------------------------------
  void BGMManager::FadeOutBGM( void )
  {
    // BGMĂȂ
    if( this->currentBGM == SOUND_ITEM_ID_NULL ) {
      GFL_PRINT( "FIELD-SOUND: Can't fade out BGM (Don't playing BGM)\n" );
      GFL_ASSERT(0);
      return;
    } 

    m_sound_system->PauseBGM( this->fadeOutFrame );

#ifdef DEBUG_PRINT_ON
    GFL_PRINT( "FIELD-SOUND: BGM fade out start\n" );
#endif
  }

  //---------------------------------------------------------------------------------
  /**
   * @brief BGMޔ
   *
   * @param fieldSound
   */
  //---------------------------------------------------------------------------------
  void BGMManager::PushBGM( void )
  {
    // BGMĂȂ
    if( this->currentBGM == SOUND_ITEM_ID_NULL ) {
      GFL_PRINT( "FIELD-SOUND: Can't push BGM (Don't playing BGM)\n" );
      GFL_ASSERT(0);
      return;
    } 
    // ޔ𐔃I[o[
    if( m_sound_system->GetMaxPushBGMCount() <= m_sound_system->GetPushedBGMCount() ) {
      GFL_PRINT( "FIELD-SOUND: Can't push BGM (Push count over)\n" );
      GFL_ASSERT(0);
      return;
    }

    // ޔ
    m_sound_system->PushBGM();

    // XV
    this->currentBGM = SOUND_ITEM_ID_NULL; 
  }

  //---------------------------------------------------------------------------------
  /**
   * @brief BGM𕜋A
   *
   * @param fieldSound
   */
  //---------------------------------------------------------------------------------
  void BGMManager::PopBGM( void )
  {
    // ޔĂȂ
    if( m_sound_system->GetPushedBGMCount() <= 0 ) {
      GFL_PRINT( "FIELD-SOUND: Can't pop BGM (There is no pushed BGM)\n" );
      GFL_ASSERT(0);
      return;
    }

    // A
    m_sound_system->PopBGM();

    // XV
    this->currentBGM = m_sound_system->GetCurrentBGM();
  }

#if 0
  //---------------------------------------------------------------------------------
  /**
   * @brief BGM̕[hJn
   *
   * @param fieldSound
   *
   * @return [hJnꍇ true
   *         łȂ false
   */
  //---------------------------------------------------------------------------------
  bool BGMManager::DivLoadBGM_start( void )
  {
    // ǂݍBGM\񂳂ĂȂ
    if( this->requestBGM == SOUND_ITEM_ID_NULL ) {
      GFL_PRINT( "FIELD-SOUND: Can't start load BGM (Don't know target BGM)\n" );
      GFL_ASSERT(0);
      return false;
    }

    // [hJn
    //PMSND_PlayBGMdiv( this->requestBGM, &(this->loadSeq), true );

    // XV
    this->loadBGM = this->requestBGM;
    this->requestBGM = SOUND_ITEM_ID_NULL;
    this->currentBGM = SOUND_ITEM_ID_NULL;

#ifdef DEBUG_PRINT_ON
    GFL_PRINT( "FIELD-SOUND: div load BGM(%d) start\n", this->loadBGM );
#endif

    return true;
  }
#endif

#if 0
  //---------------------------------------------------------------------------------
  /**
   * @brief BGM̕[hi߂
   *
   * @param fieldSound
   *
   * @return [hI, ĐJnꂽꍇ true
   *         [hȂ@false
   */
  //---------------------------------------------------------------------------------
  bool BGMManager::DivLoadBGM_load( void )
  {
    bool loadFinished;

    // [hJnĂȂ
    if( this->loadBGM == SOUND_ITEM_ID_NULL ) {
      GFL_PRINT( "FIELD-SOUND: Can't load BGM (Don't know target BGM)\n" );
      GFL_ASSERT(0); 
    }

    // [hs
    loadFinished = true;//PMSND_PlayBGMdiv( this->loadBGM, &(this->loadSeq), false );

    // [h(ĐJn)
    if( loadFinished ) {
      // XV
      this->currentBGM = this->loadBGM;
      this->loadBGM = SOUND_ITEM_ID_NULL; 
#ifdef DEBUG_PRINT_ON
      GFL_PRINT( "FIELD-SOUND: div load BGM(%d) finish\n", this->currentBGM );
#endif
    } 
    return loadFinished;
  }
#endif

  //---------------------------------------------------------------------------------
  /**
   * @brief BGM̑Đs
   *
   * @param fieldSound
   */
  //---------------------------------------------------------------------------------
  void BGMManager::ForcePlayBGM( void )
  {
    // ĐBGMw肳ĂȂ
    if( this->requestBGM == SOUND_ITEM_ID_NULL ) {
      GFL_PRINT( "FIELD-SOUND: Can't force play BGM (Don't know target BGM)\n" );
      GFL_ASSERT(0);
      return;
    }

    // Đ
    m_sound_system->PlayBGM( this->requestBGM, 0 );

    // XV
    this->currentBGM = this->requestBGM;
    this->requestBGM = SOUND_ITEM_ID_NULL;
    this->loadBGM = SOUND_ITEM_ID_NULL;

#ifdef DEBUG_PRINT_ON
    GFL_PRINT( "FIELD-SOUND: force play BGM(%d) finish\n", this->currentBGM );
#endif
  } 



  // BGM̔񓯊Jn
  void BGMManager::StartAsyncPrepareBGM( void )
  {
    // ĐBGMw肳ĂȂ
    if( this->requestBGM == SOUND_ITEM_ID_NULL ) {
      GFL_PRINT( "FIELD-SOUND: Can't force play BGM (Don't know target BGM)\n" );
      GFL_ASSERT(0);
      return;
    } 

    GFL_ASSERT( m_bgm_prepare_thread == NULL );
    if( m_bgm_prepare_thread == NULL ) { 
      m_bgm_prepare_thread = GFL_NEW_LOW( m_heap_for_work ) 
        BGMPrepareThread( m_heap_for_work, m_sound_system, this->requestBGM );
      m_bgm_prepare_thread->Start( BGM_PREPARE_THREAD_PRIORITY );

      this->loadBGM = this->requestBGM;
      this->requestBGM = SOUND_ITEM_ID_NULL;
      this->currentBGM = SOUND_ITEM_ID_NULL;
    }
  }

  // BGM̔񓯊ǂ𒲂ׂ
  bool BGMManager::IsAsyncPrepareBGMFinished( void ) const
  {
    if( m_bgm_prepare_thread ) {
      return !m_bgm_prepare_thread->IsAlive();
    }
    return true;
  }

  // BGM̔񓯊
  void BGMManager::EndAsyncPrepareBGM( void )
  {
    GFL_ASSERT( m_bgm_prepare_thread );
    if( m_bgm_prepare_thread ) {
      GFL_DELETE m_bgm_prepare_thread;
      m_bgm_prepare_thread = NULL;

      m_sound_system->PlayPreparedBGM( this->fadeInFrame );
      this->currentBGM = this->loadBGM;
      this->loadBGM = SOUND_ITEM_ID_NULL; 
    }
  }



#if PM_DEBUG
  //---------------------------------------------------------------------------------
  /**
   * @brief NGXgL[\
   *
   * @param fieldSound
   */
  //---------------------------------------------------------------------------------
  void BGMManager::DebugPrint_RequestQueue( void ) const
  {
    GFL_PRINT( "FIELD-SOUND-QUEUE: " );
    u32 request_count = m_request_queue.GetDataNum();
    for( u32 i=0; i<request_count; i++ ) {
      const RequestData& request_data = m_request_queue.GetData(i);
      switch( request_data.request_type ) {
      case REQUEST_FADE_IN:     GFL_PRINT( "FADE IN" );  break;
      case REQUEST_FADE_OUT:    GFL_PRINT( "FADE OUT" );  break;
      case REQUEST_PUSH:        GFL_PRINT( "PUSH" );  break;
      case REQUEST_POP:         GFL_PRINT( "POP" );  break;
      case REQUEST_CHANGE:      GFL_PRINT( "CHANGE" );  break;
      case REQUEST_STAND_BY:    GFL_PRINT( "STAND BY" );  break;
      case REQUEST_FORCE_PLAY:  GFL_PRINT( "FORCE_PLAY" );  break;
      }
      GFL_PRINT( " | " );
    }
    GFL_PRINT( "\n" );
  }

  //---------------------------------------------------------------------------------
  /**
   * @brief So͂
   *
   * @param fieldSound
   */
  //---------------------------------------------------------------------------------
  void BGMManager::DebugPrint_AllInfo( void ) const
  { 
    GFL_PRINT( "FIELD-SOUND: state = %d\n", this->state );
    GFL_PRINT( "FIELD-SOUND: currentBGM = %d\n", this->currentBGM );
    GFL_PRINT( "FIELD-SOUND: requestBGM = %d\n", this->requestBGM );
    GFL_PRINT( "FIELD-SOUND: loadBGM = %d\n", this->loadBGM );
    GFL_PRINT( "FIELD-SOUND: loadSeq = %d\n", this->loadSeq );
    GFL_PRINT( "FIELD-SOUND: fadeInFrame = %d\n", this->fadeInFrame );
    GFL_PRINT( "FIELD-SOUND: fadeOutFrame = %d\n", this->fadeOutFrame );
    GFL_PRINT( "FIELD-SOUND: request = %d\n", this->request );
    DebugPrint_RequestQueue( );
  }
#endif

} //namespace snd
