//=================================================================================================
/**
 * @file PokemonSoundSystem.cpp
 * @brief |PTEhNX
 * @author obata_toshihiro
 * @date 2011.06.28
 */
//=================================================================================================
#include <gflib.h>
#include <sound/PokemonSoundSystem.h>
#include <sound/SoundLevelManager.h>


namespace snd {




  // BGMtF[h̃t[
  const u32 PokemonSoundSystem::BGM_FADE_FRAME[ BGM_FADE_NUM ] = 
  {
    /* BGM_FADE_NONE */ 0,
    /* BGM_FADE_VERY_FAST */ 6,
    /* BGM_FADE_FAST */ 30,
    /* BGM_FADE_MIDDLE */ 60,
    /* BGM_FADE_SLOW */ 90,
  };


  // BGM{[
  const u32 PokemonSoundSystem::BGM_VOLUME[ BGM_VOLUME_NUM ] = 
  {
    /* BGM_VOLUME_LOW */ 0.5f,
    /* BGM_VOLUME_NORMAL */ 1.0f,
    /* BGM_VOLUME_HIGH */ 2.0f,
  };





  //-----------------------------------------------------------------------------------------------
  // RXgN^
  //-----------------------------------------------------------------------------------------------
  PokemonSoundSystem::PokemonSoundSystem( 
      gfl::heap::HeapBase* heap, gfl::heap::HeapBase* device_heap ) : 
    gfl::base::Singleton<PokemonSoundSystem>(),
    m_sound_system( NULL ),
    m_bgm_level_manager( NULL )
  {
    this->CreateSoundSystem( heap, device_heap );
    this->CreateBGMLevelManager( heap );
  }


  // TEhVXe𐶐
  void PokemonSoundSystem::CreateSoundSystem(
      gfl::heap::HeapBase* heap, gfl::heap::HeapBase* device_heap )
  {
    GFL_ASSERT( m_sound_system == NULL );
    gfl::snd::SoundSystem::InitParam system_param;
    system_param.pAppHeap = heap;
    system_param.pDevHeap = device_heap;
    system_param.soundHeapSize = 1 * 256 * 1024;
    system_param.soundHandleNum = 4;
    system_param.sound3DActorNum = 2;
    system_param.bcsarFilePath = "rom:/sound/xy_sound.bcsar";
    system_param.stackSizeOfDivLoadThread = 4096; 
    m_sound_system = GFL_NEW( heap ) gfl::snd::SoundSystem( system_param );
  }


  // BGM̊KwǗVXe쐬
  void PokemonSoundSystem::CreateBGMLevelManager( gfl::heap::HeapBase* heap )
  {
    GFL_ASSERT( m_sound_system );
    GFL_ASSERT( m_bgm_level_manager == NULL );
    SoundLevelManager::Settings manager_setting;
    manager_setting.min_sound_handle_index = SOUND_HANDLE_BGM_LV1;
    manager_setting.max_sound_handle_index = SOUND_HANDLE_BGM_LV3; 
    m_bgm_level_manager = GFL_NEW( heap ) SoundLevelManager( heap, m_sound_system, manager_setting );
  }






  //-----------------------------------------------------------------------------------------------
  // fXgN^
  //-----------------------------------------------------------------------------------------------
  PokemonSoundSystem::~PokemonSoundSystem()
  {
    this->DeleteBGMLevelManager();
    this->DeleteSoundSystem();
  } 

  // BGM̊KwǗVXej
  void PokemonSoundSystem::DeleteBGMLevelManager( void )
  {
    GFL_ASSERT( m_bgm_level_manager );
    GFL_DELETE m_bgm_level_manager;
  }

  // TEhVXej
  void PokemonSoundSystem::DeleteSoundSystem( void )
  {
    GFL_ASSERT( m_sound_system );
    GFL_DELETE m_sound_system;
  }







  //-----------------------------------------------------------------------------------------------
  // t[̍XV
  //-----------------------------------------------------------------------------------------------
  void PokemonSoundSystem::Update( void )
  {
    m_sound_system->Update();
  }







  //-----------------------------------------------------------------------------------------------
  // BGMĐ
  //-----------------------------------------------------------------------------------------------
  void PokemonSoundSystem::PlayBGM( u32 sound_item_id, BgmFadeSpeed fade_in_speed )
  {
    this->StopBGM( BGM_FADE_NONE );
    m_bgm_level_manager->LoadSoundData( sound_item_id );
    u32 sound_handle_index = m_bgm_level_manager->GetSoundHandleIndex();
    u32 fade_in_frame = BGM_FADE_FRAME[ fade_in_speed ];
    m_sound_system->Play( sound_handle_index, sound_item_id, fade_in_frame );
  }


  //-----------------------------------------------------------------------------------------------
  // BGM~
  //-----------------------------------------------------------------------------------------------
  void PokemonSoundSystem::StopBGM( BgmFadeSpeed fade_out_speed )
  {
    u32 sound_handle_index = m_bgm_level_manager->GetSoundHandleIndex();
    u32 fade_out_frame = BGM_FADE_FRAME[ fade_out_speed ];
    m_sound_system->Stop( sound_handle_index, fade_out_frame );
    m_bgm_level_manager->UnloadSoundData();
  }


  //-----------------------------------------------------------------------------------------------
  // BGMꎞ~
  //-----------------------------------------------------------------------------------------------
  void PokemonSoundSystem::PauseBGM( BgmFadeSpeed fade_out_speed )
  {
    u32 sound_handle_index = m_bgm_level_manager->GetSoundHandleIndex();
    u32 fade_out_frame = BGM_FADE_FRAME[ fade_out_speed ];
    m_sound_system->Pause( sound_handle_index, true, fade_out_frame );
  }


  //-----------------------------------------------------------------------------------------------
  // ꎞ~BGMĊJ
  //-----------------------------------------------------------------------------------------------
  void PokemonSoundSystem::PauseReleaseBGM( BgmFadeSpeed fade_in_speed )
  {
    u32 sound_handle_index = m_bgm_level_manager->GetSoundHandleIndex();
    u32 fade_in_frame = BGM_FADE_FRAME[ fade_in_speed ];
    m_sound_system->Pause( sound_handle_index, false, fade_in_frame );
  }







  //-----------------------------------------------------------------------------------------------
  // BGM̃{[ύX
  //-----------------------------------------------------------------------------------------------
  void PokemonSoundSystem::ChangeBGMVolume( BgmVolumeLevel volume_level, u32 change_frame )
  {
    u32 sound_handle_index = m_bgm_level_manager->GetSoundHandleIndex();
    f32 volume = BGM_VOLUME[ volume_level ];
    m_sound_system->SetVolume( sound_handle_index, volume, change_frame );
  }

  //-----------------------------------------------------------------------------------------------
  // BGM̃sb`ύX
  //-----------------------------------------------------------------------------------------------
  void PokemonSoundSystem::ChangeBGMPitch( f32 value )
  {
    u32 sound_handle_index = m_bgm_level_manager->GetSoundHandleIndex();
    m_sound_system->SetPitch( sound_handle_index, value );
  }

  //-----------------------------------------------------------------------------------------------
  // BGM̍EpύX
  //-----------------------------------------------------------------------------------------------
  void PokemonSoundSystem::ChangeBGMPan( f32 value )
  {
    u32 sound_handle_index = m_bgm_level_manager->GetSoundHandleIndex();
    m_sound_system->SetPan( sound_handle_index, value );
  }

  //-----------------------------------------------------------------------------------------------
  // BGM̑OpύX
  //-----------------------------------------------------------------------------------------------
  void PokemonSoundSystem::ChangeBGMSurroundPan( f32 value )
  {
    u32 sound_handle_index = m_bgm_level_manager->GetSoundHandleIndex();
    m_sound_system->SetSurroundPan( sound_handle_index, value );
  }

  //-----------------------------------------------------------------------------------------------
  // BGM̃e|ύX
  //-----------------------------------------------------------------------------------------------
  void PokemonSoundSystem::ChangeBGMTempoRatio( f32 value )
  {
    u32 sound_handle_index = m_bgm_level_manager->GetSoundHandleIndex();
    m_sound_system->SetTempoRatio( sound_handle_index, value );
  }








  //-----------------------------------------------------------------------------------------------
  // BGMޔ
  //-----------------------------------------------------------------------------------------------
  bool PokemonSoundSystem::PushBGM( void )
  {
    return m_bgm_level_manager->PushSoundData();
  }


  //-----------------------------------------------------------------------------------------------
  // BGM𕜋A
  //-----------------------------------------------------------------------------------------------
  bool PokemonSoundSystem::PopBGM( void )
  {
    return m_bgm_level_manager->PopSoundData();
  }








  //-----------------------------------------------------------------------------------------------
  // SEĐ
  //-----------------------------------------------------------------------------------------------
  void PokemonSoundSystem::PlaySE( u32 sound_item_id )
  {
    this->PlaySEVolume( sound_item_id, 0.0f );
  }

  //-----------------------------------------------------------------------------------------------
  // {[wSEĐ
  //-----------------------------------------------------------------------------------------------
  void PokemonSoundSystem::PlaySEVolume( u32 sound_item_id, f32 volume )
  {
    m_sound_system->Play( SOUND_HANDLE_SE, sound_item_id, volume );
  }

  //-----------------------------------------------------------------------------------------------
  // SĂSE~
  //-----------------------------------------------------------------------------------------------
  void PokemonSoundSystem::StopAllSE( void )
  {
    m_sound_system->StopPlayerSound( PLAYER_SE_SYS, 0 );
    m_sound_system->StopPlayerSound( PLAYER_SE_1, 0 );
    m_sound_system->StopPlayerSound( PLAYER_SE_2, 0 );
    m_sound_system->StopPlayerSound( PLAYER_SE_3, 0 );
    m_sound_system->StopPlayerSound( PLAYER_SE_4, 0 );
  }







#if PM_DEBUG 
  // TEhVXe擾
  gfl::snd::SoundSystem* PokemonSoundSystem::GetSoundSystem( void )
  {
    return m_sound_system;
  }
  // BGM̊KwǗNX擾
  SoundLevelManager* PokemonSoundSystem::GetBGMLevelManager( void )
  {
    return m_bgm_level_manager;
  }
#endif



} // namespace snd
