//=================================================================================================
/**
 * @file SoundLevelManager.cpp
 * @brief TEhKwǗNX
 * @author obata_toshihiro
 * @date 2011.06.27
 */
//=================================================================================================
#include <gflib.h>
#include <sound/SoundLevelManager.h>


namespace snd {



  //-----------------------------------------------------------------------------------------------
  // RXgN^
  //-----------------------------------------------------------------------------------------------
  SoundLevelManager::SoundLevelManager( 
      gfl::heap::HeapBase* heap, gfl::snd::SoundSystem* sound_system, const Settings& settings ) :
    m_sound_system( sound_system ),
    m_min_sound_handle_index( settings.min_sound_handle_index ),
    m_level_stack( NULL ),
    m_max_level( settings.max_sound_handle_index - settings.min_sound_handle_index ), 
    m_current_level( 0 )
  {
    u32 level_count = m_max_level + 1;
    m_level_stack = GFL_NEW( heap ) LevelState[ level_count ];
    for( u32 i=0; i<level_count; i++ ) {
      m_level_stack[i].valid_flag = false;
      m_level_stack[i].sound_item_id = 0;
      m_level_stack[i].sound_handle_index = settings.min_sound_handle_index + i;
      m_level_stack[i].sound_heap_level_before_load = -1;
      m_level_stack[i].sound_heap_level_after_load = -1;
    }
  }


  //-----------------------------------------------------------------------------------------------
  // fXgN^
  //-----------------------------------------------------------------------------------------------
  SoundLevelManager::~SoundLevelManager()
  {
    GFL_DELETE_ARRAY m_level_stack;
  }







  //-----------------------------------------------------------------------------------------------
  // ݂̊Kw̃TEhACeID擾
  //-----------------------------------------------------------------------------------------------
  u32 SoundLevelManager::GetSoundItemId( void )
  {
    LevelState* level_state = this->GetCurrentLevelState();
    return level_state->sound_item_id;
  }

  //-----------------------------------------------------------------------------------------------
  // ݂̊Kw̃TEhnh̃CfbNX擾
  //-----------------------------------------------------------------------------------------------
  u32	SoundLevelManager::SoundLevelManager::GetSoundHandleIndex( void )
  {
    LevelState* level_state = this->GetCurrentLevelState();
    return level_state->sound_handle_index;
  }

  // ݂̊Kw̏Ԃ擾
  SoundLevelManager::LevelState* SoundLevelManager::GetCurrentLevelState( void )
  {
    return this->GetLevelState( m_current_level );
  }

  // w肵Kw̏Ԃ擾
  SoundLevelManager::LevelState* SoundLevelManager::GetLevelState( u32 level )
  {
    GFL_ASSERT( level <= m_max_level );
    return &m_level_stack[ level ];
  }







  //-----------------------------------------------------------------------------------------------
  // TEhf[^ǂݍ
  //-----------------------------------------------------------------------------------------------
  bool SoundLevelManager::LoadSoundData( u32 sound_item_id )
  {
    LevelState* current_state = this->GetCurrentLevelState();
    if( current_state->valid_flag == false ) {
      bool load_result = m_sound_system->LoadSoundData( sound_item_id );
      if( load_result ) {
        current_state->valid_flag = true;
        current_state->sound_item_id = sound_item_id;
        current_state->sound_heap_level_after_load = m_sound_system->SaveHeapState();
      }
      return load_result;
    }
    return false;
  }


  //-----------------------------------------------------------------------------------------------
  // TEhf[^j
  //-----------------------------------------------------------------------------------------------
  void SoundLevelManager::UnloadSoundData( void )
  {
    LevelState* current_state = this->GetCurrentLevelState();
    if( current_state->valid_flag ) {
      m_sound_system->LoadHeapState( current_state->sound_heap_level_before_load );
      current_state->valid_flag = false;
    }
  }





  //-----------------------------------------------------------------------------------------------
  // TEhf[^ۑ
  //-----------------------------------------------------------------------------------------------
  bool SoundLevelManager::PushSoundData( void )
  {
    LevelState* current_state = this->GetCurrentLevelState();
    if( ( current_state->valid_flag ) && ( m_current_level < m_max_level ) ) {
      u32 next_level = m_current_level + 1;
      this->InitLevelState( next_level, current_state->sound_heap_level_after_load );
      m_current_level = next_level;
      return true;
    }
    return false;
  }

  // Kw̏Ԃ
  void SoundLevelManager::InitLevelState( u32 level, s32 initial_heap_level )
  {
    LevelState* level_state = this->GetLevelState( level );
    level_state->valid_flag = false;
    level_state->sound_heap_level_before_load = initial_heap_level;
    level_state->sound_heap_level_after_load = initial_heap_level;
  }



  //-----------------------------------------------------------------------------------------------
  // TEhf[^𕜌
  //-----------------------------------------------------------------------------------------------
  bool SoundLevelManager::PopSoundData( void )
  {
    if( 0 < m_current_level ) {
      u32 next_level = m_current_level - 1;
      LevelState* next_state = this->GetLevelState( next_level );
      m_sound_system->LoadHeapState( next_state->sound_heap_level_after_load );
      m_current_level = next_level;
      return true;
    }
    return false;
  }





} // namespace snd
