const shortcutsContainer = document.getElementById('shortcuts');
const addBtn = document.getElementById('add-shortcut-btn');
const modal = document.getElementById('modal');
const cancelBtn = document.getElementById('cancel-btn');
const saveBtn = document.getElementById('save-btn');
const nameInput = document.getElementById('shortcut-name');
const urlInput = document.getElementById('shortcut-url');
const addFolderBtn = document.getElementById('add-folder-btn');
const folderModal = document.getElementById('folder-modal');
const folderModalTitle = document.getElementById('folder-modal-title');
const folderNameInput = document.getElementById('folder-name');
const fSaturation = document.getElementById('f-saturation');
const fPointer = document.getElementById('f-pointer');
const fHue = document.getElementById('f-hue');
const fAlpha = document.getElementById('f-alpha');
const fPreview = document.getElementById('f-preview');
const folderColorHex = document.getElementById('folder-color-hex');
const fSwatches = document.getElementById('f-swatches');
const folderShapeSelect = document.getElementById('folder-shape-select');
const folderSaveBtn = document.getElementById('folder-save-btn');
const folderCancelBtn = document.getElementById('folder-cancel-btn');
const folderViewModal = document.getElementById('folder-view-modal');
const folderViewTitle = document.getElementById('folder-view-title');
const folderShortcutsGrid = document.getElementById('folder-shortcuts-grid');
const folderViewBack = document.getElementById('folder-view-back');
const folderEditBtn = document.getElementById('folder-edit-btn');
const addShortcutInFolderBtn = document.getElementById('add-shortcut-in-folder-btn');
const settingsBtn = document.getElementById('settings-btn');
const settingsModal = document.getElementById('settings-modal');
const closeSettingsBtn = document.getElementById('close-settings-btn');
const columnsSlider = document.getElementById('columns-count-slider');
const columnsCountVal = document.getElementById('columns-count-val');
const fontSizeSlider = document.getElementById('font-size-slider');
const fontSizeVal = document.getElementById('font-size-val');
const exportBtn = document.getElementById('export-btn');
const importBtn = document.getElementById('import-btn');
const importFileInput = document.getElementById('import-file-input');
const bgColorInput = document.getElementById('bg-color-input');
const bgHexInput = document.getElementById('bg-color-hex');
const iconShapeSelector = document.getElementById('icon-shape-selector');
const folderSelect = document.getElementById('shortcut-folder-select');
const folderSelectorGroup = document.getElementById('folder-selector-group');
const confirmModal = document.getElementById('confirm-modal');
const confirmTitle = document.getElementById('confirm-title');
const confirmMessage = document.getElementById('confirm-message');
const confirmApproveBtn = document.getElementById('confirm-approve-btn');
const confirmCancelBtn = document.getElementById('confirm-cancel-btn');
const shortcutLock = document.getElementById('shortcut-lock');
const folderLock = document.getElementById('folder-lock');

// Mobile Drawer Elements
const mobileMenuBtn = document.getElementById('mobile-menu-btn');
const mobileDrawer = document.getElementById('mobile-drawer');
const mobileDrawerOverlay = document.getElementById('mobile-drawer-overlay');
const closeDrawerBtn = document.getElementById('close-drawer-btn');
const drawerAddShortcut = document.getElementById('drawer-add-shortcut');
const drawerAddFolder = document.getElementById('drawer-add-folder');
const drawerSettings = document.getElementById('drawer-settings');

// State
let editingIndex = -1;
let editingFolderIndex = -1;
let currentFolderIndex = -1;
let draggedIndex = -1;
let draggedFromInside = false;
let isSuccessfulMove = false;
let editingFolderIsNested = false;
const defaultSettings = { columnsCount: 5, bgColor: '#1c2027', iconShape: 'circle', fontSize: 13 };

const defaultShortcuts = [
    { name: 'YouTube', url: 'https://youtube.com', type: 'shortcut' },
    { name: 'Facebook', url: 'https://facebook.com', type: 'shortcut' },
    { name: 'Instagram', url: 'https://instagram.com', type: 'shortcut' },
    { name: 'X', url: 'https://x.com', type: 'shortcut' },
    { name: 'WhatsApp', url: 'https://web.whatsapp.com', type: 'shortcut' },
    { name: 'TikTok', url: 'https://tiktok.com', type: 'shortcut' },
    { name: 'ChatGPT', url: 'https://chatgpt.com', type: 'shortcut' },
    { name: 'GitHub', url: 'https://github.com', type: 'shortcut' },
    { name: 'Amazon', url: 'https://amazon.com', type: 'shortcut' },
    { name: 'AliExpress', url: 'https://aliexpress.com', type: 'shortcut' }
];

const GLOBE_SVG = `<svg class="fallback-svg" viewBox="-8 0 512 512" xmlns="http://www.w3.org/2000/svg"><path d="M336.5 160C322 70.7 287.8 8 248 8s-74 62.7-88.5 152h177zM152 256c0 22.2 1.2 43.5 3.3 64h185.3c2.1-20.5 3.3-41.8 3.3-64s-1.2-43.5-3.3-64H155.3c-2.1 20.5-3.3 41.8-3.3 64zm324.7-96c-28.6-67.9-86.5-120.4-158-141.6 24.4 33.8 41.2 84.7 50 141.6h108zM177.2 18.4C105.8 39.6 47.8 92.1 19.3 160h108c8.7-56.9 25.5-107.8 49.9-141.6zM487.4 192H372.7c2.1 21 3.3 42.5 3.3 64s-1.2 43-3.3 64h114.6c5.5-20.5 8.6-41.8 8.6-64s-3.1-43.5-8.5-64zM120 256c0-21.5 1.2-43 3.3-64H8.6C3.2 212.5 0 233.8 0 256s3.2 43.5 8.6 64h114.6c-2-21-3.2-42.5-3.2-64zm39.5 96c14.5 89.3 48.7 152 88.5 152s74-62.7 88.5-152h-177zm159.3 141.6c71.4-21.2 129.4-73.7 158-141.6h-108c-8.8 56.9-25.6 107.8-50 141.6zM19.3 352c28.6 67.9 86.5 120.4 158 141.6-24.4-33.8-41.2-84.7-50-141.6h-108z"/></svg>`;

function showConfirm(title, message, onApprove) {
    if (!confirmModal) return;
    confirmTitle.innerText = title;
    confirmMessage.innerText = message;
    confirmModal.classList.add('active');

    confirmApproveBtn.onclick = () => {
        confirmModal.classList.remove('active');
        onApprove();
    };
    confirmCancelBtn.onclick = () => {
        confirmModal.classList.remove('active');
    };
}

// --- Premium Color Picker Logic ---
// --- Premium Color Picker Logic (Refactored for Multiple Instances) ---
const PRESETS = ['#ff4d4d', '#ff9f43', '#feca57', '#ffda79', '#48dbfb', '#1dd1a1', '#54a0ff', '#5f27cd', '#ff78b7', '#a29bfe', '#dfe6e9', '#636e72', '#2c2c2c', '#3c4043', '#000000', '#ffffff'];

class PremiumColorPicker {
    constructor(prefix, defaultColor) {
        this.prefix = prefix;
        this.h = 0; this.s = 0; this.v = 46; this.a = 1;

        // Elements
        this.elSaturation = document.getElementById(`${prefix}-saturation`);
        this.elPointer = document.getElementById(`${prefix}-pointer`);
        this.elHue = document.getElementById(`${prefix}-hue`);
        this.elAlpha = document.getElementById(`${prefix}-alpha`);
        this.elPreview = document.getElementById(`${prefix}-preview`);
        this.elHex = document.getElementById(prefix === 'f' ? 'folder-color-hex' : 'folder-icon-color-hex'); // ID naming inconsistency handling
        this.elSwatches = document.getElementById(`${prefix}-swatches`);

        if (this.elSaturation) this.init();
        this.setColor(defaultColor);
    }

    init() {
        // Saturation/Value Area
        this.elSaturation.onmousedown = (e) => {
            this.handleSaturation(e);
            const move = (ev) => this.handleSaturation(ev);
            const up = () => { window.removeEventListener('mousemove', move); window.removeEventListener('mouseup', up); };
            window.addEventListener('mousemove', move);
            window.addEventListener('mouseup', up);
        };

        // Sliders
        if (this.elHue) this.elHue.oninput = (e) => { this.h = e.target.value; this.updateUI(); };
        if (this.elAlpha) this.elAlpha.oninput = (e) => { this.a = e.target.value / 100; this.updateUI(); };

        // Hex Input
        if (this.elHex) {
            this.elHex.oninput = (e) => {
                const val = e.target.value;
                if (/^#([A-Fa-f0-9]{3}){1,2}$|^#([A-Fa-f0-9]{4}){1,2}$/.test(val)) {
                    this.setFromHex(val);
                }
            };
        }

        // Swatches
        if (this.elSwatches) {
            this.elSwatches.innerHTML = '';
            PRESETS.forEach(color => {
                const s = document.createElement('div');
                s.className = 'swatch';
                s.style.background = color;
                s.onclick = () => this.setColor(color);
                this.elSwatches.appendChild(s);
            });
        }
    }

    handleSaturation(e) {
        const rect = this.elSaturation.getBoundingClientRect();
        let x = ((e.clientX - rect.left) / rect.width) * 100;
        let y = ((e.clientY - rect.top) / rect.height) * 100;
        this.s = Math.min(100, Math.max(0, x));
        this.v = Math.min(100, Math.max(0, 100 - y));
        this.updateUI();
    }

    updateUI() {
        if (!this.elSaturation) return;
        const rgb = hsvToRgb(this.h, this.s, this.v);
        const hex = rgbToHex(rgb[0], rgb[1], rgb[2], this.a);

        this.elSaturation.style.backgroundColor = `hsl(${this.h}, 100%, 50%)`;
        this.elPreview.style.backgroundColor = hex;
        this.elHex.value = hex;
        this.elPointer.style.left = this.s + '%';
        this.elPointer.style.top = (100 - this.v) + '%';

        const opaqueHex = rgbToHex(rgb[0], rgb[1], rgb[2], 1);
        this.elAlpha.style.setProperty('color', opaqueHex);
    }

    setColor(color) {
        this.setFromHex(color || '#6b6e74');
    }

    setFromHex(hex) {
        const hsv = hexToHsv(hex);
        this.h = hsv.h; this.s = hsv.s; this.v = hsv.v; this.a = hsv.a;
        if (this.elHue) this.elHue.value = this.h;
        if (this.elAlpha) this.elAlpha.value = this.a * 100;
        this.updateUI();
    }

    getHex() {
        const rgb = hsvToRgb(this.h, this.s, this.v);
        return rgbToHex(rgb[0], rgb[1], rgb[2], this.a);
    }
}

// Helpers
function hsvToRgb(h, s, v) {
    h /= 360; s /= 100; v /= 100;
    let r, g, b, i = Math.floor(h * 6), f = h * 6 - i, p = v * (1 - s), q = v * (1 - f * s), t = v * (1 - (1 - f) * s);
    switch (i % 6) {
        case 0: r = v; g = t; b = p; break; case 1: r = q; g = v; b = p; break; case 2: r = p; g = v; b = t; break;
        case 3: r = p; g = q; b = v; break; case 4: r = t; g = p; b = v; break; case 5: r = v; g = p; b = q; break;
    }
    return [Math.round(r * 255), Math.round(g * 255), Math.round(b * 255)];
}

function hexToHsv(hex) {
    if (!hex) return { h: 0, s: 0, v: 46, a: 1 };
    let r = 0, g = 0, b = 0, a = 1;
    if (hex.startsWith('#')) {
        if (hex.length === 4) {
            r = parseInt(hex[1] + hex[1], 16); g = parseInt(hex[2] + hex[2], 16); b = parseInt(hex[3] + hex[3], 16);
        } else if (hex.length >= 7) {
            r = parseInt(hex.substring(1, 3), 16); g = parseInt(hex.substring(3, 5), 16); b = parseInt(hex.substring(5, 7), 16);
            if (hex.length === 9) a = parseInt(hex.substring(7, 9), 16) / 255;
        }
    }
    r /= 255; g /= 255; b /= 255;
    let max = Math.max(r, g, b), min = Math.min(r, g, b), d = max - min, h, s = (max === 0 ? 0 : d / max), v = max;
    if (max === min) h = 0;
    else {
        switch (max) {
            case r: h = (g - b) / d + (g < b ? 6 : 0); break;
            case g: h = (b - r) / d + 2; break;
            case b: h = (r - g) / d + 4; break;
        }
        h /= 6;
    }
    return { h: h * 360, s: s * 100, v: v * 100, a };
}

function rgbToHex(r, g, b, a = 1) {
    const componentToHex = (c) => c.toString(16).padStart(2, '0');
    let hex = "#" + componentToHex(r) + componentToHex(g) + componentToHex(b);
    if (a < 1) hex += componentToHex(Math.round(a * 255));
    return hex;
}

// Initialize Pickers
const bgPicker = new PremiumColorPicker('f', '#6b6e74');
// Note: We'll initialize the icon picker below once loaded, or if script runs at end body, it's fine.
const iconPicker = new PremiumColorPicker('fi', '#202124');

// --- End Picker Logic ---

function init() {
    chrome.storage.sync.get(['shortcuts', 'settings'], (res) => {
        const settings = res.settings || defaultSettings;
        applySettings(settings); // Always apply color settings
        document.body.classList.add('ready');

        if (!res.shortcuts || res.shortcuts.length === 0) {
            // First run: Save defaults to storage so they are editable
            chrome.storage.sync.set({ shortcuts: defaultShortcuts }, () => {
                renderShortcuts(defaultShortcuts);
            });
        } else {
            renderShortcuts(res.shortcuts);
        }
    });
}

function renderShortcuts(shortcuts, container = shortcutsContainer, isInsideFolder = false) {
    if (!container) return;
    container.innerHTML = '';
    shortcuts.forEach((item, index) => {
        const isFolder = item.type === 'folder';
        const itemEl = document.createElement(isFolder ? 'div' : 'a');

        itemEl.className = `shortcut-item ${isFolder ? 'folder ' + (item.shape || 'circle') : ''} ${item.locked ? 'is-locked' : ''}`;
        itemEl.draggable = !item.locked;

        if (isFolder) {
            itemEl.style.setProperty('--folder-color', item.color || '#8ab4f8');
            itemEl.style.setProperty('--folder-icon-color', item.iconColor || '#202124');
        } else {
            const fullUrl = item.url.startsWith('http') ? item.url : 'https://' + item.url;
            itemEl.href = fullUrl;
        }

        let iconHtml = '';
        if (isFolder) {
            iconHtml = `<div class="icon-wrapper"><svg viewBox="0 0 24 24"><path d="M10 4H4c-1.1 0-1.99.9-1.99 2L2 18c0 1.1.9 2 2 2h16c1.1 0 2-.9 2-2V8c0-1.1-.9-2-2-2h-8l-2-2z"/></svg></div>`;
        } else {
            const globeWithColor = GLOBE_SVG.replace('<svg ', `<svg fill="#778da9" `);
            const globeDataUri = `data:image/svg+xml;base64,${btoa(unescape(encodeURIComponent(globeWithColor)))}`;

            if (item.url && item.url.includes('.')) {
                try {
                    const urlStr = item.url.startsWith('http') ? item.url : 'https://' + item.url;

                    if (urlStr.includes('banana.ovo.re')) {
                        // استخدام أيقونة النجمة الأصلية الخاصة بالموقع مباشرة
                        const starIcon = `<svg style="width:24px; height:24px;" viewBox="0 0 24 24" fill="none" stroke="#6366f1" stroke-width="2" stroke-linecap="round" stroke-linejoin="round"><path d="M12 3l-1.9 5.8-5.8 1.9 5.8 1.9 1.9 5.8 1.9-5.8 5.8-1.9-5.8-1.9z"/></svg>`;
                        iconHtml = `<div class="icon-wrapper">${starIcon}</div>`;
                    } else {
                        const googleV2 = `https://t2.gstatic.com/faviconV2?client=SOCIAL&type=FAVICON&fallback_opts=TYPE,SIZE,URL&url=${encodeURIComponent(urlStr)}&size=128`;
                        iconHtml = `<div class="icon-wrapper">
                            <img src="${googleV2}" onerror="this.onerror=null; this.src='${globeDataUri}';" alt="">
                        </div>`;
                    }
                } catch (e) {
                    iconHtml = `<div class="icon-wrapper">${globeWithColor}</div>`;
                }
            } else {
                iconHtml = `<div class="icon-wrapper">${globeWithColor}</div>`;
            }
        }

        itemEl.innerHTML = `
            <div class="lock-indicator"><svg viewBox="0 0 24 24"><path d="M12 2C9.243 2 7 4.243 7 7v3H6c-1.103 0-2 .897-2 2v8c0 1.103.897 2 2 2h12c1.103 0 2-.897 2-2v-8c0-1.103-.897-2-2-2h-1V7c0-2.757-2.243-5-5-5zm3 8H9V7c0-1.654 1.346-3 3-3s3 1.346 3 3v3z"/></svg></div>
            ${iconHtml}
            <div class="shortcut-name">${item.name || 'Unnamed'}</div>
            <div class="more-btn" data-index="${index}"><svg viewBox="0 0 24 24"><path d="M12 8c1.1 0 2-.9 2-2s-.9-2-2-2-2 .9-2 2 .9 2 2 2zm0 2c-1.1 0-2 .9-2 2s.9 2 2 2 2-.9 2-2-.9-2-2-2zm0 6c-1.1 0-2 .9-2 2s.9 2 2 2 2-.9 2-2-.9-2-2-2z"/></svg></div>
        `;

        itemEl.ondragstart = (e) => {
            draggedIndex = index;
            draggedFromInside = isInsideFolder;
            isSuccessfulMove = false;
            itemEl.classList.add('dragging');
        };

        itemEl.ondragend = () => {
            itemEl.classList.remove('dragging');
            document.querySelectorAll('.drag-hover').forEach(el => el.classList.remove('drag-hover'));
        };

        itemEl.ondragover = (e) => {
            e.preventDefault();
            if (draggedIndex !== index || draggedFromInside !== isInsideFolder) {
                itemEl.classList.add('drag-hover');
            }
        };

        itemEl.ondragleave = () => itemEl.classList.remove('drag-hover');

        itemEl.ondrop = (e) => {
            e.preventDefault();
            e.stopPropagation();
            handleDropOnItem(index, isInsideFolder, item.type === 'folder');
        };

        itemEl.onclick = (e) => {
            if (e.target.closest('.more-btn')) {
                e.preventDefault();
                e.stopPropagation();
                e.stopPropagation();
                if (isFolder) openEditFolderModal(index, isInsideFolder);
                else openEditModal(index, isInsideFolder);
                return;
            }

            if (isFolder) {
                e.preventDefault();
                openFolder(index);
            } else {
                e.preventDefault();
                const url = item.url.startsWith('http') ? item.url : 'https://' + item.url;
                window.location.href = url;
            }
        };

        container.appendChild(itemEl);
    });
}

function handleDropOnItem(targetIndex, isTargetInside, isTargetFolder) {
    isSuccessfulMove = true;
    chrome.storage.sync.get(['shortcuts'], (res) => {
        let shortcuts = res.shortcuts || [];

        let targetFolder = (isTargetFolder && !isTargetInside) ? shortcuts[targetIndex] : null;
        let itemToMove;

        if (draggedFromInside && currentFolderIndex > -1) {
            itemToMove = shortcuts[currentFolderIndex].items.splice(draggedIndex, 1)[0];
        } else {
            itemToMove = shortcuts.splice(draggedIndex, 1)[0];
        }

        if (!itemToMove) return;

        if (targetFolder) {
            // Case 1: Dropping ONTO a folder -> Move item INSIDE
            // Restriction: Can't move a folder inside a folder (keep 1 level deep)
            if (itemToMove.type === 'folder') {
                // If trying to move folder into folder, just reorder instead
                shortcuts.splice(targetIndex, 0, itemToMove);
            } else {
                if (!targetFolder.items) targetFolder.items = [];
                targetFolder.items.push(itemToMove);
            }
        } else {
            // Case 2: Reordering items (Swap/Insert)
            let destList = (isTargetInside && currentFolderIndex > -1) ? shortcuts[currentFolderIndex].items : shortcuts;
            destList.splice(targetIndex, 0, itemToMove);
        }

        saveAndSync(shortcuts);
    });
}

function handleDropOutside(e) {
    if (draggedIndex === -1 || !draggedFromInside) return;
    isSuccessfulMove = true;
    chrome.storage.sync.get(['shortcuts'], (res) => {
        let shortcuts = res.shortcuts || [];
        const item = shortcuts[currentFolderIndex].items.splice(draggedIndex, 1)[0];
        if (item) shortcuts.push(item);
        saveAndSync(shortcuts);
    });
}

function saveAndSync(shortcuts) {
    chrome.storage.sync.set({ shortcuts }, () => {
        renderShortcuts(shortcuts);
        if (currentFolderIndex > -1 && shortcuts[currentFolderIndex]) {
            renderShortcuts(shortcuts[currentFolderIndex].items, folderShortcutsGrid, true);
        }
    });
}

if (folderViewModal) {
    folderViewModal.ondragover = (e) => { if (draggedFromInside) { e.preventDefault(); folderViewModal.classList.add('drop-out-hover'); } };
    folderViewModal.ondragleave = () => folderViewModal.classList.remove('drop-out-hover');
    folderViewModal.ondrop = (e) => { folderViewModal.classList.remove('drop-out-hover'); handleDropOutside(e); };
}

function openFolder(index) {
    chrome.storage.sync.get(['shortcuts'], (res) => {
        currentFolderIndex = index;
        const folder = res.shortcuts[index];
        if (folderViewTitle) folderViewTitle.innerText = folder.name;
        renderShortcuts(folder.items || [], folderShortcutsGrid, true);
        if (folderViewModal) folderViewModal.classList.add('active');
    });
}

function closeFolder() {
    if (folderViewModal) folderViewModal.classList.remove('active');
    currentFolderIndex = -1;
    init();
}

function populateFolderSelect(s) {
    if (!folderSelect) return;
    folderSelect.innerHTML = '<option value="home">Home</option>';
    s.forEach((folder, fIdx) => {
        if (folder.type === 'folder') {
            const opt = document.createElement('option');
            opt.value = fIdx;
            opt.textContent = folder.name;
            folderSelect.appendChild(opt);
        }
    });
}

function openModal() {
    chrome.storage.sync.get(['shortcuts'], (res) => {
        const s = res.shortcuts || [];
        populateFolderSelect(s);
        if (folderSelect) folderSelect.value = (currentFolderIndex > -1) ? currentFolderIndex : 'home';
        if (folderSelectorGroup) folderSelectorGroup.style.display = 'block';
        if (shortcutLock) shortcutLock.checked = false;
        if (modal) modal.classList.add('active');
        if (nameInput) nameInput.focus();
    });
}

function closeModal() {
    if (modal) modal.classList.remove('active');
    if (nameInput) nameInput.value = '';
    if (urlInput) urlInput.value = '';
    editingIndex = -1;
    const b = document.getElementById('modal-delete-btn');
    if (b) b.remove();
}

function openEditModal(index, isInside) {
    chrome.storage.sync.get(['shortcuts'], (res) => {
        const s = res.shortcuts || [];
        const item = isInside ? s[currentFolderIndex].items[index] : s[index];
        if (nameInput) nameInput.value = item.name;
        if (urlInput) urlInput.value = item.url;
        editingIndex = index;
        const title = document.querySelector('#modal h2');
        if (title) title.innerText = 'Edit Shortcut';

        populateFolderSelect(s);
        if (folderSelect) folderSelect.value = isInside ? currentFolderIndex : 'home';
        if (folderSelectorGroup) folderSelectorGroup.style.display = 'block';
        if (shortcutLock) shortcutLock.checked = !!item.locked;

        if (!document.getElementById('modal-delete-btn')) {
            const b = document.createElement('button');
            b.id = 'modal-delete-btn';
            b.className = 'btn secondary';
            b.style.marginRight = 'auto';
            b.innerText = 'Remove';
            b.onclick = () => { deleteShortcut(index, isInside); closeModal(); };
            const actions = document.querySelector('#modal .modal-actions');
            if (actions) actions.prepend(b);
        }
        if (modal) modal.classList.add('active');
    });
}

function saveShortcut() {
    const name = nameInput ? nameInput.value.trim() : '';
    const url = urlInput ? urlInput.value.trim() : '';

    // Safety check: if empty, return to prevent saving bad data.
    if (!name || !url) {
        if (nameInput && !name) nameInput.focus();
        else if (urlInput) urlInput.focus();
        return;
    }

    chrome.storage.sync.get(['shortcuts'], (res) => {
        let s = res.shortcuts || [];
        const isCurrentlyInside = currentFolderIndex > -1;
        const targetVal = folderSelect ? folderSelect.value : 'home';
        const targetFolderIdx = targetVal === 'home' ? -1 : parseInt(targetVal);

        const locked = shortcutLock ? shortcutLock.checked : false;
        let itemData = { name, url, type: 'shortcut', locked };

        if (editingIndex > -1) {
            // EDITING existing shortcut
            let sourceList = isCurrentlyInside ? s[currentFolderIndex].items : s;
            const sourceLocIdx = isCurrentlyInside ? currentFolderIndex : -1;

            if (sourceLocIdx === targetFolderIdx) {
                // SAME LOCATION -> Update
                if (sourceList[editingIndex]) {
                    sourceList[editingIndex] = { ...sourceList[editingIndex], name, url, locked };
                }
            } else {
                // DIFFERENT LOCATION -> Move
                // CRITICAL FIX: Grab reference to target folder object BEFORE splicing source list.
                // If we splice root list first, indices of subsequent folders shift, causing data loss.
                let targetFolderRef = null;
                if (targetFolderIdx > -1) {
                    targetFolderRef = s[targetFolderIdx];
                }

                const oldItem = sourceList[editingIndex];
                itemData = { ...oldItem, name, url, locked };
                sourceList.splice(editingIndex, 1);

                if (targetFolderRef) {
                    if (!targetFolderRef.items) targetFolderRef.items = [];
                    targetFolderRef.items.push(itemData);
                } else {
                    // Moving to Root (Home)
                    s.push(itemData);
                }
            }
        } else {
            // NEW shortcut
            if (targetFolderIdx > -1) {
                if (s[targetFolderIdx]) {
                    if (!s[targetFolderIdx].items) s[targetFolderIdx].items = [];
                    s[targetFolderIdx].items.push(itemData);
                }
            } else {
                s.push(itemData);
            }
        }

        chrome.storage.sync.set({ shortcuts: s }, () => {
            // Re-render
            if (currentFolderIndex > -1) {
                if (s[currentFolderIndex] && s[currentFolderIndex].type === 'folder') {
                    renderShortcuts(s[currentFolderIndex].items, folderShortcutsGrid, true);
                } else {
                    closeFolder();
                }
            }
            renderShortcuts(s);
            closeModal();
        });
    });
}

function autoFillShortcutName() {
    if (editingIndex > -1) return; // لا تقم بالملء التلقائي عند تعديل اختصار موجود

    // إذا كان المستخدم قد كتب اسماً بالفعل، لا تقم بتبديله
    if (nameInput && nameInput.value.trim() !== '') return;

    const url = urlInput.value.trim();
    if (!url || !url.includes('.')) return;

    try {
        let domain = url;
        if (!domain.startsWith('http')) domain = 'https://' + domain;

        const urlObj = new URL(domain);
        let hostname = urlObj.hostname;

        // إزالة www. وغيرها
        hostname = hostname.replace(/^www\./, '');

        let parts = hostname.split('.');
        let name = parts[0];

        // معالجة النطاقات مثل co.uk
        if (name.length <= 2 && parts.length > 2) name = parts[1];

        if (name) {
            // تكبير أول حرف
            name = name.charAt(0).toUpperCase() + name.slice(1);
            if (nameInput) nameInput.value = name;
        }
    } catch (e) {
        // ليس رابطاً صالحاً بعد
    }
}

function handleSaveConfirm() {
    if (editingIndex > -1 || editingFolderIndex > -1) {
        showConfirm('Save Changes', 'Are you sure you want to save these modifications?', () => {
            if (editingFolderIndex > -1) saveFolder();
            else saveShortcut();
        });
    } else {
        if (editingFolderIndex === -1 && folderModal.classList.contains('active')) saveFolder();
        else saveShortcut();
    }
}

function deleteShortcut(index, isInside) {
    showConfirm('Delete Shortcut', 'Are you sure you want to remove this shortcut?', () => {
        chrome.storage.sync.get(['shortcuts'], (res) => {
            let s = res.shortcuts || [];
            if (isInside) s[currentFolderIndex].items.splice(index, 1); else s.splice(index, 1);
            chrome.storage.sync.set({ shortcuts: s }, () => {
                if (isInside) renderShortcuts(s[currentFolderIndex].items, folderShortcutsGrid, true);
                renderShortcuts(s);
            });
        });
    });
}

function openEditFolderModal(index, isInside = false) {
    chrome.storage.sync.get(['shortcuts'], (res) => {
        let s = res.shortcuts || [];
        const f = isInside ? s[currentFolderIndex].items[index] : s[index];

        if (folderNameInput) folderNameInput.value = f.name;
        if (folderShapeSelect) folderShapeSelect.value = f.shape || 'circle';


        bgPicker.setColor(f.color || '#6b6e74');
        iconPicker.setColor(f.iconColor || '#202124');
        if (folderLock) folderLock.checked = !!f.locked;


        editingFolderIndex = index;
        editingFolderIsNested = isInside;

        if (folderModalTitle) folderModalTitle.innerText = 'Edit Folder';
        if (!document.getElementById('folder-delete-btn')) {
            const b = document.createElement('button');
            b.id = 'folder-delete-btn';
            b.className = 'btn secondary';
            b.style.marginRight = 'auto';
            b.innerText = 'Remove';
            b.onclick = () => { deleteFolder(index, isInside); closeFolderModal(); };
            const actions = document.getElementById('folder-modal-actions');
            if (actions) actions.prepend(b);
        }
        if (folderModal) folderModal.classList.add('active');
    });
}

function saveFolder() {
    const name = folderNameInput ? folderNameInput.value.trim() : '';
    if (!name) return;
    chrome.storage.sync.get(['shortcuts'], (res) => {
        let s = res.shortcuts || [];
        const color = bgPicker.getHex();
        const iconColor = iconPicker.getHex();
        const shape = folderShapeSelect ? folderShapeSelect.value : 'circle';
        const locked = folderLock ? folderLock.checked : false;
        // Note: We don't change 'items' if editing, or init empty if new.
        // But for update we just merge props.

        if (editingFolderIndex > -1) {
            // Editing existing folder
            if (editingFolderIsNested && currentFolderIndex > -1) {
                // Nested folder update
                const target = s[currentFolderIndex].items[editingFolderIndex];
                s[currentFolderIndex].items[editingFolderIndex] = { ...target, name, color, iconColor, shape, locked };
            } else {
                // Root folder update
                const target = s[editingFolderIndex];
                s[editingFolderIndex] = { ...target, name, color, iconColor, shape, locked };
            }
        } else {
            // Creating new folder (Always at root currently, unless we decide to support adding nested folders via button)
            // For now, "Add Folder" button is main UI, so it adds to root. 
            // Only Drag & Drop creates nested structures effectively, or if we supported "Add Folder" inside folder view (which we might not have UI for).
            // But let's assume root for new folders.
            s.push({ name, color, iconColor, shape, type: 'folder', items: [], locked });
        }

        chrome.storage.sync.set({ shortcuts: s }, () => {
            if (currentFolderIndex > -1 && editingFolderIsNested) {
                renderShortcuts(s[currentFolderIndex].items, folderShortcutsGrid, true);
            } else {
                renderShortcuts(s);
            }
            closeFolderModal();
        });
    });
}

function deleteFolder(index, isInside = false) {
    showConfirm('Delete Folder', 'Are you sure you want to delete this folder and all its contents?', () => {
        chrome.storage.sync.get(['shortcuts'], (res) => {
            let s = res.shortcuts || [];
            if (isInside && currentFolderIndex > -1) {
                s[currentFolderIndex].items.splice(index, 1);
            } else {
                s.splice(index, 1);
            }

            chrome.storage.sync.set({ shortcuts: s }, () => {
                if (isInside && currentFolderIndex > -1) {
                    renderShortcuts(s[currentFolderIndex].items, folderShortcutsGrid, true);
                } else {
                    renderShortcuts(s);
                }
                // If we deleted the folder we are currently viewing (not possible via logic usually, but akin to closeFolder)
                if (!isInside && currentFolderIndex === index) closeFolder();
            });
        });
    });
}

function closeFolderModal() {
    if (folderModal) folderModal.classList.remove('active');
    if (folderNameInput) folderNameInput.value = '';
    if (folderLock) folderLock.checked = false;

    // Reset pickers to defaults
    bgPicker.setColor('#6b6e74');
    iconPicker.setColor('#202124');

    editingFolderIndex = -1;
    editingFolderIsNested = false;
    if (folderModalTitle) folderModalTitle.innerText = 'Add Folder';
    const b = document.getElementById('folder-delete-btn');
    if (b) b.remove();
}

function applySettings(s) {
    document.documentElement.style.setProperty('--columns-count', s.columnsCount);
    document.documentElement.style.setProperty('--bg-color', s.bgColor);
    document.documentElement.style.setProperty('--font-size', (s.fontSize || 13) + 'px');

    // Ensure background overlay is reset/hidden or just shows color if we use it for that
    const bgOverlay = document.querySelector('.background-overlay');
    if (bgOverlay) {
        bgOverlay.style.backgroundImage = 'none';
        bgOverlay.style.backgroundColor = s.bgColor; // Use overlay for color to fit consistent design if needed, or body
    }
    // Also set body background just in case
    document.body.style.backgroundColor = s.bgColor;
    document.documentElement.style.backgroundColor = s.bgColor;


    if (shortcutsContainer) {
        shortcutsContainer.classList.remove('shape-circle', 'shape-square', 'shape-rounded');
        shortcutsContainer.classList.add(`shape-${s.iconShape || 'circle'}`);
    }

    if (columnsSlider) { columnsSlider.value = s.columnsCount; if (columnsCountVal) columnsCountVal.innerText = s.columnsCount; }
    if (fontSizeSlider) { fontSizeSlider.value = s.fontSize || 13; if (fontSizeVal) fontSizeVal.innerText = (s.fontSize || 13) + 'px'; }
    if (iconShapeSelector) iconShapeSelector.value = s.iconShape || 'circle';
    if (bgColorInput) bgColorInput.value = s.bgColor;
    if (bgHexInput) bgHexInput.value = s.bgColor;
}

function updateSettings() {
    const s = {
        columnsCount: columnsSlider ? columnsSlider.value : 5,
        fontSize: fontSizeSlider ? fontSizeSlider.value : 13,
        iconShape: iconShapeSelector ? iconShapeSelector.value : 'circle',
        bgColor: bgColorInput ? bgColorInput.value : '#383b43'
    };
    applySettings(s);
    chrome.storage.sync.set({ settings: s });
}

// Global Listeners
if (addBtn) addBtn.onclick = openModal;
if (cancelBtn) cancelBtn.onclick = closeModal;
if (saveBtn) saveBtn.onclick = handleSaveConfirm;
if (addFolderBtn) addFolderBtn.onclick = () => { if (folderModal) folderModal.classList.add('active'); if (folderNameInput) folderNameInput.focus(); };
if (folderCancelBtn) folderCancelBtn.onclick = closeFolderModal;
if (folderSaveBtn) folderSaveBtn.onclick = handleSaveConfirm;
if (folderViewBack) folderViewBack.onclick = closeFolder;
if (addShortcutInFolderBtn) addShortcutInFolderBtn.onclick = openModal;
if (folderEditBtn) folderEditBtn.onclick = () => { if (currentFolderIndex > -1) openEditFolderModal(currentFolderIndex); };
if (settingsBtn) settingsBtn.onclick = () => { if (settingsModal) settingsModal.classList.add('active'); };
if (closeSettingsBtn) closeSettingsBtn.onclick = () => { if (settingsModal) settingsModal.classList.remove('active'); };
if (columnsSlider) columnsSlider.oninput = updateSettings;
if (fontSizeSlider) fontSizeSlider.oninput = updateSettings;
if (iconShapeSelector) iconShapeSelector.onchange = updateSettings;
if (bgColorInput) bgColorInput.oninput = updateSettings;
if (urlInput) urlInput.oninput = autoFillShortcutName;
if (exportBtn) exportBtn.onclick = () => chrome.storage.sync.get(['shortcuts', 'settings'], (res) => {
    const shortcuts = res.shortcuts || [];
    const settings = res.settings || defaultSettings;

    const countShortcuts = (items) => {
        let total = 0;
        items.forEach(item => {
            if (item.type === 'folder' && item.items) {
                total += countShortcuts(item.items);
            } else if (item.type !== 'folder') {
                total += 1;
            }
        });
        return total;
    };

    const total = countShortcuts(shortcuts);

    const backupData = {
        version: "2.0",
        shortcuts: shortcuts,
        settings: settings
    };

    const blob = new Blob([JSON.stringify(backupData, null, 2)], { type: 'application/json' });
    const url = URL.createObjectURL(blob);
    const a = document.createElement('a');
    a.href = url; a.download = `full_backup_${total}_shortcuts.json`; a.click();
});
if (importBtn) importBtn.onclick = () => { if (importFileInput) importFileInput.click(); };
if (importFileInput) {
    importFileInput.onchange = (e) => {
        const f = e.target.files[0];
        if (f) {
            const r = new FileReader();
            r.onload = (ev) => {
                try {
                    const data = JSON.parse(ev.target.result);

                    // تحكم ذكي: التحقق هل الملف هو النظام الجديد أم القديم (مجرد روابط)
                    let shortcutsToImport = [];
                    let settingsToImport = null;

                    if (data.shortcuts && Array.isArray(data.shortcuts)) {
                        // نظام الداتا الجديد الشامل
                        shortcutsToImport = data.shortcuts;
                        settingsToImport = data.settings;
                    } else if (Array.isArray(data)) {
                        // دعم النظام القديم (الروابط فقط) لضمان عدم ضياع بياناتك القديمة
                        shortcutsToImport = data;
                    }

                    chrome.storage.sync.set({
                        shortcuts: shortcutsToImport,
                        ...(settingsToImport && { settings: settingsToImport })
                    }, () => {
                        renderShortcuts(shortcutsToImport);
                        if (settingsToImport) applySettings(settingsToImport);
                        importFileInput.value = '';
                        showConfirm('Success', 'Everything has been restored successfully!', () => { });
                    });
                } catch (err) {
                    showConfirm('Error', 'Invalid backup file.', () => { });
                }
            };
            r.readAsText(f);
        }
    };
}

window.onclick = (e) => {
    if (e.target === modal) closeModal();
    if (e.target === folderModal) closeFolderModal();
    if (e.target === folderViewModal) closeFolder();
    if (e.target === settingsModal) settingsModal.classList.remove('active');
    if (e.target === confirmModal) confirmModal.classList.remove('active');
};
window.onkeydown = (e) => {
    if (e.key === 'Escape') {
        closeModal();
        closeFolderModal();
        closeFolder();
        if (settingsModal) settingsModal.classList.remove('active');
        if (confirmModal) confirmModal.classList.remove('active');
    }
};

// --- User Onboarding Tutorial ---
const tutorialOverlay = document.getElementById('tutorial-overlay');
const tutorialTooltip = document.getElementById('tutorial-tooltip');
const tutorialStepElem = document.getElementById('tutorial-step');
const tutorialTitleElem = document.getElementById('tutorial-title');
const tutorialTextElem = document.getElementById('tutorial-text');
const tutorialNextBtn = document.getElementById('tutorial-next');
const tutorialSkipBtn = document.getElementById('tutorial-skip');

const tutorialSteps = [
    {
        targetId: 'settings-btn',
        title: 'Customize Your Board',
        text: 'Access settings here to change columns, background images, colors, and more.',
        position: 'bottom-right'
    },
    {
        targetId: 'add-shortcut-btn',
        title: 'Add Shortcuts',
        text: 'Click here to add your favorite websites to the dashboard.',
        position: 'bottom-right'
    },
    {
        targetId: 'add-folder-btn',
        title: 'Create Folders',
        text: 'Organize your shortcuts into folders for a cleaner look.',
        position: 'bottom-right'
    },
    {
        targetId: 'search-box-form',
        title: 'Quick Search',
        text: 'Search Google directly from here. It is fast and focused.',
        position: 'bottom'
    },
    {
        targetId: null, // Center
        title: 'Drag & Drop',
        text: 'You can rearrange icons or move them into folders simply by dragging them.',
        position: 'center'
    }
];

let currentStep = 0;

function startTutorial() {
    if (!tutorialOverlay || !tutorialTooltip) return;
    currentStep = 0;
    tutorialOverlay.classList.add('active');
    tutorialTooltip.classList.add('active');
    showStep(currentStep);
}

function showStep(index) {
    if (index >= tutorialSteps.length) {
        endTutorial();
        return;
    }

    const step = tutorialSteps[index];

    // Update Text
    tutorialStepElem.innerText = `Step ${index + 1} of ${tutorialSteps.length}`;
    tutorialTitleElem.innerText = step.title;
    tutorialTextElem.innerText = step.text;
    tutorialNextBtn.innerText = (index === tutorialSteps.length - 1) ? 'Finish' : 'Next';

    // Remove old highlights
    document.querySelectorAll('.highlight-target').forEach(el => el.classList.remove('highlight-target'));

    // Position Tooltip
    let target = null;
    if (step.isElement) target = step.targetId;
    else if (step.targetId) target = document.getElementById(step.targetId);

    if (target) {
        target.classList.add('highlight-target');

        const rect = target.getBoundingClientRect();
        const tooltipRect = tutorialTooltip.getBoundingClientRect();

        let top, left;

        // Simple positioning logic
        if (step.position === 'bottom-right') {
            top = rect.bottom + 15;
            left = rect.left;
        } else if (step.position === 'bottom-left') {
            top = rect.bottom + 15;
            left = rect.right - tooltipRect.width;
        } else if (step.position === 'bottom') {
            top = rect.bottom + 15;
            left = rect.left + (rect.width / 2) - (tooltipRect.width / 2);
        } else if (step.position === 'top') {
            top = rect.top - tooltipRect.height - 15;
            left = rect.left + (rect.width / 2) - (tooltipRect.width / 2);
        }

        // Boundary checks (keep on screen)
        left = Math.max(10, Math.min(window.innerWidth - tooltipRect.width - 10, left));
        top = Math.max(10, Math.min(window.innerHeight - tooltipRect.height - 10, top));

        tutorialTooltip.style.top = `${top}px`;
        tutorialTooltip.style.left = `${left}px`;
        tutorialTooltip.style.transform = 'none';

    } else {
        // Center position
        tutorialTooltip.style.top = '50%';
        tutorialTooltip.style.left = '50%';
        tutorialTooltip.style.transform = 'translate(-50%, -50%)';
    }
}

function endTutorial() {
    tutorialOverlay.classList.remove('active');
    tutorialTooltip.classList.remove('active');
    document.querySelectorAll('.highlight-target').forEach(el => el.classList.remove('highlight-target'));

    // Mark as seen
    chrome.storage.local.set({ tutorialSeen: true });
}

if (tutorialNextBtn) tutorialNextBtn.onclick = () => showStep(++currentStep);
if (tutorialSkipBtn) tutorialSkipBtn.onclick = endTutorial;

// Check if first time
function checkTutorial() {
    chrome.storage.local.get(['tutorialSeen'], (res) => {
        if (!res.tutorialSeen) {
            // Small delay to ensure UI is ready
            setTimeout(startTutorial, 1000);
        }
    });
}

// Mobile Drawer Logic
function toggleDrawer(show) {
    if (mobileDrawer) mobileDrawer.classList.toggle('active', show);
    if (mobileDrawerOverlay) mobileDrawerOverlay.classList.toggle('active', show);
}

if (mobileMenuBtn) mobileMenuBtn.onclick = () => toggleDrawer(true);
if (closeDrawerBtn) closeDrawerBtn.onclick = () => toggleDrawer(false);
if (mobileDrawerOverlay) mobileDrawerOverlay.onclick = () => toggleDrawer(false);

if (drawerAddShortcut) drawerAddShortcut.onclick = () => { toggleDrawer(false); openModal(); };
if (drawerAddFolder) drawerAddFolder.onclick = () => {
    toggleDrawer(false);
    if (folderModal) folderModal.classList.add('active');
    if (folderNameInput) folderNameInput.focus();
};
if (drawerSettings) drawerSettings.onclick = () => { toggleDrawer(false); if (settingsModal) settingsModal.classList.add('active'); };

init();
// Add checkTutorial call after init
checkTutorial();
